<?php
// serve_survey_photo.php — Serve survey photos with disk caching
// PHOTO-016: Optimized with /tmp/survey_photo_cache/ for pre-cached thumbs and on-demand proxy
//
// Two modes:
//   ?key={hash}&size=thumb  → serve pre-cached thumbnail (written by getimagelisting1.php Step 5)
//   ?path={encoded_path}    → proxy from local server, cache on disk for 7 days

$cacheDir = '/tmp/survey_photo_cache';
$cacheTTL = 604800; // 7 days in seconds

// Ensure cache directory exists
if (!is_dir($cacheDir)) {
    @mkdir($cacheDir, 0777, true);
}

// Probabilistic garbage collection (~1% of requests)
// Deletes files older than TTL to prevent inode exhaustion
if (mt_rand(1, 100) === 1) {
    $gcFiles = @glob($cacheDir . '/*');
    if ($gcFiles) {
        $now = time();
        foreach ($gcFiles as $gcFile) {
            if (is_file($gcFile)) {
                $age = $now - filemtime($gcFile);
                if ($age > $cacheTTL) {
                    @unlink($gcFile);
                }
            }
        }
    }
}

// === MODE 1: Serve pre-cached thumbnail by key ===
$key = isset($_GET['key']) ? preg_replace('/[^a-f0-9]/', '', $_GET['key']) : '';
$size = isset($_GET['size']) ? $_GET['size'] : '';

if ($key !== '' && $size === 'thumb') {
    $cacheFile = $cacheDir . '/thumb_' . $key . '.webp';

    if (file_exists($cacheFile)) {
        $mtime = filemtime($cacheFile);
        if ($mtime !== false && (time() - $mtime) < $cacheTTL) {
            header('Content-Type: image/webp');
            header('Content-Length: ' . filesize($cacheFile));
            header('Cache-Control: public, max-age=86400');
            header('X-Cache: HIT');
            readfile($cacheFile);
            exit;
        }
        // Expired — delete stale cache
        @unlink($cacheFile);
    }

    // Cache miss for thumb — /tmp/ was cleared or server rebooted.
    // Fallback: if a ?path= was also provided, proxy it on-demand.
    // The mobile app's thumb_link URLs don't include path=, but the listing call
    // also stored the thumb path in the key. We can't reconstruct the path from just
    // the hash, so we return a transparent 1x1 placeholder to avoid a broken image.
    // The next listing call will re-populate the cache.
    //
    // If caller provides ?fallback_path=, use that to proxy on-demand.
    $fallbackPath = isset($_GET['fallback_path']) ? trim($_GET['fallback_path']) : '';
    if ($fallbackPath !== '') {
        // Proxy the thumbnail from local on-demand and re-cache it
        $localUrl = 'https://upload.aeihawaii.com/serve_photo.php?path=' . urlencode($fallbackPath);
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $localUrl);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $thumbBody = curl_exec($ch);
        $thumbHttp = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($thumbBody !== false && $thumbHttp === 200 && strlen($thumbBody) > 0) {
            // Re-cache for next time
            $tmpFile = $cacheDir . '/tmp_' . $key . '_' . getmypid();
            if (@file_put_contents($tmpFile, $thumbBody) !== false) {
                @rename($tmpFile, $cacheFile);
            } else {
                @unlink($tmpFile);
            }
            header('Content-Type: image/webp');
            header('Content-Length: ' . strlen($thumbBody));
            header('Cache-Control: public, max-age=86400');
            header('X-Cache: MISS-RECOVERED');
            echo $thumbBody;
            exit;
        }
    }

    // Last resort: return a 1x1 transparent WebP placeholder (67 bytes)
    // so the grid doesn't show broken images. Next listing call will re-cache.
    $placeholder = base64_decode('UklGRlYAAABXRUJQVlA4IEoAAADQAQCdASoBAAEAAkA4JZQCdAEO/hepgAAA/v5MOf/OsfjB//4H3v/Dz/6VP5PbEMf/jh///EL///+xBv//sn//1kf//WQAAAA==');
    header('Content-Type: image/webp');
    header('Content-Length: ' . strlen($placeholder));
    header('Cache-Control: no-cache');
    header('X-Cache: MISS-PLACEHOLDER');
    echo $placeholder;
    exit;
}

// === MODE 2: Proxy from local server with cache-on-write ===
$path = isset($_GET['path']) ? trim($_GET['path']) : '';
if ($path === '') {
    header('Content-Type: application/json');
    http_response_code(400);
    echo json_encode(array('error' => 'Missing path or key parameter'));
    exit;
}

// Determine cache key from path + size
$sizeParam = ($size === 'mid' || $size === 'thumb' || $size === 'full') ? $size : 'mid';
$cacheKey = md5($path . '|' . $sizeParam);
$cacheFile = $cacheDir . '/proxy_' . $cacheKey . '.dat';
$metaFile = $cacheDir . '/proxy_' . $cacheKey . '.meta';

// Check disk cache for proxied images
if (file_exists($cacheFile) && file_exists($metaFile)) {
    $mtime = filemtime($cacheFile);
    if ($mtime !== false && (time() - $mtime) < $cacheTTL) {
        $meta = @file_get_contents($metaFile);
        $contentType = ($meta !== false && $meta !== '') ? trim($meta) : 'image/webp';
        header('Content-Type: ' . $contentType);
        header('Content-Length: ' . filesize($cacheFile));
        header('Cache-Control: public, max-age=86400');
        header('X-Cache: HIT');
        readfile($cacheFile);
        exit;
    }
    // Expired
    @unlink($cacheFile);
    @unlink($metaFile);
}

// Cache miss — proxy from local server
// Disable output buffering
while (ob_get_level()) { ob_end_clean(); }

$localUrl = 'https://upload.aeihawaii.com/serve_photo.php?path=' . urlencode($path);

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $localUrl);
curl_setopt($ch, CURLOPT_TIMEOUT, 15);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

$responseBody = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
$curlError = curl_error($ch);
curl_close($ch);

if ($responseBody === false || $httpCode !== 200 || strlen($responseBody) === 0) {
    header('Content-Type: application/json');
    http_response_code(502);
    $msg = $curlError ? $curlError : 'Local server returned HTTP ' . $httpCode;
    echo json_encode(array('error' => 'Failed to fetch survey photo', 'detail' => $msg));
    exit;
}

// Write to cache atomically (tmp file → rename)
$tmpFile = $cacheDir . '/tmp_' . $cacheKey . '_' . getmypid();
$written = @file_put_contents($tmpFile, $responseBody);
if ($written !== false) {
    @rename($tmpFile, $cacheFile);
    @file_put_contents($metaFile, $contentType ? $contentType : 'image/webp');
} else {
    @unlink($tmpFile);
}

// Serve the response
if (!$contentType) {
    $contentType = 'image/webp';
}
header('Content-Type: ' . $contentType);
header('Content-Length: ' . strlen($responseBody));
header('Cache-Control: public, max-age=86400');
header('X-Cache: MISS');
header('X-Content-Type-Options: nosniff');
echo $responseBody;
exit;
