#!/usr/bin/env python3
"""
Cron fallback: find meter_files photos missing WebP derivatives and regenerate.

Queries meter_files for recent photos (file_type=99, webpfilename set) where
the thumbs/ or webp/ file is missing on disk. If staging/ original exists,
calls generate_thumbnails.generate() to recreate derivatives + archive copy.

Designed to catch silent failures from the nohup exec() pattern where
generate_thumbnails.py may crash without anyone noticing.

Run via cron every 30 minutes:
  */30 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/fix_missing_webp.py > /dev/null 2>&1

Requires: pymysql (already installed for process_retry_queue.py)
Python 3.6 compatible.

Author: AEI Photo System
Version: 1.0
Date: 2026-02-23
"""

import os
import sys
import fcntl
import time
import logging

SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'fix_missing_webp.log')
LOCKFILE_PATH = '/tmp/fix_missing_webp.lock'

# DB config (same creds as process_retry_queue.py / upload.php)
DB_HOST = 'localhost'
DB_USER = 'schedular'
DB_PASS = 'M1gif9!6'
DB_NAME = 'mandhdesign_schedular'

UPLOADS_DIR = '/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads'
STAGING_DIR = os.path.join(UPLOADS_DIR, 'staging')

# How far back to look for missing derivatives
LOOKBACK_DAYS = 7


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def get_db_connection():
    """Connect to the local MySQL database."""
    import pymysql
    return pymysql.connect(
        host=DB_HOST,
        user=DB_USER,
        password=DB_PASS,
        db=DB_NAME,
        charset='utf8mb4',
        cursorclass=pymysql.cursors.DictCursor,
    )


def find_missing_webp(cursor):
    """Query meter_files for recent photos missing WebP on disk."""
    sql = """
        SELECT id, unique_filename, webpfilename
        FROM meter_files
        WHERE file_type = 99
          AND webpfilename IS NOT NULL
          AND webpfilename != ''
          AND created > NOW() - INTERVAL %s DAY
        ORDER BY created DESC
    """
    cursor.execute(sql, (LOOKBACK_DAYS,))
    rows = cursor.fetchall()

    missing = []
    for row in rows:
        wfn = row['webpfilename']
        thumb_path = os.path.join(UPLOADS_DIR, 'thumbs', wfn)
        large_path = os.path.join(UPLOADS_DIR, 'webp', wfn)

        if os.path.isfile(thumb_path) and os.path.isfile(large_path):
            continue  # both exist, skip

        # Try to find staging source
        ufn = row['unique_filename']
        staging_path = os.path.join(STAGING_DIR, ufn)

        if not os.path.isfile(staging_path):
            logging.warning("MISSING id=%s webp=%s - staging source not found: %s",
                            row['id'], wfn, staging_path)
            continue

        missing.append({
            'id': row['id'],
            'unique_filename': ufn,
            'webpfilename': wfn,
            'staging_path': staging_path,
            'thumb_missing': not os.path.isfile(thumb_path),
            'large_missing': not os.path.isfile(large_path),
        })

    return missing


def regenerate(item):
    """Call generate_thumbnails.generate() to recreate derivatives + archive."""
    # Import from same directory
    sys.path.insert(0, SCRIPT_DIR)
    import generate_thumbnails

    logging.info("REGEN id=%s webp=%s source=%s (thumb_missing=%s large_missing=%s)",
                 item['id'], item['webpfilename'], item['staging_path'],
                 item['thumb_missing'], item['large_missing'])

    t0 = time.time()
    try:
        result = generate_thumbnails.generate(
            item['staging_path'],
            item['webpfilename'],
            UPLOADS_DIR,
        )
        elapsed = time.time() - t0
        created = [k for k, v in result.items() if v]
        logging.info("REGEN OK id=%s created=%s elapsed=%.1fs",
                     item['id'], ','.join(created) or 'none', elapsed)
        return True
    except Exception as e:
        elapsed = time.time() - t0
        logging.error("REGEN FAIL id=%s error=%s elapsed=%.1fs",
                      item['id'], str(e), elapsed)
        return False


def main():
    setup_logging()

    # File lock — prevent concurrent runs
    lockfile = open(LOCKFILE_PATH, 'w')
    try:
        fcntl.flock(lockfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        logging.info("Another instance running, exiting")
        return

    try:
        conn = get_db_connection()
        cursor = conn.cursor()

        missing = find_missing_webp(cursor)

        if not missing:
            logging.info("CHECK OK - no missing WebP derivatives (%d-day window)", LOOKBACK_DAYS)
            return

        logging.info("CHECK FOUND %d photos with missing WebP derivatives", len(missing))

        fixed = 0
        failed = 0
        for item in missing:
            if regenerate(item):
                fixed += 1
            else:
                failed += 1

        logging.info("DONE fixed=%d failed=%d total=%d", fixed, failed, len(missing))

    except Exception as e:
        logging.error("FATAL %s", str(e))
    finally:
        cursor.close()
        conn.close()
        fcntl.flock(lockfile, fcntl.LOCK_UN)
        lockfile.close()


if __name__ == '__main__':
    main()
