<?php
/**
 * delete_local_photo.php — Delete Dropbox-synced photo copies on local server
 *
 * Runs on upload.aeihawaii.com (aei-webserv2). Called by the production
 * delete_image.php after remote files are cleaned up.
 *
 * Looks up the local_photos table (Schedular DB) by job_id + unique_filename
 * to find the actual folder_path and filename on disk, then deletes:
 *   - {folder_path}/{filename}              — synced photo
 *   - {folder_path}/thumbs/{name}.webp      — local thumbnail
 *   - {folder_path}/thumbs/mid/{name}.webp  — local mid-size thumb
 *
 * Also removes the local_photos DB row after successful deletion.
 *
 * Parameters (GET):
 *   auth_token      — shared secret
 *   job_id          — remote job ID
 *   unique_filename — unique hash filename (e.g. 1f9b2c6b83a16e7f5a45979f46bf4e8a.jpeg)
 */

// Auth check
if (!isset($_GET['auth_token']) || $_GET['auth_token'] !== 'aei_local_delete_89806849') {
    header('HTTP/1.1 403 Forbidden');
    echo json_encode(array("status" => "error", "message" => "denied"));
    exit;
}

if (empty($_GET['job_id']) || empty($_GET['unique_filename'])) {
    header('HTTP/1.1 400 Bad Request');
    echo json_encode(array("status" => "error", "message" => "missing job_id or unique_filename"));
    exit;
}

$job_id = intval($_GET['job_id']);
$unique_filename = basename($_GET['unique_filename']);

// Connect to local Schedular DB
$conn = new mysqli('localhost', 'upload_user', 'P@55w02d778899', 'Schedular');
if ($conn->connect_error) {
    header('HTTP/1.1 500 Internal Server Error');
    echo json_encode(array("status" => "error", "message" => "db connect failed"));
    exit;
}

// Look up the photo in local_photos
$stmt = $conn->prepare(
    "SELECT id, filename, folder_path FROM local_photos
     WHERE remote_job_id = ? AND filename = ?
     LIMIT 1"
);
$stmt->bind_param("is", $job_id, $unique_filename);
$stmt->execute();
$result = $stmt->get_result();
$row = $result->fetch_assoc();
$stmt->close();

if (!$row || empty($row['folder_path'])) {
    $conn->close();
    echo json_encode(array("status" => "ok", "deleted" => 0, "message" => "not found in local_photos"));
    exit;
}

$folder = $row['folder_path'];
$filename = $row['filename'];
$photo_id = $row['id'];

// Path traversal protection: folder must be under /mnt/dropbox/
$resolvedFolder = realpath($folder);
if ($resolvedFolder === false || strpos($resolvedFolder, '/mnt/dropbox/') !== 0) {
    $conn->close();
    header('HTTP/1.1 403 Forbidden');
    echo json_encode(array("status" => "error", "message" => "invalid path"));
    exit;
}

$nameNoExt = pathinfo($filename, PATHINFO_FILENAME);

$targets = array(
    $resolvedFolder . '/' . $filename,
    $resolvedFolder . '/thumbs/' . $nameNoExt . '.webp',
    $resolvedFolder . '/thumbs/mid/' . $nameNoExt . '.webp',
);

$count = 0;
foreach ($targets as $t) {
    if (is_file($t)) {
        if (unlink($t)) { $count++; }
    }
}

// Remove the local_photos record
$stmt = $conn->prepare("DELETE FROM local_photos WHERE id = ?");
$stmt->bind_param("i", $photo_id);
$stmt->execute();
$stmt->close();

// Clean up empty job subfolder if no files remain
// folder_path is the job subfolder e.g. ".../Installation/Mitsuda, Kathi, SC-I, 2026-02-03/"
$folder_removed = false;
if (is_dir($resolvedFolder)) {
    // Check if only thumbs dirs (or nothing) remain
    $remaining = array_diff(scandir($resolvedFolder), array('.', '..', 'thumbs'));
    if (empty($remaining)) {
        // Remove thumbs subdirs first
        $thumbsDir = $resolvedFolder . '/thumbs';
        if (is_dir($thumbsDir)) {
            $midDir = $thumbsDir . '/mid';
            if (is_dir($midDir)) { @rmdir($midDir); }
            @rmdir($thumbsDir);
        }
        // Remove the job subfolder itself if now empty
        if (count(array_diff(scandir($resolvedFolder), array('.', '..'))) === 0) {
            $folder_removed = @rmdir($resolvedFolder);
        }
    }
}

$conn->close();

echo json_encode(array("status" => "ok", "deleted" => $count, "folder_removed" => $folder_removed));
