<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * WebP-Safe Thumbnail Helper (PHOTO-017)
 *
 * Replaces the original GD-only thumbnail helper with a priority chain:
 *   1. Pre-generated WebP thumb (from PHOTO-017 upload pipeline or backfill)
 *   2. Python on-demand WebP generation (handles JPEG + WebP input)
 *   3. GD fallback for JPEG/PNG sources (legacy safety net)
 *   4. Placeholder image if all else fails
 *
 * PHP 5.3's GD cannot decode WebP, so PHP never attempts to open WebP files.
 * Python/Pillow handles all WebP processing.
 *
 * Same function signature as original: thumbnail($src, $width, $height, $option, $grayscale)
 *
 * @package     AEI Photo System
 * @version     2.0 (PHOTO-017)
 * @date        2026-02-19
 */

if (!function_exists('thumbnail')) {
    function thumbnail($src, $width, $height, $option='crop', $grayscale=false) {

        // ============================================================
        // CONFIG
        // ============================================================
        $config = array();
        if (defined('ENVIRONMENT') AND is_file(APPPATH . 'config/' . ENVIRONMENT . '/thumbnail.php')) {
            include(APPPATH . 'config/' . ENVIRONMENT . '/thumbnail.php');
        } elseif (is_file(APPPATH . 'config/thumbnail.php')) {
            include(APPPATH . 'config/thumbnail.php');
        }

        $uploadsDir = '/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads';
        $baseUrl = isset($config['thumbnail_base_url']) ? $config['thumbnail_base_url'] : site_url() . 'uploads';
        $baseFolder = isset($config['thumbnail_base_folder']) ? $config['thumbnail_base_folder'] : './';
        $pythonBin = '/usr/local/bin/python3.6';
        $pythonScript = '/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/generate_thumb_ondemand.py';

        // Placeholder for broken images
        $placeholder = $baseUrl . '/placeholder.png';

        // ============================================================
        // RESOLVE SOURCE
        // ============================================================
        // Clean path (same as original)
        if ($src[0] == '/') $src = substr($src, 1);
        $src = str_replace('\\', '/', $src);

        $srcImage = $baseFolder . $src;

        // ============================================================
        // PRIORITY 1: Pre-generated WebP thumb (PHOTO-017)
        // ============================================================
        // Look up webpfilename from the DB connection in CI context.
        // We can derive it: if source is {unique_filename}, check for
        // a matching pre-generated thumb by looking at the DB or by
        // checking if a WebP thumb with matching dimensions exists.
        //
        // Strategy: check thumbs/ directory for a file named after
        // the webpfilename. Since we don't have webpfilename here,
        // we check the on-demand cache directory first.

        // Build the on-demand cache path (WebP)
        // Cache key: {filename_without_ext}_{mode[0]}{width}x{height}.webp
        $pinfo = pathinfo($src);
        $filename = isset($pinfo['filename']) ? $pinfo['filename'] : '';
        $cacheDir = $uploadsDir . '/thumbnails';
        $cacheName = $filename . '_' . $option[0] . $width . 'x' . $height . '.webp';
        $cachePath = $cacheDir . '/' . $cacheName;
        $cacheUrl = $baseUrl . '/thumbnails/' . rawurlencode($cacheName);

        // Fast path: cached WebP thumb already exists
        if (file_exists($cachePath)) {
            return $cacheUrl;
        }

        // ============================================================
        // PRIORITY 2: Legacy cached JPEG thumb (from old system)
        // ============================================================
        // Check if the original GD-generated JPEG thumb exists
        $ext = isset($pinfo['extension']) ? $pinfo['extension'] : 'jpg';
        $legacyCacheName = $filename . '_' . $option[0] . $width . 'x' . $height . '.' . $ext;

        // Match original cache_type behavior
        $cacheType = isset($config['thumbnail_cache_type']) ? $config['thumbnail_cache_type'] : 'subfolder';
        $legacyCachePath = '';
        $legacyCacheUrl = '';

        if ($cacheType == 'subfolder') {
            $subfolder = isset($config['thumbnail_subfolder_name']) ? $config['thumbnail_subfolder_name'] : 'thumbnails';
            $srcDir = dirname($srcImage);
            $legacyCachePath = $srcDir . '/' . $subfolder . '/' . $legacyCacheName;
            $legacyCacheUrl = $baseUrl . '/' . dirname($src) . '/' . $subfolder . '/' . $legacyCacheName;
        } elseif ($cacheType == 'same_path') {
            $srcDir = dirname($srcImage);
            $legacyCachePath = $srcDir . '/' . $legacyCacheName;
            $legacyCacheUrl = $baseUrl . '/' . dirname($src) . '/' . $legacyCacheName;
        }

        // If legacy cached JPEG exists and source hasn't changed, use it
        if ($legacyCachePath && file_exists($legacyCachePath)) {
            if (!file_exists($srcImage) || @filemtime($legacyCachePath) >= @filemtime($srcImage)) {
                return $legacyCacheUrl;
            }
        }

        // ============================================================
        // SOURCE FILE CHECK
        // ============================================================
        if (!file_exists($srcImage)) {
            return '<!-- File ' . $srcImage . ' does not exist -->';
        }

        // Determine if source is WebP (PHP 5.3 GD cannot handle WebP)
        $srcExt = strtolower(isset($pinfo['extension']) ? $pinfo['extension'] : '');
        $isWebP = ($srcExt === 'webp');

        // ============================================================
        // PRIORITY 3: Python on-demand generation (JPEG + WebP safe)
        // ============================================================
        // Python generates a WebP thumb via Pillow — works for any input format.
        // Uses lock file to prevent stampede (multiple requests for same missing thumb).

        if (file_exists($pythonBin) && file_exists($pythonScript)) {
            // Lock file to prevent stampede
            $lockPath = $cacheDir . '/.lock_' . $cacheName;
            $lockStale = 30; // seconds before lock is considered stale

            // Check for active lock (another request is generating this thumb)
            if (file_exists($lockPath) && (time() - @filemtime($lockPath)) < $lockStale) {
                // Another process is generating — fall through to GD or placeholder
            } else {
                // Ensure cache directory exists
                if (!is_dir($cacheDir)) {
                    @mkdir($cacheDir, 0777, true);
                }

                // Create lock file
                @file_put_contents($lockPath, getmypid());

                // Map option names to Python mode names
                $modeMap = array(
                    'crop' => 'crop',
                    'landscape' => 'landscape',
                    'portrait' => 'portrait',
                    'auto' => 'auto',
                    'exact' => 'exact'
                );
                $pyMode = isset($modeMap[$option]) ? $modeMap[$option] : 'auto';

                // Validate source filename (security: no shell injection)
                $srcBase = basename($srcImage);
                if (!preg_match('/^[A-Za-z0-9._\-]+$/', $srcBase)) {
                    // Filename has unusual characters — still safe with escapeshellarg
                    // but log it for awareness
                    log_message('debug', 'Thumbnail: unusual filename: ' . $srcBase);
                }

                // Build command — synchronous (we need the result now)
                $cmd = escapeshellarg($pythonBin)
                     . ' ' . escapeshellarg($pythonScript)
                     . ' ' . escapeshellarg($srcImage)
                     . ' ' . escapeshellarg($cachePath)
                     . ' ' . escapeshellarg((string)$width)
                     . ' ' . escapeshellarg((string)$height)
                     . ' ' . escapeshellarg($pyMode)
                     . ' 2>&1';

                $output = array();
                $returnCode = 0;
                exec($cmd, $output, $returnCode);

                // Remove lock
                @unlink($lockPath);

                // Check if Python succeeded
                if ($returnCode === 0 && file_exists($cachePath)) {
                    return $cacheUrl;
                }

                // Python failed — log and fall through
                $errorMsg = implode(' ', $output);
                log_message('error', 'Thumbnail Python failed (' . $returnCode . '): ' . $errorMsg . ' src=' . $srcImage);
            }
        }

        // ============================================================
        // PRIORITY 4: GD fallback (JPEG/PNG only — NOT WebP)
        // ============================================================
        if (!$isWebP) {
            // Use the original Resize class for JPEG/PNG/GIF
            require_once APPPATH . 'third_party/Resize.php';

            // Use original naming for GD output (JPEG format)
            $old_pinfo = pathinfo($src);
            $gdCacheName = $filename . '_' . $option[0] . $width . 'x' . $height . '.' . $ext;

            if ($cacheType == 'subfolder') {
                $subfolder = isset($config['thumbnail_subfolder_name']) ? $config['thumbnail_subfolder_name'] : 'thumbnails';
                $gdDest = dirname($srcImage) . '/' . $subfolder . '/' . $gdCacheName;
                $gdUrl = $baseUrl . '/' . dirname($src) . '/' . $subfolder . '/' . $gdCacheName;
            } else {
                $gdDest = dirname($srcImage) . '/' . $gdCacheName;
                $gdUrl = $baseUrl . '/' . dirname($src) . '/' . $gdCacheName;
            }

            // Check target directory
            $targetDir = dirname($gdDest);
            if (!is_dir($targetDir)) {
                @mkdir($targetDir, 0777, true);
            }
            if (!is_writable($targetDir)) {
                @chmod($targetDir, 0777);
            }

            $image = new Resize($srcImage);
            if ($image->resizeImage($width, $height, $option)) {
                if ($grayscale) {
                    $image->setGrayscale();
                }
                if ($image->saveImage($gdDest)) {
                    return $gdUrl;
                }
            }

            log_message('error', 'Thumbnail GD fallback failed for: ' . $srcImage);
        }

        // ============================================================
        // PRIORITY 5: Placeholder
        // ============================================================
        // WebP source + Python unavailable + GD can't handle WebP = placeholder
        log_message('error', 'Thumbnail: all methods failed for: ' . $srcImage);
        return $placeholder;
    }
}

/* End of file thumbnail_helper.php */
/* Location: ./system/application/helpers/thumbnail_helper.php */
