# AEI Photo API Project

Complete photo upload and management system for the AEI Scheduler, handling photo uploads from mobile app and web interface, with synchronization between remote (AWS) and local servers.

**Project Location:** `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/`
**Last Updated:** 2026-02-22

---

## Project Structure

```
AEI_PHOTO_API_PROJECT/
├── README.md                 # This file
├── DOCS/                     # All documentation
│   ├── PHOTO_SYSTEM_DOCUMENTATION.md   # Complete system documentation
│   ├── CREDENTIALS.md                  # Auth tokens, DB passwords, API keys
│   ├── PHOTO_SYSTEM_INTEGRATION.md     # Architecture overview
│   ├── MOBILE_API_RESTORE_GUIDE.md     # Mobile API troubleshooting
│   ├── REMOTE_UPLOAD_CHANGES.md        # Remote server changes log
│   └── CROSS_YEAR_PHOTO_LISTING_ENHANCEMENT.md  # Proposed enhancement
├── QA/                       # QA tools and test documentation
│   ├── README.md                       # QA procedures, test results
│   └── test_upload_pipeline.py         # 10-step E2E pipeline test
├── REMOTE/                   # Remote server files (18.225.0.90)
│   ├── photoapi/             # Mobile app API endpoints
│   │   ├── upload.php                  # Main upload handler
│   │   ├── sync_to_local.py            # Background local server sync (Python)
│   │   ├── generate_webp.py            # Background WebP generator (Python)
│   │   ├── getimagelisting*.php        # Image listing endpoints
│   │   ├── fetch_image.php             # Serve individual images
│   │   └── delete*.php                 # Delete photo endpoints
│   ├── controllers/          # Scheduler controllers
│   ├── models/               # Data models
│   ├── views/                # UI templates
│   ├── scaws/                # AWS SDK
│   └── upload.php.*          # Remote upload.php backups
└── LOCAL/                    # Local server files (upload.aeihawaii.com)
    ├── upload_thumb_generator.py       # Background thumbnail generator
    └── uploadlocallat_kuldeep.php.*    # Local API backups
```

---

## System Overview

| Server | Domain | IP | Purpose |
|--------|--------|-----|---------|
| Remote (AWS) | aeihawaii.com | 18.225.0.90 | Scheduler, Photo API, S3 integration |
| Local | upload.aeihawaii.com | 72.235.242.139 | Photo storage, Dropbox sync |

### Photo Upload Flow (Post-PHOTO-021)

```
Mobile App / Scheduler Web
         │
         ▼
┌─────────────────────────────────────────────────────┐
│  REMOTE SERVER (18.225.0.90)                        │
│                                                      │
│  All upload paths → uploads/staging/{filename}       │
│                                                      │
│  1. Save to uploads/staging/    (ALL uploads here)   │
│  2. INSERT meter_files          (DB source of truth)│
│  3. nohup generate_thumbnails.py &  (2-tier WebP)  ─┐
│  │   → uploads/thumbs/  (200x200 Q70)               │
│  │   → uploads/webp/    (1024px Q80)                 │
│  4. nohup sync_to_local.py &    (archival) ─────────┐│
│  5. Return response (immediate ~0.5s)               ││
└─────────────────────────────────────────────────────┘│
                                                       ▼
                              ┌─────────────────────────────────────┐
                              │  LOCAL SERVER (upload.aeihawaii.com) │
                              │  uploadlocallat_kuldeep.php          │
                              │  - DB-driven customer folder lookup  │
                              │  - Auto-creates if not found         │
                              │  - Saves to Dropbox storage          │
                              │  - Background thumbs (200x200+800px)│
                              └─────────────────────────────────────┘
```

### Photo Display Flow (Post-PHOTO-021)

```
Scheduler Web UI: meter_files DB → uploads/thumbs/ (grid) + uploads/webp/ (lightbox)
Mobile App:       meter_files DB via API → uploads/thumbs/ + uploads/webp/
```

Both platforms use `meter_files` DB as single source of truth. No filesystem scanning.

---

## Key Features

| Feature | Description |
|---------|-------------|
| **Staging-First Upload** | ALL uploads (mobile + scheduler) land in `uploads/staging/` — unified pipeline |
| **2-Tier WebP** | `generate_thumbnails.py` creates thumbs (200x200 Q70), webp (1024px Q80) — hi-res tier removed (PHOTO-022) |
| **DB-Only Listing** | `meter_files` is single source of truth — no filesystem scanning on any platform |
| **Async Local Sync** | Background `sync_to_local.py` with 3 retries + queue — mobile gets instant response |
| **ID-Based Lookup** | Uses `customer_id` for accurate customer folder matching on local server |
| **Auto-Create Folders** | Creates new customer folders + DB records when not found |
| **Background Thumbnails** | Local Python creates WebP thumbs (200x200) + mid (800px) asynchronously |
| **HEIC Support** | Handles iPhone photo format via ImageMagick |
| **QA Test Suite** | 52-step E2E test covering 10 phases (`QA/test_upload_pipeline.py`) |

---

## Documentation

| Document | Location | Purpose |
|----------|----------|---------|
| **Process Map (AUTHORITATIVE)** | [DOCS/PHOTO_SYSTEM_PROCESS_MAP.md](DOCS/PHOTO_SYSTEM_PROCESS_MAP.md) | Post-PHOTO-021 unified process map — START HERE |
| System Documentation | [DOCS/PHOTO_SYSTEM_DOCUMENTATION.md](DOCS/PHOTO_SYSTEM_DOCUMENTATION.md) | Historical docs (partially outdated — see process map) |
| Credentials | [DOCS/CREDENTIALS.md](DOCS/CREDENTIALS.md) | All auth tokens, DB passwords, API keys |
| Integration Guide | [DOCS/PHOTO_SYSTEM_INTEGRATION.md](DOCS/PHOTO_SYSTEM_INTEGRATION.md) | Architecture overview (partially outdated — see process map) |
| Mobile API Guide | [DOCS/MOBILE_API_RESTORE_GUIDE.md](DOCS/MOBILE_API_RESTORE_GUIDE.md) | Troubleshooting mobile uploads |
| QA Test Suite | [QA/README.md](QA/README.md) | QA procedures, test results, troubleshooting |
| Enhancement Log | [ENHANCEMENTS/ENHANCEMENT_LOG.md](ENHANCEMENTS/ENHANCEMENT_LOG.md) | All 21 enhancements tracked |

---

## Remote Server Files (`REMOTE/`)

### Photo API Endpoints (`photoapi/`)

| File | Purpose |
|------|---------|
| `upload.php` | Main upload endpoint (mobile app) — saves photo, triggers background WebP + sync |
| `sync_to_local.py` | Background Python script — POSTs file to local server via `requests` |
| `generate_webp.py` | Background Python script — converts JPEG to WebP via Pillow |
| `getimagelisting*.php` | Image listing endpoints (scan `webp/` folder) |
| `fetch_image.php` | Serve individual WebP images to mobile app |
| `delete*.php` | Delete photo endpoints |

**Authentication:** See [DOCS/CREDENTIALS.md](DOCS/CREDENTIALS.md)

### Controllers (`controllers/`)

| File | Purpose |
|------|---------|
| `phototab.php` | Photo tab UI + S3 integration + background WebP generation |
| `preupload.php` | Premeasure photo upload |
| `loginapi.php` | Mobile app API endpoints |

### Remote Server Paths

| This Backup | Remote Server |
|-------------|---------------|
| `REMOTE/photoapi/` | `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/` |
| `REMOTE/controllers/` | `/var/www/vhosts/.../scheduler/system/application/controllers/` |
| `REMOTE/views/` | `/var/www/vhosts/.../scheduler/system/application/views/` |
| `REMOTE/scaws/` | `/var/www/vhosts/aeihawaii.com/httpdocs/scaws/` |

---

## Local Server Files (`LOCAL/`)

| File | Purpose |
|------|---------|
| `uploadlocallat_kuldeep.php.*` | Local upload API backups |
| `upload_thumb_generator.py` | Background thumbnail generator script |

**Production Location:** `/var/www/html/upload/`

| File | Purpose |
|------|---------|
| `uploadlocallat_kuldeep.php` | Main upload API (receives sync from remote) |
| `upload_thumb_generator.py` | Background gallery thumbnail generator |
| `db_config.php` | Database connection settings |
| `.env` | Auth token storage |

---

## Recent Changes

### 2026-02-22 (PHOTO-022)
- **Hi-res tier removed**: `generate_thumbnails.py` no longer generates `uploads/hi-res/` (2048px Q82) — 2-tier only (thumbs + webp)
- **full_link → webp/**: API `full_link` changed from `/hi-res/` to `/webp/` (same as `link` — best available remote copy)

### 2026-02-21 (PHOTO-021)
- **Staging direct upload**: Scheduler uploads go directly to `uploads/staging/` — no JPEG in uploads/ root, no CI resize
- **DB-only listing**: Scheduler photo_index uses `meter_files` DB only — `$s3files = array()`, s3files view block removed
- **API full_link fix**: `getimagelisting.php` full_link corrected (PHOTO-021); changed to `/webp/` when hi-res tier removed (PHOTO-022)
- **QA expanded**: test_upload_pipeline.py updated to 52 steps across 10 phases, 42/44 PASS

### 2026-02-20 (PHOTO-020)
- **Gallery UX**: All photos loaded at once (no "Show more"), 4-column grid, instant lightbox

### 2026-02-19 (PHOTO-017)
- **WebP generation**: `generate_thumbnails.py` v4.0 creates thumbs (200x200 Q70), webp (1024px Q80); hi-res (2048px Q82) later removed in PHOTO-022
- **Staging flow**: Both mobile and scheduler save to staging/, generate derivatives from full-size source
- **Mobile thumb integration**: `thumb_link` in API responses, Flutter app uses thumbs for grids

### 2026-02-05

### 2026-02-04
- **Async background WebP**: Remote server generates WebP via `generate_webp.py` in background (`nohup &`) — mobile app gets instant JSON response
- Local server returns JSON immediately, triggers background Python thumbnails (`thumbs/` 200x200, `thumbs/mid/` 800px)
- Servers are now independent — no cross-server WebP dependency
- Fixed photo_type bug in WebP filenames (was hardcoded "-S", now dynamic)
- Added auto-create customer folder feature with database auto-update
- SQL injection protection: `intval()`, `mysqli_real_escape_string()`, dynamic `filesize()`
- Fixed hardcoded "2025" year in `upload.php`, `getimagelisting.php`, `fetch_image.php`
- Added cURL error logging and `CURLOPT_CONNECTTIMEOUT` (5s)
- Fixed mobile API sync URL (was pointing to wrong server)
- Fixed local server `.env` permissions and MySQL user

### 2026-01-30
- Added `customer_id` to remote-local sync
- Integrated `unified_customers` database lookup
- Added SQL injection protection

---

## Quick Reference

### Run E2E QA Test
```bash
python3 QA/test_upload_pipeline.py
python3 QA/test_upload_pipeline.py --job-id 108946    # specific job
python3 QA/test_upload_pipeline.py --skip-cleanup      # inspect test data
```

### Monitor Sync Log (Remote)
```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -f /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log"
```

### Monitor Local Server Uploads
```bash
tail -f /var/log/apache2/upload_access.log | grep "18.225.0.90"
```

### Check Database Mappings
```bash
mysql -u upload_user -p'***' Schedular -e "  # See DOCS/CREDENTIALS.md for password
SELECT COUNT(*) as total,
       SUM(CASE WHEN folder_path IS NOT NULL THEN 1 ELSE 0 END) as with_folders
FROM unified_customers;"
```

### SSH to Remote Server
```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90
```

### Ensure Firewall Whitelist (Local Server)
```bash
# Required after local server reboot
sudo ipset test trusted_whitelist 18.225.0.90 || sudo ipset add trusted_whitelist 18.225.0.90
```

---

## Known Behaviors

### Cross-Year Folder Routing (Remote vs Local)

When a customer has jobs spanning multiple years (e.g., premeasure in 2025, installation in 2026), the two servers route photos differently:

- **Remote server** (`upload.php`): Always creates folders under the job's year (`2026 Customers/`). No database lookup — path is derived from `job_date`.
- **Local server** (`uploadlocallat_kuldeep.php`): Uses `unified_customers` DB lookup with fallback. If a 2026 entry doesn't exist, it reuses the customer's **2025 folder**.

This means cross-year customers may have photos in different year folders on each server. The mobile app gallery (`getimagelisting.php`) only shows photos from the current job's year folder — premeasure photos from a prior year are not visible in the installation job's gallery.

See [DOCS/PHOTO_SYSTEM_DOCUMENTATION.md](DOCS/PHOTO_SYSTEM_DOCUMENTATION.md) — "Cross-Year Folder Routing" section for the full comparison table and examples.

---

## Related Projects

| Project | Location | Purpose |
|---------|----------|---------|
| Customer Mapper | `/var/opt/map_dropbox/` | Maps customers to Dropbox folders (daily cron) |
| Local Server Security | `/var/www/html/security/` | Firewall, ipset whitelist, fail2ban |
| Photo Organizer | `/var/opt/move_photos/` | Batch photo organization (dormant) |

---

*Project maintained at: `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/`*
