#!/usr/bin/env python3
"""
AEI Photo Upload Pipeline — End-to-End QA Test

Tests the complete upload flow:
  Phase 1 — Upload & Response:
    1. POST test image to remote API (upload.php)
    2. Verify JSON response + timing
  Phase 2 — Remote Artifacts:
    3. Verify file saved to /mnt/dropbox/ on remote
    4. Verify WebP generated in webp/ folder
    5. Verify file in uploads/staging/ (all uploads)
    6. Verify WebP copy to scheduler/uploads/webp/
    7. Verify meter_files DB insert
    8. Verify sync_to_local.py triggered (log entry)
  Phase 3 — Local Server:
    9. Verify file received on local server
    10. Verify thumbnails generated on local server
  Phase 4 — Async Sync Enhancement (ASYNC_SYNC_ENHANCEMENT.md):
    11. No synchronous cURL in upload.php (curl_init/exec/close removed)
    12. Echo JSON is last PHP statement before ?>
    13. sync_to_local.py exists with correct permissions (755)
    14. process_retry_queue.py exists and is executable (755)
    15. queue/ and queue/failed/ directories exist with correct permissions
    16. Cron entry for process_retry_queue.py (ec2-user, */15)
    17. logs/ directory exists and is writable (777)
    18. Python 3.6 requests module available
    19. Queue empty after successful upload (no false queuing)
  Phase 5 — Photo Folder Path Enhancement (PHOTO_FOLDER_PATH_ENHANCEMENT.md):
    20. folder_path column exists in meter_files
    21. folder_path populated for test upload
    22. folder_path matches actual file location
    23. getimagelisting.php returns test image
    24. fetch_image.php serves image via folder_path
    25. Survey path spacing fix deployed (local)
  Phase 6 — Reconciliation Enhancement (PHOTO-011):
    26. check_photos.php exists on local server
    27. check_photos.php rejects invalid auth token
    28. check_photos.php enforces batch limit (>200 rejected)
    29. check_photos.php detects known-missing file
    30. check_photos.php correctly checks uploaded test file
    31. Reconciliation constants in process_retry_queue.py
    32. Reconciliation functions in process_retry_queue.py
    33. .reconcile_started marker file exists on remote
  Phase 7 — PHOTO-009 Unified Storage (DB-only, flat storage, delete):
    34. Upload does NOT write to /mnt/dropbox/ hierarchy
    35. getimagelisting.php returns correct structure (fetch_image.php URLs)
    36. fetch_image.php serves WebP with correct Content-Type
    37. delete.php deletes test photo (file + WebP + DB row)
    38. delete.php rejects invalid auth token
    39. getimagelisting.php returns empty after delete
  Phase 8 — Remote Server Capacity Audit:
    40. PHP post_max_size >= 50M
    41. PHP memory_limit >= 256M
    42. PHP max_execution_time is 0 or >= 300
    43. Disk space > 10GB free on web partition
    44. Python 3.6 Pillow module available
    45. scheduler/uploads/ and webp/ writable by Apache
  Phase 9 — Large Photo Upload:
    46. Upload largest test photo (~11MB), verify success + timing
    47. Verify WebP generated for large photo
  Phase 10 — PHOTO-022/023 Staging Direct + 2-Tier WebP (1280px):
    48. staging/ directory exists and is writable
    49. Mobile API upload writes to uploads/ (correct behavior)
    50. 2-tier WebP derivatives: thumbs/, webp/ (hi-res removed PHOTO-022)
    51. getimagelisting full_link points to /webp/ (PHOTO-022)
    52. All photos in listing are from meter_files DB (no s3files)
  Phase 11 — Complete Photo Deletion:
    53. delete_image.php rejects invalid auth token
    54. delete_image.php backward compat (old single 'img' format)
    55. delete_image.php batch delete removes staging + thumbs + webp
    56. delete_local_photo.php rejects invalid auth token
    57. delete_local_photo.php requires job_id + unique_filename
    58. delete_local_photo.php returns ok for non-existent photo
    59. delete_local_photo.php local_photos lookup + file deletion
  Phase 12 — PHOTO-023/024 Error Logging, Cron Fallback, No Archive Copy:
    60. No archive copy in uploads/ root (PHOTO-024 removed)
    61. webp_generation.log has timestamped entries after upload
    62. fix_missing_webp.py runs without error
    63. delete_image.php works without archive copy (PHOTO-024)

  Batch Mode (--batch):
    Uploads N real photos from QA/test_photos/ sequentially, verifies all
    remote artifacts, reports timing stats, then cleans up.

Usage:
  python3 test_upload_pipeline.py                           # Full test (default job)
  python3 test_upload_pipeline.py --job-id 108946           # Test with specific job
  python3 test_upload_pipeline.py --skip-cleanup            # Leave test data for inspection
  python3 test_upload_pipeline.py --remote-only             # Skip local server checks
  python3 test_upload_pipeline.py --batch                   # Upload 200 real photos
  python3 test_upload_pipeline.py --batch --batch-size 50   # Upload 50 real photos
  python3 test_upload_pipeline.py --remote-only --batch     # Batch, remote-only

Prerequisites:
  - pip3 install requests Pillow paramiko
  - SSH key at /root/.ssh/aei_remote.pem
  - Network access to aeihawaii.com (remote) and upload.aeihawaii.com (local)
  - For --batch: real photos in QA/test_photos/ directory
"""

import argparse
import base64
import glob as glob_mod
import io
import json
import os
import re
import sys
import time
import subprocess

try:
    import requests
    from PIL import Image
    import urllib3
    urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
except ImportError:
    print("ERROR: pip3 install requests Pillow")
    sys.exit(1)

# ── Configuration ──────────────────────────────────────────────────

REMOTE_API_URL = "https://aeihawaii.com/photoapi/upload.php"
REMOTE_AUTH_TOKEN = "aei@89806849"
SSH_KEY = "/root/.ssh/aei_production.pem"
SSH_USER = "Julian"
SSH_HOST = "18.225.0.90"
DB_USER = "schedular"
DB_PASS = "M1gif9!6"
DB_NAME = "mandhdesign_schedular"
TEST_FILENAME = "qa_pipeline_test.jpg"
DEFAULT_JOB_ID = "108946"
REMOTE_PHOTOAPI = "/var/www/vhosts/aeihawaii.com/httpdocs/photoapi"
REMOTE_SCHEDULER = "/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads"
REMOTE_DELETE_URL = "https://aeihawaii.com/photoapi/delete.php"
REMOTE_DELETE_IMAGE_URL = "https://aeihawaii.com/photoapi/delete_image.php"
LOCAL_DELETE_URL = "https://upload.aeihawaii.com/delete_local_photo.php"
LOCAL_DELETE_AUTH = "aei_local_delete_89806849"
REMOTE_LISTING_URL = "https://aeihawaii.com/photoapi/getimagelisting.php"
LOCAL_SSH_USER = "aeiuser"
LOCAL_SSH_HOST = "192.168.141.219"   # aei-webserv2 (upload.aeihawaii.com)
BATCH_PREFIX = "qa_batch_"
TEST_PHOTOS_DIR = os.path.join(os.path.dirname(os.path.abspath(__file__)), "test_photos")

# ── Helpers ────────────────────────────────────────────────────────

class Colors:
    PASS = "\033[92m"
    FAIL = "\033[91m"
    WARN = "\033[93m"
    INFO = "\033[94m"
    RESET = "\033[0m"
    BOLD = "\033[1m"

def ssh_cmd(cmd, timeout=30):
    """Run command on remote production server (18.225.0.90) via SSH."""
    result = subprocess.run(
        ["ssh", "-i", SSH_KEY, "-o", "ConnectTimeout=10",
         "-o", "StrictHostKeyChecking=no",
         f"{SSH_USER}@{SSH_HOST}", cmd],
        capture_output=True, text=True, timeout=timeout
    )
    return result.stdout.strip(), result.stderr.strip(), result.returncode

def local_ssh_cmd(cmd, timeout=30):
    """Run command on local upload server (aei-webserv2 / upload.aeihawaii.com) via SSH."""
    result = subprocess.run(
        ["ssh", "-o", "ConnectTimeout=10",
         "-o", "StrictHostKeyChecking=no",
         f"{LOCAL_SSH_USER}@{LOCAL_SSH_HOST}", cmd],
        capture_output=True, text=True, timeout=timeout
    )
    return result.stdout.strip(), result.stderr.strip(), result.returncode

def result_line(step_num, name, passed, detail=""):
    """Print a formatted test result line."""
    status = f"{Colors.PASS}PASS{Colors.RESET}" if passed else f"{Colors.FAIL}FAIL{Colors.RESET}"
    detail_str = f" — {detail}" if detail else ""
    print(f"  [{status}] {step_num:>2}. {name}{detail_str}")
    return passed

def create_test_image():
    """Generate a small test JPEG and return base64."""
    img = Image.new("RGB", (200, 200), color="blue")
    buf = io.BytesIO()
    img.save(buf, format="JPEG", quality=80)
    return base64.b64encode(buf.getvalue()).decode()

# ── Test Steps ─────────────────────────────────────────────────────

def test_upload(job_id, filename=None, image_data=None):
    """POST test image to remote API and return response + timing."""
    if filename is None:
        filename = TEST_FILENAME
    if image_data is None:
        image_data = create_test_image()

    payload = {
        "auth_token": REMOTE_AUTH_TOKEN,
        "file_name": filename,
        "image_data": image_data,
        "job_id": str(job_id),
    }

    start = time.time()
    try:
        resp = requests.post(REMOTE_API_URL, json=payload, timeout=60, verify=False)
    except Exception as e:
        return None, 0, str(e)
    elapsed = time.time() - start

    try:
        data = resp.json()
    except Exception:
        data = {"raw": resp.text[:200]}

    return data, elapsed, None


def test_remote_checks(job_id, uploaded_filename=None):
    """Steps 3-8: Verify all remote server artifacts.

    Args:
        uploaded_filename: The actual filename on disk (from API response 'path' field).
            upload.php prepends a unique ID, so the on-disk name differs from TEST_FILENAME.
            If None, falls back to TEST_FILENAME for backward compatibility.
    """
    results = {}
    disk_filename = uploaded_filename or TEST_FILENAME

    # Build single SSH command that checks everything
    # Note: disk_filename = on-disk name (with uniqid prefix), TEST_FILENAME = unique_filename in DB
    cmd = f"""
    echo '===FILES==='
    find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' 2>/dev/null | head -1

    echo '===WEBP==='
    JOBDIR=$(find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' -exec dirname {{}} \\; 2>/dev/null | head -1)
    [ -n "$JOBDIR" ] && find "$JOBDIR/webp/" -name '*.webp' 2>/dev/null | head -1 || echo ''

    echo '===SCHEDULER==='
    ls '{REMOTE_SCHEDULER}/staging/{disk_filename}' 2>/dev/null || echo ''

    echo '===SCHEDULER_WEBP==='
    WEBPNAME=$(mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SELECT webpfilename FROM meter_files WHERE unique_filename='{TEST_FILENAME}' AND webpfilename IS NOT NULL AND webpfilename != '' ORDER BY id DESC LIMIT 1;" 2>/dev/null)
    [ -n "$WEBPNAME" ] && ls '{REMOTE_SCHEDULER}/webp/'"$WEBPNAME" 2>/dev/null || echo ''

    echo '===SCHEDULER_THUMBS==='
    [ -n "$WEBPNAME" ] && ls '{REMOTE_SCHEDULER}/thumbs/'"$WEBPNAME" 2>/dev/null || echo ''

    echo '===NOT_IN_UPLOADS_ROOT==='
    ls '{REMOTE_SCHEDULER}/{disk_filename}' 2>/dev/null || echo 'CORRECT_NOT_FOUND'

    echo '===METER_FILES==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SELECT id,job_id,file_size,webpfilename FROM meter_files WHERE unique_filename='{TEST_FILENAME}' ORDER BY id DESC LIMIT 1;" 2>/dev/null

    echo '===SYNC_LOG==='
    cat {REMOTE_PHOTOAPI}/logs/sync_to_local.log 2>/dev/null | grep '{TEST_FILENAME}' | tail -1
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    # Parse sections
    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["file_saved"] = section("FILES")
    results["webp_generated"] = section("WEBP")
    results["scheduler_copy"] = section("SCHEDULER")
    results["scheduler_webp"] = section("SCHEDULER_WEBP")
    results["scheduler_thumbs"] = section("SCHEDULER_THUMBS")
    results["not_in_uploads_root"] = section("NOT_IN_UPLOADS_ROOT")
    results["meter_files"] = section("METER_FILES")
    results["sync_log"] = section("SYNC_LOG")

    return results


def test_local_checks(customer_id, last_name, first_name, job_date, test_filename=None):
    """Steps 9-10: Verify file received on local server with thumbnails.

    Runs queries and file checks on aei-webserv2 (upload.aeihawaii.com) via SSH,
    since that's where the Schedular DB and /mnt/dropbox/ filesystem live.
    """
    if test_filename is None:
        test_filename = TEST_FILENAME

    results = {}

    LOCAL_DB_USER = "upload_user"
    LOCAL_DB_PASS = "P@55w02d778899"
    LOCAL_DB_NAME = "Schedular"

    # Run all checks on aei-webserv2 in a single SSH call
    job_year = job_date[:4] if job_date else "2026"
    idx = last_name[0].upper() if last_name else "?"

    check_script = (
        "#!/bin/bash\n"
        "# Query unified_customers for folder paths\n"
        "FOLDERS=$(mysql -u {db_user} -p'{db_pass}' {db_name} -N -e \""
        "SELECT folder_path FROM unified_customers "
        "WHERE remote_customer_id = {cust_id} AND folder_path IS NOT NULL "
        "ORDER BY folder_year DESC;\" 2>/dev/null)\n"
        "if [ -z \"$FOLDERS\" ]; then\n"
        "  FOLDERS=$(mysql -u {db_user} -p'{db_pass}' {db_name} -N -e \""
        "  SELECT folder_path FROM unified_customers "
        "  WHERE first_name = '{first}' AND last_name = '{last}' "
        "  AND folder_path IS NOT NULL ORDER BY folder_year DESC;\" 2>/dev/null)\n"
        "fi\n"
        "echo \"DB_LOOKUP=$FOLDERS\"\n"
        "# Add fallback paths\n"
        "FOLDERS=\"$FOLDERS\n/mnt/dropbox/{year} Customers/{idx}\n/mnt/dropbox/Uploads\"\n"
        "# Search each folder for the test file\n"
        "FOUND=''\n"
        "while IFS= read -r folder; do\n"
        "  [ -z \"$folder\" ] && continue\n"
        "  [ ! -d \"$folder\" ] && continue\n"
        "  HIT=$(find \"$folder\" -maxdepth 6 -name '{filename}' 2>/dev/null | head -1)\n"
        "  if [ -n \"$HIT\" ]; then\n"
        "    FOUND=\"$HIT\"\n"
        "    break\n"
        "  fi\n"
        "done <<< \"$FOLDERS\"\n"
        "echo \"FILE_FOUND=$FOUND\"\n"
        "# Check thumbnails\n"
        "if [ -n \"$FOUND\" ]; then\n"
        "  PARENT=$(dirname \"$FOUND\")\n"
        "  WEBP=$(echo '{filename}' | sed 's/.jpg$/.webp/')\n"
        "  [ -f \"$PARENT/thumbs/$WEBP\" ] && echo \"THUMB=YES\" || echo \"THUMB=NO\"\n"
        "  [ -f \"$PARENT/thumbs/mid/$WEBP\" ] && echo \"MID=YES\" || echo \"MID=NO\"\n"
        "else\n"
        "  echo \"THUMB=NO\"\n"
        "  echo \"MID=NO\"\n"
        "fi\n"
    ).format(
        db_user=LOCAL_DB_USER, db_pass=LOCAL_DB_PASS, db_name=LOCAL_DB_NAME,
        cust_id=customer_id, first=first_name, last=last_name,
        year=job_year, idx=idx, filename=test_filename
    )

    try:
        out, err, rc = local_ssh_cmd(check_script, timeout=20)
    except Exception as e:
        results["file_found"] = ""
        results["thumbnail"] = ""
        results["mid_thumb"] = ""
        results["db_lookup"] = "SSH error: %s" % str(e)
        return results

    # Parse output
    results["db_lookup"] = ""
    results["file_found"] = ""
    results["thumbnail"] = ""
    results["mid_thumb"] = ""

    for line in out.split("\n"):
        if line.startswith("DB_LOOKUP="):
            results["db_lookup"] = line[10:].strip()
        elif line.startswith("FILE_FOUND="):
            path = line[11:].strip()
            results["file_found"] = path
        elif line.startswith("THUMB=YES"):
            results["thumbnail"] = "yes"
        elif line.startswith("MID=YES"):
            results["mid_thumb"] = "yes"

    return results


def test_async_enhancement_checks():
    """Steps 11-19: Verify async sync enhancement deployment (ASYNC_SYNC_ENHANCEMENT.md)."""
    results = {}

    cmd = f"""
    echo '===NO_CURL==='
    grep -c 'curl_init\\|curl_exec\\|curl_close' {REMOTE_PHOTOAPI}/upload.php 2>/dev/null || true

    echo '===ECHO_LAST==='
    tac {REMOTE_PHOTOAPI}/upload.php | grep -m1 -n '[a-zA-Z]' | head -1

    echo '===SYNC_PERMS==='
    stat -c '%a %U:%G' {REMOTE_PHOTOAPI}/sync_to_local.py 2>/dev/null || echo 'NOT_FOUND'

    echo '===RETRY_PERMS==='
    stat -c '%a %U:%G' {REMOTE_PHOTOAPI}/process_retry_queue.py 2>/dev/null || echo 'NOT_FOUND'

    echo '===QUEUE_DIR==='
    stat -c '%a' {REMOTE_PHOTOAPI}/queue/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===QUEUE_FAILED==='
    stat -c '%a' {REMOTE_PHOTOAPI}/queue/failed/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===CRON==='
    (sudo crontab -u ec2-user -l 2>/dev/null; sudo crontab -l 2>/dev/null) | grep process_retry || echo 'NOT_FOUND'

    echo '===LOGS_DIR==='
    stat -c '%a' {REMOTE_PHOTOAPI}/logs/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===PYTHON_REQUESTS==='
    /usr/local/bin/python3.6 -c 'import requests; print("OK")' 2>/dev/null || echo 'MISSING'

    echo '===QUEUE_COUNT==='
    grep -l '{TEST_FILENAME}' {REMOTE_PHOTOAPI}/queue/*.json 2>/dev/null | wc -l
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["no_curl"] = section("NO_CURL")
    results["echo_last"] = section("ECHO_LAST")
    results["sync_perms"] = section("SYNC_PERMS")
    results["retry_perms"] = section("RETRY_PERMS")
    results["queue_dir"] = section("QUEUE_DIR")
    results["queue_failed"] = section("QUEUE_FAILED")
    results["cron"] = section("CRON")
    results["logs_dir"] = section("LOGS_DIR")
    results["python_requests"] = section("PYTHON_REQUESTS")
    results["queue_count"] = section("QUEUE_COUNT")

    return results


def test_folder_path_checks():
    """Steps 20-22: Verify folder_path column and population in meter_files."""
    results = {}

    cmd = f"""
    echo '===COLUMN==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SHOW COLUMNS FROM meter_files LIKE 'folder_path';" 2>/dev/null

    echo '===FOLDER_PATH==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SELECT folder_path FROM meter_files WHERE unique_filename='{TEST_FILENAME}' ORDER BY id DESC LIMIT 1;" 2>/dev/null
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["column_info"] = section("COLUMN")
    results["folder_path"] = section("FOLDER_PATH")

    return results


def test_image_api_checks(job_id):
    """Steps 23-24: Verify getimagelisting.php and fetch_image.php."""
    results = {}

    # Step 23: POST to getimagelisting.php
    try:
        listing_url = "https://aeihawaii.com/photoapi/getimagelisting.php"
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": str(job_id),
        }
        resp = requests.post(listing_url, json=payload, timeout=30, verify=False)
        try:
            data = resp.json()
        except Exception:
            data = None
        results["listing_status"] = resp.status_code
        results["listing_data"] = data
    except Exception as e:
        results["listing_status"] = 0
        results["listing_data"] = None
        results["listing_error"] = str(e)

    # Step 24: GET one image link from the listing
    results["fetch_status"] = 0
    results["fetch_content_type"] = ""
    results["fetch_size"] = 0

    if results["listing_data"] and isinstance(results["listing_data"], list):
        # Find a link URL in the response
        link_url = None
        for entry in results["listing_data"]:
            if isinstance(entry, dict) and entry.get("link"):
                link_url = entry["link"]
                break

        if link_url:
            try:
                img_resp = requests.get(link_url, timeout=30, verify=False)
                results["fetch_status"] = img_resp.status_code
                results["fetch_content_type"] = img_resp.headers.get("Content-Type", "")
                results["fetch_size"] = len(img_resp.content)
                results["fetch_url"] = link_url
            except Exception as e:
                results["fetch_error"] = str(e)
        else:
            results["fetch_error"] = "no link URL in listing response"
    else:
        results["fetch_error"] = "no valid listing data"

    return results


def test_survey_spacing_fix():
    """Step 25: Verify survey path spacing fix on local server (aei-webserv2 via SSH)."""
    results = {}
    local_file = "/var/www/html/upload/uploadlocallat_kuldeep.php"

    try:
        cmd = (
            "if [ -f '%s' ]; then echo 'EXISTS=YES'; "
            "BAD=$(grep -c '\\-S,[^ ]' '%s' 2>/dev/null || echo 0); echo \"BAD=$BAD\"; "
            "GOOD=$(grep -c '\\-S, ' '%s' 2>/dev/null || echo 0); echo \"GOOD=$GOOD\"; "
            "else echo 'EXISTS=NO'; fi"
        ) % (local_file, local_file, local_file)
        out, err, rc = local_ssh_cmd(cmd, timeout=10)
    except Exception as e:
        results["file_exists"] = False
        results["error"] = str(e)
        return results

    results["file_exists"] = "EXISTS=YES" in out

    if not results["file_exists"]:
        return results

    results["bad_count"] = 0
    results["good_count"] = 0
    for line in out.split("\n"):
        if line.startswith("BAD="):
            try:
                results["bad_count"] = int(line[4:].strip())
            except ValueError:
                pass
        elif line.startswith("GOOD="):
            try:
                results["good_count"] = int(line[5:].strip())
            except ValueError:
                pass

    return results


SURVEY_JOB_TYPES = {'PM', 'WM', 'AS', 'RPM', 'GCPM'}
CHECK_PHOTOS_URL = "https://upload.aeihawaii.com/check_photos.php"
LOCAL_AUTH_TOKEN = "remote_token"


def test_check_photos_endpoint(job_info, local_file_found):
    """Steps 26-30: Verify check_photos.php endpoint on local server (aei-webserv2)."""
    results = {}

    # Step 26: File exists on local server (aei-webserv2 via SSH)
    try:
        out, _, _ = local_ssh_cmd("[ -f /var/www/html/upload/check_photos.php ] && echo YES || echo NO", timeout=5)
        results["file_exists"] = "YES" in out
    except Exception:
        results["file_exists"] = False

    # Step 27: Auth rejection
    try:
        resp = requests.post(CHECK_PHOTOS_URL,
            json={"auth_token": "WRONG_TOKEN", "photos": []},
            timeout=10, verify=False)
        results["auth_rejected"] = resp.status_code == 403
        results["auth_status"] = resp.status_code
    except Exception as e:
        results["auth_rejected"] = False
        results["auth_error"] = str(e)

    # Step 28: Batch limit enforcement (>200 rejected)
    try:
        big_batch = [{"filename": "test%d.jpg" % i, "customer_id": 1,
                      "first_name": "T", "last_name": "U",
                      "job_date": "2026-01-01", "photo_type": "I",
                      "job_type": "WS"} for i in range(201)]
        resp = requests.post(CHECK_PHOTOS_URL,
            json={"auth_token": LOCAL_AUTH_TOKEN, "photos": big_batch},
            timeout=10, verify=False)
        results["batch_rejected"] = resp.status_code == 400
        results["batch_status"] = resp.status_code
    except Exception as e:
        results["batch_rejected"] = False
        results["batch_error"] = str(e)

    # Step 29: Known-missing file detected
    try:
        jtype = (job_info.get("job_type") or "WS").upper().strip()
        photo_type = "S" if jtype in SURVEY_JOB_TYPES else "I"
        resp = requests.post(CHECK_PHOTOS_URL,
            json={"auth_token": LOCAL_AUTH_TOKEN, "photos": [{
                "filename": "qa_nonexistent_999.jpg",
                "customer_id": int(job_info.get("customer_id", 0)),
                "first_name": job_info.get("first_name", ""),
                "last_name": job_info.get("last_name", ""),
                "job_date": job_info.get("job_date", ""),
                "photo_type": photo_type, "job_type": jtype,
            }]}, timeout=10, verify=False)
        data = resp.json()
        results["missing_detected"] = "qa_nonexistent_999.jpg" in data.get("missing", [])
        results["missing_response"] = data
    except Exception as e:
        results["missing_detected"] = False
        results["missing_error"] = str(e)

    # Step 30: Check test file (should match local_file_found state)
    try:
        jtype = (job_info.get("job_type") or "WS").upper().strip()
        photo_type = "S" if jtype in SURVEY_JOB_TYPES else "I"
        resp = requests.post(CHECK_PHOTOS_URL,
            json={"auth_token": LOCAL_AUTH_TOKEN, "photos": [{
                "filename": TEST_FILENAME,
                "customer_id": int(job_info.get("customer_id", 0)),
                "first_name": job_info.get("first_name", ""),
                "last_name": job_info.get("last_name", ""),
                "job_date": job_info.get("job_date", ""),
                "photo_type": photo_type, "job_type": jtype,
            }]}, timeout=10, verify=False)
        data = resp.json()
        test_found = TEST_FILENAME not in data.get("missing", []) and data.get("found", 0) > 0
        results["test_file_found"] = test_found
        results["test_file_response"] = data
    except Exception as e:
        results["test_file_found"] = False
        results["test_file_error"] = str(e)

    # Whether the local file was actually present (for step 30 pass/fail logic)
    results["local_file_expected"] = local_file_found

    return results


def test_reconciliation_remote():
    """Steps 31-33: Verify reconciliation code deployed on remote server."""
    results = {}

    cmd = (
        "echo '===CONSTANTS===' && "
        "grep -cE '^(CHECK_PHOTOS_URL|RECONCILE_DAYS|RECONCILE_BATCH_SIZE) =' "
        "%s/process_retry_queue.py 2>/dev/null; "
        "echo '===FUNCTIONS===' && "
        "grep -c 'def should_run_reconcile\\|def mark_reconcile_run\\|def reconcile_local_sync\\|def get_reconcile_start_date' "
        "%s/process_retry_queue.py 2>/dev/null; "
        "echo '===MARKER===' && "
        "cat %s/logs/.reconcile_started 2>/dev/null || echo 'NOT_FOUND'"
    ) % (REMOTE_PHOTOAPI, REMOTE_PHOTOAPI, REMOTE_PHOTOAPI)

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["constants_count"] = section("CONSTANTS")
    results["functions_count"] = section("FUNCTIONS")
    results["marker"] = section("MARKER")

    return results


def test_enh009_no_dropbox_write():
    """Step 34: Verify upload.php does NOT write to /mnt/dropbox/ hierarchy."""
    cmd = (
        "find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' "
        "-name '%s' 2>/dev/null | head -1" % TEST_FILENAME
    )
    out, err, rc = ssh_cmd(cmd, timeout=15)
    # PASS if file is NOT found in /mnt/dropbox/ (PHOTO-009 removed hierarchy writes)
    return not bool(out.strip()), out.strip() if out.strip() else "not in /mnt/dropbox/"


def get_webpfilename_for_test():
    """Look up the webpfilename for our test file from meter_files DB."""
    cmd = (
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
        "ORDER BY id DESC LIMIT 1;\" 2>/dev/null"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME)
    out, _, _ = ssh_cmd(cmd, timeout=10)
    return out.strip() if out.strip() else None


def test_enh009_listing_structure(job_id):
    """Step 35: Verify getimagelisting.php returns correct structure.

    PHOTO-022 removed hi-res tier. full_link now points to /webp/ (same as link).
    The listing should contain direct static URLs for link (webp/), thumb_link (thumbs/),
    and full_link (webp/).
    """
    # First, get the webpfilename from DB so we can find it in the listing
    webpname = get_webpfilename_for_test()

    try:
        payload = {"auth_token": REMOTE_AUTH_TOKEN, "job_id": str(job_id)}
        resp = requests.post(REMOTE_LISTING_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
    except Exception as e:
        return False, str(e), None

    if not isinstance(data, list):
        return False, "response is not a list: %s" % type(data).__name__, None

    # Find our test file in the listing by webpfilename
    test_entry = None
    for entry in data:
        if isinstance(entry, dict) and entry.get("link", ""):
            link = entry["link"]
            # Match by webpfilename (from DB) or by unique_filename as fallback
            if webpname and webpname in link:
                test_entry = entry
                break
            if TEST_FILENAME.replace(".jpg", "") in link:
                test_entry = entry
                break

    if not test_entry:
        return False, "test file not in listing (%d entries, webp=%s)" % (
            len(data), webpname), None

    # Validate structure: link should be direct static URL (webp/ or fetch_image.php)
    link = test_entry.get("link", "")
    full_link = test_entry.get("full_link", "")
    has_link = bool(link)
    has_fields = all(k in test_entry for k in ("link", "job_id", "photo_type"))
    # PHOTO-022: full_link should point to /webp/ (hi-res removed)
    full_link_webp = "/webp/" in full_link
    ok = has_link and has_fields and full_link_webp
    detail = "link=%s, full_link=%s, fields=%s" % (
        "ok" if has_link else "MISSING",
        "webp/ (PHOTO-022)" if full_link_webp else (full_link[:60] if full_link else "MISSING/wrong"),
        "ok" if has_fields else "missing keys"
    )
    return ok, detail, test_entry


def test_enh009_fetch_image(listing_entry):
    """Step 36: Verify fetch_image.php serves WebP with correct Content-Type."""
    if not listing_entry or not isinstance(listing_entry, dict) or not listing_entry.get("link"):
        return False, "no listing entry to test (step 35 failed)", 0

    link = listing_entry["link"]
    try:
        resp = requests.get(link, timeout=30, verify=False)
    except Exception as e:
        return False, str(e), 0

    ct = resp.headers.get("Content-Type", "")
    ok = resp.status_code == 200 and ct.startswith("image/")
    detail = "HTTP %d, %s, %d bytes" % (resp.status_code, ct, len(resp.content))
    return ok, detail, len(resp.content)


def test_enh009_delete(job_id):
    """Step 37: Verify delete.php deletes test photo (file + WebP + DB row)."""
    # First get the webpfilename from DB
    db_cmd = (
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT webpfilename, unique_filename FROM meter_files "
        "WHERE unique_filename='%s' ORDER BY id DESC LIMIT 1;\" 2>/dev/null"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME)
    db_out, _, _ = ssh_cmd(db_cmd, timeout=10)

    # Call delete.php
    try:
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": int(job_id),
            "file_name": TEST_FILENAME,
        }
        resp = requests.post(REMOTE_DELETE_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "delete request failed: %s" % str(e), None

    status = data.get("status", "")
    deleted = data.get("deleted", [])
    ok = status == "ok" and len(deleted) > 0

    if not ok:
        return False, "status=%s, deleted=%s, errors=%s" % (
            status, deleted, data.get("errors", [])), data

    # Verify files actually removed from disk
    verify_cmd = (
        "echo '===JPG===' && "
        "ls '%s/staging/%s' '%s/%s' 2>/dev/null || echo 'GONE' && "
        "echo '===DB===' && "
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT COUNT(*) FROM meter_files WHERE unique_filename='%s';\" 2>/dev/null"
    ) % (REMOTE_SCHEDULER, TEST_FILENAME, REMOTE_SCHEDULER, TEST_FILENAME,
         DB_USER, DB_PASS, DB_NAME, TEST_FILENAME)
    v_out, _, _ = ssh_cmd(verify_cmd, timeout=10)

    jpg_gone = "GONE" in v_out
    sections = v_out.split("===")
    db_count = "0"
    for i, s in enumerate(sections):
        if s.strip() == "DB" and i + 1 < len(sections):
            db_count = sections[i + 1].strip()

    # Delete removes one row at a time; orphan rows from prior runs are acceptable
    # Success = file removed from disk + delete API returned ok
    detail = "deleted=%s, file=%s, db_rows_remaining=%s" % (
        deleted, "removed" if jpg_gone else "STILL EXISTS",
        db_count
    )
    return jpg_gone, detail, data


def test_enh009_delete_auth_rejected(job_id):
    """Step 38: Verify delete.php rejects invalid auth token."""
    try:
        payload = {
            "auth_token": "WRONG_TOKEN",
            "job_id": int(job_id),
            "file_name": TEST_FILENAME,
        }
        resp = requests.post(REMOTE_DELETE_URL, json=payload, timeout=15, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "request failed: %s" % str(e)

    ok = data.get("status") == "error" and "token" in data.get("message", "").lower()
    return ok, "status=%s, message=%s" % (data.get("status"), data.get("message", ""))


def test_enh009_listing_empty_after_delete(job_id, webpname=None):
    """Step 39: Verify getimagelisting.php has no test file after delete."""
    try:
        payload = {"auth_token": REMOTE_AUTH_TOKEN, "job_id": str(job_id)}
        resp = requests.post(REMOTE_LISTING_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
    except Exception as e:
        return False, str(e)

    if not isinstance(data, list):
        # Could be an error dict if job has no photos at all — that's fine
        return True, "no listing (non-list response)"

    # Check our test file is NOT in the listing (match by webpfilename or unique_filename)
    for entry in data:
        if not isinstance(entry, dict):
            continue
        link = entry.get("link", "")
        if webpname and webpname in link:
            return False, "test file still in listing (%d entries)" % len(data)
        if TEST_FILENAME.replace(".jpg", "") in link:
            return False, "test file still in listing (%d entries)" % len(data)

    return True, "test file absent (%d other entries)" % len(data)


def test_capacity_audit():
    """Steps 40-45: Remote server capacity audit via SSH."""
    results = {}

    cmd = """
    echo '===POST_MAX==='
    php -r 'echo ini_get("post_max_size");' 2>/dev/null

    echo '===MEMORY==='
    php -r 'echo ini_get("memory_limit");' 2>/dev/null

    echo '===MAX_EXEC==='
    php -r 'echo ini_get("max_execution_time");' 2>/dev/null

    echo '===DISK==='
    df -BG /var/www/ 2>/dev/null | tail -1 | awk '{print $4}'

    echo '===PILLOW==='
    /usr/local/bin/python3.6 -c 'from PIL import Image; print("OK")' 2>/dev/null || echo 'MISSING'

    echo '===WRITABLE==='
    UPLOADS_DIR='/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads'
    echo "uploads=$(stat -c '%a' $UPLOADS_DIR 2>/dev/null || echo 'MISSING')"
    echo "staging=$(stat -c '%a' $UPLOADS_DIR/staging 2>/dev/null || echo 'MISSING')"
    echo "webp=$(stat -c '%a' $UPLOADS_DIR/webp 2>/dev/null || echo 'MISSING')"
    echo "thumbs=$(stat -c '%a' $UPLOADS_DIR/thumbs 2>/dev/null || echo 'MISSING')"
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["post_max"] = section("POST_MAX")
    results["memory"] = section("MEMORY")
    results["max_exec"] = section("MAX_EXEC")
    results["disk_free"] = section("DISK")
    results["pillow"] = section("PILLOW")
    results["writable"] = section("WRITABLE")

    return results


def parse_php_size(val):
    """Convert PHP size string (e.g. '100M', '512M', '2G') to MB."""
    val = val.strip().upper()
    if not val or val == '-1':
        return -1  # unlimited
    try:
        if val.endswith('G'):
            return int(val[:-1]) * 1024
        elif val.endswith('M'):
            return int(val[:-1])
        elif val.endswith('K'):
            return int(val[:-1]) // 1024
        else:
            return int(val) // (1024 * 1024)
    except (ValueError, TypeError):
        return 0


def get_largest_test_photo():
    """Find the largest photo in test_photos/ directory."""
    if not os.path.isdir(TEST_PHOTOS_DIR):
        return None, 0

    largest = None
    largest_size = 0
    for f in os.listdir(TEST_PHOTOS_DIR):
        fpath = os.path.join(TEST_PHOTOS_DIR, f)
        if os.path.isfile(fpath) and f.lower().endswith(('.jpg', '.jpeg', '.png')):
            sz = os.path.getsize(fpath)
            if sz > largest_size:
                largest = fpath
                largest_size = sz
    return largest, largest_size


def test_large_photo_upload(job_id):
    """Step 46: Upload largest test photo, verify success."""
    photo_path, photo_size = get_largest_test_photo()
    if not photo_path:
        return False, "no test photos found in %s" % TEST_PHOTOS_DIR, 0

    size_mb = photo_size / (1024 * 1024)

    # Read and base64-encode the photo
    with open(photo_path, "rb") as f:
        image_data = base64.b64encode(f.read()).decode()

    # Use a unique filename to avoid collision
    ext = os.path.splitext(photo_path)[1]
    large_filename = "qa_pipeline_test_large%s" % ext

    # Upload
    data, elapsed, error = test_upload(job_id, filename=large_filename, image_data=image_data)
    if error:
        return False, "upload failed: %s" % error, elapsed

    ok = data and data.get("success") == 1
    detail = "%.1fMB photo, %.1fs upload" % (size_mb, elapsed)
    if not ok:
        detail += ", response=%s" % json.dumps(data)[:100]

    return ok, detail, elapsed, large_filename


def test_large_photo_webp(large_filename, wait=30):
    """Step 47: Verify WebP generated for large photo."""
    if not large_filename:
        return False, "no large photo uploaded"

    # Poll DB for webpfilename (server generates a unique name, not based on original)
    for attempt in range(wait // 5):
        time.sleep(5)
        cmd = (
            "mysql -u %s -p'%s' %s -N -e \""
            "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
            "AND webpfilename IS NOT NULL AND webpfilename != '' "
            "ORDER BY id DESC LIMIT 1;\" 2>/dev/null"
        ) % (DB_USER, DB_PASS, DB_NAME, large_filename)
        out, _, _ = ssh_cmd(cmd, timeout=10)
        webpname = out.strip()
        if webpname:
            # Verify the file exists on disk
            verify_cmd = "ls '%s/webp/%s' 2>/dev/null" % (REMOTE_SCHEDULER, webpname)
            v_out, _, _ = ssh_cmd(verify_cmd, timeout=10)
            if v_out.strip():
                return True, "WebP found after %ds: %s" % ((attempt + 1) * 5, webpname)

    return False, "WebP not generated after %ds" % wait


def cleanup_large_photo(large_filename, job_id):
    """Clean up large photo test artifacts."""
    if not large_filename:
        return
    try:
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": int(job_id),
            "file_name": large_filename,
        }
        requests.post(REMOTE_DELETE_URL, json=payload, timeout=30, verify=False)
    except Exception:
        pass
    # Fallback: direct SSH cleanup
    cmd = (
        "sudo rm -f '%s/%s' && "
        "sudo rm -f %s/webp/*%s* && "
        "mysql -u %s -p'%s' %s -e \"DELETE FROM meter_files WHERE unique_filename='%s';\" 2>/dev/null"
    ) % (REMOTE_SCHEDULER, large_filename, REMOTE_SCHEDULER,
         os.path.splitext(large_filename)[0], DB_USER, DB_PASS, DB_NAME, large_filename)
    ssh_cmd(cmd, timeout=15)


def get_test_photos(max_count=200):
    """Read up to max_count photos from test_photos/ directory, sorted by size."""
    if not os.path.isdir(TEST_PHOTOS_DIR):
        return []

    photos = []
    for f in os.listdir(TEST_PHOTOS_DIR):
        fpath = os.path.join(TEST_PHOTOS_DIR, f)
        if os.path.isfile(fpath) and f.lower().endswith(('.jpg', '.jpeg', '.png')):
            photos.append((fpath, os.path.getsize(fpath)))

    # Sort by size ascending (upload smallest first to catch errors early)
    photos.sort(key=lambda x: x[1])
    return photos[:max_count]


def run_batch_test(job_id, batch_size=200, skip_cleanup=False):
    """Upload N real photos from test_photos/, verify, report, clean up."""
    print(f"\n{Colors.BOLD}{'=' * 60}{Colors.RESET}")
    print(f"{Colors.BOLD}  BATCH UPLOAD TEST — up to {batch_size} photos{Colors.RESET}")
    print(f"{'=' * 60}\n")

    photos = get_test_photos(batch_size)
    if not photos:
        print(f"  {Colors.FAIL}ERROR: No photos found in {TEST_PHOTOS_DIR}{Colors.RESET}")
        return False

    total = len(photos)
    total_bytes = sum(sz for _, sz in photos)
    print(f"  Found {total} photos ({total_bytes / (1024*1024):.1f} MB total)")
    print(f"  Uploading sequentially to job {job_id}...\n")

    # Track results
    successes = 0
    failures = 0
    times = []
    uploaded_filenames = []
    errors_list = []

    start_all = time.time()

    for i, (photo_path, photo_size) in enumerate(photos):
        # Create a unique batch filename
        orig_name = os.path.basename(photo_path)
        batch_name = "%s%04d_%s" % (BATCH_PREFIX, i, orig_name)

        # Read and encode
        try:
            with open(photo_path, "rb") as f:
                image_data = base64.b64encode(f.read()).decode()
        except Exception as e:
            failures += 1
            errors_list.append((batch_name, "read error: %s" % str(e)))
            continue

        # Upload
        data, elapsed, error = test_upload(job_id, filename=batch_name, image_data=image_data)

        if error:
            failures += 1
            errors_list.append((batch_name, error))
        elif data and data.get("success") == 1:
            successes += 1
            times.append(elapsed)
            uploaded_filenames.append(batch_name)
        else:
            failures += 1
            resp_str = json.dumps(data)[:100] if data else "no response"
            errors_list.append((batch_name, resp_str))

        # Progress indicator every 10 photos
        if (i + 1) % 10 == 0 or (i + 1) == total:
            elapsed_total = time.time() - start_all
            rate = (i + 1) / elapsed_total if elapsed_total > 0 else 0
            print(f"  [{i+1:>4}/{total}] {successes} ok, {failures} fail "
                  f"({elapsed_total:.0f}s elapsed, {rate:.1f} photos/s)")

    total_elapsed = time.time() - start_all

    # ── Wait for WebP generation ──
    print(f"\n  Waiting 30s for WebP background generation...")
    time.sleep(30)

    # ── Verify remote artifacts ──
    print(f"\n{Colors.INFO}  Verifying remote artifacts...{Colors.RESET}")

    # Check DB count (unique_filename has qa_batch_ prefix)
    db_cmd = (
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT COUNT(*) FROM meter_files WHERE unique_filename LIKE '%s%%';\" 2>/dev/null"
    ) % (DB_USER, DB_PASS, DB_NAME, BATCH_PREFIX)
    db_out, _, _ = ssh_cmd(db_cmd, timeout=15)
    db_count = int(db_out.strip()) if db_out.strip().isdigit() else 0

    # Check WebP count via DB (webpfilenames are server-generated, not prefixed)
    webp_cmd = (
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT COUNT(*) FROM meter_files WHERE unique_filename LIKE '%s%%' "
        "AND webpfilename IS NOT NULL AND webpfilename != '';\" 2>/dev/null"
    ) % (DB_USER, DB_PASS, DB_NAME, BATCH_PREFIX)
    webp_out, _, _ = ssh_cmd(webp_cmd, timeout=15)
    webp_count = int(webp_out.strip()) if webp_out.strip().isdigit() else 0

    # Check listing count (get total images for this customer via API)
    listing_count = 0
    try:
        payload = {"auth_token": REMOTE_AUTH_TOKEN, "job_id": str(job_id)}
        resp = requests.post(REMOTE_LISTING_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
        if isinstance(data, list):
            # The listing returns ALL customer photos; count the ones from batch
            # by querying the DB for batch webpfilenames and matching
            webpnames_cmd = (
                "mysql -u %s -p'%s' %s -N -e \""
                "SELECT webpfilename FROM meter_files WHERE unique_filename LIKE '%s%%' "
                "AND webpfilename IS NOT NULL AND webpfilename != '';\" 2>/dev/null"
            ) % (DB_USER, DB_PASS, DB_NAME, BATCH_PREFIX)
            wn_out, _, _ = ssh_cmd(webpnames_cmd, timeout=15)
            batch_webpnames = set(wn_out.strip().split("\n")) if wn_out.strip() else set()
            for entry in data:
                if isinstance(entry, dict):
                    link = entry.get("link", "")
                    # Extract img= param from link
                    for wn in batch_webpnames:
                        if wn and wn in link:
                            listing_count += 1
                            break
    except Exception:
        pass

    # ── Report ──
    print(f"\n{Colors.BOLD}  Batch Upload Results{Colors.RESET}")
    print(f"  {'─' * 40}")
    print(f"  Photos attempted:    {total}")
    print(f"  Uploads succeeded:   {successes}")
    print(f"  Uploads failed:      {failures}")
    if times:
        print(f"  Upload time avg:     {sum(times)/len(times):.2f}s")
        print(f"  Upload time min:     {min(times):.2f}s")
        print(f"  Upload time max:     {max(times):.2f}s")
    print(f"  Total elapsed:       {total_elapsed:.1f}s")
    print(f"  DB rows (meter_files): {db_count}/{successes}")
    print(f"  WebP files generated:  {webp_count}/{successes}")
    print(f"  In listing response:   {listing_count}/{successes}")

    if errors_list:
        print(f"\n  {Colors.FAIL}Failures:{Colors.RESET}")
        for fname, err in errors_list[:10]:
            print(f"    {fname}: {err}")
        if len(errors_list) > 10:
            print(f"    ... and {len(errors_list) - 10} more")

    # ── Cleanup ──
    if not skip_cleanup and uploaded_filenames:
        print(f"\n{Colors.INFO}  Cleaning up {len(uploaded_filenames)} batch photos...{Colors.RESET}")
        cleanup_batch(uploaded_filenames, job_id)
        print(f"  Batch cleanup done.")

    success_rate = successes / total if total > 0 else 0
    batch_ok = success_rate >= 0.95 and db_count >= successes * 0.95
    color = Colors.PASS if batch_ok else Colors.FAIL
    print(f"\n  {color}{Colors.BOLD}Batch result: {'PASS' if batch_ok else 'FAIL'} "
          f"({success_rate*100:.1f}% success rate){Colors.RESET}")
    print(f"{'=' * 60}\n")

    return batch_ok


def cleanup_batch(filenames, job_id):
    """Clean up batch test photos via delete.php + SSH fallback."""
    # Delete in chunks of 20 via delete.php
    for i in range(0, len(filenames), 20):
        chunk = filenames[i:i+20]
        try:
            payload = {
                "auth_token": REMOTE_AUTH_TOKEN,
                "job_id": int(job_id),
                "files": chunk,
            }
            requests.post(REMOTE_DELETE_URL, json=payload, timeout=60, verify=False)
        except Exception:
            pass

    # SSH fallback: delete any remaining files and DB rows
    cmd = (
        "cd '%s' && sudo rm -f %s* 2>/dev/null; "
        "cd '%s/webp' && sudo rm -f %s* 2>/dev/null; "
        "mysql -u %s -p'%s' %s -e \""
        "DELETE FROM meter_files WHERE unique_filename LIKE '%s%%';\" 2>/dev/null"
    ) % (REMOTE_SCHEDULER, BATCH_PREFIX,
         REMOTE_SCHEDULER, BATCH_PREFIX,
         DB_USER, DB_PASS, DB_NAME, BATCH_PREFIX)
    ssh_cmd(cmd, timeout=30)


def test_enh021_staging_and_3tier(job_id):
    """Steps 48-52: PHOTO-022/023 tests — staging dir, 2-tier WebP (1280px), DB-only listing.

    Uses the mobile API upload (which writes to uploads/ flat) and verifies:
      48. staging/ directory exists and is writable on remote
      49. Upload file in uploads/ (mobile API — correct behavior)
      50. 2-tier WebP derivatives: thumbs/, webp/ (hi-res removed PHOTO-022)
      51. getimagelisting full_link points to /webp/ (PHOTO-022)
      52. All photos in listing are from meter_files DB (no s3files/scandir)
    """
    results = {}

    # Upload a fresh test file via mobile API
    resp_data, elapsed, error = test_upload(job_id)
    if error:
        results["upload_ok"] = False
        results["upload_error"] = error
        return results

    results["upload_ok"] = resp_data.get("success") == 1
    # upload.php prepends unique ID to filename
    disk_filename = resp_data.get("path", TEST_FILENAME) if resp_data else TEST_FILENAME
    results["disk_filename"] = disk_filename

    # Wait for background WebP generation (3-tier needs time)
    time.sleep(20)

    # Single SSH command to check all artifacts
    # Note: disk_filename is the actual file on disk (with unique prefix from upload.php)
    # TEST_FILENAME is the unique_filename stored in meter_files DB
    cmd = (
        "echo '===STAGING_DIR===' && "
        "stat -c '%%a' '%s/staging' 2>/dev/null || echo 'NOT_FOUND' && "
        "echo '===IN_STAGING===' && "
        "ls '%s/staging/%s' 2>/dev/null || echo 'NOT_FOUND' && "
        "echo '===WEBP_DB===' && "
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
        "AND webpfilename IS NOT NULL AND webpfilename != '' "
        "ORDER BY id DESC LIMIT 1;\" 2>/dev/null && "
        "echo '===THUMBS===' && "
        "WEBPNAME=$(mysql -u %s -p'%s' %s -N -e \""
        "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
        "ORDER BY id DESC LIMIT 1;\" 2>/dev/null) && "
        "[ -n \"$WEBPNAME\" ] && "
        "ls \"%s/thumbs/$WEBPNAME\" 2>/dev/null || echo 'NOT_FOUND' && "
        "echo '===WEBP_FILE===' && "
        "WEBPNAME=$(mysql -u %s -p'%s' %s -N -e \""
        "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
        "ORDER BY id DESC LIMIT 1;\" 2>/dev/null) && "
        "[ -n \"$WEBPNAME\" ] && "
        "ls \"%s/webp/$WEBPNAME\" 2>/dev/null || echo 'NOT_FOUND' && "
        "echo '===END==='"
    ) % (REMOTE_SCHEDULER,
         REMOTE_SCHEDULER, disk_filename,
         DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         REMOTE_SCHEDULER,
         DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         REMOTE_SCHEDULER)

    out, err, rc = ssh_cmd(cmd, timeout=20)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["staging_dir_perms"] = section("STAGING_DIR")
    results["staging_dir_exists"] = section("STAGING_DIR") != "NOT_FOUND"
    results["in_staging"] = section("IN_STAGING") != "NOT_FOUND"
    results["webpname"] = section("WEBP_DB")
    results["has_thumbs"] = section("THUMBS") != "NOT_FOUND"
    results["has_webp"] = section("WEBP_FILE") != "NOT_FOUND"

    # Check listing for full_link → webp (PHOTO-022: hi-res removed)
    try:
        payload = {"auth_token": REMOTE_AUTH_TOKEN, "job_id": str(job_id)}
        resp = requests.post(REMOTE_LISTING_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
        if isinstance(data, list):
            results["listing_count"] = len(data)
            # Find our test entry
            webpname = results.get("webpname", "")
            for entry in data:
                if isinstance(entry, dict):
                    link = entry.get("link", "")
                    if webpname and webpname in link:
                        results["full_link"] = entry.get("full_link", "")
                        break
        else:
            results["listing_count"] = 0
    except Exception as e:
        results["listing_error"] = str(e)
        results["listing_count"] = -1

    return results


# ── Phase 11: Complete Photo Deletion (delete_image.php + delete_local_photo.php) ──

def test_delete_image_batch(job_id):
    """Test delete_image.php batch format — deletes staging + thumbs + webp."""
    # Get file info from DB first
    db_cmd = (
        "mysql -u %s -p'%s' %s -N -e \""
        "SELECT unique_filename, webpfilename, original_filename, folder_path "
        "FROM meter_files WHERE unique_filename='%s' AND job_id=%s "
        "ORDER BY id DESC LIMIT 1;\" 2>/dev/null"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME, job_id)
    db_out, _, _ = ssh_cmd(db_cmd, timeout=10)

    if not db_out or db_out.strip() == "":
        return False, "no meter_files row found for test file", {}

    parts = db_out.strip().split("\t")
    unique = parts[0] if len(parts) > 0 else ""
    webp = parts[1] if len(parts) > 1 else ""
    original = parts[2] if len(parts) > 2 else ""
    folder = parts[3] if len(parts) > 3 else ""

    # Verify files exist before delete
    pre_check = (
        "echo STAGING=$( [ -f '%s/staging/%s' ] && echo YES || echo NO ) && "
        "echo THUMBS=$( [ -f '%s/thumbs/%s' ] && echo YES || echo NO ) && "
        "echo WEBP=$( [ -f '%s/webp/%s' ] && echo YES || echo NO )"
    ) % (REMOTE_SCHEDULER, unique, REMOTE_SCHEDULER, webp, REMOTE_SCHEDULER, webp)
    pre_out, _, _ = ssh_cmd(pre_check, timeout=10)

    # Call delete_image.php with batch format
    try:
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": job_id,
            "files": [{
                "unique_filename": unique,
                "webpfilename": webp,
                "original_filename": original,
                "folder_path": folder,
            }]
        }
        resp = requests.post(REMOTE_DELETE_IMAGE_URL, json=payload, timeout=30, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "request failed: %s" % str(e), {}

    status = data.get("status", "")
    deleted_count = data.get("deleted", 0)

    # Verify files removed
    post_check = (
        "echo STAGING=$( [ -f '%s/staging/%s' ] && echo YES || echo NO ) && "
        "echo THUMBS=$( [ -f '%s/thumbs/%s' ] && echo YES || echo NO ) && "
        "echo WEBP=$( [ -f '%s/webp/%s' ] && echo YES || echo NO )"
    ) % (REMOTE_SCHEDULER, unique, REMOTE_SCHEDULER, webp, REMOTE_SCHEDULER, webp)
    post_out, _, _ = ssh_cmd(post_check, timeout=10)

    staging_gone = "STAGING=NO" in post_out
    thumbs_gone = "THUMBS=NO" in post_out
    webp_gone = "WEBP=NO" in post_out
    all_gone = staging_gone and thumbs_gone and webp_gone

    detail = "status=%s, deleted=%d, staging=%s, thumbs=%s, webp=%s | before: %s" % (
        status, deleted_count,
        "gone" if staging_gone else "STILL EXISTS",
        "gone" if thumbs_gone else "STILL EXISTS",
        "gone" if webp_gone else "STILL EXISTS",
        pre_out.replace("\n", ", "))

    ok = status == "ok" and all_gone
    return ok, detail, {"unique": unique, "webp": webp}


def test_delete_image_auth_rejected():
    """Test delete_image.php rejects invalid auth token."""
    try:
        payload = {"auth_token": "WRONG", "job_id": "1", "files": []}
        resp = requests.post(REMOTE_DELETE_IMAGE_URL, json=payload, timeout=15, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "request failed: %s" % str(e)

    ok = data.get("status") == "error" and "token" in data.get("message", "").lower()
    return ok, "status=%s, message=%s" % (data.get("status"), data.get("message", ""))


def test_delete_image_backward_compat(job_id):
    """Test delete_image.php backward compat with old single 'img' format."""
    try:
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": job_id,
            "img": "nonexistent_file_that_should_not_exist.jpg"
        }
        resp = requests.post(REMOTE_DELETE_IMAGE_URL, json=payload, timeout=15, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "request failed: %s" % str(e)

    # Should return ok with 0 deleted (file doesn't exist, but no error)
    ok = data.get("status") == "ok" and data.get("deleted", -1) == 0
    return ok, "status=%s, deleted=%s" % (data.get("status"), data.get("deleted"))


def test_delete_local_photo_auth():
    """Test delete_local_photo.php rejects invalid auth token."""
    try:
        resp = requests.get(LOCAL_DELETE_URL, params={"auth_token": "WRONG"},
                            timeout=10, verify=False)
        ok = resp.status_code == 403
        return ok, "HTTP %d — %s" % (resp.status_code, resp.text[:100])
    except Exception as e:
        return False, str(e)


def test_delete_local_photo_missing_params():
    """Test delete_local_photo.php requires job_id and unique_filename."""
    try:
        resp = requests.get(LOCAL_DELETE_URL, params={"auth_token": LOCAL_DELETE_AUTH},
                            timeout=10, verify=False)
        ok = resp.status_code == 400
        data = resp.json()
        return ok, "HTTP %d — %s" % (resp.status_code, data.get("message", ""))
    except Exception as e:
        return False, str(e)


def test_delete_local_photo_not_found():
    """Test delete_local_photo.php returns ok with deleted=0 for non-existent photo."""
    try:
        resp = requests.get(LOCAL_DELETE_URL, params={
            "auth_token": LOCAL_DELETE_AUTH,
            "job_id": "999999",
            "unique_filename": "nonexistent.jpg"
        }, timeout=10, verify=False)
        data = resp.json()
        ok = data.get("status") == "ok" and data.get("deleted", -1) == 0
        return ok, "status=%s, deleted=%s, message=%s" % (
            data.get("status"), data.get("deleted"), data.get("message", ""))
    except Exception as e:
        return False, str(e)


def test_delete_local_photo_lookup(job_id, unique_filename):
    """Test delete_local_photo.php looks up local_photos and deletes files."""
    try:
        resp = requests.get(LOCAL_DELETE_URL, params={
            "auth_token": LOCAL_DELETE_AUTH,
            "job_id": job_id,
            "unique_filename": unique_filename
        }, timeout=15, verify=False)
        data = resp.json()
        status = data.get("status", "")
        deleted = data.get("deleted", 0)
        folder_removed = data.get("folder_removed", False)
        # ok if status is ok (deleted >= 0 is fine; file may not have synced yet)
        ok = status == "ok"
        return ok, "status=%s, files_deleted=%s, folder_removed=%s" % (
            status, deleted, folder_removed)
    except Exception as e:
        return False, str(e)


# ── Phase 12: PHOTO-023/024 Error Logging, Cron Fallback, No Archive Copy ──

def test_photo024_no_archive_copy(disk_filename):
    """Verify generate_thumbnails.py does NOT create an archive copy in uploads/ root (PHOTO-024)."""
    cmd = "ls '%s/%s' 2>/dev/null && echo 'FOUND' || echo 'NOT_FOUND'" % (
        REMOTE_SCHEDULER, disk_filename)
    out, _, _ = ssh_cmd(cmd, timeout=10)
    not_found = "NOT_FOUND" in out
    return not_found, "correctly absent from uploads/" if not_found else "ERROR: archive copy still being created"


def test_photo023_webp_log_has_entries():
    """Check webp_generation.log has timestamped entries from recent upload."""
    cmd = "tail -5 %s/logs/webp_generation.log 2>/dev/null || echo 'NO_LOG'" % REMOTE_PHOTOAPI
    out, _, _ = ssh_cmd(cmd, timeout=10)
    if "NO_LOG" in out or not out.strip():
        return False, "webp_generation.log empty or missing"
    has_timestamp = bool(re.search(r'\[\d{4}-\d{2}-\d{2}', out))
    has_ok_or_start = "START" in out or "OK" in out
    return has_timestamp and has_ok_or_start, out.strip().split('\n')[-1][:80]


def test_photo023_fix_missing_webp():
    """Run fix_missing_webp.py and verify it exits cleanly."""
    cmd = (
        "/usr/local/bin/python3.6 %s/fix_missing_webp.py 2>&1; "
        "echo EXIT_CODE=$?"
    ) % REMOTE_PHOTOAPI
    out, _, _ = ssh_cmd(cmd, timeout=60)
    exit_ok = "EXIT_CODE=0" in out
    # Check log for output
    log_cmd = "tail -3 %s/logs/fix_missing_webp.log 2>/dev/null" % REMOTE_PHOTOAPI
    log_out, _, _ = ssh_cmd(log_cmd, timeout=10)
    has_check = "CHECK" in log_out
    detail = log_out.strip().split('\n')[-1][:80] if log_out.strip() else "no log output"
    return exit_ok and has_check, detail


def test_photo024_delete_no_archive_ref(job_id, unique_filename, webpfilename):
    """Verify delete_image.php works without archive copy handling (PHOTO-024)."""
    # Call delete_image.php — should delete staging + thumbs + webp only
    payload = {
        "auth_token": REMOTE_AUTH_TOKEN,
        "job_id": str(job_id),
        "files": [{"unique_filename": unique_filename, "webpfilename": webpfilename}]
    }
    try:
        resp = requests.post(REMOTE_DELETE_IMAGE_URL, json=payload, timeout=15, verify=False)
        data = resp.json()
    except Exception as e:
        return False, "request failed: %s" % str(e)

    ok = data.get("status") == "ok"
    detail = "status=%s, deleted=%s, errors=%s" % (
        data.get("status", "?"), data.get("deleted", "?"), data.get("errors", "?"))
    return ok, detail


def cleanup_remote():
    """Remove test artifacts from remote server."""
    cmd = f"""
    # Clean small test file
    JOBDIR=$(find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' -exec dirname {{}} \\; 2>/dev/null | head -1)
    [ -n "$JOBDIR" ] && sudo rm -rf "$JOBDIR"

    sudo rm -f '{REMOTE_SCHEDULER}/{TEST_FILENAME}'
    sudo rm -f '{REMOTE_SCHEDULER}/staging/{TEST_FILENAME}'
    sudo rm -f {REMOTE_SCHEDULER}/webp/*$(date +%m-%d-%Y)*.webp
    sudo rm -f {REMOTE_SCHEDULER}/thumbs/*$(date +%m-%d-%Y)*.webp
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -e "DELETE FROM meter_files WHERE unique_filename = '{TEST_FILENAME}';" 2>/dev/null
    sudo truncate -s 0 {REMOTE_PHOTOAPI}/logs/sync_to_local.log 2>/dev/null
    for qf in {REMOTE_PHOTOAPI}/queue/*.json; do
        [ -f "$qf" ] && grep -q '{TEST_FILENAME}' "$qf" 2>/dev/null && sudo rm -f "$qf"
    done
    echo 'Remote cleanup done'
    """
    ssh_cmd(cmd, timeout=15)


def cleanup_local():
    """Remove test artifacts from local server (aei-webserv2 via SSH)."""
    try:
        cmd = (
            "for p in $(find '/mnt/dropbox/2025 Customers/' '/mnt/dropbox/2026 Customers/' "
            "'/mnt/dropbox/Uploads/' -maxdepth 8 -name '%s' 2>/dev/null); do "
            "PARENT=$(dirname \"$p\"); rm -rf \"$PARENT\"; done; "
            "echo 'local cleanup done'"
        ) % TEST_FILENAME
        local_ssh_cmd(cmd, timeout=15)
    except Exception:
        pass


# ── Main ───────────────────────────────────────────────────────────

def main():
    parser = argparse.ArgumentParser(description="AEI Photo Upload Pipeline E2E Test")
    parser.add_argument("--job-id", default=DEFAULT_JOB_ID, help="Job ID to test with")
    parser.add_argument("--skip-cleanup", action="store_true", help="Leave test data for inspection")
    parser.add_argument("--remote-only", action="store_true", help="Skip local server checks")
    parser.add_argument("--wait", type=int, default=60,
                        help="Seconds to wait for background processes (default 60)")
    parser.add_argument("--batch", action="store_true",
                        help="Run batch upload test with real photos from test_photos/")
    parser.add_argument("--batch-size", type=int, default=200,
                        help="Number of photos to upload in batch mode (default 200)")
    args = parser.parse_args()

    print(f"\n{Colors.BOLD}AEI Photo Upload Pipeline — E2E Test{Colors.RESET}")
    print(f"{'=' * 60}")
    print(f"  Job ID:    {args.job_id}")
    print(f"  File:      {TEST_FILENAME}")
    print(f"  Remote:    {REMOTE_API_URL}")
    print(f"  Batch:     {'%d photos' % args.batch_size if args.batch else 'no'}")
    print(f"  Timestamp: {time.strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"{'=' * 60}\n")

    passed = 0
    failed = 0
    # Total steps: 63 full, or 55 remote-only (skip steps 9-10, 25-30)
    total = 63 if not args.remote_only else 55
    local_file_found = False  # Tracked for Phase 6 reconciliation test

    # ── Phase 1: Upload small file (Steps 1-2) ──
    print(f"{Colors.INFO}Phase 1 — Upload...{Colors.RESET}")
    resp_data, elapsed, error = test_upload(args.job_id)

    if error:
        result_line(1, "API POST", False, error)
        result_line(2, "Response time", False, "upload failed")
        failed += 2
        print(f"\n{Colors.FAIL}Upload failed — cannot continue.{Colors.RESET}")
        return 1
    else:
        ok = resp_data.get("success") == 1 and resp_data.get("status") == "ok"
        if result_line(1, "API response", ok, json.dumps(resp_data)):
            passed += 1
        else:
            failed += 1

        fast = elapsed < 2.0
        if result_line(2, "Response time", fast, f"{elapsed:.3f}s {'(< 2s)' if fast else '(SLOW > 2s)'}"):
            passed += 1
        else:
            failed += 1

    # ── Wait for background processes ──
    print(f"\n{Colors.INFO}Waiting {args.wait}s for background processes...{Colors.RESET}")
    time.sleep(args.wait)

    # ── Phase 2: Remote checks (Steps 3-8) ──
    print(f"\n{Colors.INFO}Phase 2 — Remote artifacts...{Colors.RESET}")
    # upload.php prepends a unique ID to the filename (e.g., "699ace04_qa_pipeline_test.jpg")
    uploaded_filename = resp_data.get("path", TEST_FILENAME) if resp_data else TEST_FILENAME
    remote = test_remote_checks(args.job_id, uploaded_filename=uploaded_filename)

    # PHOTO-009 removed /mnt/dropbox/ hierarchy writes — files now go to flat scheduler/uploads/ only
    # Steps 3-4 now verify the file is NOT in /mnt/dropbox/ (PHOTO-009 behavior)
    no_dropbox = not bool(remote["file_saved"])
    if result_line(3, "File NOT in /mnt/dropbox/ (PHOTO-009)", no_dropbox,
                   "correct — flat storage only" if no_dropbox else
                   "UNEXPECTED: %s" % remote["file_saved"]):
        passed += 1
    else:
        failed += 1

    no_dropbox_webp = not bool(remote["webp_generated"])
    if result_line(4, "WebP NOT in /mnt/dropbox/ (PHOTO-009)", no_dropbox_webp,
                   "correct — flat storage only" if no_dropbox_webp else
                   "UNEXPECTED: %s" % remote["webp_generated"]):
        passed += 1
    else:
        failed += 1

    # All uploads land in staging/ (mobile PHOTO-017, scheduler PHOTO-021)
    if result_line(5, "File in uploads/staging/", bool(remote["scheduler_copy"]),
                   "present in staging/" if remote["scheduler_copy"] else "NOT FOUND in staging/"):
        passed += 1
    else:
        failed += 1

    if result_line(6, "WebP in scheduler/uploads/webp/", bool(remote["scheduler_webp"]),
                   os.path.basename(remote["scheduler_webp"]) if remote["scheduler_webp"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    if result_line(7, "meter_files DB insert", bool(remote["meter_files"]),
                   remote["meter_files"][:80] if remote["meter_files"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    sync_ok = "OK" in remote.get("sync_log", "") or "200" in remote.get("sync_log", "")
    sync_err = "ERROR" in remote.get("sync_log", "")
    if result_line(8, "sync_to_local.py executed", bool(remote["sync_log"]),
                   remote["sync_log"][:80] if remote["sync_log"] else "NO LOG ENTRY"):
        passed += 1
    else:
        failed += 1

    # ── Fetch job details for local lookup ──
    job_info = {"customer_id": "0", "last_name": "", "first_name": "", "job_date": "", "job_type": ""}
    try:
        info_cmd = (f"mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e \""
                    f"SELECT js.customer_id, cs.last_name, cs.first_name, js.job_date, "
                    f"COALESCE(jt.intials, '') "
                    f"FROM jobs js LEFT JOIN customers cs ON js.customer_id=cs.id "
                    f"LEFT JOIN job_types jt ON js.job_type_id = jt.id "
                    f"WHERE js.id={args.job_id} LIMIT 1;\"")
        info_out, _, _ = ssh_cmd(info_cmd, timeout=10)
        if info_out:
            parts = info_out.split("\t")
            if len(parts) >= 5:
                job_info = {"customer_id": parts[0], "last_name": parts[1],
                            "first_name": parts[2], "job_date": parts[3],
                            "job_type": parts[4]}
    except Exception:
        pass

    # ── Phase 3: Local checks (Steps 9-10) ──
    if not args.remote_only:
        print(f"\n{Colors.INFO}Phase 3 — Local server...{Colors.RESET}")
        local = test_local_checks(
            customer_id=job_info["customer_id"],
            last_name=job_info["last_name"],
            first_name=job_info["first_name"],
            job_date=job_info["job_date"],
        )

        local_file_found = bool(local["file_found"])
        if result_line(9, "File received on local server", local_file_found,
                       local["file_found"].split("/mnt/dropbox/")[-1] if local["file_found"] else "NOT FOUND"):
            passed += 1
        else:
            failed += 1

        thumbs_ok = bool(local["thumbnail"]) and bool(local["mid_thumb"])
        if result_line(10, "Local thumbnails generated", thumbs_ok,
                       "thumbs/ + thumbs/mid/" if thumbs_ok else "MISSING"):
            passed += 1
        else:
            failed += 1

    # ── Phase 4: Async sync enhancement checks (Steps 11-19) ──
    print(f"\n{Colors.INFO}Phase 4 — Async sync enhancement deployment...{Colors.RESET}")
    async_checks = test_async_enhancement_checks()

    # Step 11: No synchronous cURL in upload.php
    curl_count = async_checks["no_curl"].strip().split("\n")[0] if async_checks["no_curl"] else "0"
    no_curl = curl_count == "0"
    if result_line(11, "No synchronous cURL in upload.php", no_curl,
                   "clean (no curl_init/exec/close)" if no_curl else f"{curl_count} cURL references found"):
        passed += 1
    else:
        failed += 1

    # Step 12: Echo JSON is last PHP statement before ?>
    echo_last_line = async_checks["echo_last"]
    echo_ok = "echo" in echo_last_line.lower() and "json_encode" in echo_last_line.lower()
    if result_line(12, "Echo JSON is last statement", echo_ok,
                   "echo json_encode() is final statement" if echo_ok else f"last line: {echo_last_line[:60]}"):
        passed += 1
    else:
        failed += 1

    # Step 13: sync_to_local.py permissions
    sync_perms = async_checks["sync_perms"]
    sync_ok = sync_perms != "NOT_FOUND" and sync_perms.startswith("755")
    if result_line(13, "sync_to_local.py permissions", sync_ok,
                   sync_perms if sync_perms != "NOT_FOUND" else "FILE NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 14: process_retry_queue.py permissions
    retry_perms = async_checks["retry_perms"]
    retry_ok = retry_perms != "NOT_FOUND" and retry_perms.startswith("755")
    if result_line(14, "process_retry_queue.py permissions", retry_ok,
                   retry_perms if retry_perms != "NOT_FOUND" else "FILE NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 15: queue/ and queue/failed/ directories
    queue_dir = async_checks["queue_dir"]
    queue_failed = async_checks["queue_failed"]
    dirs_ok = queue_dir == "777" and queue_failed != "NOT_FOUND"
    detail_parts = []
    detail_parts.append(f"queue/={'ok' if queue_dir == '777' else queue_dir}")
    detail_parts.append(f"failed/={'ok' if queue_failed != 'NOT_FOUND' else 'MISSING'}")
    if result_line(15, "queue/ and queue/failed/ directories", dirs_ok,
                   ", ".join(detail_parts)):
        passed += 1
    else:
        failed += 1

    # Step 16: Cron entry for process_retry_queue.py
    cron = async_checks["cron"]
    cron_ok = "process_retry" in cron and "NOT_FOUND" not in cron
    if result_line(16, "Cron entry for retry processor", cron_ok,
                   cron[:70] if cron_ok else "NOT FOUND in ec2-user or root crontab"):
        passed += 1
    else:
        failed += 1

    # Step 17: logs/ directory writable
    logs_dir = async_checks["logs_dir"]
    logs_ok = logs_dir == "777"
    if result_line(17, "logs/ directory writable", logs_ok,
                   f"permissions={logs_dir}" if logs_dir != "NOT_FOUND" else "DIRECTORY NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 18: Python requests module
    py_req = async_checks["python_requests"]
    py_ok = py_req == "OK"
    if result_line(18, "Python 3.6 requests module", py_ok,
                   "available" if py_ok else "MISSING — pip3.6 install requests"):
        passed += 1
    else:
        failed += 1

    # Step 19: Queue empty after successful upload
    queue_count = async_checks["queue_count"]
    queue_empty = queue_count == "0"
    sync_was_queued = "QUEUED" in remote.get("sync_log", "")
    if sync_was_queued and not queue_empty:
        # Sync failed and was correctly queued — queue is working as designed
        result_line(19, "Queue captured failed sync", True,
                    f"{queue_count} item(s) queued (sync failed — check connectivity)")
        passed += 1
    elif result_line(19, "Queue empty (no false queuing)", queue_empty,
                     "0 items in queue/" if queue_empty else f"{queue_count} items in queue/ (unexpected — sync showed OK)"):
        passed += 1
    else:
        failed += 1

    # ── Phase 5: Photo folder path enhancement (Steps 20-25) ──
    print(f"\n{Colors.INFO}Phase 5 — Photo folder path enhancement...{Colors.RESET}")

    # Steps 20-22: folder_path column checks
    fp_checks = test_folder_path_checks()

    # Step 20: folder_path column exists
    col_info = fp_checks["column_info"]
    col_exists = bool(col_info) and "folder_path" in col_info.lower()
    col_detail = col_info[:80] if col_info else "COLUMN NOT FOUND"
    if result_line(20, "folder_path column in meter_files", col_exists, col_detail):
        passed += 1
    else:
        failed += 1

    # Step 21: folder_path — PHOTO-009 uses flat storage, so folder_path is intentionally empty
    folder_path_val = fp_checks["folder_path"]
    fp_empty = not folder_path_val or folder_path_val == "NULL" or folder_path_val.strip() == ""
    if result_line(21, "folder_path empty (PHOTO-009 flat storage)", fp_empty,
                   "correct — flat storage, no folder path" if fp_empty else
                   "UNEXPECTED: %s" % folder_path_val):
        passed += 1
    else:
        failed += 1

    # Step 22: Verify file is in staging/ (all uploads land there)
    flat_file_ok = bool(remote.get("scheduler_copy"))
    if result_line(22, "File in uploads/staging/ (all uploads)", flat_file_ok,
                   "present in staging/" if flat_file_ok else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Steps 23-24: Image API checks
    api_checks = test_image_api_checks(args.job_id)

    # Step 23: getimagelisting.php returns test image
    listing_data = api_checks.get("listing_data")
    if isinstance(listing_data, list):
        listing_ok = len(listing_data) > 0
        if result_line(23, "getimagelisting.php returns images", listing_ok,
                       f"{len(listing_data)} image(s) returned"):
            passed += 1
        else:
            failed += 1
    else:
        err = api_checks.get("listing_error", "invalid response")
        if result_line(23, "getimagelisting.php returns images", False,
                       f"HTTP {api_checks.get('listing_status', '?')} — {err}"):
            passed += 1
        else:
            failed += 1

    # Step 24: fetch_image.php serves image
    fetch_status = api_checks.get("fetch_status", 0)
    fetch_ct = api_checks.get("fetch_content_type", "")
    fetch_size = api_checks.get("fetch_size", 0)
    fetch_ok = fetch_status == 200 and fetch_ct.startswith("image/")
    if fetch_ok:
        if result_line(24, "fetch_image.php serves image", True,
                       f"{fetch_ct}, {fetch_size:,} bytes"):
            passed += 1
        else:
            failed += 1
    else:
        fetch_err = api_checks.get("fetch_error", f"HTTP {fetch_status}, {fetch_ct}")
        if result_line(24, "fetch_image.php serves image", False, fetch_err):
            passed += 1
        else:
            failed += 1

    # Step 25: Survey path spacing fix (local only)
    if not args.remote_only:
        spacing = test_survey_spacing_fix()

        if not spacing.get("file_exists"):
            if result_line(25, "Survey path spacing fix", False,
                           "uploadlocallat_kuldeep.php not found"):
                passed += 1
            else:
                failed += 1
        else:
            bad = spacing.get("bad_count", -1)
            good = spacing.get("good_count", 0)
            spacing_ok = bad == 0 and good == 2
            if spacing_ok:
                detail = f"0 bad, {good} correct '-S, ' instances"
            else:
                detail = f"{bad} bad '-S,' (no space), {good} correct '-S, ' (expected 2)"
            if result_line(25, "Survey path spacing fix", spacing_ok, detail):
                passed += 1
            else:
                failed += 1

    # ── Phase 6: Reconciliation enhancement (Steps 26-33) ──
    print(f"\n{Colors.INFO}Phase 6 — Reconciliation enhancement (PHOTO-011)...{Colors.RESET}")

    # Steps 26-30: Local endpoint tests (skip in remote-only mode)
    if not args.remote_only:
        cp_checks = test_check_photos_endpoint(job_info, local_file_found)

        # Step 26: check_photos.php exists
        if result_line(26, "check_photos.php exists on local", cp_checks["file_exists"],
                       "/var/www/html/upload/check_photos.php"):
            passed += 1
        else:
            failed += 1

        # Step 27: Auth rejection
        auth_ok = cp_checks.get("auth_rejected", False)
        if result_line(27, "Auth validation rejects bad token", auth_ok,
                       "HTTP 403 returned" if auth_ok else
                       "HTTP %s (expected 403)" % cp_checks.get("auth_status", cp_checks.get("auth_error", "?"))):
            passed += 1
        else:
            failed += 1

        # Step 28: Batch limit
        batch_ok = cp_checks.get("batch_rejected", False)
        if result_line(28, "Batch limit enforced (>200)", batch_ok,
                       "HTTP 400 returned for 201 photos" if batch_ok else
                       "HTTP %s (expected 400)" % cp_checks.get("batch_status", cp_checks.get("batch_error", "?"))):
            passed += 1
        else:
            failed += 1

        # Step 29: Missing file detected
        missing_ok = cp_checks.get("missing_detected", False)
        if result_line(29, "Detects known-missing file", missing_ok,
                       "qa_nonexistent_999.jpg in missing array" if missing_ok else
                       "not reported as missing"):
            passed += 1
        else:
            failed += 1

        # Step 30: Test file check matches reality
        test_found = cp_checks.get("test_file_found", False)
        expected_found = cp_checks.get("local_file_expected", False)
        step30_ok = test_found == expected_found
        if expected_found:
            detail = "correctly reported as found" if step30_ok else "reported missing but file exists locally"
        else:
            detail = "correctly reported as missing (sync did not reach local)" if step30_ok else "reported found but file not on local"
        if result_line(30, "Test file check matches local state", step30_ok, detail):
            passed += 1
        else:
            failed += 1

    # Steps 31-33: Remote reconciliation deployment (always run)
    recon = test_reconciliation_remote()

    # Step 31: Constants present
    const_count = recon.get("constants_count", "0")
    const_ok = const_count == "3"
    if result_line(31, "Reconciliation constants in retry processor", const_ok,
                   "%s/3 constants found" % const_count):
        passed += 1
    else:
        failed += 1

    # Step 32: Functions present
    func_count = recon.get("functions_count", "0")
    func_ok = func_count == "4"
    if result_line(32, "Reconciliation functions in retry processor", func_ok,
                   "%s/4 functions found" % func_count):
        passed += 1
    else:
        failed += 1

    # Step 33: .reconcile_started marker exists
    marker = recon.get("marker", "NOT_FOUND")
    marker_ok = marker != "NOT_FOUND" and len(marker) >= 10
    if result_line(33, ".reconcile_started marker on remote", marker_ok,
                   "started %s" % marker if marker_ok else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # ── Phase 7: PHOTO-009 Unified Storage (Steps 34-39) ──
    print(f"\n{Colors.INFO}Phase 7 — PHOTO-009 Unified Storage...{Colors.RESET}")

    # Clean up ALL prior test rows before re-upload (avoid orphan DB rows affecting tests)
    pre_clean_cmd = (
        "mysql -u %s -p'%s' %s -e \""
        "DELETE FROM meter_files WHERE unique_filename='%s';\" 2>/dev/null && "
        "sudo rm -f '%s/%s' && "
        "sudo rm -f '%s/staging/%s' && "
        "sudo rm -f %s/webp/*qa_pipeline_test* 2>/dev/null && "
        "sudo rm -f %s/thumbs/*qa_pipeline_test* 2>/dev/null && "
        "echo 'pre-clean done'"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         REMOTE_SCHEDULER, TEST_FILENAME,
         REMOTE_SCHEDULER, TEST_FILENAME,
         REMOTE_SCHEDULER, REMOTE_SCHEDULER)
    ssh_cmd(pre_clean_cmd, timeout=15)

    # Re-upload fresh test file for PHOTO-009 delete testing
    print(f"  (Re-uploading test file for PHOTO-009 delete tests...)")
    resp_data2, _, err2 = test_upload(args.job_id)
    if err2:
        print(f"  {Colors.WARN}Re-upload failed: {err2} — PHOTO-009 tests may be affected{Colors.RESET}")
    time.sleep(5)  # Brief wait for WebP generation

    # Step 34: Upload does NOT write to /mnt/dropbox/
    no_dropbox, detail34 = test_enh009_no_dropbox_write()
    if result_line(34, "Upload skips /mnt/dropbox/ hierarchy (PHOTO-009)", no_dropbox, detail34):
        passed += 1
    else:
        failed += 1

    # Step 35: getimagelisting.php structure check (PHOTO-022: full_link → webp/)
    listing_ok, detail35, listing_entry = test_enh009_listing_structure(args.job_id)
    if result_line(35, "getimagelisting structure + full_link → webp/", listing_ok, detail35):
        passed += 1
    else:
        failed += 1

    # Step 36: fetch_image.php serves WebP
    fetch_ok, detail36, _ = test_enh009_fetch_image(listing_entry)
    if result_line(36, "fetch_image.php serves WebP image", fetch_ok, detail36):
        passed += 1
    else:
        failed += 1

    # Save webpfilename before delete (for post-delete verification)
    test_webpname = get_webpfilename_for_test()

    # Step 37: delete.php deletes test photo
    del_ok, detail37, _ = test_enh009_delete(args.job_id)
    if result_line(37, "delete.php removes file + WebP + DB row", del_ok, detail37):
        passed += 1
    else:
        failed += 1

    # Step 38: delete.php rejects bad auth
    auth_ok, detail38 = test_enh009_delete_auth_rejected(args.job_id)
    if result_line(38, "delete.php rejects invalid auth token", auth_ok, detail38):
        passed += 1
    else:
        failed += 1

    # Step 39: listing empty after delete
    empty_ok, detail39 = test_enh009_listing_empty_after_delete(args.job_id, webpname=test_webpname)
    if result_line(39, "Listing has no test file after delete", empty_ok, detail39):
        passed += 1
    else:
        failed += 1

    # ── Phase 8: Remote Server Capacity Audit (Steps 40-45) ──
    print(f"\n{Colors.INFO}Phase 8 — Remote server capacity audit...{Colors.RESET}")
    capacity = test_capacity_audit()

    # Step 40: post_max_size >= 50M
    post_max_mb = parse_php_size(capacity.get("post_max", "0"))
    post_ok = post_max_mb >= 50 or post_max_mb == -1
    if result_line(40, "PHP post_max_size >= 50M", post_ok,
                   "%s (%dMB)" % (capacity.get("post_max", "?"), post_max_mb)):
        passed += 1
    else:
        failed += 1

    # Step 41: memory_limit >= 128M (128M proven sufficient for 10MB+ photos)
    mem_mb = parse_php_size(capacity.get("memory", "0"))
    mem_ok = mem_mb >= 128 or mem_mb == -1
    if result_line(41, "PHP memory_limit >= 128M", mem_ok,
                   "%s (%dMB)" % (capacity.get("memory", "?"), mem_mb)):
        passed += 1
    else:
        failed += 1

    # Step 42: max_execution_time 0 or >= 300
    max_exec = capacity.get("max_exec", "0").strip()
    try:
        max_exec_val = int(max_exec)
    except (ValueError, TypeError):
        max_exec_val = -999
    exec_ok = max_exec_val == 0 or max_exec_val >= 300
    if result_line(42, "PHP max_execution_time unlimited or >= 300", exec_ok,
                   "%s%s" % (max_exec, " (unlimited)" if max_exec_val == 0 else "s")):
        passed += 1
    else:
        failed += 1

    # Step 43: Disk space > 10GB free
    disk_str = capacity.get("disk_free", "0G").replace("G", "")
    try:
        disk_gb = int(disk_str)
    except (ValueError, TypeError):
        disk_gb = 0
    disk_ok = disk_gb > 10
    if result_line(43, "Disk space > 10GB free", disk_ok,
                   "%dGB free" % disk_gb):
        passed += 1
    else:
        failed += 1

    # Step 44: Pillow available
    pillow_ok = capacity.get("pillow", "") == "OK"
    if result_line(44, "Python 3.6 Pillow module available", pillow_ok,
                   "available" if pillow_ok else "MISSING"):
        passed += 1
    else:
        failed += 1

    # Step 45: uploads/, staging/, webp/, thumbs/ writable (hi-res removed PHOTO-022)
    writable_str = capacity.get("writable", "")
    uploads_writable = "uploads=777" in writable_str or "uploads=775" in writable_str
    staging_writable = "staging=777" in writable_str or "staging=775" in writable_str
    webp_writable = "webp=777" in writable_str or "webp=775" in writable_str
    thumbs_writable = "thumbs=777" in writable_str or "thumbs=775" in writable_str
    dirs_writable = uploads_writable and staging_writable and webp_writable and thumbs_writable
    if result_line(45, "All upload dirs writable (uploads/staging/webp/thumbs)", dirs_writable,
                   writable_str.replace("\n", ", ")):
        passed += 1
    else:
        failed += 1

    # ── Phase 9: Large Photo Upload (Steps 46-47) ──
    large_filename = None
    if os.path.isdir(TEST_PHOTOS_DIR):
        print(f"\n{Colors.INFO}Phase 9 — Large photo upload...{Colors.RESET}")

        # Step 46: Upload largest photo
        large_result = test_large_photo_upload(args.job_id)
        if len(large_result) == 4:
            large_ok, detail46, elapsed46, large_filename = large_result
        else:
            large_ok, detail46, elapsed46 = large_result[0], large_result[1], large_result[2]
            large_filename = None
        if result_line(46, "Upload largest test photo", large_ok, detail46):
            passed += 1
        else:
            failed += 1

        # Step 47: WebP generated for large photo
        if large_filename:
            webp_ok, detail47 = test_large_photo_webp(large_filename)
        else:
            webp_ok, detail47 = False, "no large photo to check"
        if result_line(47, "WebP generated for large photo", webp_ok, detail47):
            passed += 1
        else:
            failed += 1

        # Clean up large photo
        if not args.skip_cleanup and large_filename:
            cleanup_large_photo(large_filename, args.job_id)
    else:
        print(f"\n{Colors.WARN}Phase 9 — Skipped (no test_photos/ directory){Colors.RESET}")
        result_line(46, "Upload largest test photo", False, "test_photos/ not found")
        result_line(47, "WebP generated for large photo", False, "skipped")
        failed += 2

    # ── Phase 10: PHOTO-022/023 Staging Direct + 2-Tier WebP 1280px (Steps 48-52) ──
    print(f"\n{Colors.INFO}Phase 10 — PHOTO-022/023: Staging direct + 2-tier WebP (1280px)...{Colors.RESET}")

    # Pre-clean for PHOTO-022/023 tests
    enh021_clean = (
        "mysql -u %s -p'%s' %s -e \""
        "DELETE FROM meter_files WHERE unique_filename='%s';\" 2>/dev/null && "
        "sudo rm -f '%s/staging/%s' && "
        "sudo rm -f '%s/%s' && "
        "echo 'enh021 pre-clean done'"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         REMOTE_SCHEDULER, TEST_FILENAME,
         REMOTE_SCHEDULER, TEST_FILENAME)
    ssh_cmd(enh021_clean, timeout=15)

    enh021 = test_enh021_staging_and_3tier(args.job_id)

    if not enh021.get("upload_ok"):
        print(f"  {Colors.WARN}Upload failed for PHOTO-022/023 tests: {enh021.get('upload_error', 'unknown')}{Colors.RESET}")
        for step in range(48, 53):
            result_line(step, "PHOTO-022/023 test (upload failed)", False, "upload failed")
            failed += 1
    else:
        # Step 48: staging/ directory exists and writable
        staging_ok = enh021["staging_dir_exists"]
        if result_line(48, "staging/ directory exists + writable (PHOTO-021)", staging_ok,
                       "permissions=%s" % enh021.get("staging_dir_perms", "?") if staging_ok else "NOT FOUND"):
            passed += 1
        else:
            failed += 1

        # Step 49: Mobile API upload → staging/ (PHOTO-017 changed upload.php to use staging/)
        if result_line(49, "Mobile API upload in staging/ (PHOTO-017)", enh021["in_staging"],
                       "present in staging/" if enh021["in_staging"] else "NOT in staging/"):
            passed += 1
        else:
            failed += 1

        # Step 50: WebP derivatives (thumbs + webp required; hi-res removed PHOTO-022)
        tier2_ok = enh021["has_thumbs"] and enh021["has_webp"]
        tier2_detail = "thumbs=%s, webp=%s" % (
            "ok" if enh021["has_thumbs"] else "MISSING",
            "ok" if enh021["has_webp"] else "MISSING")
        if result_line(50, "2-tier WebP derivatives (thumbs+webp, 1280px PHOTO-023)", tier2_ok, tier2_detail):
            passed += 1
        else:
            failed += 1

        # Step 51: full_link → /webp/ (PHOTO-022: hi-res removed)
        full_link = enh021.get("full_link", "")
        fl_ok = "/webp/" in full_link
        if result_line(51, "full_link points to /webp/ (PHOTO-022)", fl_ok,
                       full_link[:80] if full_link else "not found in listing"):
            passed += 1
        else:
            failed += 1

        # Step 52: Listing is DB-only (all entries have expected structure)
        listing_count = enh021.get("listing_count", 0)
        listing_ok = listing_count > 0
        if result_line(52, "Listing returns DB-sourced photos", listing_ok,
                       "%d photos from meter_files" % listing_count if listing_ok else "empty or error"):
            passed += 1
        else:
            failed += 1

    # ── Phase 11: Complete Photo Deletion (Steps 53-59) ──
    print(f"\n{Colors.INFO}Phase 11 — Complete Photo Deletion (delete_image.php + delete_local_photo.php)...{Colors.RESET}")

    # Step 53: delete_image.php rejects invalid auth
    del_auth_ok, del_auth_detail = test_delete_image_auth_rejected()
    if result_line(53, "delete_image.php rejects bad auth token", del_auth_ok, del_auth_detail):
        passed += 1
    else:
        failed += 1

    # Step 54: delete_image.php backward compat (old single 'img' format)
    del_compat_ok, del_compat_detail = test_delete_image_backward_compat(args.job_id)
    if result_line(54, "delete_image.php backward compat (single img)", del_compat_ok, del_compat_detail):
        passed += 1
    else:
        failed += 1

    # Step 55: delete_image.php batch delete (staging + thumbs + webp)
    del_batch_ok, del_batch_detail, del_batch_info = test_delete_image_batch(args.job_id)
    if result_line(55, "delete_image.php batch delete (staging+thumbs+webp)", del_batch_ok, del_batch_detail):
        passed += 1
    else:
        failed += 1

    # Step 56: delete_local_photo.php rejects bad auth
    loc_auth_ok, loc_auth_detail = test_delete_local_photo_auth()
    if result_line(56, "delete_local_photo.php rejects bad auth", loc_auth_ok, loc_auth_detail):
        passed += 1
    else:
        failed += 1

    # Step 57: delete_local_photo.php requires params
    loc_params_ok, loc_params_detail = test_delete_local_photo_missing_params()
    if result_line(57, "delete_local_photo.php requires job_id+unique_filename", loc_params_ok, loc_params_detail):
        passed += 1
    else:
        failed += 1

    # Step 58: delete_local_photo.php returns ok for non-existent photo
    loc_notfound_ok, loc_notfound_detail = test_delete_local_photo_not_found()
    if result_line(58, "delete_local_photo.php handles non-existent photo", loc_notfound_ok, loc_notfound_detail):
        passed += 1
    else:
        failed += 1

    # Step 59: delete_local_photo.php local_photos lookup + delete
    test_unique = del_batch_info.get("unique", TEST_FILENAME)
    loc_lookup_ok, loc_lookup_detail = test_delete_local_photo_lookup(args.job_id, test_unique)
    if result_line(59, "delete_local_photo.php local_photos lookup", loc_lookup_ok, loc_lookup_detail):
        passed += 1
    else:
        failed += 1

    # ── Phase 12: PHOTO-023/024 Error Logging, Cron Fallback, No Archive Copy (Steps 60-63) ──
    print(f"\n{Colors.INFO}Phase 12 — PHOTO-023/024: Error logging, cron fallback, no archive copy...{Colors.RESET}")

    # Upload a fresh file for PHOTO-023 tests
    p12_clean = (
        "mysql -u %s -p'%s' %s -e \""
        "DELETE FROM meter_files WHERE unique_filename='%s';\" 2>/dev/null && "
        "sudo rm -f '%s/staging/%s' '%s/%s' && "
        "sudo truncate -s 0 %s/logs/webp_generation.log 2>/dev/null && "
        "echo 'p12 pre-clean done'"
    ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME,
         REMOTE_SCHEDULER, TEST_FILENAME, REMOTE_SCHEDULER, TEST_FILENAME,
         REMOTE_PHOTOAPI)
    ssh_cmd(p12_clean, timeout=15)

    p12_resp, p12_elapsed, p12_error = test_upload(args.job_id)
    p12_disk = p12_resp.get("path", TEST_FILENAME) if p12_resp else TEST_FILENAME
    time.sleep(15)  # Wait for background generate_thumbnails.py

    if p12_error or not p12_resp or p12_resp.get("success") != 1:
        print(f"  {Colors.WARN}Upload failed for PHOTO-023/024 tests: {p12_error or 'bad response'}{Colors.RESET}")
        for step in range(60, 64):
            result_line(step, "PHOTO-023/024 test (upload failed)", False, "upload failed")
            failed += 1
    else:
        # Step 60: No archive copy in uploads/ root (PHOTO-024)
        arch_ok, arch_detail = test_photo024_no_archive_copy(p12_disk)
        if result_line(60, "No archive copy in uploads/ root (PHOTO-024)", arch_ok, arch_detail):
            passed += 1
        else:
            failed += 1

        # Step 61: webp_generation.log has timestamped entries
        log_ok, log_detail = test_photo023_webp_log_has_entries()
        if result_line(61, "webp_generation.log has timestamped entries", log_ok, log_detail):
            passed += 1
        else:
            failed += 1

        # Step 62: fix_missing_webp.py runs without error
        cron_ok, cron_detail = test_photo023_fix_missing_webp()
        if result_line(62, "fix_missing_webp.py runs clean (cron fallback)", cron_ok, cron_detail):
            passed += 1
        else:
            failed += 1

        # Step 63: delete_image.php works without archive copy (PHOTO-024)
        # Get webpfilename from DB for the test file
        webp_cmd = (
            "mysql -u %s -p'%s' %s -N -e \""
            "SELECT webpfilename FROM meter_files WHERE unique_filename='%s' "
            "ORDER BY id DESC LIMIT 1;\" 2>/dev/null"
        ) % (DB_USER, DB_PASS, DB_NAME, TEST_FILENAME)
        webp_out, _, _ = ssh_cmd(webp_cmd, timeout=10)
        p12_webp = webp_out.strip()

        del_ok, del_detail = test_photo024_delete_no_archive_ref(
            args.job_id, TEST_FILENAME, p12_webp)
        if result_line(63, "delete_image.php works without archive (PHOTO-024)", del_ok, del_detail):
            passed += 1
        else:
            failed += 1

    # ── Summary ──
    print(f"\n{'=' * 60}")
    all_pass = failed == 0
    color = Colors.PASS if all_pass else Colors.FAIL
    print(f"  {color}{Colors.BOLD}Result: {passed}/{total} passed, {failed} failed{Colors.RESET}")
    if sync_err and not sync_ok:
        print(f"  {Colors.WARN}Note: sync_to_local.py ran but got an error (local server may be unreachable){Colors.RESET}")
    print(f"{'=' * 60}")

    # ── Cleanup ──
    if not args.skip_cleanup:
        print(f"\n{Colors.INFO}Cleaning up test data...{Colors.RESET}")
        cleanup_remote()
        if not args.remote_only:
            cleanup_local()
        print("  Done.")
    else:
        print(f"\n{Colors.WARN}Skipping cleanup (--skip-cleanup). Test data left for inspection.{Colors.RESET}")

    # ── Batch test (after main test) ──
    batch_ok = True
    if args.batch:
        batch_ok = run_batch_test(
            args.job_id,
            batch_size=args.batch_size,
            skip_cleanup=args.skip_cleanup,
        )

    print()
    if args.batch:
        return 0 if (all_pass and batch_ok) else 1
    return 0 if all_pass else 1


if __name__ == "__main__":
    sys.exit(main())
