# PHOTO-026: Chunked Resumable Uploads

## Status: PLANNING

## Problem

The current photo upload pipeline sends entire files as base64-encoded JSON in a single HTTP POST. This creates three reliability issues for field technicians on cellular networks:

1. **All-or-nothing transfers** — A 10MB photo encoded as ~13MB base64 must complete in one request. If the connection drops at 90%, the entire upload is lost and must restart from zero.
2. **Memory pressure** — Base64 encoding inflates file size by 33%. A 10MB JPEG becomes ~13MB in the JSON payload, plus ~10MB decoded server-side = ~23MB peak memory per upload.
3. **No progress visibility** — The Flutter app shows "uploading..." with no byte-level progress. Users can't tell if an upload is stalled or progressing.

PHOTO-018 (Background Upload Queue) already solved the UI-blocking and retry problems with a SQLite-backed queue. This enhancement builds on that foundation by replacing the transfer protocol itself.

## Reference Implementation

The **premeasure** project (`/var/www/html/PROJECTS/premeasure/`) implements a production-ready chunked upload system with:
- Adaptive chunk sizing (512KB–2MB based on connection quality)
- 4-phase protocol (init → chunk → finalize → status)
- Server-side `.part` file assembly with exclusive file locking
- Deterministic file IDs (SHA-1) enabling resume across sessions
- Abandoned chunk cleanup (48-hour TTL)
- Per-chunk retry with exponential backoff

This plan adapts the premeasure approach for the Photo API's constraints (PHP 5.3 server, Flutter client, existing staging → WebP pipeline).

## Dependencies

| ID | Name | Status | Relationship |
|----|------|--------|-------------|
| PHOTO-018 | Background Upload Queue | Planning | Phase 1 (SQLite queue) is prerequisite; this replaces Phase 2's multipart approach with chunked protocol |
| PHOTO-017 | Derivatives-Only Remote + Staging | Deployed | Defines the staging/ pipeline this feeds into |
| PHOTO-009 | Unified Storage | Deployed | DB-only listing, webpfilename convention |

## Constraints

| Constraint | Impact |
|-----------|--------|
| **PHP 5.3.29** on production | No namespaces, no `[]` array syntax, no `finally`, no generators, no `json_encode` options |
| **No `crypto.subtle`** on server | File ID generation must be PHP 5.3-compatible (use `sha1()` directly) |
| **Existing pipeline** | Chunks must ultimately produce a file in `uploads/staging/` that feeds the existing `generate_thumbnails.py` + `sync_to_local.py` pipeline |
| **Backward compatibility** | Current base64 JSON uploads must continue working during rollout (Flutter app versions in the field) |
| **Flutter (Dart)** | Chunking logic is Dart, not JavaScript — premeasure JS is reference only |

---

## Architecture Overview

```
Flutter App                          Production Server (PHP 5.3)
-----------                          --------------------------

1. Camera capture
   ↓
2. Generate file_id
   (sha1 of name+size+mtime)
   ↓
3. POST /photoapi/upload_chunked.php
   action=init                  →    Create .meta.json + allocate session
                                ←    {next_chunk: 0} or {next_chunk: N} (resume)
   ↓
4. Split file into chunks
   (adaptive: 512KB–2MB)
   ↓
5. POST action=chunk             →   Append to .part file (flock exclusive)
   chunk_index=0, data=...      ←    {next_chunk: 1, received_bytes: N}
   ↓ (repeat for each chunk)
6. POST action=chunk             →   Append chunk
   chunk_index=N, data=...      ←    {next_chunk: N+1, received_bytes: M}
   ↓
7. POST action=finalize          →   Verify size, move .part → staging/
                                     Trigger generate_thumbnails.py
                                     Trigger sync_to_local.py
                                     Insert meter_files DB row
                                ←    {success: 1, path: "webpfilename.webp"}
```

---

## Phase 1: Server-Side Chunk Handler (PHP 5.3)

### New File: `photoapi/upload_chunked.php`

Separate endpoint from existing `upload.php` (which continues to serve legacy base64 clients).

#### Actions

**`init`** — Establish or resume an upload session

```
POST parameters:
  action       = "init"
  auth_token   = "aei@89806849"
  file_id      = 40-char hex (sha1)
  file_name    = original filename
  file_size    = total bytes (integer)
  total_chunks = ceil(file_size / chunk_size)
  chunk_size   = bytes per chunk
  job_id       = scheduler job ID

Response (new):
  {"status":"ok", "file_id":"abc...", "next_chunk":0, "received_bytes":0, "resumed":false}

Response (resume):
  {"status":"ok", "file_id":"abc...", "next_chunk":3, "received_bytes":3145728, "resumed":true}
```

Server logic:
1. Validate auth token
2. Validate file_id format: `/^[a-f0-9]{40}$/`
3. Check for existing `.meta.json` in chunks directory
4. If exists → return current state (resume)
5. If new → create `.meta.json` with metadata, return `next_chunk: 0`

**`chunk`** — Receive one chunk

```
POST parameters:
  action      = "chunk"
  auth_token  = "aei@89806849"
  file_id     = 40-char hex
  chunk_index = 0-based integer
  chunk_data  = raw binary (multipart file upload)

Response:
  {"status":"ok", "next_chunk":4, "received_bytes":4194304, "complete":false}
```

Server logic:
1. Validate auth + file_id
2. Load `.meta.json`
3. Verify chunk_index is sequential (`== count(received_chunks)`)
4. If duplicate (already received), return current state (idempotent)
5. If out-of-order, return `{"status":"error", "retryable":true, "next_chunk":N}`
6. Open `.part` file with `flock(LOCK_EX)`, append chunk data, flush, unlock
7. Update `.meta.json`: add chunk_index to received_chunks, update received_bytes and last_activity
8. Return updated state

**`finalize`** — Assemble and feed into pipeline

```
POST parameters:
  action     = "finalize"
  auth_token = "aei@89806849"
  file_id    = 40-char hex

Response:
  {"success":1, "status":"ok", "message":"Image uploaded", "path":"webpfilename.webp"}
```

Server logic:
1. Validate auth + file_id
2. Load `.meta.json`, verify all chunks received
3. Verify `.part` file size matches expected `file_size`
4. Look up job metadata from DB (customer name, job type, date — same as current upload.php)
5. Generate `webpfilename` (same convention as upload.php)
6. Move `.part` → `uploads/staging/{uniqid}_{filename}`
7. Insert `meter_files` DB row (same fields as upload.php)
8. Trigger `generate_thumbnails.py` via `nohup` (same as upload.php)
9. Trigger `sync_to_local.py` via `nohup` (same as upload.php)
10. Delete `.meta.json`
11. Return success response (same format as upload.php)

**`status`** — Query upload state (for resume after app restart)

```
POST parameters:
  action     = "status"
  auth_token = "aei@89806849"
  file_id    = 40-char hex

Response:
  {"status":"ok", "next_chunk":3, "received_bytes":3145728, "total_chunks":10, "complete":false}
```

#### Chunk Storage

```
/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/.chunks/
  ├── {file_id}.meta.json     ← Upload session metadata
  └── {file_id}.part          ← Accumulated binary data
```

The `.chunks/` directory needs:
- `chmod 777` (Apache writes as `apache` user)
- `.htaccess` with `Deny from all` (prevent direct HTTP access)

#### Meta File Structure (PHP 5.3 compatible)

```json
{
  "file_id": "a1b2c3d4e5...",
  "original_name": "IMG_1234.jpg",
  "total_size": 8388608,
  "total_chunks": 8,
  "chunk_size": 1048576,
  "received_bytes": 3145728,
  "received_chunks": [0, 1, 2],
  "job_id": "12345",
  "created_at": "2026-02-23 15:30:45",
  "last_activity": "2026-02-23 15:31:20"
}
```

#### Abandoned Chunk Cleanup

Add to existing photoapi cron or create new cron job:

```php
// cleanup_chunks.php — run every 6 hours
$maxAge = 48 * 3600; // 48 hours
$chunksDir = '/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/.chunks/';
$now = time();
foreach (glob($chunksDir . '*.meta.json') as $metaFile) {
    if (($now - filemtime($metaFile)) > $maxAge) {
        $fileId = basename($metaFile, '.meta.json');
        @unlink($metaFile);
        @unlink($chunksDir . $fileId . '.part');
    }
}
```

#### PHP 5.3 Compatibility Notes

| Pattern | Use | Avoid |
|---------|-----|-------|
| `array()` | Yes | `[]` array shorthand |
| `json_encode($data)` | Yes | `json_encode($data, JSON_PRETTY_PRINT)` (flag added in 5.4) |
| `flock()` + `fopen('ab')` | Yes | `file_put_contents` with `LOCK_EX` + `FILE_APPEND` (unreliable in 5.3) |
| String concatenation | Yes | Namespaces, traits, `finally` |
| `sha1()` | Yes | `hash()` with exotic algorithms |
| `$_FILES['chunk_data']` | Yes | `php://input` for multipart (need `move_uploaded_file`) |

---

## Phase 2: Flutter Client (Dart)

### Modify: `lib/services/upload_queue_service.dart`

Replace the current `_processItem()` method's base64 JSON upload with chunked protocol.

#### File ID Generation (Dart)

```dart
import 'dart:convert';
import 'package:crypto/crypto.dart'; // add to pubspec.yaml

String generateFileId(String filePath, int fileSize, int lastModified) {
  final input = '$filePath|$fileSize|$lastModified';
  final bytes = utf8.encode(input);
  final digest = sha1.convert(bytes);
  return digest.toString(); // 40-char hex
}
```

#### Adaptive Chunk Sizing (Dart)

```dart
import 'package:connectivity_plus/connectivity_plus.dart';

Map<String, dynamic> getUploadParams() {
  // connectivity_plus already a dependency (PHOTO-018)
  final result = await Connectivity().checkConnectivity();

  if (result == ConnectivityResult.mobile) {
    // Conservative for cellular
    return {'chunkSize': 512 * 1024, 'concurrent': 1}; // 512KB
  } else if (result == ConnectivityResult.wifi) {
    return {'chunkSize': 2 * 1024 * 1024, 'concurrent': 1}; // 2MB
  }
  return {'chunkSize': 1024 * 1024, 'concurrent': 1}; // 1MB default
}
```

Note: Concurrency stays at 1 (sequential uploads) since the queue already handles ordering. Chunk concurrency within a single file is unnecessary — sequential chunks are simpler and the server appends sequentially.

#### Upload Flow (Dart pseudocode)

```dart
Future<void> _processItem(UploadQueueItem item) async {
  final file = File(item.filePath);
  final fileSize = await file.length();
  final fileId = generateFileId(item.filePath, fileSize, item.lastModified);
  final params = getUploadParams();
  final chunkSize = params['chunkSize'];
  final totalChunks = (fileSize / chunkSize).ceil();

  // 1. INIT — establish or resume session
  final initResponse = await _postJson(uploadChunkedUrl, {
    'action': 'init',
    'auth_token': authToken,
    'file_id': fileId,
    'file_name': item.fileName,
    'file_size': fileSize,
    'total_chunks': totalChunks,
    'chunk_size': chunkSize,
    'job_id': item.jobId,
  });

  int nextChunk = initResponse['next_chunk'];
  int receivedBytes = initResponse['received_bytes'];

  // Update progress for resumed uploads
  _updateProgress(item.id, receivedBytes, fileSize);

  // 2. CHUNK — send each chunk with retry
  final raf = await file.open(mode: FileMode.read);
  try {
    for (int i = nextChunk; i < totalChunks; i++) {
      await raf.setPosition(i * chunkSize);
      final remaining = fileSize - (i * chunkSize);
      final readSize = remaining < chunkSize ? remaining : chunkSize;
      final chunkData = await raf.read(readSize);

      final chunkResponse = await _postChunk(
        fileId, i, chunkData,
        maxRetries: 5,
      );

      receivedBytes = chunkResponse['received_bytes'];
      _updateProgress(item.id, receivedBytes, fileSize);
    }
  } finally {
    await raf.close();
  }

  // 3. FINALIZE — trigger server pipeline
  final finalResponse = await _postJson(uploadChunkedUrl, {
    'action': 'finalize',
    'auth_token': authToken,
    'file_id': fileId,
  });

  return finalResponse;
}
```

#### Chunk POST (multipart, not base64)

```dart
Future<Map> _postChunk(String fileId, int chunkIndex, List<int> data, {int maxRetries = 5}) async {
  int attempts = 0;
  while (attempts < maxRetries) {
    try {
      final request = http.MultipartRequest('POST', Uri.parse(uploadChunkedUrl));
      request.fields['action'] = 'chunk';
      request.fields['auth_token'] = authToken;
      request.fields['file_id'] = fileId;
      request.fields['chunk_index'] = chunkIndex.toString();
      request.files.add(http.MultipartFile.fromBytes('chunk_data', data, filename: 'chunk'));

      final response = await request.send().timeout(Duration(minutes: 2));
      final body = await response.stream.bytesToString();
      final json = jsonDecode(body);

      if (json['status'] == 'ok') return json;
      if (json['retryable'] == false) throw UploadError(json['message']);

      // Retryable error — backoff and retry
      attempts++;
      await Future.delayed(Duration(seconds: pow(2, attempts).toInt()));
    } on TimeoutException {
      attempts++;
      await Future.delayed(Duration(seconds: pow(2, attempts).toInt()));
    }
  }
  throw UploadError('Max retries exceeded for chunk $chunkIndex');
}
```

### New Dependency: `pubspec.yaml`

```yaml
dependencies:
  crypto: ^3.0.3  # For SHA-1 file ID generation
  # connectivity_plus already present from PHOTO-018
  # sqflite already present from PHOTO-018
```

### Progress Reporting

The existing `UploadQueueService` exposes streams. Add byte-level progress:

```dart
// New stream for byte-level progress
final _progressController = StreamController<UploadProgress>.broadcast();
Stream<UploadProgress> get onProgress => _progressController.stream;

class UploadProgress {
  final int itemId;
  final int bytesUploaded;
  final int totalBytes;
  double get fraction => totalBytes > 0 ? bytesUploaded / totalBytes : 0;
}
```

The `clock_status_banner.dart` or a new `upload_progress_widget.dart` can subscribe to this stream and show a per-file progress bar.

---

## Phase 3: Resume Across App Restart

The SQLite queue from PHOTO-018 already persists items across restarts. Add:

1. Store `file_id` in the queue item's SQLite row when init succeeds
2. On app restart, for any item with status `uploading`:
   - Call `action=status` with stored `file_id`
   - Server returns `next_chunk` → resume from there
   - If server returns "not found" (cleaned up after 48h), restart from chunk 0

#### SQLite Schema Addition

```sql
ALTER TABLE upload_queue ADD COLUMN file_id TEXT DEFAULT NULL;
ALTER TABLE upload_queue ADD COLUMN chunks_sent INTEGER DEFAULT 0;
ALTER TABLE upload_queue ADD COLUMN total_chunks INTEGER DEFAULT 0;
```

---

## Migration & Backward Compatibility

| Client Version | Server Endpoint | Protocol |
|----------------|----------------|----------|
| Current Flutter (pre-PHOTO-018) | `upload.php` | Base64 JSON |
| PHOTO-018 Flutter (queue, no chunking) | `upload.php` | Base64 JSON (queued) |
| PHOTO-026 Flutter (chunked) | `upload_chunked.php` | Multipart chunked |

- `upload.php` remains unchanged and continues serving older app versions
- `upload_chunked.php` is a new endpoint — no risk to existing clients
- Flutter app can feature-flag: try chunked endpoint first, fall back to base64 if server returns 404 (older server)

---

## File Summary

### New Files

| File | Location | Language | Purpose |
|------|----------|----------|---------|
| `upload_chunked.php` | `photoapi/` (remote) | PHP 5.3 | Chunked upload handler (init/chunk/finalize/status) |
| `cleanup_chunks.php` | `photoapi/` (remote) | PHP 5.3 | Cron job to clean abandoned uploads |
| `.chunks/.htaccess` | `photoapi/.chunks/` (remote) | Apache | `Deny from all` |

### Modified Files

| File | Location | Language | Changes |
|------|----------|----------|---------|
| `upload_queue_service.dart` | `lib/services/` (Flutter) | Dart | Replace base64 upload with chunked protocol |
| `upload_queue_item.dart` | `lib/models/` (Flutter) | Dart | Add file_id, chunks_sent, total_chunks fields |
| `pubspec.yaml` | Flutter root | YAML | Add `crypto` dependency |

### Unchanged Files

| File | Why |
|------|-----|
| `upload.php` | Backward compatibility for older app versions |
| `generate_thumbnails.py` | Called by finalize action (same as before) |
| `sync_to_local.py` | Called by finalize action (same as before) |
| `meter_files` table | Same INSERT, same fields |

---

## Performance Comparison

| Metric | Current (Base64 JSON) | PHOTO-026 (Chunked) |
|--------|----------------------|---------------------|
| **10MB photo over good WiFi** | ~2s (single POST) | ~3s (init + 5 chunks + finalize) |
| **10MB photo over 3G** | ~30s or fail | ~35s total, resumable at any point |
| **10MB photo, connection drop at 80%** | **Restart from 0** (30s wasted) | **Resume from chunk 4/5** (~6s to finish) |
| **Memory (server)** | ~23MB peak | ~2MB peak (one chunk at a time) |
| **Memory (client)** | ~13MB (full base64 in RAM) | ~2MB (one chunk in RAM) |
| **Progress visibility** | None | Per-chunk byte-accurate |
| **Concurrent uploads** | All-or-nothing per file | Chunk-level granularity |

### When Chunking Adds Overhead

For small files (<2MB), the init/finalize round-trips add ~200ms latency. The chunk threshold should be set at **2MB** — files below this size continue using the existing base64 path (routed automatically by the Flutter queue service).

---

## Testing Plan

### Server Testing (upload_chunked.php)

```bash
# 1. Test init — new upload
curl -X POST https://aeihawaii.com/photoapi/upload_chunked.php \
  -d "action=init&auth_token=aei@89806849&file_id=test123abc...&file_name=test.jpg&file_size=5242880&total_chunks=5&chunk_size=1048576&job_id=12345"

# 2. Test chunk upload
curl -X POST https://aeihawaii.com/photoapi/upload_chunked.php \
  -F "action=chunk" -F "auth_token=aei@89806849" -F "file_id=test123abc..." \
  -F "chunk_index=0" -F "chunk_data=@/tmp/test_chunk_0.bin"

# 3. Test resume (call init again with same file_id)
# Should return next_chunk > 0

# 4. Test finalize
curl -X POST https://aeihawaii.com/photoapi/upload_chunked.php \
  -d "action=finalize&auth_token=aei@89806849&file_id=test123abc..."

# 5. Verify staging file exists
ls -la /var/www/vhosts/.../scheduler/uploads/staging/

# 6. Verify meter_files row created
mysql -e "SELECT * FROM meter_files ORDER BY id DESC LIMIT 1"

# 7. Verify WebP generated (wait 5s for background process)
ls -la /var/www/vhosts/.../scheduler/uploads/webp/
ls -la /var/www/vhosts/.../scheduler/uploads/thumbs/
```

### Flutter Testing

1. **Normal upload** — Select photo, verify chunks sent, verify photo appears in grid
2. **Kill app mid-upload** — Reopen app, verify upload resumes from correct chunk
3. **Airplane mode mid-upload** — Enable airplane mode, wait, disable, verify resume
4. **Small file (<2MB)** — Verify falls back to base64 path
5. **Large file (>10MB)** — Verify chunks are 2MB on WiFi, 512KB on cellular
6. **Duplicate photo** — Verify dedup prevents re-upload (existing PHOTO-018 behavior)

### Rollback

- **Server**: Delete `upload_chunked.php` and `.chunks/` directory. No impact on `upload.php`.
- **Flutter**: Revert `upload_queue_service.dart` to base64 path. Queue items already uploaded are unaffected.

---

## Implementation Order

| Step | Scope | Effort | Prerequisite |
|------|-------|--------|-------------|
| 1. Create `upload_chunked.php` | Server (PHP 5.3) | 1 day | None |
| 2. Create `.chunks/` directory + `.htaccess` | Server config | 10 min | Step 1 |
| 3. Test with curl | Server validation | 1 hour | Step 2 |
| 4. Create `cleanup_chunks.php` + cron | Server | 30 min | Step 1 |
| 5. Implement PHOTO-018 Phase 1 (SQLite queue) | Flutter | 2-3 days | None (parallel with steps 1-4) |
| 6. Add chunked upload to queue service | Flutter | 1-2 days | Steps 3 + 5 |
| 7. Add progress UI widget | Flutter | 0.5 day | Step 6 |
| 8. Integration testing | Both | 1 day | Steps 6 + 7 |

**Total estimate**: 5-7 days (server + Flutter, some parallelizable)

---

## Open Questions

1. **Chunk threshold**: 2MB recommended (premeasure uses 5MB). For phone photos averaging 5-8MB, 2MB means 3-4 chunks per photo — good granularity for resume without excessive round-trips. Confirm?
2. **Auth token evolution**: Current hardcoded token `aei@89806849` works but is a known security gap (PHOTO-013). Should PHOTO-026 maintain the same token for consistency, or should auth be upgraded as part of this work?
3. **PHOTO-018 sequencing**: Should PHOTO-018 Phase 1 (SQLite queue) be implemented first as a standalone enhancement, then PHOTO-026 adds chunking on top? Or merge both into a single implementation?
