#!/usr/bin/env python3
"""
Remote Image Processor for AEI Photo System

Generates WebP derivatives from a source JPEG/PNG on the remote server.
Called by upload.php, preupload.php, and phototab.php after photo upload.

Produces two tiers:
    1. Thumb    — 200x200, Q70 WebP              → {uploads_dir}/thumbs/{webpfilename}
    2. Standard — 1280px max dimension, Q75 WebP  → {uploads_dir}/webp/{webpfilename}

The source image should be the highest-quality copy available:
  - Mobile flow:     staging/{filename} (full-size original)
  - Scheduler flow:  staging/{filename} (copy made before CI in-place resize)

Full-size archival is handled by sync_to_local.py (separate process).
No copies are written to uploads/ root.

Usage:
    python3.6 generate_thumbnails.py <source_image> <webpfilename> <uploads_dir>

Arguments:
    source_image:  Full path to the source image (typically staging/{filename})
    webpfilename:  The webpfilename from meter_files (e.g. "SmithJohnAC-I02-18-2026_IMGabc123.webp")
    uploads_dir:   Base uploads directory (e.g. /var/www/.../scheduler/uploads)

Output:
    {uploads_dir}/thumbs/{webpfilename}  (200x200 WebP, Q70)
    {uploads_dir}/webp/{webpfilename}    (1280px max WebP, Q75)

Idempotent: skips any output file that already exists.
Atomic writes: temp file + rename prevents serving partial images.
Python 3.6 compatible.

Author: AEI Photo System
Version: 7.0 (PHOTO-024: remove archive copy, archival via sync_to_local.py only)
Date: 2026-02-23
"""

import os
import sys
import signal
import tempfile
import time
from PIL import Image, ImageOps

# Configuration
THUMBNAIL_SIZE = (200, 200)
MAX_LARGE = 1280
WEBP_LARGE_QUALITY = 75
WEBP_THUMB_QUALITY = 70
TIMEOUT_SECONDS = 120


def timeout_handler(signum, frame):
    print("Error: Image processing timed out after %d seconds" % TIMEOUT_SECONDS, file=sys.stderr)
    sys.exit(1)


def load_image(file_path):
    """Load image, extract EXIF data, and apply EXIF rotation.
    Returns (img, exif_bytes) where exif_bytes may be None."""
    img = Image.open(file_path)
    # Extract EXIF before transpose (which strips orientation tag)
    exif_data = img.info.get('exif')
    img = ImageOps.exif_transpose(img)
    # Convert to RGB if needed
    if img.mode in ('RGBA', 'LA', 'P'):
        bg = Image.new('RGB', img.size, (255, 255, 255))
        if img.mode == 'P':
            img = img.convert('RGBA')
        bg.paste(img, mask=img.split()[-1] if 'A' in img.mode else None)
        img = bg
    elif img.mode != 'RGB':
        img = img.convert('RGB')
    return img, exif_data


def resize_max_dimension(img, max_dim):
    """Resize so largest dimension is max_dim. No upscale."""
    width, height = img.size
    if width <= max_dim and height <= max_dim:
        return img.copy()

    if width > height:
        new_width = max_dim
        new_height = int(height * (max_dim / float(width)))
    else:
        new_height = max_dim
        new_width = int(width * (max_dim / float(height)))

    return img.resize((new_width, new_height), Image.LANCZOS)


def atomic_save(img, output_path, quality, exif=None):
    """Save image atomically: write to temp file, then rename.
    Preserves EXIF data from source if provided."""
    out_dir = os.path.dirname(output_path)
    fd, tmp_path = tempfile.mkstemp(suffix='.webp', dir=out_dir)
    try:
        os.close(fd)
        save_kwargs = {'quality': quality}
        if exif:
            save_kwargs['exif'] = exif
        img.save(tmp_path, 'WEBP', **save_kwargs)
        os.chmod(tmp_path, 0o644)
        os.rename(tmp_path, output_path)
    except Exception:
        if os.path.isfile(tmp_path):
            os.unlink(tmp_path)
        raise


def generate(source_path, webpfilename, uploads_dir):
    """
    Generate 2-tier WebP derivatives from source image.
    Returns dict of created files (path or None if skipped).
    """
    # Output paths
    thumb_path = os.path.join(uploads_dir, 'thumbs', webpfilename)
    large_path = os.path.join(uploads_dir, 'webp', webpfilename)

    # Check what already exists
    thumb_exists = os.path.isfile(thumb_path)
    large_exists = os.path.isfile(large_path)

    if thumb_exists and large_exists:
        return {'thumb': None, 'large': None}

    # Load image once (with EXIF data for passthrough)
    img, exif_data = load_image(source_path)

    # Ensure directories exist
    for d in [os.path.join(uploads_dir, 'thumbs'),
              os.path.join(uploads_dir, 'webp')]:
        if not os.path.isdir(d):
            os.makedirs(d, mode=0o777)

    created = {'thumb': None, 'large': None}

    # Generate thumbnail (200x200, Q70)
    if not thumb_exists:
        thumb = img.copy()
        thumb.thumbnail(THUMBNAIL_SIZE, Image.LANCZOS)
        atomic_save(thumb, thumb_path, WEBP_THUMB_QUALITY, exif=exif_data)
        created['thumb'] = thumb_path

    # Generate standard (1280px max dimension, Q75)
    if not large_exists:
        large = resize_max_dimension(img, MAX_LARGE)
        atomic_save(large, large_path, WEBP_LARGE_QUALITY, exif=exif_data)
        created['large'] = large_path

    return created


def _timestamp():
    """ISO-8601 timestamp for log output."""
    return time.strftime('%Y-%m-%d %H:%M:%S')


if __name__ == "__main__":
    if len(sys.argv) < 4:
        print("Usage: python3.6 generate_thumbnails.py <source_image> <webpfilename> <uploads_dir>")
        sys.exit(1)

    source_image = sys.argv[1]
    webpfilename = sys.argv[2]
    uploads_dir = sys.argv[3]

    print("[%s] START source=%s webp=%s" % (_timestamp(), source_image, webpfilename))

    if not os.path.isfile(source_image):
        print("[%s] ERROR Source file not found: %s" % (_timestamp(), source_image), file=sys.stderr)
        sys.exit(1)

    signal.signal(signal.SIGALRM, timeout_handler)
    signal.alarm(TIMEOUT_SECONDS)

    t0 = time.time()
    try:
        result = generate(source_image, webpfilename, uploads_dir)
        elapsed = time.time() - t0
        created = [k for k, v in result.items() if v]
        skipped = [k for k, v in result.items() if v is None]
        print("[%s] OK created=%s skipped=%s elapsed=%.1fs" % (
            _timestamp(), ','.join(created) or 'none', ','.join(skipped) or 'none', elapsed))
    except Exception as e:
        elapsed = time.time() - t0
        print("[%s] FAIL error=%s elapsed=%.1fs" % (_timestamp(), str(e), elapsed), file=sys.stderr)
        sys.exit(1)

    signal.alarm(0)
    sys.exit(0)
