# PHOTO-018: Background Upload Queue

## Status: Planning (2026-02-19)

## Problem

The Flutter mobile app blocks the UI during photo uploads. Users pick photos, then wait while each file is base64-encoded in memory and sent as a JSON POST body — sometimes minutes for multi-photo uploads. They cannot navigate away, take more photos, or do any other work. Large photos (3-5MB JPEG) encoded as base64 (~6.7MB) cause RAM spikes; batch uploads multiply this.

## Solution: Two-Phase Rollout

### Phase 1 — Foreground Upload Queue (app-only, no server changes)
Non-blocking upload queue as a Dart singleton service. User picks photos → added to persistent queue → can immediately navigate away. Queue processes items sequentially in the foreground. Uploads pause when app is backgrounded, resume on reopen.

### Phase 2 — True Background Upload (app + server change)
Add multipart/form-data support to `upload.php` so iOS NSURLSession / Android WorkManager can stream files directly from disk. Eliminates base64 RAM overhead entirely. Uploads continue when app is minimized/closed.

## Architecture Decisions

- **WebP size**: 1024px Q80 standard tier + 2048px Q82 hi-res tier + 200x200 Q70 thumbnails. All three tiers generated by `generate_thumbnails.py` v4.0 on production.
- **No compression before upload**: Send original full-res JPEG to server. Server writes to `staging/`, generates 3-tier WebP via Pillow/LANCZOS, syncs full-size to local for archival. This preserves the PHOTO-017 pipeline.
- **Don't enable `_compressImage()`** for upload payloads — only for local UI previews if needed. The local archive must remain original quality.
- **Queue persistence**: `sqflite` (SQLite) recommended over `shared_preferences` (brittle for large queues, no atomic updates, 1MB practical limit on some platforms).
- **File copying on enqueue**: When a user picks a photo via `image_picker`, the OS provides a temporary path that may be cleaned up before the queue processes it. On enqueue, immediately copy the file to `getApplicationDocumentsDirectory()/upload_queue/` and store that permanent path in SQLite. Delete the local copy only after successful upload.

## Dependencies
- PHOTO-017 (Derivatives-Only Remote + Staging) — deployed, defines the server pipeline

## Prerequisites (before Phase 1)
- **WebP quality verified at Q80**: `generate_thumbnails.py` and `backfill_thumbnails.py` use MAX_LARGE=1024, Q80 for the webp/ tier. Already deployed to remote (2026-02-21).

---

## Phase 1: Foreground Upload Queue

### Goal
Stop user frustration. Let them pick photos and move on. Queue uploads run in the foreground, one at a time, with retry and persistence.

### New Dependencies (pubspec.yaml)

```yaml
dependencies:
  sqflite: ^2.3.0        # Persistent upload queue (SQLite)
  # Already present: connectivity_plus, shared_preferences, path_provider, http
```

### New Files

| File | Purpose |
|------|---------|
| `lib/services/upload_queue_service.dart` | Singleton queue manager — the core of Phase 1 |
| `lib/models/upload_queue_item.dart` | Queue item model + SQLite schema |
| `lib/widgets/upload_status_widget.dart` | Small UI widget showing queue progress |

### Modified Files

| File | Change |
|------|--------|
| `lib/pages/job_photos_full_view.dart` | Replace blocking `_uploadToServerMany` with queue enqueue |
| `lib/pages/job_details_page.dart` | Replace blocking `_uploadToServer` with queue enqueue |
| `lib/main.dart` | Initialize `UploadQueueService` singleton on app start |
| `pubspec.yaml` | Add `sqflite` dependency |

### Queue Item Model (`upload_queue_item.dart`)

```dart
enum UploadStatus { queued, uploading, uploaded, failed }

class UploadQueueItem {
  final int? id;            // SQLite auto-increment
  final String filePath;    // Absolute path to picked image on device
  final int jobId;          // Server job ID
  final String fileName;    // Original filename
  final String dedupKey;    // filePath + fileSize + lastModified (dedup)
  final UploadStatus status;
  final int retryCount;     // 0-5
  final DateTime createdAt;
  final String? errorMessage;

  // SQLite table definition
  static const String createTableSQL = '''
    CREATE TABLE IF NOT EXISTS upload_queue (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      file_path TEXT NOT NULL,
      job_id INTEGER NOT NULL,
      file_name TEXT NOT NULL,
      dedup_key TEXT NOT NULL UNIQUE,
      status TEXT NOT NULL DEFAULT 'queued',
      retry_count INTEGER NOT NULL DEFAULT 0,
      created_at TEXT NOT NULL,
      error_message TEXT
    )
  ''';
}
```

### Upload Queue Service (`upload_queue_service.dart`)

Core singleton that manages the persistent queue:

```
┌──────────────────────────────────────────────┐
│            UploadQueueService                 │
│                                              │
│  SQLite DB ←→ Queue State                    │
│                                              │
│  enqueue(filePath, jobId, fileName)          │
│    → dedupe check (dedupKey)                 │
│    → copy file to app docs dir (see below)   │
│    → INSERT into upload_queue (permanent path)│
│    → _processNext() if idle                  │
│                                              │
│  _processNext()                              │
│    → SELECT first WHERE status = 'queued'    │
│    → UPDATE status = 'uploading'             │
│    → read file bytes (one file at a time)    │
│    → base64 encode                           │
│    → POST to upload.php (existing JSON API)  │
│    → on success: UPDATE status = 'uploaded'  │
│    → on failure: retry logic (see below)     │
│    → _processNext() (loop)                   │
│                                              │
│  Notifiers:                                  │
│    queueCount (ValueNotifier<int>)           │
│    currentItem (ValueNotifier<String?>)      │
│    isProcessing (ValueNotifier<bool>)        │
│                                              │
│  Lifecycle:                                  │
│    init() — open DB, resume pending items    │
│    pause() — stop processing                 │
│    resume() — restart processing             │
│    dispose() — close DB                      │
│                                              │
└──────────────────────────────────────────────┘
```

#### Retry Logic

```
On upload failure:
  if retryCount < 5:
    UPDATE status = 'queued', retryCount += 1
    wait: 2^retryCount seconds (2, 4, 8, 16, 32s)
    _processNext()
  else:
    UPDATE status = 'failed', errorMessage = reason
    _processNext()  // skip to next item
```

#### Token Expiry Handling

```
On HTTP 401 response:
  1. Pause queue
  2. Read current token from TokenProvider
  3. If token changed since last attempt → retry immediately (another request refreshed it)
  4. If no refresh mechanism → set all 'uploading' items back to 'queued'
  5. Notify UI: "Session expired — please log in to resume uploads"
  6. On next successful login → resume()
```

**Implementation note:** Step 5 requires a global event mechanism to pop a login modal over whatever screen the user is currently viewing. Two approaches:
- **NavigatorKey approach**: Store a `GlobalKey<NavigatorState>` in a service locator; `UploadQueueService` uses it to push a login route
- **Stream/ChangeNotifier approach**: `UploadQueueService` exposes an `onAuthRequired` stream; `main.dart`'s `MaterialApp` builder listens and shows a login dialog

The stream approach is cleaner since it doesn't couple the queue service to navigation.

#### Deduplication

```
dedupKey = "${filePath}:${fileSize}:${lastModified.millisecondsSinceEpoch}"

On enqueue:
  INSERT OR IGNORE into upload_queue (... dedup_key ...)
  → UNIQUE constraint prevents duplicates
  → No need to compute file hash (expensive for large files)
```

#### File Copying on Enqueue (Critical Mobile Quirk)

On iOS and Android, `image_picker` returns a temporary file path that the OS may clean up at any time (cache eviction, reboot). If the queue hasn't processed the item yet, the file disappears.

```dart
Future<bool> enqueue({required String filePath, required int jobId, required String fileName}) async {
  // 1. Compute dedupKey from original file
  final file = File(filePath);
  if (!file.existsSync()) return false;
  final stat = file.statSync();
  final dedupKey = '$filePath:${stat.size}:${stat.modified.millisecondsSinceEpoch}';

  // 2. Check dedup before copying (avoid wasting I/O)
  if (await _existsInQueue(dedupKey)) return false;

  // 3. Copy to permanent app storage
  final appDir = await getApplicationDocumentsDirectory();
  final queueDir = Directory('${appDir.path}/upload_queue');
  if (!queueDir.existsSync()) queueDir.createSync(recursive: true);
  final permanentPath = '${queueDir.path}/${DateTime.now().millisecondsSinceEpoch}_$fileName';
  await file.copy(permanentPath);

  // 4. INSERT with permanent path
  await _db.insert('upload_queue', {
    'file_path': permanentPath,  // permanent, not temp
    'job_id': jobId,
    'file_name': fileName,
    'dedup_key': dedupKey,
    'status': 'queued',
    'created_at': DateTime.now().toIso8601String(),
  });

  // 5. Start processing if idle
  _processNext();
  return true;
}
```

After successful upload, delete the permanent copy:
```dart
// In _processNext(), after successful upload:
File(item.filePath).deleteSync();  // clean up app docs copy
```

#### Network Awareness

```dart
// Use connectivity_plus (already installed)
ConnectivityResult.none → pause()
ConnectivityResult.wifi/mobile → resume()
```

#### App Lifecycle

```dart
// In main.dart or a WidgetsBindingObserver
void didChangeAppLifecycleState(AppLifecycleState state) {
  if (state == AppLifecycleState.paused) {
    UploadQueueService.instance.pause();
  } else if (state == AppLifecycleState.resumed) {
    UploadQueueService.instance.resume();
  }
}
```

### UI Changes

#### job_photos_full_view.dart — Replace blocking upload

```dart
// BEFORE (blocking):
Future<void> _uploadToServerMany(List<File> files) async {
  setState(() { _isUploading = true; ... });
  for (final file in files) {
    await _uploadSingle(file);  // blocks UI
  }
  await _fetchPhotos();
  setState(() => _isUploading = false);
}

// AFTER (non-blocking):
Future<void> _uploadToServerMany(List<File> files) async {
  int enqueued = 0;
  for (final file in files) {
    final added = await UploadQueueService.instance.enqueue(
      filePath: file.path,
      jobId: widget.jobId,
      fileName: file.path.split('/').last,
    );
    if (added) enqueued++;
  }
  if (mounted) {
    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(content: Text('$enqueued photo(s) queued for upload')),
    );
  }
  // Optionally: listen for upload completions and refresh grid
}
```

#### job_details_page.dart — Same pattern

Replace `_uploadToServer(file)` call with `UploadQueueService.instance.enqueue(...)`.

#### Upload Status Widget

Small persistent indicator shown in the app bar or as a banner:

```
┌─────────────────────────────────────┐
│  ↑ Uploading 3 of 7 photos...      │
│  ████████░░░░░░░░  (43%)           │
└─────────────────────────────────────┘
```

Or as a compact badge on the app bar:
```
[↑ 4]  ← 4 photos queued
```

Shows:
- Number of items in queue
- Current upload progress (item N of M)
- Tap to expand: list of queued/failed items with retry button
- Failed items shown in red with "Retry" / "Remove" actions

### Post-Upload Grid Refresh

When an upload completes for the currently-viewed job, auto-refresh the photo grid:

```dart
// In UploadQueueService:
final onUploadComplete = StreamController<int>.broadcast(); // emits jobId

// In job_photos_full_view.dart initState:
_uploadSub = UploadQueueService.instance.onUploadComplete.listen((jobId) {
  if (jobId == widget.jobId) _fetchPhotos();
});
```

---

## Phase 2: True Background Upload (multipart/form-data)

### Goal
Uploads continue when app is minimized or closed. Eliminate base64 RAM overhead.

### New Dependencies (pubspec.yaml)

```yaml
dependencies:
  background_downloader: ^9.5.0  # Native background transfers (iOS/Android)
  # sqflite stays for queue state
```

### Server Change: `upload.php` accepts BOTH formats

```php
<?php
header('Content-Type: application/json');

$uploadsBase = '/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads';
$stagingDir = $uploadsBase . '/staging';
if (!is_dir($stagingDir)) { @mkdir($stagingDir, 0777, true); }

// === Detect upload format ===
if (isset($_FILES['image_data'])) {
    // --- Phase 2: Multipart form upload ---
    // Use move_uploaded_file() — zero RAM overhead, OS moves the temp file directly
    $filename_insert = basename($_POST['file_name'] ?: $_FILES['image_data']['name']);
    $job_id_input = $_POST['job_id'];
    $auth_token = $_POST['auth_token'];

    // Validate token early before moving file
    if ($auth_token !== 'aei@89806849') {
        echo json_encode(array("status" => "error", "message" => "Invalid token"));
        exit;
    }

    // Move uploaded file directly to staging/ — no file_get_contents, no RAM spike
    // Prefix with uniqid to prevent concurrent overwrites (multiple field workers
    // may upload "image.jpg" simultaneously)
    $staging_name = uniqid() . '_' . $filename_insert;
    $stagingPath = $stagingDir . '/' . $staging_name;
    if (!move_uploaded_file($_FILES['image_data']['tmp_name'], $stagingPath)) {
        echo json_encode(array("status" => "error", "message" => "Failed to save uploaded file"));
        exit;
    }
} else {
    // --- Legacy: JSON body with base64 ---
    $data = json_decode(file_get_contents("php://input"), true);
    if (!isset($data['auth_token'], $data['file_name'], $data['image_data'], $data['job_id'])) {
        echo json_encode(array("status" => "error", "message" => "Missing fields"));
        exit;
    }

    $auth_token = $data['auth_token'];
    if ($auth_token !== 'aei@89806849') {
        echo json_encode(array("status" => "error", "message" => "Invalid token"));
        exit;
    }

    $imageData = base64_decode($data['image_data']);
    if ($imageData === false) {
        echo json_encode(array("status" => "error", "message" => "Invalid base64"));
        exit;
    }
    $filename_insert = basename($data['file_name']);
    $job_id_input = $data['job_id'];

    // Same uniqid prefix for legacy path — prevents concurrent overwrites
    $staging_name = uniqid() . '_' . $filename_insert;
    $stagingPath = $stagingDir . '/' . $staging_name;
    if (!file_put_contents($stagingPath, $imageData)) {
        echo json_encode(array("status" => "error", "message" => "Failed to save image"));
        exit;
    }
}

// === From here on, both paths have $stagingPath, $filename_insert, $job_id_input ===
// ... rest of upload.php unchanged (DB lookup, WebP generation, DB insert, sync) ...
```

**Key optimization**: The multipart path uses `move_uploaded_file()` instead of `file_get_contents()`. PHP's temp upload file is moved directly to `staging/` — zero PHP memory overhead for the image data. The legacy JSON/base64 path still uses `file_put_contents()` (must decode base64 in memory).

**Multipart contract:**
```
POST /photoapi/upload.php
Content-Type: multipart/form-data

Fields:
  auth_token: "aei@89806849"
  job_id: "12345"
  file_name: "photo.jpg"  (optional — falls back to uploaded filename)

File part:
  image_data: <binary JPEG file>
```

### Flutter Changes for Phase 2

Replace the queue's `_processItem()` to use `background_downloader`:

```dart
import 'package:background_downloader/background_downloader.dart';

// Instead of http.post with base64 JSON:
final task = UploadTask(
  url: 'https://aeihawaii.com/photoapi/upload.php',
  filename: item.fileName,
  fields: {
    'auth_token': 'aei@89806849',
    'job_id': item.jobId.toString(),
    'file_name': item.fileName,
  },
  fileField: 'image_data',
  updates: Updates.statusAndProgress,
);

final result = await FileDownloader().upload(task);
```

This streams the file directly from disk — no base64 encoding, no full-file-in-memory. The OS handles the transfer natively, including when the app is backgrounded.

### Phase 2 Backward Compatibility

- Legacy JSON endpoint remains functional (scheduler phototab.php, preupload.php still use it)
- Mobile app Phase 1 users (older app versions) still work via JSON path
- Phase 2 mobile app uses multipart for new uploads
- Response format is identical regardless of upload method

---

## File Inventory

### Phase 1 (App Only)

| Action | File | Location |
|--------|------|----------|
| **New** | `upload_queue_service.dart` | `lib/services/` |
| **New** | `upload_queue_item.dart` | `lib/models/` |
| **New** | `upload_status_widget.dart` | `lib/widgets/` |
| Modified | `job_photos_full_view.dart` | `lib/pages/` — enqueue instead of block |
| Modified | `job_details_page.dart` | `lib/pages/` — enqueue instead of block |
| Modified | `main.dart` | `lib/` — init queue service, lifecycle observer |
| Modified | `pubspec.yaml` | Add `sqflite` |

### Phase 2 (App + Server)

| Action | File | Location |
|--------|------|----------|
| Modified | `upload.php` | `photoapi/` — accept multipart + legacy JSON |
| Modified | `upload_queue_service.dart` | `lib/services/` — switch to background_downloader |
| Modified | `pubspec.yaml` | Add `background_downloader` |

---

## Verification

### Phase 1
1. Pick 5 photos → snackbar shows "5 photos queued" → user can navigate away immediately
2. Return to photos tab → grid updates as uploads complete
3. Kill app mid-queue → reopen → queue resumes from where it left off
4. Turn airplane mode on → queue pauses → turn off → queue resumes
5. Pick same 5 photos again → no duplicates added (dedupe)
6. Simulate 401 → queue pauses → log in again → queue resumes
7. 5 consecutive failures → item marked as `failed` → next item proceeds
8. Check server: staging/ has full-size, webp/ has 1024px, thumbs/ has 200x200

### Phase 2
9. Pick photo, immediately background the app → upload completes in background
10. `curl -F 'image_data=@photo.jpg' -F 'auth_token=aei@89806849' -F 'job_id=12345' upload.php` → same response as JSON
11. Legacy JSON POST still works (backward compat)
12. Check: no base64 in memory during multipart upload (profile with DevTools)

---

## Risk Assessment

| Risk | Mitigation |
|------|------------|
| SQLite DB corruption on crash | WAL mode (default in sqflite), atomic transactions |
| Picked file deleted before upload | Copy to app docs on enqueue (permanent path); check `existsSync()` before processing |
| RAM spike from base64 (Phase 1) | One file at a time; Phase 2 eliminates base64 entirely |
| iOS kills background transfer (Phase 2) | Queue persists state; resumes on next app open |
| Token expires during long queue | Pause + prompt login + resume pattern |
| Server rejects multipart (Phase 2) | Feature flag: try multipart first, fall back to JSON if 400/415 |

## Rollback
- Phase 1: Revert Flutter changes, remove sqflite. Blocking upload restored.
- Phase 2: Revert upload.php from ORIGINAL/. Revert Flutter to Phase 1 (still non-blocking).

## Estimates
- Phase 1: 2-3 days Flutter development
- Phase 2: 1 day server + 2-3 days Flutter = 3-4 days additional
- Phase 1 alone solves ~90% of the user frustration
