#!/usr/bin/env python3
"""
On-Demand Thumbnail Generator for AEI Photo System (PHOTO-017)

Called by thumbnail_helper.php when a pre-generated thumb is missing.
Handles both JPEG and WebP input; always outputs WebP.

Usage:
    python3.6 generate_thumb_ondemand.py <source_image> <output_path> <width> <height> <mode>

Arguments:
    source_image:  Full path to source image (JPEG, PNG, or WebP)
    output_path:   Full path for the output WebP file
    width:         Target width in pixels
    height:        Target height in pixels (0 = proportional)
    mode:          Resize mode: crop | landscape | portrait | auto | exact

Output:
    Writes WebP thumbnail to output_path (atomic: temp file + rename).
    Prints the output path on success.
    Exit 0 on success (including if output already exists).
    Exit 1 on error.

Idempotent: exits immediately if output_path already exists.
Python 3.6 compatible. No f-strings.

Author: AEI Photo System
Version: 1.0 (PHOTO-017)
Date: 2026-02-19
"""

import os
import sys
import tempfile
from PIL import Image, ImageOps


WEBP_QUALITY = 80
MAX_RUNTIME = 30  # seconds


def load_image(file_path):
    """Load image and apply EXIF rotation."""
    img = Image.open(file_path)
    img = ImageOps.exif_transpose(img)
    return img


def resize_crop(img, width, height):
    """Resize to fill width x height, then center-crop."""
    if width <= 0 or height <= 0:
        return img

    src_w, src_h = img.size
    # Scale to cover the target area
    scale = max(float(width) / src_w, float(height) / src_h)
    # Don't upscale
    scale = max(scale, 1.0) if (src_w < width or src_h < height) else scale

    new_w = int(src_w * scale)
    new_h = int(src_h * scale)

    if new_w != src_w or new_h != src_h:
        img = img.resize((new_w, new_h), Image.LANCZOS)

    # Center crop
    left = (new_w - width) // 2
    top = (new_h - height) // 2
    img = img.crop((left, top, left + width, top + height))
    return img


def resize_landscape(img, width, height):
    """Resize by fixed width, proportional height. If height > 0, cap it."""
    if width <= 0:
        return img

    src_w, src_h = img.size
    if src_w <= width:
        return img

    new_w = width
    new_h = int(src_h * (float(width) / src_w))

    if height > 0 and new_h > height:
        new_h = height

    img = img.resize((new_w, new_h), Image.LANCZOS)
    return img


def resize_portrait(img, width, height):
    """Resize by fixed height, proportional width."""
    if height <= 0:
        return img

    src_w, src_h = img.size
    if src_h <= height:
        return img

    new_h = height
    new_w = int(src_w * (float(height) / src_h))

    if width > 0 and new_w > width:
        new_w = width

    img = img.resize((new_w, new_h), Image.LANCZOS)
    return img


def resize_auto(img, width, height):
    """Resize to fit within width x height, maintaining aspect ratio. No upscale."""
    if width <= 0 and height <= 0:
        return img

    src_w, src_h = img.size

    if width <= 0:
        width = src_w
    if height <= 0:
        height = src_h

    ratio = max(float(src_h) / height, float(src_w) / width)
    ratio = max(ratio, 1.0)  # no upscale

    new_w = int(src_w / ratio)
    new_h = int(src_h / ratio)

    if new_w != src_w or new_h != src_h:
        img = img.resize((new_w, new_h), Image.LANCZOS)
    return img


def resize_exact(img, width, height):
    """Resize to exact dimensions (may distort)."""
    if width <= 0 or height <= 0:
        return img
    img = img.resize((width, height), Image.LANCZOS)
    return img


def generate(source_path, output_path, width, height, mode):
    """Generate thumbnail. Returns output_path on success, None if skipped."""

    # Idempotent: skip if output exists
    if os.path.isfile(output_path):
        return None

    img = load_image(source_path)

    # Convert to RGB if needed (RGBA/palette → white background)
    if img.mode in ('RGBA', 'LA', 'P'):
        bg = Image.new('RGB', img.size, (255, 255, 255))
        if img.mode == 'P':
            img = img.convert('RGBA')
        bg.paste(img, mask=img.split()[-1] if 'A' in img.mode else None)
        img = bg
    elif img.mode != 'RGB':
        img = img.convert('RGB')

    # Apply resize mode
    if mode == 'crop':
        img = resize_crop(img, width, height)
    elif mode == 'landscape':
        img = resize_landscape(img, width, height)
    elif mode == 'portrait':
        img = resize_portrait(img, width, height)
    elif mode == 'exact':
        img = resize_exact(img, width, height)
    else:
        img = resize_auto(img, width, height)

    # Ensure output directory exists
    out_dir = os.path.dirname(output_path)
    if not os.path.isdir(out_dir):
        os.makedirs(out_dir, mode=0o777)

    # Atomic write: temp file in same directory, then rename
    fd, tmp_path = tempfile.mkstemp(suffix='.webp', dir=out_dir)
    try:
        os.close(fd)
        img.save(tmp_path, 'WEBP', quality=WEBP_QUALITY)
        os.chmod(tmp_path, 0o644)
        os.rename(tmp_path, output_path)
    except Exception:
        # Clean up temp file on failure
        if os.path.isfile(tmp_path):
            os.unlink(tmp_path)
        raise

    return output_path


if __name__ == "__main__":
    if len(sys.argv) < 6:
        print("Usage: python3.6 generate_thumb_ondemand.py <source> <output> <width> <height> <mode>")
        sys.exit(1)

    source = sys.argv[1]
    output = sys.argv[2]
    width = int(sys.argv[3])
    height = int(sys.argv[4])
    mode = sys.argv[5]

    if not os.path.isfile(source):
        print("Error: Source not found: %s" % source, file=sys.stderr)
        sys.exit(1)

    # Validate mode
    valid_modes = ('crop', 'landscape', 'portrait', 'auto', 'exact')
    if mode not in valid_modes:
        print("Error: Invalid mode '%s'. Use: %s" % (mode, ', '.join(valid_modes)), file=sys.stderr)
        sys.exit(1)

    try:
        result = generate(source, output, width, height, mode)
        if result:
            print(result)
        # Exit 0 even if skipped (idempotent)
    except Exception as e:
        print("Error: %s" % str(e), file=sys.stderr)
        sys.exit(1)

    sys.exit(0)
