# PHOTO-016: Survey Photos in Mobile App (Optimized)

## Summary
Merges local survey photos (from `/mnt/dropbox/` via PHOTO-006's API) into the mobile app's photo grid. The remote server proxies both listing and images since mobile phones can't reach the local server directly.

**Optimization over original PHOTO-016:** Pre-caches thumbnails on remote disk during the listing call, eliminating 112+ individual proxy requests. Grid loads in ~0.7s instead of ~50s.

## Architecture

```
LISTING (1 request):
  Mobile → getimagelisting1.php Step 5
         → local:get_survey_photos.php?include_thumbs=1 (JSON + base64 thumbs)
         → Decodes base64, writes thumbs to /tmp/survey_photo_cache/
         → Returns thumb_link = serve_survey_photo.php?key={hash}&size=thumb

GRID VIEW (instant — thumbs pre-cached):
  Mobile → serve_survey_photo.php?key={hash}&size=thumb → readfile() from cache

DETAIL VIEW (on tap — proxy + cache):
  Mobile → serve_survey_photo.php?path={encoded}&size=mid → curl to local + cache
```

## Files Changed

| File | Action | Deploy Path |
|------|--------|-------------|
| `get_survey_photos.php` | Modified | local: `/var/www/html/upload/get_survey_photos.php` |
| `getimagelisting1.php` | Modified | remote: `httpdocs/photoapi/getimagelisting1.php` |
| `serve_survey_photo.php` | **Rewritten** | remote: `httpdocs/photoapi/serve_survey_photo.php` |

## Dependencies
- PHOTO-006 (`get_survey_photos.php` + `serve_photo.php` on local server) must be deployed
- PHOTO-017 (direct static URLs in getimagelisting1.php) must be deployed
- Local server must be reachable from remote (AWS IP in `trusted_whitelist` ipset)

## What Changed

### get_survey_photos.php (LOCAL — Modified)
Added `include_thumbs=1` query parameter. When set, each photo entry includes `thumbnail_base64` (base64-encoded WebP thumbnail) and `thumbnail_content_type`. Files over 20KB are skipped to prevent response bloat. Backward compatible — without the parameter, behavior is unchanged.

### getimagelisting1.php (REMOTE — Modified)
Added Step 5 after PHOTO-017's Step 4: calls local `get_survey_photos.php?customer_id={id}&type=survey&include_thumbs=1`. For each survey photo:
- Decodes base64 thumbnail → writes to `/tmp/survey_photo_cache/thumb_{md5}.webp`
- `thumb_link` = `serve_survey_photo.php?key={md5}&size=thumb` (pre-cached, instant)
- `link` = `serve_survey_photo.php?path={mid_path}` (on-demand proxy)
- `full_link` = `serve_survey_photo.php?path={orig_path}` (on-demand proxy)
- `source` = `"local"` to distinguish from uploaded photos
- Silent failure: if local unreachable, uploaded photos still returned

### serve_survey_photo.php (REMOTE — Rewritten)
Two serving modes with disk caching at `/tmp/survey_photo_cache/`:
- `?key={hash}&size=thumb` → serve from cache (pre-populated by listing call). Cache HIT = readfile().
- `?path={encoded}` → proxy from local with cache-on-write (7-day TTL). Atomic writes via tmp+rename.
- Returns `X-Cache: HIT` or `X-Cache: MISS` header for debugging.

## Performance

| Metric | Old PHOTO-016 | This Version |
|--------|-------------|--------------|
| Grid load requests | 1 + N proxied images | 1 (thumbs pre-cached) |
| Grid load time (112 photos) | ~50s | ~0.7s |
| Grid bandwidth | 5.6MB | ~670KB (in listing JSON) |
| Detail view (first) | 0.45s (no cache) | ~0.65s (proxy + cache) |
| Detail view (repeat) | 0.45s (no cache) | <10ms (disk cache hit) |
| fail2ban risk | High (N requests) | None (1 listing request) |

## Rollback
1. Replace `getimagelisting1.php` with `ORIGINAL/getimagelisting1.php`
2. Replace `get_survey_photos.php` with `ORIGINAL/get_survey_photos.php`
3. Delete `serve_survey_photo.php` from remote server (or leave — harmless without Step 5)
4. Optionally: `rm -rf /tmp/survey_photo_cache/`

## Deployment
```bash
# LOCAL: get_survey_photos.php
sudo cp /var/www/html/upload/get_survey_photos.php /var/www/html/upload/get_survey_photos.php.bak
sudo cp NEW/get_survey_photos.php /var/www/html/upload/get_survey_photos.php

# REMOTE: getimagelisting1.php + serve_survey_photo.php
scp -i /root/.ssh/aei_remote.pem NEW/serve_survey_photo.php Julian@18.225.0.90:/tmp/
scp -i /root/.ssh/aei_remote.pem NEW/getimagelisting1.php Julian@18.225.0.90:/tmp/

# On remote:
sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/getimagelisting1.php /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/getimagelisting1.php.bak
sudo cp /tmp/serve_survey_photo.php /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/
sudo cp /tmp/getimagelisting1.php /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/
sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/serve_survey_photo.php
sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/getimagelisting1.php
```

## Verification
```bash
# 1. Test local API with include_thumbs:
curl -s "https://upload.aeihawaii.com/get_survey_photos.php?customer_id=4955&type=survey&include_thumbs=1" | python3 -c "import sys,json; d=json.load(sys.stdin); photos=[p for y in d['photo_types']['survey'].values() for f in y['folders'] for p in f['photos']]; print(f'{len(photos)} photos, {sum(1 for p in photos if \"thumbnail_base64\" in p)} with thumbs')"

# 2. Test full listing from remote:
curl -s -X POST https://aeihawaii.com/photoapi/getimagelisting1.php \
  -H 'Content-Type: application/json' \
  -d '{"auth_token":"aei@89806849","job_id":106361}' | python3 -c "import sys,json; imgs=json.load(sys.stdin); survey=[i for i in imgs if i.get('source')=='local']; print(f'{len(imgs)} total, {len(survey)} survey')"

# 3. Test cache hit/miss:
curl -sD- 'https://aeihawaii.com/photoapi/serve_survey_photo.php?key=HASH&size=thumb' 2>&1 | grep X-Cache
# Second request should show X-Cache: HIT

# 4. Test proxy caching:
curl -sD- 'https://aeihawaii.com/photoapi/serve_survey_photo.php?path=ENCODED_PATH' 2>&1 | grep X-Cache
# First: MISS, Second: HIT
```

## Status
**Deployed** — 2026-02-19. All 3 files live, verified with functional tests.
