#!/usr/bin/env python3
"""
Process the retry queue for failed local server syncs.
Reads JSON files from queue/, retries POST to local server,
moves to queue/failed/ after max age exceeded.

Retry schedule (time-based):
  - First 24 hours:  every cron run (~15 min intervals, ~96 attempts)
  - After 24 hours:  once every 6 hours (~24 more attempts over 6 days)
  - After 7 days:    moved to queue/failed/

Also detects "stuck" items — files that were uploaded but never
queued for sync (e.g. sync_to_local.py hung before enqueueing).

Designed to run via cron every 15 minutes:
  */15 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py > /dev/null 2>&1

Requires: requests (pip3 install requests), pymysql (pip3 install pymysql)
"""

import os
import sys
import json
import glob
import fcntl
import shutil
import logging
import random
from datetime import datetime

UPLOAD_URL = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php'
AUTH_TOKEN = 'remote_token'
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'retry_queue.log')
QUEUE_DIR = os.path.join(SCRIPT_DIR, 'queue')
FAILED_DIR = os.path.join(QUEUE_DIR, 'failed')
CONNECT_TIMEOUT = 10
READ_TIMEOUT = 60
MAX_AGE_HOURS = 168           # 7 days — move to failed/ after this
REDUCED_AFTER_HOURS = 24      # first 24h: retry every cron run (15min)
REDUCED_INTERVAL_HOURS = 6    # after 24h: retry once every 6 hours
LOCKFILE_PATH = '/tmp/process_retry_queue.lock'

# DB config for stuck detection (same creds as upload.php)
DB_HOST = 'localhost'
DB_USER = 'schedular'
DB_PASS = 'M1gif9!6'
DB_NAME = 'mandhdesign_schedular'

# Survey photo types — these job_type initials map to photo_type 'S'
SURVEY_JOB_TYPES = {'PM', 'WM', 'AS', 'RPM', 'GCPM'}


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def sync_file(file_path, file_name, job_id, customer_id, job_type,
              last_name, first_name, job_date, photo_type):
    """POST file and metadata to local server.

    Returns (success, error_reason) tuple.
    """
    import requests

    try:
        with open(file_path, 'rb') as f:
            files = {'file': (file_name, f)}
            data = {
                'auth_token':  AUTH_TOKEN,
                'file_name':   file_name,
                'job_id':      job_id,
                'customer_id': customer_id,
                'job_type':    job_type,
                'last_name':   last_name,
                'first_name':  first_name,
                'job_date':    job_date,
                'photo_type':  photo_type,
            }
            resp = requests.post(
                UPLOAD_URL,
                files=files,
                data=data,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )

        if resp.status_code == 200:
            return True, None
        else:
            return False, "HTTP %d body=%s" % (resp.status_code, resp.text[:200])

    except Exception as e:
        return False, "%s: %s" % (type(e).__name__, str(e))


def get_queued_filenames():
    """Build a set of file_name values already in queue/ and queue/failed/."""
    names = set()
    for search_dir in [QUEUE_DIR, FAILED_DIR]:
        pattern = os.path.join(search_dir, '*.json')
        for qpath in glob.glob(pattern):
            try:
                with open(qpath, 'r') as f:
                    data = json.load(f)
                names.add(data.get('file_name', ''))
            except (ValueError, IOError):
                continue
    return names


def detect_stuck_items():
    """Scan meter_files DB for items uploaded 30min-24h ago that have no queue entry.

    If the source file still exists on disk and isn't already queued, create a
    new queue JSON so process_queue() picks it up on the next run.
    """
    try:
        import pymysql
    except ImportError:
        logging.warning("STUCK_DETECT pymysql not installed, skipping stuck detection")
        return

    # Cache existing queued filenames to avoid duplicate enqueuing
    queued_names = get_queued_filenames()

    try:
        conn = pymysql.connect(
            host=DB_HOST, user=DB_USER, passwd=DB_PASS, db=DB_NAME,
            charset='utf8', connect_timeout=10,
        )
    except Exception as e:
        logging.warning("STUCK_DETECT DB connection failed: %s — skipping", e)
        return

    try:
        with conn.cursor(pymysql.cursors.DictCursor) as cur:
            cur.execute("""
                SELECT mf.id, mf.unique_filename, mf.folder_path, mf.created,
                       js.customer_id, cs.first_name, cs.last_name, js.job_date,
                       jt.intials AS job_type, js.job_pid
                FROM meter_files mf
                LEFT JOIN jobs js ON mf.job_id = js.job_pid
                LEFT JOIN customers cs ON js.customer_id = cs.id
                LEFT JOIN job_types jt ON js.job_type_id = jt.id
                WHERE mf.created BETWEEN NOW() - INTERVAL 24 HOUR AND NOW() - INTERVAL 30 MINUTE
                  AND mf.folder_path IS NOT NULL
            """)
            rows = cur.fetchall()

        if not rows:
            logging.info("STUCK_DETECT no candidate rows found")
            return

        stuck_count = 0
        skip_count = 0

        for row in rows:
            filename = row.get('unique_filename') or ''
            folder = row.get('folder_path') or ''
            if not filename or not folder:
                continue

            file_path = os.path.join(folder, filename)

            # Skip if source file no longer exists on disk
            if not os.path.isfile(file_path):
                continue

            # Skip if already in queue or failed
            if filename in queued_names:
                skip_count += 1
                continue

            # Derive photo_type: survey job types → 'S', else → 'I'
            jtype = (row.get('job_type') or '').upper().strip()
            photo_type = 'S' if jtype in SURVEY_JOB_TYPES else 'I'

            # Build queue JSON payload
            now = datetime.now()
            rand = random.randint(100000, 999999)
            queue_filename = 'stuck_%s_%d.json' % (now.strftime('%Y%m%d_%H%M%S'), rand)

            payload = {
                'file_path':      file_path,
                'file_name':      filename,
                'job_id':         row.get('job_pid') or '',
                'customer_id':    row.get('customer_id') or '',
                'job_type':       jtype,
                'last_name':      row.get('last_name') or '',
                'first_name':     row.get('first_name') or '',
                'job_date':       str(row.get('job_date') or ''),
                'photo_type':     photo_type,
                'queued_at':      now.strftime('%Y-%m-%d %H:%M:%S'),
                'retry_count':    0,
                'last_error':     'stuck_detected',
                'stuck_detected': True,
                'meter_file_id':  row.get('id'),
                'created':        str(row.get('created') or ''),
            }

            if not os.path.isdir(QUEUE_DIR):
                os.makedirs(QUEUE_DIR, mode=0o777, exist_ok=True)

            queue_path = os.path.join(QUEUE_DIR, queue_filename)
            with open(queue_path, 'w') as f:
                json.dump(payload, f, indent=2)

            queued_names.add(filename)  # prevent dups within this run
            stuck_count += 1
            logging.warning("STUCK_DETECTED %s (meter_file %s, created %s) -> enqueued as %s",
                            filename, row.get('id'), row.get('created'), queue_filename)

        logging.info("STUCK_DETECT complete: %d stuck enqueued, %d already queued, %d total checked",
                     stuck_count, skip_count, len(rows))

    except Exception as e:
        logging.error("STUCK_DETECT query error: %s", e)
    finally:
        conn.close()


def parse_timestamp(ts):
    """Parse a 'YYYY-MM-DD HH:MM:SS' string into a datetime, or None."""
    if not ts:
        return None
    try:
        return datetime.strptime(ts, '%Y-%m-%d %H:%M:%S')
    except (ValueError, TypeError):
        return None


def process_queue():
    """Scan queue/ for JSON files and retry each one.

    Retry schedule:
      - First 24 hours:  every cron run (15 min)
      - After 24 hours:  once every 6 hours
      - After 7 days:    move to failed/
    """
    if not os.path.isdir(QUEUE_DIR):
        return

    pattern = os.path.join(QUEUE_DIR, '*.json')
    queue_files = sorted(glob.glob(pattern))

    if not queue_files:
        return

    logging.info("Processing %d queued item(s)", len(queue_files))
    now = datetime.now()

    for queue_path in queue_files:
        filename = os.path.basename(queue_path)

        try:
            with open(queue_path, 'r') as f:
                payload = json.load(f)
        except (ValueError, IOError) as e:
            logging.error("SKIP %s -> cannot read JSON: %s", filename, str(e))
            continue

        retry_count = payload.get('retry_count', 0)
        file_path = payload.get('file_path', '')
        file_name = payload.get('file_name', '')

        # Calculate age from queued_at
        queued_at = parse_timestamp(payload.get('queued_at', ''))
        age_hours = (now - queued_at).total_seconds() / 3600.0 if queued_at else 0

        # Expired: older than 7 days -> move to failed/
        if age_hours >= MAX_AGE_HOURS:
            if not os.path.isdir(FAILED_DIR):
                os.makedirs(FAILED_DIR, mode=0o777, exist_ok=True)
            failed_path = os.path.join(FAILED_DIR, filename)
            shutil.move(queue_path, failed_path)
            logging.warning("EXPIRED %s -> moved to failed/ after %.1f hours, %d retries (last error: %s)",
                            file_name, age_hours, retry_count, payload.get('last_error', 'unknown'))
            continue

        # After 24 hours: only retry if last attempt was >= 6 hours ago
        if age_hours >= REDUCED_AFTER_HOURS:
            last_retry = parse_timestamp(payload.get('last_retry', ''))
            if last_retry:
                hours_since_last = (now - last_retry).total_seconds() / 3600.0
                if hours_since_last < REDUCED_INTERVAL_HOURS:
                    continue  # skip — not time yet

        # Source file gone -> skip (nothing to send)
        if not os.path.isfile(file_path):
            os.remove(queue_path)
            logging.info("SKIP %s -> source file no longer exists, removing from queue", file_name)
            continue

        # Attempt retry
        success, error_reason = sync_file(
            file_path=file_path,
            file_name=file_name,
            job_id=payload.get('job_id', ''),
            customer_id=payload.get('customer_id', ''),
            job_type=payload.get('job_type', ''),
            last_name=payload.get('last_name', ''),
            first_name=payload.get('first_name', ''),
            job_date=payload.get('job_date', ''),
            photo_type=payload.get('photo_type', ''),
        )

        if success:
            os.remove(queue_path)
            logging.info("RETRY OK %s -> succeeded on retry %d after %.1fh (queued %s)",
                         file_name, retry_count + 1, age_hours, payload.get('queued_at', ''))
        else:
            # Update retry count and error, rewrite JSON
            payload['retry_count'] = retry_count + 1
            payload['last_error'] = error_reason or 'Unknown error'
            payload['last_retry'] = now.strftime('%Y-%m-%d %H:%M:%S')
            with open(queue_path, 'w') as f:
                json.dump(payload, f, indent=2)
            phase = "15min" if age_hours < REDUCED_AFTER_HOURS else "6hr"
            logging.warning("RETRY FAIL %s -> attempt %d, age %.1fh (%s phase): %s",
                            file_name, retry_count + 1, age_hours, phase, error_reason)

    logging.info("Queue processing complete")


if __name__ == "__main__":
    setup_logging()

    # Lockfile: prevent overlapping cron runs
    lock_fp = open(LOCKFILE_PATH, 'w')
    try:
        fcntl.flock(lock_fp, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        logging.info("Another instance is running (lockfile %s held), exiting", LOCKFILE_PATH)
        sys.exit(0)

    process_queue()
    detect_stuck_items()
