<?php
header('Content-Type: application/json');

// Load database configuration for unified_customers lookup
require_once __DIR__ . '/db_config.php';

// Load environment variables
if (file_exists(__DIR__ . '/.env')) {
    $env = parse_ini_file(__DIR__ . '/.env');
    define('AUTH_TOKEN', $env['AUTH_TOKEN'] ?? '');
} else {
    define('AUTH_TOKEN', 'remote_token');
}

/**
 * Look up customer folder path by remote customer_id and job year
 * Copied from uploadlocallat_kuldeep.php to avoid modifying production upload handler
 */
function getCustomerFolderByCustomerId($customer_id, $job_year = null) {
    if (empty($customer_id)) {
        return null;
    }

    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("check_photos: DB connection failed for customer lookup");
        return null;
    }

    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE remote_customer_id = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ii", $customer_id, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE remote_customer_id = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        $conn->close();
        return null;
    }

    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

/**
 * Look up customer folder path by name (fallback method)
 * Copied from uploadlocallat_kuldeep.php
 */
function getCustomerFolderByName($first_name, $last_name, $job_year = null) {
    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("check_photos: DB connection failed for name lookup");
        return null;
    }

    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE first_name = ? AND last_name = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ssi", $first_name, $last_name, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE first_name = ? AND last_name = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        $conn->close();
        return null;
    }

    $stmt->bind_param("ss", $first_name, $last_name);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

// --- Request handling ---

// Only accept POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["status" => "error", "message" => "POST required"]);
    exit;
}

// Read JSON body
$raw = file_get_contents('php://input');
$body = json_decode($raw, true);

if (!$body || !is_array($body)) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Invalid JSON body"]);
    exit;
}

// Validate auth token
if (!isset($body['auth_token']) || $body['auth_token'] !== AUTH_TOKEN) {
    http_response_code(403);
    echo json_encode(["status" => "error", "message" => "Invalid or missing auth token"]);
    exit;
}

// Validate photos array
if (!isset($body['photos']) || !is_array($body['photos'])) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Missing photos array"]);
    exit;
}

// Enforce batch size limit
if (count($body['photos']) > 200) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Max 200 photos per batch"]);
    exit;
}

$missing = [];
$found_count = 0;

foreach ($body['photos'] as $photo) {
    $filename    = $photo['filename'] ?? '';
    $customer_id = isset($photo['customer_id']) ? intval($photo['customer_id']) : 0;
    $first_name  = $photo['first_name'] ?? '';
    $last_name   = $photo['last_name'] ?? '';
    $job_date    = $photo['job_date'] ?? '';
    $photo_type  = $photo['photo_type'] ?? 'I';
    $job_type    = $photo['job_type'] ?? '';

    if (empty($filename)) {
        continue;
    }

    // Parse job year from job_date
    $timestamp = strtotime($job_date);
    $job_year = $timestamp ? (int)date('Y', $timestamp) : (int)date('Y');

    // Look up customer folder
    $customer_folder = null;
    if ($customer_id) {
        $customer_folder = getCustomerFolderByCustomerId($customer_id, $job_year);
    }
    if (!$customer_folder && $first_name && $last_name) {
        $customer_folder = getCustomerFolderByName($first_name, $last_name, $job_year);
    }

    $file_found = false;

    // Check 1: Customer folder path (Survey or Installation subfolder)
    if ($customer_folder && is_dir($customer_folder)) {
        $customer_name = $last_name . ", " . $first_name;
        $type_suffix = ($photo_type === 'S') ? '-S, ' : '-I, ';
        $subfolder = ($photo_type === 'S') ? 'Survey' : 'Installation';
        $job_date_fmt = $timestamp ? date('Y-m-d', $timestamp) : $job_date;

        $expected_path = $customer_folder . '/' . $subfolder . '/' .
                         $customer_name . ', ' . $job_type . $type_suffix . $job_date_fmt . '/' .
                         $filename;

        if (file_exists($expected_path)) {
            $file_found = true;
        }
    }

    // Check 2: Uploads fallback folder
    if (!$file_found) {
        $customer_name_fb = ($last_name && $first_name)
            ? $last_name . ", " . $first_name
            : '';
        $job_date_fmt = $timestamp ? date('Y-m-d', $timestamp) : $job_date;

        if ($customer_name_fb) {
            $uploads_path = '/mnt/dropbox/Uploads/[' . $customer_name_fb . '] [' .
                            $job_type . '] [' . $job_date_fmt . ']/' . $filename;
            if (file_exists($uploads_path)) {
                $file_found = true;
            }
        }
    }

    if ($file_found) {
        $found_count++;
    } else {
        $missing[] = $filename;
    }
}

$checked = count($body['photos']);
echo json_encode([
    "status"  => "success",
    "checked" => $checked,
    "found"   => $found_count,
    "missing" => $missing
]);
