# CUSTOMER_FEEDBACK_FIX Enhancement

**Created:** 2026-02-07
**Status:** Planning
**Risk Level:** MEDIUM — Changes customer-facing email content and click handling

---

## Problem Statement

The customer review email system sends a "Congratulations" email with embedded thumbs up/down image buttons. Multiple issues are causing false negative (thumbs-down) reports:

### Current Data (as of 2026-02-07)
- **1,145 total clicks** across **887 unique jobs**
- **1,139 thumbs up** / **6 thumbs down**
- **166 jobs** have multiple clicks (duplicates/corrections)
- **3,619 emails sent** total (90 in 2026, 33 read, 57 unread)

### Evidence of Accidental Clicks
| Customer | Click Sequence | Time Between | Pattern |
|----------|---------------|--------------|---------|
| Larry Chu | down, up | 26 seconds | Misclick, then correction |
| Tim Harrison | down, down | 4 seconds | Double-tap (2 alert emails sent) |
| kuldeep banga (test) | up,up,up,down,up,down | various | Developer testing confirms ease of misclick |

---

## Root Causes

### 1. Layout — Buttons Too Close Together
The thumbs up (154x171px) and thumbs down (183x171px) images are placed directly adjacent with **zero spacing and no text labels**:

```html
<a href='http://aeihawaii.com/trackthumb.php?ssid={job_id}'>
  <img src='http://aeihawaii.com/scheduler/assets/images/thumbup.png'>
</a><a href='http://aeihawaii.com/emailtrack.php?ssid={job_id}'>
  <img src='http://aeihawaii.com/scheduler/assets/images/thumbdown.png'>
</a>
```

- **No spacing** between clickable areas — easy to fat-finger on mobile
- **No text labels** — when images are blocked by email client, customer sees two empty boxes
- **Unequal sizes** — thumbup is 154px wide, thumbdown is 183px wide
- **No `alt` text** — blocked images show nothing

### 2. No Click Deduplication
Every click inserts a new `trackthumb` row. No check for existing responses:
- Double-taps record twice
- Corrections (down then up) record both
- Each thumbs-down triggers a separate alert email to support

### 3. Immediate Alert on Thumbs Down
`reviewbg/sentmailtoadmin()` sends an alert email to `support@alternateenergyhawaii.com` instantly on every thumbs-down click. No grace period, no confirmation, no dedup check.

### 4. HTTP Links (Not HTTPS)
All email links use `http://aeihawaii.com/...`. Many email clients and corporate firewalls:
- Flag HTTP links as unsafe
- Rewrite them through proxy URLs
- Block them entirely
- Show security warnings that discourage clicking

### 5. Broken Sales Contact Info
In `emailcustomer.php` `sendemail()`, the email body references `$data["sales_user"]["phone"]` and `$data["sales_user"]["email"]`, but `$data["sales_user"]` is **never populated**. The variable is undefined, resulting in blank contact info in the email footer. The `error_reporting(0)` at the top of the file suppresses the PHP notice.

### 6. Fragile Click Recording
`trackthumb.php` and `emailtrack.php` use `exec("wget http://...")` to call the backend — a fire-and-forget HTTP call via wget subprocess. No error checking, no retry, no confirmation that the click was actually saved.

---

## Proposed Fixes

### Fix 1: Redesign Email Button Layout

Replace the current adjacent images with properly sized, labeled, spaced buttons that work on mobile. Both buttons should be:
- **Equal size** (e.g., 140x60px each)
- **Clearly labeled** with text ON the image (not relying on `alt` text alone)
- **Spaced apart** — at minimum 30px gap between them, using an HTML table layout (tables are the only reliable spacing method in HTML email)
- **Mobile-friendly** — sized to fit side by side on a 320px viewport with comfortable tap targets (minimum 44px touch target per Apple/Google guidelines)
- **Include `alt` text** — "I'm Happy" / "Needs Improvement" for when images are blocked

**New button images needed:**
- `feedback_happy.png` — green/positive, text: "Great Job!" (140x60px)
- `feedback_unhappy.png` — red/negative, text: "Needs Improvement" (140x60px)

**Email HTML structure (table-based for email compatibility):**
```html
<table cellpadding="0" cellspacing="0" width="100%" style="margin: 20px 0;">
  <tr>
    <td align="center" width="50%" style="padding: 10px 15px 10px 0;">
      <a href="https://aeihawaii.com/trackthumb.php?ssid={job_id}">
        <img src="https://aeihawaii.com/scheduler/assets/images/feedback_happy.png"
             alt="I'm Happy - Great Job!" width="140" height="60"
             style="border:0; display:block;">
      </a>
    </td>
    <td align="center" width="50%" style="padding: 10px 0 10px 15px;">
      <a href="https://aeihawaii.com/emailtrack.php?ssid={job_id}">
        <img src="https://aeihawaii.com/scheduler/assets/images/feedback_unhappy.png"
             alt="Needs Improvement" width="140" height="60"
             style="border:0; display:block;">
      </a>
    </td>
  </tr>
</table>
```

### Fix 2: Click Deduplication + Last-Click-Wins

Modify `reviewbg.php` `trackthumb()` and `sentmailtoadmin()` to:

1. **Check for existing response** for this `job_id` before inserting
2. If a response already exists:
   - **UPDATE** the existing row instead of inserting a new one (last-click-wins)
   - Skip sending the alert email if this is a correction within a grace period
3. If no response exists:
   - INSERT as normal

```php
// In trackthumb():
$existing = $this->db->query("SELECT id, thumbup FROM trackthumb WHERE job_id = " .
    $this->db->escape($job_id) . " ORDER BY id DESC LIMIT 1")->row_array();

if ($existing) {
    // Update existing — last click wins
    $this->db->where('id', $existing['id']);
    $this->db->update('trackthumb', array('thumbup' => 1, 'created' => time()));
} else {
    // First click — insert new
    $this->db->insert('trackthumb', array('job_id' => $job_id, 'created' => time(), 'thumbup' => 1));
}
```

### Fix 3: Delayed Alert for Thumbs Down

Instead of sending the support alert email immediately, add a grace period:

**Option A (Simple):** Check if a thumbs-up already exists for this job. If the customer previously clicked thumbs-up and now clicks thumbs-down, still record it, but note it may be accidental. If they click thumbs-up again within the grace window, suppress the alert entirely.

**Option B (Recommended):** Record the thumbs-down in the database immediately, but delay the alert email by having a cron job (or the existing `process_retry_queue.py` cycle) check for unresolved thumbs-down entries older than 5 minutes. If the customer corrected to thumbs-up within that window, no alert is sent.

For initial implementation, **Option A** is simpler — just check for duplicates before sending:

```php
// In sentmailtoadmin():
$existing = $this->db->query("SELECT id, thumbup FROM trackthumb WHERE job_id = " .
    $this->db->escape($job_id) . " ORDER BY id DESC LIMIT 1")->row_array();

if ($existing) {
    // Update to thumbs down
    $this->db->where('id', $existing['id']);
    $this->db->update('trackthumb', array('thumbup' => 0, 'created' => time()));
} else {
    // First click
    $this->db->insert('trackthumb', array('job_id' => $job_id, 'created' => time(), 'thumbup' => 0));
}

// Only send alert if this is the FIRST thumbs-down (not a re-click)
if (!$existing || $existing['thumbup'] == 1) {
    // Send alert email to support...
}
```

### Fix 4: Switch All Links to HTTPS

Update all email links from `http://` to `https://`:
- `http://aeihawaii.com/trackthumb.php` -> `https://aeihawaii.com/trackthumb.php`
- `http://aeihawaii.com/emailtrack.php` -> `https://aeihawaii.com/emailtrack.php`
- `http://aeihawaii.com/scheduler/assets/images/...` -> `https://aeihawaii.com/scheduler/assets/images/...`
- Tracking pixel URL in `sendemail()`: `base_url()` already generates HTTP — verify and fix

**Files affected:**
- `index_emailac.php` (email compose template)
- `emailcustomer.php` `sendemail()` (tracking pixel URL)

### Fix 5: Fix Sales Contact Info in Email Footer

In `emailcustomer.php` `sendemail()`, the email body references `$data["sales_user"]` which is never populated. Fix by querying the sales rep for the job:

```php
// Before building $message, look up the sales user from the job
$sales_query = "SELECT u.first_name, u.last_name, u.phone, u.email
                FROM jobs js
                LEFT JOIN users u ON js.user_id = u.id
                WHERE js.id = " . $this->db->escape($job_id);
$sales_user = $this->db->query($sales_query)->row_array();

// Then in the email body, use $sales_user['phone'] and $sales_user['email']
```

### Fix 6: Replace wget with Direct PHP Call

Replace the fragile `exec("wget ...")` in `trackthumb.php` and `emailtrack.php` with direct PHP includes or cURL calls that can verify success:

**Option A (Simplest):** Make the click handlers call the scheduler controller directly via cURL with error checking.

**Option B (Recommended):** Move the trackthumb/sentmailtoadmin logic OUT of the scheduler controller and INTO the web root scripts themselves. This eliminates the loopback HTTP call entirely:

```php
// trackthumb.php — direct DB insert instead of wget
<?php
$job_id = intval($_GET['ssid']);
if ($job_id <= 0) { header("Location: https://alternateenergyhawaii.com/thumbs-up"); die; }

$sdb = mysqli_connect("localhost", "schedular", "M1gif9!6", "mandhdesign_schedular");
// ... dedup check + insert/update ...
mysqli_close($sdb);
header("Location: https://alternateenergyhawaii.com/thumbs-up");
die;
?>
```

This is more reliable because it removes the dependency on wget, the loopback HTTP call, and the CodeIgniter session framework.

---

## Files Changed

### Modified Files (6)

| File | Remote Path | Change |
|------|------------|--------|
| `trackthumb.php` | `/var/www/vhosts/aeihawaii.com/httpdocs/trackthumb.php` | Replace wget with direct DB, add dedup, HTTPS redirect |
| `emailtrack.php` | `/var/www/vhosts/aeihawaii.com/httpdocs/emailtrack.php` | Replace wget with direct DB, add dedup + delayed alert, HTTPS redirect |
| `reviewbg.php` | `scheduler/system/application/controllers/reviewbg.php` | Add dedup logic to trackthumb() and sentmailtoadmin() |
| `emailcustomer.php` | `scheduler/system/application/controllers/emailcustomer.php` | Fix sales_user query in sendemail() |
| `index_emailac.php` | `scheduler/system/application/views/emailcustomer/index_emailac.php` | New button layout with spacing/labels, HTTPS links |

### New Files (2)

| File | Remote Path | Purpose |
|------|------------|---------|
| `feedback_happy.png` | `scheduler/assets/images/feedback_happy.png` | New thumbs-up button image (140x60, green, labeled) |
| `feedback_unhappy.png` | `scheduler/assets/images/feedback_unhappy.png` | New thumbs-down button image (140x60, red, labeled) |

### Unchanged Files

| File | Why |
|------|-----|
| `thumblist.php` | Admin listing page — reads from `trackthumb` table, no logic changes needed |
| `thumblist/index.php` (view) | Display-only view of admin thumb list |
| Old button images | Keep `thumbup.png` and `thumbdown.png` on server for old emails still in inboxes |

---

## Current System Flow

```
Admin clicks "Email to Customer" on job page
    |
    v
emailcustomer/index/{job_id}  -->  index_emailac.php (compose popup)
    |  Pre-fills message + thumbup/thumbdown image buttons
    v
emailcustomer/sendemail  -->  Sends via Mailtrap SMTP
    |  Inserts customerreviewtracking row
    |  Embeds 1x1 tracking pixel (confirm_read_mail)
    v
Customer receives email
    |
    +-- Opens email --> tracking pixel fires --> confirm_read_mail updates read_by_mail
    |
    +-- Clicks Thumbs UP --> trackthumb.php
    |       |  exec("wget .../reviewbg/trackthumb/{job_id}")  <-- FRAGILE
    |       |  INSERT into trackthumb (thumbup=1)
    |       +-- Redirect to alternateenergyhawaii.com/thumbs-up
    |
    +-- Clicks Thumbs DOWN --> emailtrack.php
            |  exec("wget .../reviewbg/sentmailtoadmin/{job_id}")  <-- FRAGILE
            |  INSERT into trackthumb (thumbup=0)
            |  SEND alert email to support@alternateenergyhawaii.com  <-- IMMEDIATE, NO DEDUP
            +-- Redirect to alternateenergyhawaii.com/thumbs-down
```

## Proposed System Flow

```
Admin clicks "Email to Customer" on job page
    |
    v
emailcustomer/index/{job_id}  -->  index_emailac.php (compose popup)
    |  Pre-fills message + NEW labeled buttons with spacing
    |  Sales rep contact info populated correctly
    v
emailcustomer/sendemail  -->  Sends via Mailtrap SMTP (HTTPS links)
    |  Inserts customerreviewtracking row
    |  Embeds 1x1 tracking pixel
    v
Customer receives email (buttons visible even with images blocked via alt text)
    |
    +-- Clicks "Great Job!" --> trackthumb.php (HTTPS)
    |       |  Direct DB: check existing, INSERT or UPDATE (last-click-wins)
    |       +-- Redirect to alternateenergyhawaii.com/thumbs-up
    |
    +-- Clicks "Needs Improvement" --> emailtrack.php (HTTPS)
            |  Direct DB: check existing, INSERT or UPDATE (last-click-wins)
            |  Only send alert if FIRST thumbs-down for this job (not a re-click)
            +-- Redirect to alternateenergyhawaii.com/thumbs-down
```

---

## Button Design Requirements

### Current Buttons (Problems)
- `thumbup.png`: 154x171px (portrait, no text, just a thumb icon)
- `thumbdown.png`: 183x171px (portrait, different width, no text)
- No spacing between them
- No alt text
- Too tall for email — pushes content down

### New Buttons (Requirements)
- **Both exactly 140x60px** (landscape, equal size)
- **Text ON the image:** "Great Job!" (green) / "Needs Improvement" (red)
- **Minimum 30px gap** between buttons in email layout
- **Mobile-friendly:** Two buttons side by side at 140px each = 280px + 30px gap = 310px total, fits within 320px mobile viewport
- **Touch target:** 60px height exceeds the 44px minimum recommended by Apple/Google
- **Alt text:** Descriptive fallback when images are blocked
- **File format:** PNG with transparent or white background
- **Style:** Rounded corners, clear iconography (checkmark/X or thumb icons), high contrast text

---

## Deployment Plan

### Step 1: Create New Button Images
Design and create `feedback_happy.png` and `feedback_unhappy.png` (140x60px each).

### Step 2: Modify Click Handlers
Update `trackthumb.php` and `emailtrack.php`:
- Direct DB connection (remove wget)
- Deduplication (last-click-wins)
- HTTPS redirect to landing pages
- Input sanitization (intval job_id)

### Step 3: Update Email Template
Modify `index_emailac.php`:
- New table-based button layout with spacing
- HTTPS image URLs and click URLs
- Alt text on button images

### Step 4: Fix Email Sender
Update `emailcustomer.php` `sendemail()`:
- Query sales rep from jobs.user_id -> users table
- Populate phone and email in footer
- HTTPS tracking pixel URL

### Step 5: Update Controller (Optional)
If keeping the scheduler controller as a secondary path, update `reviewbg.php` with dedup logic.

### Step 6: Deploy & Test
1. Upload new images to `scheduler/assets/images/`
2. Deploy modified PHP files
3. Send test email to verify layout, spacing, link functionality
4. Test both click paths — verify dedup works, alert logic correct
5. Test with images blocked — verify alt text is readable

---

## Rollback

All original files preserved in `ENHANCEMENTS/CUSTOMER_FEEDBACK_FIX/ORIGINAL/`:
- `trackthumb.php`
- `emailtrack.php`
- `reviewbg.php`
- `emailcustomer.php`
- `index_emailac.php`

Old button images (`thumbup.png`, `thumbdown.png`) remain on server — existing emails in customer inboxes will still render correctly.

To rollback: restore original files via SCP from `ORIGINAL/` directory.
