# UNIFIED_STORAGE Enhancement

**Created:** 2026-02-06
**Updated:** 2026-02-17
**Status:** Deployed (Phase 1 + Phase 2 live on remote server)
**Risk Level:** HIGH - Architectural change affecting all photo upload and retrieval paths

---

## Problem Statement

The remote server (18.225.0.90) currently maintains **two completely separate storage systems** for every uploaded photo, creating quadruple redundancy:

| Copy | Location | Consumer | Purpose |
|------|----------|----------|---------|
| 1. Hierarchy Original | `/mnt/dropbox/.../image.jpg` | Nothing actively reads this | Exists to maintain folder structure |
| 2. Hierarchy WebP | `/mnt/dropbox/.../webp/image.webp` | `getimagelisting.php` (mobile app) | Mobile app photo browser |
| 3. Flat Original | `scheduler/uploads/hash.jpg` | Scheduler UI, download, local sync | Scheduler operations |
| 4. Flat WebP | `scheduler/uploads/webp/hash.webp` | Scheduler gallery, premeasure gallery | Scheduler thumbnail display |

The mobile app uses **Method A (filesystem scanning)** - it constructs folder paths and calls `scandir()` to enumerate WebP files. The scheduler uses **Method B (database lookup)** - it queries `meter_files` for filenames and reads from the flat directory.

This dual approach:
- **Doubles storage usage** on the remote server (~985GB disk)
- **Creates complex failure modes** when folder paths are wrong (the PHOTO_LISTING_FIXES bugs)
- **Requires every upload path** to write to two locations and generate two WebP copies
- **Makes `/mnt/dropbox/` on the remote server a "fake Dropbox"** - it mimics the local server's hierarchy but serves no Dropbox sync purpose

## Proposed Solution

Unify all consumers onto **Method B (database lookup)**. The mobile app API endpoints will query `meter_files` instead of scanning directories, and serve images from the existing `scheduler/uploads/webp/` flat directory.

```
CURRENT:
  Mobile App  -> scandir(/mnt/dropbox/.../webp/)  -> fetch from hierarchy
  Scheduler   -> query meter_files                 -> fetch from scheduler/uploads/

PROPOSED:
  Mobile App  -> query meter_files                 -> fetch from scheduler/uploads/webp/
  Scheduler   -> query meter_files                 -> fetch from scheduler/uploads/
```

---

## Comprehensive Impact Analysis

### Files That CHANGE (Active Code)

#### 1. `photoapi/getimagelisting.php` - REWRITE
**Current:** Scans `/mnt/dropbox/.../webp/` directories via `scandir()`
**Proposed:** Pure database query on `meter_files` for all customer job photos
- Remove `scanWebpDir()` helper
- Remove folder path reconstruction fallback
- Remove `/mnt/aeiserver/` legacy scan
- Return links pointing to flat `scheduler/uploads/webp/` via `fetch_image.php`

#### 2. `photoapi/fetch_image.php` - REWRITE
**Current:** Reads WebP from `/mnt/dropbox/.../webp/{filename}` (primary) or reconstructed path (fallback)
**Proposed:** Reads WebP from `scheduler/uploads/webp/{filename}` using `meter_files.webpfilename`
- Remove folder_path lookup
- Remove path reconstruction fallback
- Serve directly from flat WebP directory
- Add fallback: try `scheduler/uploads/{unique_filename}` if WebP missing (serve original)

#### 3. `photoapi/delete.php` - REWRITE
**Current:** Deletes from `/mnt/dropbox/2025 Customers/.../webp/` (hardcoded 2025 bug)
**Proposed:** Deletes from `scheduler/uploads/webp/` AND removes `meter_files` DB row
- Fix: currently does NOT delete DB records (only unlinks files)
- Fix: currently has hardcoded 2025 year
- New: delete from flat WebP directory
- New: also delete from `scheduler/uploads/` (original)
- New: delete corresponding `meter_files` row

#### 4. `photoapi/upload.php` - MODIFY (stop hierarchy writes)
**Current:** Writes original to `/mnt/dropbox/` hierarchy + `scheduler/uploads/`, generates WebP to both locations
**Proposed:** Write original to `scheduler/uploads/` only, generate WebP to `scheduler/uploads/webp/` only
- Remove: mkdir cascade for `/mnt/dropbox/{Year} Customers/...` (lines 56-94)
- Remove: `file_put_contents()` to hierarchy path (line 98) — rewrite to save directly to `$filePath_u`
- Remove: WebP generation to hierarchy — change `generate_webp.py` destination from `$final_folder_webp . $webpfilename` to `$scheduler_webp_dest`, and remove the `&& cp` shell chain (lines 126-131)
- Remove: dead code `copy($filePath, $final_folder)` at line 82 — `$filePath` is undefined at this point, so this `copy()` silently fails on every Installation upload. Goes away naturally with hierarchy removal.
- Keep: `scheduler/uploads/` write
- Keep: `scheduler/uploads/webp/` write (now the direct destination for `generate_webp.py`)
- Keep: `meter_files` INSERT (folder_path becomes unnecessary but keep for backward compat)
- ~~Modify: `sync_to_local.py` call — pass `scheduler/uploads/{filename}` as file_path instead of hierarchy path~~ **DONE** (2026-02-09, PHOTO-011) — `$filePath` → `$filePath_u` on line 142

#### 5. `controllers/phototab.php` `dropImageUpload()` - MODIFY
**Current:** Copies to hierarchy + generates WebP to both locations
**Proposed:** Stop hierarchy writes
- Remove: mkdir cascade for `/mnt/dropbox/` hierarchy (lines 661-693)
- Remove: `copy($filePath, $final_folder)` to hierarchy (line 704)
- Remove: WebP generation to hierarchy — change `generate_webp.py` destination from `$final_folder_webp . $webpfilename` to `$scheduler_webp_dest`, remove `&& cp` (lines 734-739)
- Keep: `scheduler/uploads/` (already the primary path via CodeIgniter)
- Keep: `scheduler/uploads/webp/` write (now the direct destination for `generate_webp.py`)
- Keep: `meter_files` INSERT
- Note: `sync_to_local.py` call (line 712) already passes `$filePath` which is `scheduler/uploads/` — no change needed

#### 6. `controllers/preupload.php` `submit()` - MODIFY
**Current:** Copies to hierarchy + generates WebP to both locations
**Proposed:** Stop hierarchy writes
- Remove: mkdir cascade for `/mnt/dropbox/` hierarchy (lines 118-150)
- Remove: `copy($filePath, $final_folder)` to hierarchy (line 160)
- Remove: WebP generation to hierarchy — change `generate_webp.py` destination from `$final_folder_webp . $webpfilename` to `$scheduler_webp_dest`, remove `&& cp` (lines 190-195)
- Keep: `scheduler/uploads/` (already the primary path via CodeIgniter)
- Keep: `scheduler/uploads/webp/` write (now the direct destination for `generate_webp.py`)
- Keep: `meter_files` INSERT
- Note: `sync_to_local.py` call (line 168) already passes `$filePath` which is `scheduler/uploads/` — no change needed

#### 7. `photoapi/sync_to_local.py` - ~~MODIFY (path change)~~ **DONE** (2026-02-09, PHOTO-011)
**Current:** ~~`file_path` arg points to `/mnt/dropbox/.../image.jpg`~~
**Deployed:** `file_path` arg now points to `scheduler/uploads/{filename}`
- Changed in `upload.php` line 142: `$filePath` → `$filePath_u`
- `phototab.php` (line 712) and `preupload.php` (line 168) already passed `scheduler/uploads/` paths — no change was needed
- `sync_to_local.py` itself is path-agnostic (opens whatever `file_path` it receives)
- No change to local server reception (`uploadlocallat_kuldeep.php`)

#### 8. `photoapi/process_retry_queue.py` - ~~MODIFY~~ **DONE** (2026-02-09, PHOTO-011)
**Current:** ~~Builds `file_path` from `meter_files.folder_path + unique_filename` (hierarchy)~~
**Deployed:** Builds path from `SCHEDULER_UPLOADS + unique_filename`
- Added `SCHEDULER_UPLOADS` constant (line 56)
- `detect_stuck_items()` (line 180): uses `SCHEDULER_UPLOADS` instead of `folder_path`
- `reconcile_local_sync()` (line 450): uses `SCHEDULER_UPLOADS` instead of `folder_path`

#### 9. `photoapi/generate_webp.py` - NO CHANGE
- Script takes source + destination as arguments
- Callers will simply pass different destination paths

### Files That DO NOT Change

| File | Why |
|------|-----|
| `controllers/phototab.php` `download_all_photos()` | Already reads from `scheduler/uploads/` |
| `controllers/phototab.php` `getimagelisting()` | Calls `getimagelistingm.php` (legacy, /mnt/aeiserver/) - separate issue |
| `controllers/preupload.php` `gallery()` | Already reads from `scheduler/uploads/webp/` |
| `controllers/preupload.php` `download_all()` | Already reads from `scheduler/uploads/` |
| `views/photo_tab/photos_index.php` | Uses `thumbnail()` helper on `unique_filename` (flat) |
| `views/admin/job_photos.php` | Uses `thumbnail()` helper on `unique_filename` (flat) |
| `views/admin/job_photos_installer.php` | Direct reference to `uploads/{unique_filename}` |
| `views/file_tab/download_photo.php` | Uses `thumbnail()` helper on `unique_filename` (flat) |

### Legacy Endpoints (Separate Issue, Not in Scope)

These endpoints scan `/mnt/aeiserver/{job_id}/` which is a different legacy path unrelated to `/mnt/dropbox/`:

| File | Purpose | Note |
|------|---------|------|
| `getimagelistingm.php` | List images from `/mnt/aeiserver/` | Called by `phototab.php getimagelisting()` |
| `getimagelistingnew.php` | Get newest image from `/mnt/aeiserver/` | Legacy mobile API |
| `getimagelistingcnt.php` | Count images in `/mnt/aeiserver/` | Legacy mobile API |
| `delete_image.php` | Delete from `/mnt/aeiserver/` | Legacy delete, called by `phototab.php delete_image_ajax()` |

These are out of scope because they reference a completely different storage location. They should be addressed in a separate cleanup enhancement.

---

## Database Impact

### `meter_files` Table

| Column | Current Use | After Change |
|--------|------------|--------------|
| `unique_filename` | Filename in `scheduler/uploads/` | **No change** - still the flat original |
| `webpfilename` | Filename in `scheduler/uploads/webp/` AND `/mnt/dropbox/.../webp/` | **Simplified** - only in `scheduler/uploads/webp/` |
| `folder_path` | Path to `/mnt/dropbox/` hierarchy directory | **Deprecated** - no longer needed for reads. Keep populated for backward compatibility and potential future use. Could be repurposed as metadata. |
| `job_id` | Links to `jobs.job_pid` | **No change** |
| `file_type` | `99` = photo | **No change** |

### No Schema Changes Required
The existing `meter_files` table already has all columns needed. The `folder_path` column becomes unused for reads but doesn't need to be dropped.

---

## Migration Strategy

### Phase 1: Make Reads Database-Only (Zero Downtime)

Modify read-path endpoints to use database lookup instead of filesystem scanning. The hierarchy files still exist, so this is non-destructive.

1. Rewrite `getimagelisting.php` - DB query instead of scandir
2. Rewrite `fetch_image.php` - serve from `scheduler/uploads/webp/`
3. Rewrite `delete.php` - delete from flat storage + DB row

**Verification:** Mobile app returns same photo count as before for test customers.

**Rollback:** Revert to original files from `ORIGINAL/` directory.

### Phase 2: Stop Hierarchy Writes (Reduces Redundancy)

Stop writing to `/mnt/dropbox/` hierarchy in all upload paths. New photos only go to flat storage.

1. Modify `upload.php` - remove hierarchy writes + change WebP destination
2. Modify `phototab.php` `dropImageUpload()` - remove hierarchy writes + change WebP destination
3. Modify `preupload.php` `submit()` - remove hierarchy writes + change WebP destination
4. ~~Modify `sync_to_local.py` - read from flat storage~~ **DONE** (2026-02-09, PHOTO-011)
5. ~~Modify `process_retry_queue.py` - check flat storage~~ **DONE** (2026-02-09, PHOTO-011)

**WebP generation detail (applies to steps 1-3):** All three files currently run `generate_webp.py` with destination `$final_folder_webp . $webpfilename` (the `/mnt/dropbox/.../webp/` path), then `&& cp` the result to `$scheduler_webp_dest`. The change is to pass `$scheduler_webp_dest` directly as the destination argument and remove the `&& cp` entirely.

**Verification:** Upload a photo via each path, verify it appears in mobile app and scheduler.

**Rollback:** Revert to original files. No data loss since hierarchy still has all pre-change photos.

### Phase 3: Cleanup (After Soak Period)

After running Phase 1+2 for a sufficient soak period (suggested: 2-4 weeks):

1. Delete remote `/mnt/dropbox/{Year} Customers/` directories
2. Optionally: stop populating `folder_path` in `meter_files` INSERT
3. Optionally: deprecate `getimagelistingm.php`, `getimagelistingnew.php`, `getimagelistingcnt.php`, `delete_image.php` (legacy endpoints)

---

## Risk Assessment

### High Risk
- **Mobile app regression** - if `getimagelisting.php` returns fewer photos than before, users see missing images. Mitigation: thorough before/after photo count comparison for multiple customers.
- **WebP files missing from `scheduler/uploads/webp/`** - older uploads may not have had the WebP copy step. The hierarchy may be the only location. Mitigation: run a pre-migration check to identify any WebP files that exist in hierarchy but not in `scheduler/uploads/webp/`, and copy them.

### Medium Risk
- ~~**Sync to local server breaks** - changing the source path in `sync_to_local.py`.~~ **RESOLVED** (2026-02-09) — deployed via PHOTO-011, QA 33/33 passed. Sync works correctly with `scheduler/uploads/` paths.
- **Delete endpoint behavior change** - currently only deletes files, not DB rows. Adding DB deletion is correct behavior but changes semantics. Mitigation: test delete flow end-to-end.

### Low Risk
- **Storage savings not realized until Phase 3** - Phases 1-2 stop new writes but old data persists. This is intentional for safety.
- **`folder_path` column becomes unused** - no schema change needed; column is simply ignored.

---

## Pre-Migration Checklist

Before implementing, these checks must be performed on the remote server:

### 1. WebP Coverage Audit
Verify that `scheduler/uploads/webp/` contains WebP files for ALL `meter_files` rows that have a `webpfilename`:
```sql
SELECT COUNT(*) AS total_with_webp,
       SUM(CASE WHEN webpfilename IS NOT NULL AND webpfilename != '' THEN 1 ELSE 0 END) AS has_webpfilename
FROM meter_files
WHERE file_type = 99;
```

Then on the filesystem:
```bash
ls /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/webp/ | wc -l
```

If the file count is significantly lower than the DB count, we need a remediation step to copy missing WebP files from the hierarchy to flat storage before Phase 1.

### 2. Photo Count Comparison
For a sample of customers, compare photo counts between:
- Current `getimagelisting.php` (filesystem scan)
- Proposed DB-only query (`SELECT COUNT(*) FROM meter_files WHERE job_id IN (...)`)

This validates that the database is complete.

### 3. Disk Usage Analysis
```bash
du -sh /mnt/dropbox/20* 2>/dev/null        # Total hierarchy size
du -sh /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/     # Flat storage size
du -sh /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/webp/ # Flat WebP size
```

---

## Architecture After Change

```
UPLOAD SOURCES
  Mobile App --------> upload.php
  Scheduler Photo ---> phototab/dropImageUpload
  Scheduler Pre -----> preupload/submit

         |
         v

REMOTE SERVER (18.225.0.90)
  SINGLE STORAGE: scheduler/uploads/ + scheduler/uploads/webp/
  +-- uploads/{hash}.jpg          (originals, CodeIgniter encrypted names)
  +-- uploads/webp/{name}.webp    (WebP thumbnails for all consumers)

  Database: meter_files
  +-- unique_filename  -> maps to uploads/{hash}.jpg
  +-- webpfilename     -> maps to uploads/webp/{name}.webp

  API Endpoints (all DB-driven):
  +-- getimagelisting.php  -> queries meter_files, returns fetch_image URLs
  +-- fetch_image.php      -> serves from uploads/webp/{webpfilename}
  +-- delete.php           -> deletes from uploads/ + uploads/webp/ + meter_files row

         |  sync_to_local.py (reads from uploads/)
         v

LOCAL SERVER (upload.aeihawaii.com)
  /mnt/dropbox/{Year} Customers/...  (unchanged - still organized hierarchy)
  +-- Original photos + thumbs/ + thumbs/mid/
```

---

## Progress Log

| Date | Change | Items | Deployed Via |
|------|--------|-------|--------------|
| 2026-02-09 | Sync source switched to `scheduler/uploads/` | Items 7, 8 (Phase 2 steps 4, 5) | PHOTO-011 |
| 2026-02-17 | Phase 1: DB-only reads implemented | Items 1-3 (getimagelisting, fetch_image, delete) | PHOTO-009 |
| 2026-02-17 | Phase 2: Hierarchy writes removed | Items 4-6 (upload, phototab, preupload) | PHOTO-009 |

## Files in This Enhancement

```
ENHANCEMENTS/PHOTO-009_UNIFIED_STORAGE/
+-- README.md               (this file)
+-- ORIGINAL/               (pre-change copies of all modified files)
|   +-- getimagelisting.php
|   +-- fetch_image.php
|   +-- delete.php
|   +-- upload.php
|   +-- phototab.php
|   +-- preupload.php
|   +-- sync_to_local.py        (no longer needed — script is path-agnostic)
|   +-- process_retry_queue.py  (done — backup in PHOTO-011/ORIGINAL/)
+-- NEW/                    (modified versions)
|   +-- (same files, with changes applied)
+-- DOCS/
    +-- pre_migration_audit.md    (results of pre-migration checks)
    +-- migration_runbook.md      (step-by-step deployment instructions)
```
