# Enhancement: Cross-Year Photo Listing for Mobile App

**Created:** 2026-02-05
**Status:** Proposed
**Affects:** `getimagelisting.php`, `fetch_image.php` (remote server)

---

## Problem

When a customer has jobs spanning multiple years (e.g., premeasure in 2025 and installation in 2026), the mobile app gallery only shows photos from the **current job's year folder**. The installer doing a 2026 job cannot see the 2025 premeasure photos.

### Root Cause

Both `getimagelisting.php` and `fetch_image.php` derive the folder path from a **single job's** `job_date`. They construct one path and scan one `webp/` folder:

```php
// Current behavior — single year, single job folder
$job_year = date('Y', strtotime($row['job_date']));
$uploadDir = '/mnt/dropbox/' . $job_year . ' Customers/'
           . $indexname . '/' . $customername . '/' . $photo_type
           . '/' . $customer_name . ', ' . $job_type_text . '-' . $photo_s
           . ', ' . $job_date . '/webp/';
```

There is no lookup by `customer_id` to find related jobs across years.

### Real-World Example

```
Customer: Blaisdell, Mitsunori (customer_id=571)

Job 105661: PM  — 2025-07-21  → /mnt/dropbox/2025 Customers/B/Blaisdell, Mitsunori/Survey/...
Job 107399: SC  — 2025-10-09  → /mnt/dropbox/2025 Customers/B/Blaisdell, Mitsunori/Installation/...
Job 108364: AC  — 2026-01-06  → /mnt/dropbox/2026 Customers/B/Blaisdell, Mitsunori/Installation/...

When viewing job 108364 (2026 AC), the app shows ONLY photos from the 2026 folder.
The 2025 PM survey photos and 2025 SC installation photos are invisible.
```

---

## Scope

### Files to Modify

| File | Location on Remote Server | Change |
|------|--------------------------|--------|
| `getimagelisting.php` | `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/` | Add cross-year customer job lookup |
| `fetch_image.php` | `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/` | Support serving images from any year via related job_id |

### Files NOT Changed

| File | Reason |
|------|--------|
| `upload.php` | Upload routing is correct — photos go to the job's own year folder |
| `getimagelistingcnt.php` | Uses legacy `/mnt/aeiserver/` path only, not year-based folders |
| `getimagelistingm.php` | Uses legacy `/mnt/aeiserver/` path only, not year-based folders |
| `getimagelistingnew.php` | Uses legacy `/mnt/aeiserver/` path only (returns single image) |
| `sync_to_local.py` | Local server sync is unaffected |
| `uploadlocallat_kuldeep.php` | Local server routing is unaffected |

---

## Current Behavior

### `getimagelisting.php` — Current Flow

```
1. Receive POST { auth_token, job_id }
2. Query: SELECT first_name, last_name, job_date, intials
          FROM jobs WHERE id = {job_id}
3. Derive: $job_year from job_date
4. Derive: $photo_type (Survey or Installation) from job type
5. Construct single path:
     /mnt/dropbox/{YEAR} Customers/{Letter}/{Customer}/{PhotoType}/{Customer}, {Type}-{S|I}, {Date}/webp/
6. Scan that ONE webp/ folder
7. Also scan legacy /mnt/aeiserver/{jobId}/ path
8. Return JSON array of fetch_image.php links (all using same job_id)
```

**Limitation:** Only scans one year folder for one job type. No awareness of other jobs for the same customer.

### `fetch_image.php` — Current Flow

```
1. Receive GET ?job_id={id}&img={filename}
2. Query: SELECT first_name, last_name, job_date, intials
          FROM jobs WHERE id = {job_id}
3. Derive folder path from that job's metadata (same logic as listing)
4. Serve file from: {constructed_path}/webp/{filename}
```

**Limitation:** Can only serve files from the folder matching the provided `job_id`. If the listing returns a file from a different job's folder, this endpoint won't find it unless it receives the correct job_id for that file.

---

## Proposed Enhancement

### `getimagelisting.php` — Enhanced Flow

```
1. Receive POST { auth_token, job_id }
2. Query: SELECT customer_id, first_name, last_name, job_date, intials
          FROM jobs WHERE id = {job_id}
                          ^^^^^^^^^^^^
                          Add customer_id to query
3. Query ALL jobs for this customer:
     SELECT id, job_date, intials
     FROM jobs js
     LEFT JOIN job_types jt ON js.job_type_id = jt.id
     WHERE js.customer_id = {customer_id}
     ORDER BY js.job_date ASC
4. For EACH related job:
   a. Derive $job_year, $photo_type, $photo_s from that job's data
   b. Construct webp/ folder path for that job
   c. If folder exists, scan for images
   d. Build links using THAT JOB'S id (not the requesting job_id)
5. Also scan legacy /mnt/aeiserver/{jobId}/ path (unchanged)
6. Return combined JSON array with job metadata per image
```

### Enhanced Response Format

Current response (flat array):
```json
[
  { "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=108364&img=photo1.webp" },
  { "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=108364&img=photo2.webp" }
]
```

Enhanced response (with job context):
```json
[
  {
    "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=105661&img=photo1.webp",
    "job_id": 105661,
    "job_type": "PM",
    "job_date": "07-21-2025",
    "photo_type": "Survey"
  },
  {
    "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=107399&img=photo2.webp",
    "job_id": 107399,
    "job_type": "SC",
    "job_date": "10-09-2025",
    "photo_type": "Installation"
  },
  {
    "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=108364&img=photo3.webp",
    "job_id": 108364,
    "job_type": "AC",
    "job_date": "01-06-2026",
    "photo_type": "Installation"
  }
]
```

The additional fields (`job_id`, `job_type`, `job_date`, `photo_type`) allow the mobile app to group or label photos by job if desired. Existing mobile app code that only reads the `link` field will continue to work.

### `fetch_image.php` — Changes

`fetch_image.php` already derives the folder path from whatever `job_id` is passed in the URL. If `getimagelisting.php` returns the correct `job_id` per image, `fetch_image.php` will serve the right file without logic changes.

However, the following improvements should be made:

1. **SQL injection protection**: Add `intval()` to `$jobId` (currently uses `basename()` which is insufficient for SQL queries)
2. **Input validation order**: Move `$jobId`/`$fileName` validation before the DB query (currently query runs before validation on lines 12-16 vs 35-39)

---

## Database Queries

### New Query: Find All Jobs for a Customer

```sql
-- Get all jobs for the same customer, with job type info
SELECT js.id AS job_id,
       js.job_date,
       jt.intials AS job_type
FROM jobs js
LEFT JOIN job_types jt ON js.job_type_id = jt.id
WHERE js.customer_id = {customer_id}
  AND js.job_date IS NOT NULL
ORDER BY js.job_date ASC
```

### Job Types → Photo Type Mapping

The photo type (Survey vs Installation) determines the subfolder. This mapping is used in `upload.php`, `getimagelisting.php`, and `fetch_image.php`:

| Job Type Initials | Photo Type | Folder | Code |
|-------------------|-----------|--------|------|
| PM, WM, AS, RPM, GCPM | Survey | `Survey/` | `S` |
| All others (SC, PV, AC, EL, SP, etc.) | Installation | `Installation/` | `I` |

---

## Folder Path Construction

For each related job, the webp/ folder path is:

```
/mnt/dropbox/{YEAR} Customers/{Letter}/{Customer}/{PhotoType}/{Customer}, {JobType}-{S|I}, {Date}/webp/
```

Where:
- `{YEAR}` = `date('Y', strtotime($job_date))` — from that specific job's date
- `{Letter}` = first character of last name
- `{Customer}` = `"LastName, FirstName"`
- `{PhotoType}` = `"Survey"` or `"Installation"` (from job type mapping above)
- `{JobType}` = job type initials (PM, SC, AC, etc.)
- `{S|I}` = `S` for Survey, `I` for Installation
- `{Date}` = `date('m-d-Y', strtotime($job_date))`

### Example: customer_id=571, Blaisdell, Mitsunori

| Job | Constructed webp/ Path |
|-----|----------------------|
| 105661 PM 2025-07-21 | `/mnt/dropbox/2025 Customers/B/Blaisdell, Mitsunori/Survey/Blaisdell, Mitsunori, PM-S, 07-21-2025/webp/` |
| 107399 SC 2025-10-09 | `/mnt/dropbox/2025 Customers/B/Blaisdell, Mitsunori/Installation/Blaisdell, Mitsunori, SC-I, 10-09-2025/webp/` |
| 108364 AC 2026-01-06 | `/mnt/dropbox/2026 Customers/B/Blaisdell, Mitsunori/Installation/Blaisdell, Mitsunori, AC-I, 01-06-2026/webp/` |

---

## Implementation Notes

### Backward Compatibility

- The enhanced response adds new fields (`job_id`, `job_type`, `job_date`, `photo_type`) alongside the existing `link` field
- Mobile app code that only reads the `link` field will continue to work without changes
- The `link` URL now uses each image's own `job_id` instead of the requesting job's ID — `fetch_image.php` resolves this correctly since it derives the path from the provided `job_id`

### Performance Considerations

- **Additional DB query**: One extra query to find all customer jobs (indexed on `customer_id`)
- **Multiple folder scans**: Each related job requires an `is_dir()` + `scandir()` on its webp/ folder
- **Typical customer**: 2-8 jobs total. Even with 10 jobs, the overhead is negligible (local filesystem checks)
- **Edge case**: Customers with many jobs (20+) across many years — still fast since each scan is a single directory read

### Existing Code Issues (to fix during this enhancement)

| Issue | File | Current | Fix |
|-------|------|---------|-----|
| SQL injection on job_id | `fetch_image.php` line 14 | `WHERE js.id=$jobId` with `basename()` | Use `intval($jobId)` |
| Validation order | `fetch_image.php` lines 12-39 | DB query runs before input validation | Move validation before query |
| Hardcoded auth token | `getimagelisting.php` line 10 | `$data['auth_token']='aei@89806849'` (bypasses auth) | Remove hardcoded override |
| Broken link format | `getimagelisting.php` line 90 | Uses `$fullPath` in URL instead of `$jobId` | Fix to use `$jobId` |
| Legacy dual-scan | `getimagelisting.php` lines 69-92 | Second scan of `/mnt/aeiserver/` with different link format | Consolidate or remove if legacy path is no longer used |

### Security

- All `job_id` values must use `intval()` before SQL queries
- Customer names in folder paths are derived from the database (not user input) for each related job
- The enhancement does not introduce new user-facing input — only `job_id` is received from the client

---

## Deployment

### Pre-Deployment

1. Backup current files on remote server:
   ```bash
   ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90
   cd /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/
   sudo cp getimagelisting.php getimagelisting.php.bak.pre_crossyear.$(date +%Y%m%d)
   sudo cp fetch_image.php fetch_image.php.bak.pre_crossyear.$(date +%Y%m%d)
   ```

2. Verify `customer_id` index exists on `jobs` table (for query performance):
   ```sql
   SHOW INDEX FROM jobs WHERE Column_name = 'customer_id';
   ```

### Deployment Steps

1. Upload modified `getimagelisting.php` to remote server
2. Upload modified `fetch_image.php` to remote server
3. Verify PHP syntax: `php -l getimagelisting.php && php -l fetch_image.php`
4. Test with a known cross-year customer (e.g., customer_id=571, Blaisdell)

### Verification

```bash
# Test: Request image listing for a 2026 job belonging to a cross-year customer
curl -s -X POST https://aeihawaii.com/photoapi/getimagelisting.php \
  -H "Content-Type: application/json" \
  -d '{"auth_token":"aei@89806849","job_id":"108364"}' | python3 -m json.tool

# Expected: Response includes images from 2025 AND 2026 job folders
# Each image's link should contain the correct job_id for its source job
```

### Rollback

Restore backup files:
```bash
sudo cp getimagelisting.php.bak.pre_crossyear.{date} getimagelisting.php
sudo cp fetch_image.php.bak.pre_crossyear.{date} fetch_image.php
```

---

## Related Documentation

| Document | Relevance |
|----------|-----------|
| [PHOTO_SYSTEM_DOCUMENTATION.md](../../DOCS/PHOTO_SYSTEM_DOCUMENTATION.md) | "Cross-Year Folder Routing" section documents remote vs local difference |
| [PHOTO_SYSTEM_INTEGRATION.md](../../DOCS/PHOTO_SYSTEM_INTEGRATION.md) | "Year-Aware Folder Selection" section documents local DB fallback |
| [REMOTE_UPLOAD_CHANGES.md](../CUSTOMER_ID_LOOKUP/REMOTE_UPLOAD_CHANGES.md) | Remote server change history |
| [CREDENTIALS.md](../../DOCS/CREDENTIALS.md) | Auth tokens and DB credentials |

---

*Enhancement proposed: 2026-02-05*
