# PHOTO_LISTING_FIXES — Mobile App Photo Listing Issues

## Problem Statement

The mobile app photo listing (`getimagelisting.php`) does not show all photos for some customers. The scheduler photo tab shows all photos correctly because it queries the `meter_files` table directly and serves images from the flat `scheduler/uploads/` directory. The mobile app, however, scans `webp/` subdirectories on disk — so any file that is missing or misplaced in `/mnt/dropbox/` is invisible to the app.

**Discovered via:** Customer 7729 (Nishimoto, Howard), Job 108642 — scheduler shows 10 photos, mobile app shows fewer.

---

## Root Causes

### Issue 1: Premeasure Year Hardcoded to 2025

**File:** `preupload.php` (Scheduler Premeasure bulk upload controller)
**Line:** 117

```php
$uploadDir='/mnt/dropbox/2025 Customers/'.$indexname;
```

The year is hardcoded to `2025`. For any job with a 2026+ date, photos are stored under `/mnt/dropbox/2025 Customers/` instead of the correct year folder.

**Impact on mobile app:** `getimagelisting.php` reconstructs the expected folder path using the dynamic year from `job_date`. For a 2026 job, it looks in `/mnt/dropbox/2026 Customers/.../webp/` — but the files are actually in `/mnt/dropbox/2025 Customers/.../webp/`. The photos are invisible.

**Impact on local server:** The local server receives these files via synchronous cURL and stores them using the year from the metadata sent by the remote server. If the metadata includes the correct year, the local copy ends up in the right place, but the remote copy is in the wrong year folder.

**Correct pattern** (already used by `phototab.php` lines 659–660):
```php
$job_year = date("Y", strtotime($norm_date));
$uploadDir = '/mnt/dropbox/' . $job_year . ' Customers/' . $indexname;
```

The variable `$norm_date` is already available in `preupload.php` (computed at lines 61–66).

### Issue 2: WebP Naming Collisions

**File:** `upload.php` (Mobile App upload endpoint)
**Lines:** 119–121

```php
$customer_namewebp = $last_name . $first_name;
$randomId = rand(100000, 999999);
$webpfilename = $customer_namewebp . "" . $job_type_text . "-" . $photo_type . $job_date . "_IMG" . $randomId . '.webp';
```

The WebP filename uses `rand(100000, 999999)` — a 6-digit random number with only 900,000 possible values. For a batch of photos uploaded to the same customer/job in quick succession, all parts of the filename are identical except `$randomId`. If two photos get the same random number, the second WebP overwrites the first.

**Observed:** Customer 7729, Job 108642 — 10 photos uploaded, but only 7 unique WebP files on disk. Three WebP files were overwritten by later uploads with the same `$randomId`.

**Impact on mobile app:** `getimagelisting.php` scans the `webp/` directory and finds only 7 files instead of 10. The original JPEG files are all present (they use the original filename from the device, which is always unique), but the WebP copies used for mobile display have collisions.

**Impact on scheduler:** No impact — the scheduler photo tab serves images from `scheduler/uploads/` using the `unique_filename` column in `meter_files`, which uses CodeIgniter's hash-based naming (guaranteed unique).

### Issue 3: Missing `folder_path` on Scheduler-Uploaded Photos

**Table:** `meter_files`
**Affected rows:** Any row inserted by `phototab.php` or `preupload.php`

When photos are uploaded via the scheduler (Photo Tab or Premeasure), the `folder_path` column in `meter_files` is set to `NULL`. Only `upload.php` (mobile app path) populates `folder_path`.

**Impact on mobile app:** `getimagelisting.php` uses a two-step lookup:
1. **Step 5a:** Check `meter_files.folder_path` — if set, scan `{folder_path}/webp/`
2. **Step 5b:** Fall back to path reconstruction from job metadata

For scheduler-uploaded photos, step 5a is skipped (NULL folder_path), so the app relies entirely on path reconstruction. Combined with Issue 1 (wrong year), the reconstructed path points to a non-existent directory.

---

## Example Case: Customer 7729, Job 108642

| meter_files ID | Upload Source | folder_path | WebP on Disk | Visible in App |
|----------------|---------------|-------------|--------------|----------------|
| 207482 | Scheduler (preupload) | NULL | In 2025 folder (wrong year) | No — reconstructs 2026 path |
| 207483 | Scheduler (preupload) | NULL | In 2025 folder (wrong year) | No — reconstructs 2026 path |
| 207484 | Scheduler (preupload) | NULL | In 2025 folder (wrong year) | No — reconstructs 2026 path |
| 207485–207491 | Mobile app | `/mnt/dropbox/2026 Customers/N/.../Survey/...` | In 2026 folder (correct) | Partially — 7 of 7 unique, but 3 WebP collisions = only ~4-7 shown |

---

## Planned Fixes

### Fix 1: Dynamic Year in Premeasure Upload

**File to modify:** `preupload.php`
**Change:** Replace hardcoded `2025` with dynamic year from `$norm_date`

```
BEFORE (line 117):
  $uploadDir='/mnt/dropbox/2025 Customers/'.$indexname;

AFTER:
  $job_year = date("Y", strtotime($norm_date));
  $uploadDir = '/mnt/dropbox/' . $job_year . ' Customers/' . $indexname;
```

**Risk:** Low — `$norm_date` is already validated and normalized on lines 61–66. This matches the pattern used by `phototab.php`.

### Fix 2: Unique WebP Filenames in Mobile Upload

**File to modify:** `upload.php`
**Change:** Replace `rand(100000, 999999)` with a more unique identifier

```
BEFORE (lines 120-121):
  $randomId = rand(100000, 999999);
  $webpfilename = $customer_namewebp . "" . $job_type_text . "-" . $photo_type . $job_date . "_IMG" . $randomId . '.webp';

AFTER:
  $uniqueId = substr(md5(uniqid(mt_rand(), true)), 0, 12);
  $webpfilename = $customer_namewebp . "" . $job_type_text . "-" . $photo_type . $job_date . "_IMG" . $uniqueId . '.webp';
```

`md5(uniqid(mt_rand(), true))` combines a high-resolution timestamp, process ID, and random seed — producing a 32-hex-char hash. Using the first 12 chars gives 16^12 = ~2.8 trillion possible values, eliminating collisions entirely. This is a common PHP pattern compatible with PHP 5.3.

**Risk:** Low — only changes the random portion of the filename. No downstream code depends on the specific format of `$randomId`. The `webpfilename` column in `meter_files` stores whatever value is generated.

### Fix 3: Populate `folder_path` in Scheduler Upload Paths

**Files to modify:** `phototab.php` and `preupload.php`
**Change:** Add `folder_path` to the `meter_files` INSERT statement

Both controllers already compute the destination folder path (`$final_folder`) before copying the file. The fix is to include this value in the database INSERT, matching what `upload.php` already does.

**For `preupload.php`:** The INSERT into `meter_files` needs to include the `folder_path` column with the value of `$final_folder`.

**For `phototab.php`:** Same change — include `folder_path` in the INSERT.

**Risk:** Low — adds data to a column that already exists and is already populated by mobile uploads. No schema change required.

---

## Files Affected

| File | Change | Server |
|------|--------|--------|
| `preupload.php` | Fix 1 (dynamic year) + Fix 3 (folder_path) | Remote |
| `upload.php` | Fix 2 (unique WebP names) | Remote |
| `phototab.php` | Fix 3 (folder_path) | Remote |

**Local server:** No changes required.

**Database:** No schema changes. `meter_files.folder_path` column already exists (`varchar(500)`).

## Original Files

Copies of the original (pre-fix) versions of modified files should be placed in `ORIGINAL/` before deploying changes.

## Deployment

All three files are on the remote server (18.225.0.90):
- `/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/application/controllers/preupload.php`
- `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php`
- `/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/application/controllers/phototab.php`

Deploy via SCP + SSH:
```bash
scp -i /root/.ssh/aei_remote.pem <file> Julian@18.225.0.90:/tmp/
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "sudo cp /tmp/<file> <destination>"
```

## Existing Data Remediation

After deploying the fixes, existing misplaced photos need to be addressed:

1. **Move misplaced 2026 photos from 2025 folder** — Any photos uploaded via premeasure for 2026 jobs are sitting in `/mnt/dropbox/2025 Customers/`. These need to be moved to `/mnt/dropbox/2026 Customers/` on the remote server.

2. **Backfill `folder_path` for existing scheduler uploads** — Query `meter_files` rows where `file_type = 99` and `folder_path IS NULL`, then compute and populate the correct path from the associated job's customer/date/type data.

3. **Regenerate collided WebP files** — For any `meter_files` rows where the `webpfilename` file doesn't exist on disk (overwritten by a collision), regenerate the WebP from the original JPEG in `scheduler/uploads/`.

These remediation steps can be scripted and run once after deploying the code fixes.
