# SURVEY_PHOTOS — Local Photo Browser API

API on the local server (`upload.aeihawaii.com`) that allows the remote server or smartphone app to retrieve Survey, Installation, and Fusion (Service Fusion) photos for a customer.

## How It Works

1. Caller sends a request with `customer_id` (or `name`) to the local server
2. The API queries the local `unified_customers` table to find the customer's folder(s)
3. It scans `{folder_path}/Survey/`, `{folder_path}/Installation/`, and `{folder_path}/Fusion/` for photos
4. Returns JSON with photo metadata and URLs for three image sizes
5. Caller fetches actual images from the returned URLs on the same host

No remote server changes required. No existing files modified.

## Base URL

```
https://upload.aeihawaii.com
```

## Endpoints

### 1. Photo Listing — `GET /get_survey_photos.php`

Returns structured JSON with all Survey, Installation, and Fusion photos for a customer.

**Parameters:**

| Param | Type | Required | Description |
|-------|------|----------|-------------|
| `customer_id` | int | Yes* | Remote customer ID (`customers.id` from the scheduler DB) |
| `name` | string | Yes* | Customer name search (min 2 chars, supports multi-word) |
| `year` | int | No | Filter to a specific year (2010-2030) |
| `type` | string | No | `survey`, `installation`, `fusion`, or `all` (default: `all`) |

*One of `customer_id` or `name` is required.

**Example — get all photos for customer 4955:**
```
GET https://upload.aeihawaii.com/get_survey_photos.php?customer_id=4955
```

**Example — survey photos only, 2025:**
```
GET https://upload.aeihawaii.com/get_survey_photos.php?customer_id=4955&year=2025&type=survey
```

**Example — search by name:**
```
GET https://upload.aeihawaii.com/get_survey_photos.php?name=Calvin%20Abe
```

**Response:**
```json
{
  "success": true,
  "customer_id": 4955,
  "customer_name": "Abe, Calvin",
  "years": [2025],
  "photo_types": {
    "survey": {
      "2025": {
        "folders": [
          {
            "name": "Calvin Abe premeasure 2025-06-18",
            "photo_count": 112,
            "photos": [
              {
                "filename": "IMG_0682",
                "thumbnail_url": "serve_photo.php?path=2025+Customers%2FA%2FAbe%2C+Calvin%2FSurvey%2F...%2Fthumbs%2FIMG_0682.webp",
                "mid_url": "serve_photo.php?path=2025+Customers%2FA%2FAbe%2C+Calvin%2FSurvey%2F...%2Fthumbs%2Fmid%2FIMG_0682.webp",
                "original_url": "serve_photo.php?path=2025+Customers%2FA%2FAbe%2C+Calvin%2FSurvey%2F...%2FIMG_0682.jpeg",
                "has_thumbnail": true,
                "has_mid": true,
                "has_original": true
              }
            ]
          }
        ],
        "total_photos": 112
      }
    },
    "installation": {
      "2025": {
        "folders": [ ... ],
        "total_photos": 39
      }
    },
    "fusion": {
      "2026": {
        "folders": [
          {
            "name": "HVAC-SVC - 2026-01-15",
            "photo_count": 8,
            "photos": [ ... ]
          }
        ],
        "total_photos": 8
      }
    }
  },
  "stats": {
    "total_photos": 159,
    "survey_photos": 112,
    "installation_photos": 39,
    "fusion_photos": 8,
    "years_with_photos": ["2026", "2025"]
  }
}
```

**Error response (HTTP 400):**
```json
{
  "success": false,
  "error": "Required: customer_id (integer) or name (min 2 chars)"
}
```

**No photos found (HTTP 200):**
```json
{
  "success": true,
  "customer_id": 9999,
  "customer_name": null,
  "years": [],
  "photo_types": { "survey": [], "installation": [], "fusion": [] },
  "stats": { "total_photos": 0, "survey_photos": 0, "installation_photos": 0, "fusion_photos": 0, "years_with_photos": [] }
}
```

### 2. Serve Image — `GET /serve_photo.php?path=...`

Serves the actual image file. The `path` value comes directly from the `thumbnail_url`, `mid_url`, or `original_url` fields in the photo listing response.

**To build an absolute image URL**, prepend the base URL to the relative URL from the API response:

```
https://upload.aeihawaii.com/serve_photo.php?path=2025+Customers%2FA%2FAbe%2C+Calvin%2FSurvey%2F...%2Fthumbs%2FIMG_0682.webp
```

**Response:** Raw image bytes with appropriate `Content-Type` header (`image/webp`, `image/jpeg`, etc.)

**Caching:** Returns `Cache-Control: public, max-age=31536000` and supports `If-Modified-Since` (304 responses).

### 3. Customer Search — `GET /photo_browser_search.php?term=...`

Autocomplete endpoint for finding customers by name. Returns customers that have photo folders.

**Parameters:**

| Param | Type | Required | Description |
|-------|------|----------|-------------|
| `term` | string | Yes | Search term (min 2 chars) |

**Response:**
```json
[
  {
    "id": 5480,
    "first_name": "Calvin",
    "last_name": "Abe",
    "full_name": "Calvin Abe",
    "remote_customer_id": 4955
  }
]
```

## Image Sizes

Each photo has up to three sizes available:

| Size | Field | Typical Dimensions | Typical File Size | Format | Use Case |
|------|-------|--------------------|-------------------|--------|----------|
| Thumbnail | `thumbnail_url` | 200x150 | ~4 KB | WebP | Gallery grid, quick loading |
| Mid-size | `mid_url` | 800x600 | ~70 KB | WebP | Detail view, phone display |
| Original | `original_url` | 4032x3024 | ~2.8 MB | JPEG | Full-resolution download |

The `has_thumbnail`, `has_mid`, and `has_original` booleans indicate which sizes exist. When a size doesn't exist, its URL falls back to the next available size (thumbnail < mid < original).

## Integration from Remote Server (PHP)

```php
// Fetch survey photos for a customer
$customerId = 4955;
$url = "https://upload.aeihawaii.com/get_survey_photos.php?customer_id=" . $customerId . "&type=survey";

$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
$response = curl_exec($ch);
curl_close($ch);

$data = json_decode($response, true);

if ($data && $data['success']) {
    $baseUrl = "https://upload.aeihawaii.com/";

    foreach ($data['photo_types']['survey'] as $year => $yearData) {
        foreach ($yearData['folders'] as $folder) {
            echo $folder['name'] . " — " . $folder['photo_count'] . " photos\n";

            foreach ($folder['photos'] as $photo) {
                $thumbUrl  = $baseUrl . $photo['thumbnail_url'];  // ~4KB WebP
                $midUrl    = $baseUrl . $photo['mid_url'];         // ~70KB WebP
                $origUrl   = $baseUrl . $photo['original_url'];    // ~2.8MB JPEG
                // Use in <img> tags, download, etc.
            }
        }
    }
}
```

## Integration from Remote Server (Python)

```python
import requests

customer_id = 4955
base_url = "https://upload.aeihawaii.com"
resp = requests.get(f"{base_url}/get_survey_photos.php",
                    params={"customer_id": customer_id, "type": "survey"},
                    verify=False, timeout=10)
data = resp.json()

if data["success"]:
    for year, year_data in data["photo_types"]["survey"].items():
        for folder in year_data["folders"]:
            print(f"{folder['name']} — {folder['photo_count']} photos")
            for photo in folder["photos"]:
                thumb_url = f"{base_url}/{photo['thumbnail_url']}"
                mid_url   = f"{base_url}/{photo['mid_url']}"
                orig_url  = f"{base_url}/{photo['original_url']}"
                # Download: requests.get(thumb_url, verify=False).content
```

## Integration from JavaScript / Smartphone App

```javascript
const BASE_URL = "https://upload.aeihawaii.com";
const customerId = 4955;

const resp = await fetch(`${BASE_URL}/get_survey_photos.php?customer_id=${customerId}`);
const data = await resp.json();

if (data.success) {
    for (const [type, yearData] of Object.entries(data.photo_types)) {
        for (const [year, yd] of Object.entries(yearData)) {
            for (const folder of yd.folders) {
                for (const photo of folder.photos) {
                    // Use in <img src="...">
                    const thumbUrl = `${BASE_URL}/${photo.thumbnail_url}`;
                    const midUrl   = `${BASE_URL}/${photo.mid_url}`;
                    const origUrl  = `${BASE_URL}/${photo.original_url}`;
                }
            }
        }
    }
}
```

## Test Page

A visual browser is available for manual testing:

```
https://upload.aeihawaii.com/test_photo_browser.php
https://upload.aeihawaii.com/test_photo_browser.php?customer_id=4955
```

## Deployed Files

| File | Location on Local Server |
|------|--------------------------|
| `get_survey_photos.php` | `/var/www/html/upload/get_survey_photos.php` |
| `serve_photo.php` | `/var/www/html/upload/serve_photo.php` |
| `photo_browser_search.php` | `/var/www/html/upload/photo_browser_search.php` |
| `test_photo_browser.php` | `/var/www/html/upload/test_photo_browser.php` |

Source copies in: `ENHANCEMENTS/SURVEY_PHOTOS/NEW/`

## Dependencies

- `db_config.php` at `/var/www/html/upload/` (existing `upload_user` credentials)
- `unified_customers` table in local `Schedular` database
- `/mnt/dropbox/` mount with customer photo folders containing `Survey/`, `Installation/`, and/or `Fusion/` subdirectories
- Fusion folders follow the pattern `{customer_folder}/Fusion/{ABBREV} - {YYYY-MM-DD}/` with `thumbs/` and `thumbs/mid/` for WebP thumbnails (same structure as Survey/Installation)
- Thumbnails generated by the existing upload pipeline into `thumbs/` (WebP, 200x150) and `thumbs/mid/` (WebP, 800x600)
