<?php
// get_survey_photos.php - Local-only API endpoint for browsing Survey and Installation photos
// Queries unified_customers directly (no remote server dependency)
require_once __DIR__ . '/db_config.php';

header('Content-Type: application/json');

try {
    // Parse and validate input parameters
    $customerId = isset($_GET['customer_id']) ? intval($_GET['customer_id']) : 0;
    $nameSearch = isset($_GET['name']) ? trim($_GET['name']) : '';
    $yearFilter = isset($_GET['year']) ? intval($_GET['year']) : 0;
    $typeFilter = isset($_GET['type']) ? strtolower(trim($_GET['type'])) : 'all';

    // Validate: need either customer_id or name
    if ($customerId <= 0 && strlen($nameSearch) < 2) {
        throw new Exception('Required: customer_id (integer) or name (min 2 chars)');
    }

    // Validate year if provided
    if ($yearFilter > 0 && ($yearFilter < 2010 || $yearFilter > 2030)) {
        throw new Exception('Year must be between 2010 and 2030');
    }

    // Validate type filter
    if (!in_array($typeFilter, ['all', 'survey', 'installation'])) {
        throw new Exception('Type must be: all, survey, or installation');
    }

    // Connect to local Schedular database
    $conn = getSchedularConnection();
    if (!$conn) {
        throw new Exception('Database connection failed');
    }
    $conn->set_charset('utf8');

    // Build query based on lookup method
    $customers = [];

    if ($customerId > 0) {
        // Primary lookup: by remote_customer_id
        $sql = "SELECT id, remote_customer_id, first_name, last_name, folder_name, folder_path, folder_year
                FROM unified_customers
                WHERE remote_customer_id = ? AND folder_path IS NOT NULL";
        $params = [$customerId];
        $types = "i";

        if ($yearFilter > 0) {
            $sql .= " AND folder_year = ?";
            $params[] = $yearFilter;
            $types .= "i";
        }

        $sql .= " ORDER BY folder_year DESC";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();

        while ($row = $result->fetch_assoc()) {
            $customers[] = $row;
        }
        $stmt->close();

    } else {
        // Fallback lookup: by name search
        $searchWords = preg_split('/[\s,]+/', $nameSearch, -1, PREG_SPLIT_NO_EMPTY);
        if (empty($searchWords)) {
            throw new Exception('Invalid name search term');
        }

        $sql = "SELECT id, remote_customer_id, first_name, last_name, folder_name, folder_path, folder_year
                FROM unified_customers
                WHERE folder_path IS NOT NULL";

        $params = [];
        $types = "";

        foreach ($searchWords as $word) {
            $sql .= " AND (first_name LIKE ? OR last_name LIKE ?)";
            $params[] = "%$word%";
            $params[] = "%$word%";
            $types .= "ss";
        }

        if ($yearFilter > 0) {
            $sql .= " AND folder_year = ?";
            $params[] = $yearFilter;
            $types .= "i";
        }

        $sql .= " ORDER BY folder_year DESC LIMIT 50";

        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();

        while ($row = $result->fetch_assoc()) {
            $customers[] = $row;
        }
        $stmt->close();
    }

    $conn->close();

    if (empty($customers)) {
        echo json_encode([
            'success' => true,
            'customer_id' => $customerId ?: null,
            'customer_name' => $nameSearch ?: null,
            'years' => [],
            'photo_types' => ['survey' => [], 'installation' => []],
            'stats' => [
                'total_photos' => 0,
                'survey_photos' => 0,
                'installation_photos' => 0,
                'years_with_photos' => []
            ]
        ]);
        exit();
    }

    // Determine customer display info from first result
    $displayCustomerId = $customers[0]['remote_customer_id'] ?: $customerId;
    $displayName = trim($customers[0]['last_name'] . ', ' . $customers[0]['first_name']);

    // Collect all years from customer records
    $allYears = array_unique(array_filter(array_column($customers, 'folder_year')));
    rsort($allYears);

    // Base URL prefix for serve_photo.php (co-located in /var/www/html/upload/)
    $servePhotoBase = 'serve_photo.php?path=';

    // Scan photo directories
    $photoTypes = ['survey' => [], 'installation' => []];
    $totalSurvey = 0;
    $totalInstallation = 0;
    $yearsWithPhotos = [];

    // Which types to scan
    $typesToScan = ($typeFilter === 'all') ? ['Survey', 'Installation'] : [ucfirst($typeFilter)];

    foreach ($customers as $customer) {
        $basePath = $customer['folder_path'];
        $year = $customer['folder_year'] ? (string)$customer['folder_year'] : 'unknown';

        if (!is_dir($basePath)) {
            continue;
        }

        foreach ($typesToScan as $photoType) {
            $typePath = $basePath . '/' . $photoType;
            $typeKey = strtolower($photoType);

            if (!is_dir($typePath)) {
                continue;
            }

            // Get all subdirectories (photo sets) within Survey/ or Installation/
            $subDirs = glob($typePath . '/*', GLOB_ONLYDIR);

            foreach ($subDirs as $subDir) {
                $dirName = basename($subDir);

                // Skip thumbs directories at this level
                if (strtolower($dirName) === 'thumbs') {
                    continue;
                }

                $photos = scanPhotoFolder($subDir, $servePhotoBase);

                if (empty($photos)) {
                    continue;
                }

                // Initialize year group if needed
                if (!isset($photoTypes[$typeKey][$year])) {
                    $photoTypes[$typeKey][$year] = [
                        'folders' => [],
                        'total_photos' => 0
                    ];
                }

                $photoTypes[$typeKey][$year]['folders'][] = [
                    'name' => $dirName,
                    'photo_count' => count($photos),
                    'photos' => $photos
                ];

                $photoTypes[$typeKey][$year]['total_photos'] += count($photos);

                if ($typeKey === 'survey') {
                    $totalSurvey += count($photos);
                } else {
                    $totalInstallation += count($photos);
                }

                if (!in_array($year, $yearsWithPhotos)) {
                    $yearsWithPhotos[] = $year;
                }
            }
        }
    }

    // Sort years descending within each type
    foreach ($photoTypes as $typeKey => &$yearData) {
        krsort($yearData);
    }
    unset($yearData);

    rsort($yearsWithPhotos);

    $totalPhotos = $totalSurvey + $totalInstallation;

    echo json_encode([
        'success' => true,
        'customer_id' => $displayCustomerId ? (int)$displayCustomerId : null,
        'customer_name' => $displayName,
        'years' => array_map('intval', $allYears),
        'photo_types' => $photoTypes,
        'stats' => [
            'total_photos' => $totalPhotos,
            'survey_photos' => $totalSurvey,
            'installation_photos' => $totalInstallation,
            'years_with_photos' => $yearsWithPhotos
        ]
    ], JSON_INVALID_UTF8_SUBSTITUTE);

} catch (Exception $e) {
    error_log("Error in get_survey_photos.php: " . $e->getMessage());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

/**
 * Scan a photo sub-folder for images.
 * Prefers WebP thumbnails from thumbs/ and thumbs/mid/, falls back to originals.
 *
 * @param string $dir  Absolute path to the photo sub-folder
 * @param string $servePhotoBase  URL prefix for serve_photo.php
 * @return array  Photo entries
 */
function scanPhotoFolder($dir, $servePhotoBase) {
    $photos = [];
    $thumbsDir = $dir . '/thumbs';
    $midDir = $dir . '/thumbs/mid';

    if (is_dir($thumbsDir)) {
        // Preferred path: use WebP thumbnails
        $thumbFiles = glob($thumbsDir . '/*.webp');
        $thumbFiles = array_merge($thumbFiles, glob($thumbsDir . '/*.WEBP'));

        foreach ($thumbFiles as $thumbFile) {
            $thumbFilename = basename($thumbFile);
            $baseFilename = pathinfo($thumbFilename, PATHINFO_FILENAME);

            // Mid-size
            $midFile = $midDir . '/' . $thumbFilename;
            $hasMid = file_exists($midFile);

            // Original — try common extensions
            $hasOriginal = false;
            $originalFile = '';
            foreach (['jpeg', 'JPEG', 'jpg', 'JPG', 'png', 'PNG'] as $ext) {
                $candidate = $dir . '/' . $baseFilename . '.' . $ext;
                if (file_exists($candidate)) {
                    $originalFile = $candidate;
                    $hasOriginal = true;
                    break;
                }
            }

            // Build URLs via serve_photo.php
            $thumbUrl = $servePhotoBase . urlencode(str_replace('/mnt/dropbox/', '', $thumbFile));

            $midUrl = $hasMid
                ? $servePhotoBase . urlencode(str_replace('/mnt/dropbox/', '', $midFile))
                : $thumbUrl;

            $originalUrl = $hasOriginal
                ? $servePhotoBase . urlencode(str_replace('/mnt/dropbox/', '', $originalFile))
                : $midUrl;

            $photos[] = [
                'filename' => $baseFilename,
                'thumbnail_url' => $thumbUrl,
                'mid_url' => $midUrl,
                'original_url' => $originalUrl,
                'has_thumbnail' => true,
                'has_mid' => $hasMid,
                'has_original' => $hasOriginal
            ];
        }
    } else {
        // Fallback: no thumbs directory — use original images directly
        $imageFiles = [];
        foreach (['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp'] as $ext) {
            $imageFiles = array_merge($imageFiles, glob($dir . '/*.' . $ext));
            $imageFiles = array_merge($imageFiles, glob($dir . '/*.' . strtoupper($ext)));
        }

        // Deduplicate (e.g., IMG.jpg and IMG.JPG could both match)
        $seen = [];
        foreach ($imageFiles as $file) {
            $filename = basename($file);
            $lower = strtolower($filename);
            if (isset($seen[$lower])) {
                continue;
            }
            $seen[$lower] = true;

            $baseFilename = pathinfo($filename, PATHINFO_FILENAME);
            $webUrl = $servePhotoBase . urlencode(str_replace('/mnt/dropbox/', '', $file));

            $photos[] = [
                'filename' => $baseFilename,
                'thumbnail_url' => $webUrl,
                'mid_url' => $webUrl,
                'original_url' => $webUrl,
                'has_thumbnail' => false,
                'has_mid' => false,
                'has_original' => true
            ];
        }
    }

    return $photos;
}
?>
