#!/usr/bin/env python3
"""
AEI Survey Photos API — Test Script

Tests the get_survey_photos.php endpoint against real customer data
on the local server (upload.aeihawaii.com).

Usage:
  python3 test_survey_photos.py                          # Run all tests
  python3 test_survey_photos.py --base-url http://localhost  # Custom base URL
  python3 test_survey_photos.py -v                       # Verbose output

Prerequisites:
  pip3 install requests
"""

import argparse
import json
import sys
import time

try:
    import requests
    import urllib3
    urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
except ImportError:
    print("ERROR: pip3 install requests")
    sys.exit(1)

# ── Configuration ──────────────────────────────────────────────────

DEFAULT_BASE_URL = "https://upload.aeihawaii.com"
API_PATH = "/get_survey_photos.php"

# Known test customers (update if data changes)
# Calvin Abe — has Survey + Installation with thumbs/mid
TEST_CUSTOMER_ID = 4955
TEST_CUSTOMER_NAME = "Abe"
TEST_CUSTOMER_YEAR = 2025

# Howard Nishimoto — has Survey + Installation (2026)
TEST_CUSTOMER_ID_2 = 7729
TEST_CUSTOMER_NAME_2 = "Nishimoto"
TEST_CUSTOMER_YEAR_2 = 2026

# ── Test Framework ──────────────────────────────────────────────────

passed = 0
failed = 0
errors = []
verbose = False


def test(name, condition, detail=""):
    global passed, failed
    if condition:
        passed += 1
        status = "PASS"
    else:
        failed += 1
        errors.append(f"{name}: {detail}")
        status = "FAIL"
    print(f"  [{status}] {name}")
    if verbose and detail:
        print(f"         {detail}")


def section(title):
    print(f"\n{'='*60}")
    print(f"  {title}")
    print(f"{'='*60}")


def api_get(base_url, params):
    """Make a GET request to the survey photos API."""
    url = base_url + API_PATH
    try:
        resp = requests.get(url, params=params, timeout=30, verify=False)
        return resp
    except requests.exceptions.ConnectionError as e:
        return None


# ── Tests ──────────────────────────────────────────────────────────

def test_01_basic_customer_id_lookup(base_url):
    """Lookup by customer_id returns valid response with photos."""
    section("Test 1: Lookup by customer_id")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID})
    test("HTTP 200", resp is not None and resp.status_code == 200,
         f"status={resp.status_code if resp else 'connection failed'}")

    if not resp or resp.status_code != 200:
        return None

    data = resp.json()
    test("success=true", data.get("success") is True)
    test("customer_id matches", data.get("customer_id") == TEST_CUSTOMER_ID,
         f"got {data.get('customer_id')}")
    test("customer_name present", bool(data.get("customer_name")),
         f"got '{data.get('customer_name')}'")
    test("years is list", isinstance(data.get("years"), list))
    test("photo_types has survey", "survey" in data.get("photo_types", {}))
    test("photo_types has installation", "installation" in data.get("photo_types", {}))
    test("stats present", "stats" in data)
    test("total_photos > 0", data.get("stats", {}).get("total_photos", 0) > 0,
         f"total={data.get('stats', {}).get('total_photos', 0)}")

    return data


def test_02_survey_photos_structure(base_url):
    """Verify Survey photos have correct nested structure."""
    section("Test 2: Survey photo structure")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "type": "survey"})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)

    survey = data.get("photo_types", {}).get("survey", {})
    test("survey has year keys", len(survey) > 0, f"years: {list(survey.keys())}")

    # Check first year's folder structure
    found_photos = False
    for year, year_data in survey.items():
        folders = year_data.get("folders", [])
        test(f"year {year} has folders", len(folders) > 0, f"count={len(folders)}")

        for folder in folders:
            test("folder has name", bool(folder.get("name")))
            test("folder has photo_count", folder.get("photo_count", 0) > 0)
            test("folder has photos list", isinstance(folder.get("photos"), list))

            if folder.get("photos"):
                photo = folder["photos"][0]
                test("photo has filename", bool(photo.get("filename")))
                test("photo has thumbnail_url", bool(photo.get("thumbnail_url")))
                test("photo has mid_url", bool(photo.get("mid_url")))
                test("photo has original_url", bool(photo.get("original_url")))
                test("photo has has_thumbnail", "has_thumbnail" in photo)
                test("photo has has_mid", "has_mid" in photo)
                test("photo has has_original", "has_original" in photo)
                found_photos = True
                break
        if found_photos:
            break

    # Installation should be empty when type=survey
    installation = data.get("photo_types", {}).get("installation", {})
    test("installation empty with type=survey", len(installation) == 0,
         f"got {len(installation)} year groups")

    # Stats should reflect survey only
    stats = data.get("stats", {})
    test("installation_photos=0 with type=survey",
         stats.get("installation_photos", -1) == 0)


def test_03_installation_photos(base_url):
    """Verify Installation photos filter works."""
    section("Test 3: Installation photo filter")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "type": "installation"})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)

    installation = data.get("photo_types", {}).get("installation", {})
    test("installation has data", len(installation) > 0)

    survey = data.get("photo_types", {}).get("survey", {})
    test("survey empty with type=installation", len(survey) == 0)

    stats = data.get("stats", {})
    test("survey_photos=0 with type=installation",
         stats.get("survey_photos", -1) == 0)
    test("installation_photos > 0",
         stats.get("installation_photos", 0) > 0,
         f"count={stats.get('installation_photos', 0)}")


def test_04_name_search(base_url):
    """Verify name-based search returns results."""
    section("Test 4: Name search")

    resp = api_get(base_url, {"name": TEST_CUSTOMER_NAME})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)
    test("total_photos > 0",
         data.get("stats", {}).get("total_photos", 0) > 0,
         f"total={data.get('stats', {}).get('total_photos', 0)}")

    # Also test multi-word name search
    resp2 = api_get(base_url, {"name": "Calvin Abe"})
    data2 = resp2.json() if resp2 and resp2.status_code == 200 else {}
    test("multi-word name works", data2.get("success") is True)
    test("multi-word finds photos",
         data2.get("stats", {}).get("total_photos", 0) > 0)


def test_05_year_filter(base_url):
    """Verify year filter restricts results."""
    section("Test 5: Year filter")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "year": TEST_CUSTOMER_YEAR})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)

    # Only the filtered year should appear
    all_years_in_data = set()
    for type_data in data.get("photo_types", {}).values():
        for year_key in type_data.keys():
            all_years_in_data.add(str(year_key))

    test("only requested year in results",
         all_years_in_data <= {str(TEST_CUSTOMER_YEAR)},
         f"years found: {all_years_in_data}")

    # Test with wrong year should return empty
    resp2 = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "year": 2019})
    data2 = resp2.json() if resp2 and resp2.status_code == 200 else {}
    test("wrong year returns 0 photos",
         data2.get("stats", {}).get("total_photos", 0) == 0)


def test_06_thumbnail_urls_valid(base_url):
    """Verify thumbnail and mid URLs actually serve images."""
    section("Test 6: Thumbnail URL validation")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "type": "survey"})
    data = resp.json() if resp and resp.status_code == 200 else {}

    # Find the first photo with a thumbnail
    thumb_url = None
    mid_url = None
    original_url = None
    for type_data in data.get("photo_types", {}).values():
        for year_data in type_data.values():
            for folder in year_data.get("folders", []):
                for photo in folder.get("photos", []):
                    if photo.get("has_thumbnail"):
                        thumb_url = photo["thumbnail_url"]
                        mid_url = photo["mid_url"]
                        original_url = photo["original_url"]
                        break
                if thumb_url:
                    break
            if thumb_url:
                break
        if thumb_url:
            break

    test("found a photo with thumbnail", thumb_url is not None)

    if thumb_url:
        # serve_photo.php URL may be relative — ensure leading slash
        if not thumb_url.startswith('/') and not thumb_url.startswith('http'):
            thumb_url = '/' + thumb_url
        full_thumb_url = base_url + thumb_url
        try:
            img_resp = requests.get(full_thumb_url, timeout=15, verify=False)
            test("thumbnail serves HTTP 200", img_resp.status_code == 200,
                 f"status={img_resp.status_code}, url={full_thumb_url}")
            test("thumbnail content-type is image",
                 img_resp.headers.get("Content-Type", "").startswith("image/"),
                 f"content-type={img_resp.headers.get('Content-Type')}")
            test("thumbnail has content",
                 len(img_resp.content) > 100,
                 f"size={len(img_resp.content)} bytes")
        except Exception as e:
            test("thumbnail request succeeded", False, str(e))

    if mid_url and mid_url != thumb_url:
        if not mid_url.startswith('/') and not mid_url.startswith('http'):
            mid_url = '/' + mid_url
        full_mid_url = base_url + mid_url
        try:
            mid_resp = requests.get(full_mid_url, timeout=15, verify=False)
            test("mid-size serves HTTP 200", mid_resp.status_code == 200)
            test("mid-size is image",
                 mid_resp.headers.get("Content-Type", "").startswith("image/"))
        except Exception as e:
            test("mid-size request succeeded", False, str(e))


def test_07_second_customer(base_url):
    """Test a different customer to verify generalization."""
    section("Test 7: Second customer (Nishimoto)")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID_2})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)
    test("customer_id matches", data.get("customer_id") == TEST_CUSTOMER_ID_2)

    stats = data.get("stats", {})
    test("has photos", stats.get("total_photos", 0) > 0,
         f"total={stats.get('total_photos', 0)}")


def test_08_invalid_customer(base_url):
    """Test with non-existent customer_id."""
    section("Test 8: Invalid customer_id")

    resp = api_get(base_url, {"customer_id": 9999999})
    test("HTTP 200 (empty result)", resp is not None and resp.status_code == 200)

    if resp and resp.status_code == 200:
        data = resp.json()
        test("success=true (empty)", data.get("success") is True)
        test("total_photos=0", data.get("stats", {}).get("total_photos", 0) == 0)
        test("photo_types empty", not any(data.get("photo_types", {}).values()))


def test_09_missing_params(base_url):
    """Test with no customer_id or name."""
    section("Test 9: Missing parameters")

    resp = api_get(base_url, {})
    test("returns error", resp is not None and resp.status_code == 400)

    if resp:
        data = resp.json()
        test("success=false", data.get("success") is False)
        test("error message present", bool(data.get("error")))


def test_10_invalid_type(base_url):
    """Test with invalid type parameter."""
    section("Test 10: Invalid type parameter")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "type": "bogus"})
    test("returns error", resp is not None and resp.status_code == 400)

    if resp:
        data = resp.json()
        test("success=false", data.get("success") is False)


def test_11_all_type(base_url):
    """Verify type=all returns both survey and installation."""
    section("Test 11: type=all returns both")

    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID, "type": "all"})
    data = resp.json() if resp and resp.status_code == 200 else {}

    test("success=true", data.get("success") is True)

    stats = data.get("stats", {})
    test("survey_photos > 0", stats.get("survey_photos", 0) > 0,
         f"survey={stats.get('survey_photos', 0)}")
    test("installation_photos > 0", stats.get("installation_photos", 0) > 0,
         f"installation={stats.get('installation_photos', 0)}")
    test("total = survey + installation",
         stats.get("total_photos", 0) == stats.get("survey_photos", 0) + stats.get("installation_photos", 0))


def test_12_response_time(base_url):
    """Verify API responds within reasonable time."""
    section("Test 12: Response time")

    start = time.time()
    resp = api_get(base_url, {"customer_id": TEST_CUSTOMER_ID})
    elapsed = time.time() - start

    test(f"response under 5 seconds", elapsed < 5.0, f"took {elapsed:.2f}s")
    test(f"response under 2 seconds", elapsed < 2.0, f"took {elapsed:.2f}s")


# ── Main ──────────────────────────────────────────────────────────

def main():
    global verbose

    parser = argparse.ArgumentParser(description="Test get_survey_photos.php API")
    parser.add_argument("--base-url", default=DEFAULT_BASE_URL,
                        help=f"Base URL (default: {DEFAULT_BASE_URL})")
    parser.add_argument("-v", "--verbose", action="store_true",
                        help="Show extra detail on each test")
    args = parser.parse_args()

    verbose = args.verbose
    base_url = args.base_url.rstrip("/")

    print(f"\nAEI Survey Photos API — Test Suite")
    print(f"Target: {base_url}{API_PATH}")
    print(f"Time: {time.strftime('%Y-%m-%d %H:%M:%S')}")

    # Verify API is reachable
    try:
        resp = requests.get(base_url + API_PATH, params={"customer_id": 1}, timeout=10, verify=False)
    except requests.exceptions.ConnectionError:
        print(f"\nERROR: Cannot connect to {base_url}")
        print("Is Apache running? Try: sudo systemctl start apache2")
        sys.exit(1)

    tests = [
        test_01_basic_customer_id_lookup,
        test_02_survey_photos_structure,
        test_03_installation_photos,
        test_04_name_search,
        test_05_year_filter,
        test_06_thumbnail_urls_valid,
        test_07_second_customer,
        test_08_invalid_customer,
        test_09_missing_params,
        test_10_invalid_type,
        test_11_all_type,
        test_12_response_time,
    ]

    for test_fn in tests:
        try:
            test_fn(base_url)
        except Exception as e:
            print(f"  [ERROR] {test_fn.__name__}: {e}")

    # Summary
    total = passed + failed
    print(f"\n{'='*60}")
    print(f"  Results: {passed}/{total} passed, {failed} failed")
    print(f"{'='*60}")

    if errors:
        print("\nFailures:")
        for err in errors:
            print(f"  - {err}")

    sys.exit(0 if failed == 0 else 1)


if __name__ == "__main__":
    main()
