# Enhancement: Async Local Server Sync

**Date:** 2026-02-05 (rev1), 2026-02-09 (rev2)
**Status:** rev1 Deployed, rev2 Pending
**Reconstruction:** Retroactive (created 2026-02-05 from available backups)

## Purpose

Replace all synchronous cURL calls to the local server with background Python sync (`sync_to_local.py`) invoked via `nohup &`. Provides unified async sync with retry for **all** upload paths — mobile app, scheduler photo tab, and premeasure uploads.

### Rev1 (2026-02-05) — Deployed
- Replaced synchronous cURL in `upload.php` (mobile app) with `nohup sync_to_local.py &`
- Mobile app response time dropped from 2-10s to ~0.5s
- Added store-and-forward retry queue (`process_retry_queue.py`) for resilience

### Rev2 (2026-02-09) — Pending
- Extended async sync to `phototab.php` (scheduler uploads) and `preupload.php` (premeasure uploads)
- These still used synchronous cURL with **no retry** — if the local server was offline, sync failed silently
- Also fixed `preupload.php` WebP generation (was using cURL response body instead of `generate_webp.py`)

## Files Changed

| File | Server | Path | Change |
|------|--------|------|--------|
| `upload.php` | Remote | `photoapi/` | rev1: Removed sync cURL, added `nohup sync_to_local.py &` |
| `sync_to_local.py` | Remote | `photoapi/` | rev1: **New** - Background sync with inline retry + queue |
| `process_retry_queue.py` | Remote | `photoapi/` | rev1: **New** - Cron-driven retry processor |
| `phototab.php` | Remote | `controllers/` | rev2: Removed sync cURL, added `nohup sync_to_local.py &` |
| `preupload.php` | Remote | `controllers/` | rev2: Removed sync cURL + response-body WebP, added `nohup sync_to_local.py &` + `nohup generate_webp.py &` |

## Folder Structure

```
PHOTO-003_ASYNC_SYNC/
+-- README.md                        <- This file
+-- ASYNC_SYNC_ENHANCEMENT.md        <- Full rev1 specification
+-- NEW/                             <- Post-enhancement files
|   +-- upload.php                   (rev1)
|   +-- sync_to_local.py            (rev1, new file)
|   +-- process_retry_queue.py       (rev1, new file)
|   +-- controllers/
|       +-- phototab.php             (rev2)
|       +-- preupload.php            (rev2)
+-- ORIGINAL/                        <- Pre-enhancement files
    +-- upload.php                   (rev1)
    +-- controllers/
        +-- phototab.php             (rev2)
        +-- preupload.php            (rev2)
```

## Architecture Change

### Rev1: Mobile App Upload (upload.php) — DEPLOYED

```
BEFORE (synchronous cURL - 2-10s blocking):
  Mobile App -> POST base64 image
    1. Decode + save image              (fast)
    2. echo JSON response               (BUFFERED - not sent yet)
    3. cURL POST to local server        (2-10s BLOCKING)
    4. Copy to scheduler/uploads        (fast)
    5. generate_webp.py in background   (non-blocking)
    6. Insert meter_files               (fast)
    7. Script ends -> JSON finally sent (DELAYED 2-10s)

AFTER (background Python - ~0.5s):
  Mobile App -> POST base64 image
    1. Decode + save image              (fast)
    2. Copy to scheduler/uploads        (fast)
    3. generate_webp.py in BACKGROUND   (non-blocking)
    4. Insert meter_files               (fast)
    5. sync_to_local.py in BACKGROUND   (non-blocking - nohup &)
    6. echo JSON response               (LAST - sent immediately)
```

### Rev2: Scheduler Uploads (phototab.php, preupload.php) — PENDING

```
BEFORE (synchronous cURL - blocking, NO retry):
  Scheduler -> Upload photo(s)
    1. Save to scheduler/uploads        (fast)
    2. Copy to /mnt/dropbox/ hierarchy  (fast)
    3. cURL POST to local server        (2-10s BLOCKING, NO RETRY)
    4. Generate WebP                    (varied — see bugs below)
    5. Insert meter_files               (fast)
    6. Redirect back to page            (DELAYED by cURL)

AFTER (background Python - non-blocking, WITH retry):
  Scheduler -> Upload photo(s)
    1. Save to scheduler/uploads        (fast)
    2. Copy to /mnt/dropbox/ hierarchy  (fast)
    3. sync_to_local.py in BACKGROUND   (non-blocking - nohup &)
    4. generate_webp.py in BACKGROUND   (non-blocking - nohup &)
    5. Insert meter_files               (fast)
    6. Redirect back to page            (IMMEDIATE)
```

## Rev2 Changes Detail

### phototab.php `dropImageUpload()`

**1. Added `js.customer_id` to SQL SELECT** (needed for sync_to_local.py arg):
```php
// BEFORE:
"SELECT cs.first_name,cs.last_name,js.job_date,jt.intials FROM jobs js ..."
// AFTER:
"SELECT cs.first_name,cs.last_name,js.customer_id,js.job_date,jt.intials FROM jobs js ..."

$customer_id=$row["customer_id"];
```

**2. Replaced cURL block (old lines 707-755) with:**
```php
$syncScript = '/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py';
$syncCmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($syncScript)
    . ' ' . escapeshellarg($filePath)
    . ' ' . escapeshellarg(basename($filePath))
    . ' ' . escapeshellarg($job_id)
    . ' ' . escapeshellarg($customer_id)
    . ' ' . escapeshellarg($job_type_text)
    . ' ' . escapeshellarg($last_name)
    . ' ' . escapeshellarg($first_name)
    . ' ' . escapeshellarg($norm_date)
    . ' ' . escapeshellarg($photo_type)
    . ' > /dev/null 2>&1 &';
exec($syncCmd);
```

**WebP generation:** Unchanged — already uses `nohup generate_webp.py &` (correct pattern).

### preupload.php `submit()`

**1. Replaced cURL block (old lines 164-212) with sync_to_local.py** (same pattern as phototab.php).

**2. Replaced broken WebP generation (old lines 213-222):**

The old code used the cURL response body to create WebP files:
```php
// OLD (broken): WebP from cURL response body
$responseWithHeaders = curl_exec($ch);
list($respHeaders, $respBody) = explode("\r\n\r\n", $responseWithHeaders, 2);
file_put_contents($final_folder_webp, $respBody);  // response body as WebP?!
```

This was fragile and incorrect. Replaced with `generate_webp.py` (matching phototab.php pattern):
```php
// NEW: generate_webp.py in background (correct pattern)
$generateScript = '/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/generate_webp.py';
$source_image = '/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/' . $data['file_name'];
$cmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($generateScript)
     . ' ' . escapeshellarg($source_image)
     . ' ' . escapeshellarg($webp_dest)
     . ' 80'
     . ' && cp ' . escapeshellarg($webp_dest) . ' ' . escapeshellarg($scheduler_webp_dest)
     . ' > /dev/null 2>&1 &';
exec($cmd);
```

**3. Fixed WebP filename bug:**
Old code used hardcoded `-S` (Survey) in the scheduler/uploads/webp copy regardless of actual photo type:
```php
// OLD (bug): always "-S" regardless of photo_type
copy($final_folder_webp, ".../scheduler/uploads/webp/".$customer_namewebp."".$job_type_text."-S".$job_date...);
```
New code uses `$webpfilename` (which includes correct `$photo_type`).

**4. Cleaned up DB insert:**
```php
// OLD: reconstructed filename inline
$row['webpfilename']=$customer_namewebp."".$job_type_text."-".$photo_type.$job_date."_IMG".$randomId.'.webp';
// NEW: uses $webpfilename variable (already constructed)
$row['webpfilename']=$webpfilename;
```

## Retry Architecture (all upload paths)

```
sync_to_local.py (inline retry for transient failures):
  +-- Attempt 1 -> fail -> wait 2s
  +-- Attempt 2 -> fail -> wait 4s
  +-- Attempt 3 -> fail -> wait 8s
  +-- All failed + source file exists -> enqueue JSON to queue/

process_retry_queue.py (cron every 15min):
  +-- Read queue/*.json files
  +-- Retry POST -> success -> delete JSON
  +-- Retry POST -> fail -> increment retry_count
  +-- retry_count >= 10 -> move to queue/failed/
```

**All three upload paths now share this retry architecture:**

| Upload Path | Source | Sync | Retry | WebP |
|---|---|---|---|---|
| Mobile app | `upload.php` | `sync_to_local.py` (nohup) | Inline + queue | `generate_webp.py` (nohup) |
| Scheduler photos | `phototab.php` | `sync_to_local.py` (nohup) | Inline + queue | `generate_webp.py` (nohup) |
| Premeasure uploads | `preupload.php` | `sync_to_local.py` (nohup) | Inline + queue | `generate_webp.py` (nohup) |

## Performance Impact

| Metric | Before (rev1 only) | After (rev2) |
|--------|-------------------|-------------|
| Mobile app sync | Async + retry | Async + retry (unchanged) |
| Scheduler sync | **Synchronous, no retry** | Async + retry |
| Premeasure sync | **Synchronous, no retry** | Async + retry |
| Local server offline | Mobile: resilient, **Scheduler: silent failure** | **All paths resilient** |
| Scheduler page response | Delayed 2-10s by cURL | Immediate |

## Deployment (Rev2)

```bash
# 1. Backup current controllers on remote
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php \
     /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php.bak.pre_async_sync.$(date +%Y%m%d)
  cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php \
     /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php.bak.pre_async_sync.$(date +%Y%m%d)
"

# 2. Upload modified controllers
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/ENHANCEMENTS/PHOTO-003_ASYNC_SYNC/NEW/controllers/phototab.php \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php

scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/ENHANCEMENTS/PHOTO-003_ASYNC_SYNC/NEW/controllers/preupload.php \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php

# 3. Set permissions
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php
"

# 4. Ensure AWS IP is whitelisted on local server
sudo ipset test trusted_whitelist 18.225.0.90 2>/dev/null || \
  sudo ipset add trusted_whitelist 18.225.0.90
```

## Rollback (Rev2)

```bash
# Restore pre-rev2 controllers
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php.bak.pre_async_sync.20260209 \
          /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php.bak.pre_async_sync.20260209 \
          /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/phototab.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/system/application/controllers/preupload.php
"
```

**Warning:** Rolling back reintroduces synchronous cURL in scheduler (blocking + no retry).

## Rollback (Rev1 — upload.php)

```bash
# Restore pre-async-sync upload.php (reverts to synchronous cURL)
scp -i /root/.ssh/aei_remote.pem ORIGINAL/upload.php \
   Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php

# Remove retry cron
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
   "sudo crontab -u ec2-user -l | grep -v process_retry | sudo crontab -u ec2-user -"
```

## Reconstruction Quality

| File | ORIGINAL | NEW |
|------|----------|-----|
| `upload.php` | **Approximate** - nearest pre-async-sync backup | **Approximate** - post-async-sync state |
| `sync_to_local.py` | N/A (new file) | **Exact** - deployed version |
| `process_retry_queue.py` | N/A (new file) | **Exact** - deployed retry processor |
| `controllers/phototab.php` | **Exact** - current deployed version | **Exact** - cURL replaced with sync_to_local.py |
| `controllers/preupload.php` | **Exact** - current deployed version | **Exact** - cURL + response-body WebP replaced |

## Bugs Fixed in Rev2

| Bug | File | Description |
|-----|------|-------------|
| Silent sync failure | `phototab.php` | Scheduler uploads failed silently when local server offline — no retry |
| Silent sync failure | `preupload.php` | Premeasure uploads failed silently when local server offline — no retry |
| WebP from response body | `preupload.php` | Used cURL response body as WebP data instead of generating locally |
| Hardcoded `-S` in WebP | `preupload.php` | WebP copy to `scheduler/uploads/webp/` always used `-S` (Survey) suffix regardless of photo type |

## Full Documentation

See `ASYNC_SYNC_ENHANCEMENT.md` (in this folder) for complete rev1 specification including:
- sync_to_local.py argument list and behavior
- process_retry_queue.py constants and queue JSON format
- Inline retry details
- Deployment issues discovered during E2E testing
- Monitoring and troubleshooting commands
