# Async Local Server Sync Enhancement

**Date:** 2026-02-05
**Status:** Deployed
**Files Changed:** `upload.php`, `sync_to_local.py` (new)
**Server:** Remote (18.225.0.90)

---

## Problem

The synchronous cURL call in `upload.php` blocked the mobile app response for 2-10+ seconds per upload. PHP buffers the `echo` JSON response until the script ends, so the mobile app waited for the entire local server round-trip before receiving any response.

The local server sync is an internal copy for office storage — the mobile app has no dependency on it.

### Impact

- Mobile app experienced slow uploads and occasional timeouts
- If the local server was unreachable (firewall, reboot), every mobile upload was delayed by the cURL connect timeout (previously unbounded, then 5s)
- Users perceived the system as unreliable during local server maintenance windows

---

## Solution

Replaced the synchronous cURL block (old lines 105-162 in `upload.php`) with a background Python script (`sync_to_local.py`) invoked via `nohup ... &`, matching the proven pattern already used by `generate_webp.py`.

### Architecture Change

```
BEFORE (synchronous cURL):
  Mobile App → POST base64 image
    1. Decode + save image                    (fast)
    2. echo JSON response                     (BUFFERED — not sent yet)
    3. cURL POST to local server              (2-10s BLOCKING)
    4. Copy to scheduler/uploads              (fast)
    5. generate_webp.py in background         (non-blocking)
    6. Insert meter_files                     (fast)
    7. Script ends → JSON finally sent        (DELAYED 2-10s)

AFTER (background Python):
  Mobile App → POST base64 image
    1. Decode + save image                    (fast)
    2. Copy to scheduler/uploads              (fast)
    3. generate_webp.py in BACKGROUND         (non-blocking — nohup &)
    4. Insert meter_files                     (fast)
    5. sync_to_local.py in BACKGROUND         (non-blocking — nohup &)
    6. echo JSON response                     (LAST — sent immediately)
    7. Script ends → mobile app has response  (~0.1-0.5s total)
       ... local sync happens 2-10s later     (background)
       ... WebP appears 1-3s later            (background)
```

### Key Design Decision

The `echo` JSON response was moved to the **last line** of `upload.php` (before `?>`). Since PHP buffers output and sends it when the script ends, placing `echo` last — after all fast operations and background launches — ensures the mobile app receives the response as quickly as possible. The `nohup ... &` processes continue running after the PHP script exits.

---

## New File: `sync_to_local.py`

**Remote Path:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py`
**Local Path:** `REMOTE/photoapi/sync_to_local.py`
**Permissions:** `755`, `ec2-user:ec2-user`
**Runtime:** Python 3.6 (`/usr/local/bin/python3.6`), requires `requests` module

### Arguments (9 positional, via `escapeshellarg()`)

| # | Argument | Source in upload.php |
|---|----------|---------------------|
| 1 | `file_path` | `$filePath` (full path to saved image) |
| 2 | `file_name` | `basename($filePath)` |
| 3 | `job_id` | `$job_id` (from `js.job_pid`) |
| 4 | `customer_id` | `$customer_id` (from `js.customer_id`) |
| 5 | `job_type` | `$job_type_text` (PM, SC, AC, etc.) |
| 6 | `last_name` | `$last_name` |
| 7 | `first_name` | `$first_name` |
| 8 | `job_date` | `$job_date_t` (YYYY-MM-DD) |
| 9 | `photo_type` | `$photo_type` (S or I) |

### Behavior

1. Validates that `file_path` exists on disk
2. Opens the file and POSTs it as multipart form data to `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php`
3. Includes all metadata fields (auth_token, customer_id, job_type, etc.)
4. Logs result to `photoapi/logs/sync_to_local.log`
5. Timeouts: connect=10s, read=60s (generous since running in background)
6. Exits 0 on success, 1 on failure (no impact on upload.php — already exited)

### Invocation in upload.php

```php
// Sync to local server in background (non-blocking)
$syncScript = __DIR__ . '/sync_to_local.py';
$syncCmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($syncScript)
    . ' ' . escapeshellarg($filePath)
    . ' ' . escapeshellarg(basename($filePath))
    . ' ' . escapeshellarg($job_id)
    . ' ' . escapeshellarg($customer_id)
    . ' ' . escapeshellarg($job_type_text)
    . ' ' . escapeshellarg($last_name)
    . ' ' . escapeshellarg($first_name)
    . ' ' . escapeshellarg($job_date_t)
    . ' ' . escapeshellarg($photo_type)
    . ' > /dev/null 2>&1 &';
exec($syncCmd);
```

### Security

- All 9 arguments passed through `escapeshellarg()` — prevents shell injection
- Auth token for the local server is hardcoded in `sync_to_local.py` (not passed via CLI)
- `verify=False` on HTTPS — the local server uses a self-signed certificate

---

## What Was Removed from upload.php

The entire synchronous cURL block was deleted:

```php
// REMOVED — these lines no longer exist in upload.php:
$remoteUploadUrl = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php';
$remoteAuthToken = '***';
$postFields = array(
    'auth_token'  => $remoteAuthToken,
    'file'        => '@' . $filePath,
    'file_name'   => basename($filePath),
    'job_id'      => $job_id,
    'customer_id' => $customer_id,
    'job_type'    => $job_type_text,
    'last_name'   => $last_name,
    'first_name'  => $first_name,
    'job_date'    => $job_date_t,
    'photo_type'  => $photo_type,
);
$ch = curl_init($remoteUploadUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
$response = curl_exec($ch);
// ... error handling ...
curl_close($ch);
```

**`curl_init()`, `curl_exec()`, `curl_close()` should NOT be present in the current `upload.php`.**

---

## Logging

**Log File:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log`

### Log Format

```
2026-02-05 08:23:15 INFO OK photo.jpg -> HTTP 200 (12345)
2026-02-05 08:23:16 WARNING FAIL photo.jpg -> HTTP 500 body={"status":"error"...}
2026-02-05 08:23:17 ERROR ERROR photo.jpg -> ConnectionError: ...
```

### Log Directory Permissions

The `logs/` directory must be `777` because Apache runs as `apache` user while files are owned by `ec2-user`:

```bash
# Required permissions
sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/
sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/
```

### Monitoring

```bash
# Watch sync activity in real-time
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -f /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log"

# Check for recent failures
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "grep -E 'WARNING|ERROR' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log | tail -20"
```

---

## Performance Impact

| Metric | Before (sync cURL) | After (background Python) |
|--------|-------------------|--------------------------|
| Mobile app wait | 2-10+ seconds | ~0.1-0.5 seconds |
| Local server down impact | 5s timeout delay per upload | Zero impact on mobile app |
| Echo position in script | Line 99 (buffered, delayed) | Last line (sent immediately) |
| Sync mechanism | PHP cURL (synchronous) | Python requests (background nohup) |
| Error visibility | `error_log()` in PHP error log | Dedicated `logs/sync_to_local.log` |
| Local server maintenance | Degrades mobile upload speed | No mobile app impact |

---

## Prerequisites / Dependencies

### Remote Server

| Requirement | Details |
|-------------|---------|
| Python 3.6 | `/usr/local/bin/python3.6` (no `python3` symlink) |
| `requests` module | `pip3.6 install requests` |
| `logs/` directory | Must exist with `chmod 777` |
| `sync_to_local.py` | Must be `chmod 755`, `chown ec2-user:ec2-user` |

### Local Server (Firewall)

The local server's firewall uses an ipset whitelist with a DROP default INPUT policy. The AWS IP **must** be whitelisted for the background sync to connect:

```bash
# Check if whitelisted
sudo ipset test trusted_whitelist 18.225.0.90

# Add if missing (required after local server reboot)
sudo ipset add trusted_whitelist 18.225.0.90
```

The whitelist is volatile (cleared on reboot). The IP is persisted in `/var/www/html/security/whitelist.json` and reloaded on boot. See `/var/www/html/security/README.md`.

---

## Deployment

```bash
# 1. Create logs directory on remote
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo mkdir -p /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
"

# 2. Backup current upload.php
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
     /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.$(date +%Y%m%d)
"

# 3. Upload sync_to_local.py
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/sync_to_local.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
"

# 4. Upload modified upload.php
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/upload.php \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"

# 5. Ensure AWS IP is whitelisted on local server firewall
sudo ipset test trusted_whitelist 18.225.0.90 2>/dev/null || \
  sudo ipset add trusted_whitelist 18.225.0.90
```

---

## Rollback

```bash
# Restore pre-async-sync backup (reverts to synchronous cURL — slower but functional)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"
# sync_to_local.py can remain on disk — only invoked if upload.php calls it
```

**Warning:** Rolling back reintroduces synchronous cURL, which will:
- Add 2-10s delay to every mobile app upload
- Cause 5s timeout delays if local server is unreachable
- The `echo` position also reverts, so response timing changes

---

## Backups

| Location | File | Description |
|----------|------|-------------|
| Remote server | `upload.php.bak.pre_async_sync.20260205` | Before async sync change (has sync cURL) |
| Project | `REMOTE/photoapi/upload.php.bak.pre_async_webp.20260204` | Before async WebP (also has sync cURL) |

---

## Deployment Issues Discovered

These issues were found during E2E testing after initial deployment:

| Issue | Root Cause | Fix |
|-------|-----------|-----|
| `sync_to_local.log` not created | `photoapi/logs/` had `755` permissions. Apache `apache` user can't write. | `sudo chmod 777 photoapi/logs/` |
| `sync_to_local.py` connection timeout | AWS IP (18.225.0.90) not in `trusted_whitelist` ipset on local server. Firewall INPUT policy is DROP. | `sudo ipset add trusted_whitelist 18.225.0.90` |
| `file_put_contents` failed on upload | `/mnt/dropbox/2026 Customers/` had `775` (ec2-user:ec2-user). Apache runs as `apache`, not in ec2-user group. | `sudo chmod 777 '/mnt/dropbox/2026 Customers/'` |

**Key Lesson:** The remote server's Apache runs as `apache` user, while files deployed via SSH are owned by `ec2-user`. Any directory that Apache needs to write to must be `777` or have `apache` in the owner/group.

---

## Troubleshooting

### Sync Not Working (No Log Entries)

```bash
# 1. Check if sync_to_local.py exists and is executable
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py"

# 2. Check if logs/ directory is writable
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/"

# 3. Check if Python requests module is available
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "/usr/local/bin/python3.6 -c 'import requests; print(\"OK\")'"

# 4. Check if upload.php still has the sync_to_local.py invocation
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "grep 'sync_to_local' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
```

### Sync Failing (ERROR in Log)

```bash
# 1. Check local server firewall
sudo ipset test trusted_whitelist 18.225.0.90

# 2. Test connectivity from remote to local
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "curl -sk -o /dev/null -w '%{http_code}' https://upload.aeihawaii.com/"

# 3. Check local server Apache is running
systemctl status apache2
```

---

## Store-and-Forward Retry Queue

**Date Added:** 2026-02-05
**Files:** `sync_to_local.py` (modified), `process_retry_queue.py` (new)

### Problem

`sync_to_local.py` was fire-and-forget. If the local server was down, firewalled, or returned an error, the photo never reached the office server — and nobody knew. The mobile app user saw "Success" because `upload.php` already responded.

### Solution

Two-phase store-and-forward: on failure, `sync_to_local.py` saves the payload to a `queue/` directory as a JSON file. A cron-driven `process_retry_queue.py` retries those files every 15 minutes.

### Architecture

```
sync_to_local.py (called by upload.php via nohup &)
  ├─ POST to local server succeeds → done (same as before)
  └─ POST fails (connection error, HTTP error) AND source file exists
      └─ Serialize payload to queue/sync_{timestamp}_{random6}.json
          └─ process_retry_queue.py (cron every 15 min)
              ├─ Retry POST → success → delete JSON file
              ├─ Retry POST → fail → increment retry_count, update JSON
              └─ retry_count >= 10 → move to queue/failed/
```

**Exception:** If the source `file_path` doesn't exist on disk, the failure is NOT queued — the photo is gone and retrying is pointless.

### Queue Directory

**Remote Path:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/`
**Local Path:** `REMOTE/photoapi/queue/`
**Permissions:** `777`, `ec2-user:ec2-user` (Apache writes here as `apache` user)

### Queue JSON Format

**Filename:** `sync_{YYYYMMDD}_{HHMMSS}_{random6}.json`

```json
{
  "file_path": "/mnt/dropbox/2026 Customers/H/How, Clifford/Installation/.../photo.jpg",
  "file_name": "photo.jpg",
  "job_id": "12345",
  "customer_id": "18093",
  "job_type": "SC",
  "last_name": "How",
  "first_name": "Clifford",
  "job_date": "2026-02-13",
  "photo_type": "I",
  "queued_at": "2026-02-05 14:30:22",
  "retry_count": 0,
  "last_error": "ConnectionError: Connection timed out"
}
```

After a failed retry, `last_retry` timestamp is added and `retry_count` is incremented.

### `process_retry_queue.py`

**Remote Path:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py`
**Local Path:** `REMOTE/photoapi/process_retry_queue.py`
**Permissions:** `755`, `ec2-user:ec2-user`
**Log:** `photoapi/logs/retry_queue.log`

**Cron entry (ec2-user):**
```
*/15 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py > /dev/null 2>&1
```

**Constants:**
| Constant | Value |
|----------|-------|
| `MAX_RETRIES` | 10 (~2.5 hours at 15-min intervals) |
| `QUEUE_DIR` | `photoapi/queue/` |
| `FAILED_DIR` | `photoapi/queue/failed/` |
| `UPLOAD_URL` | Same as `sync_to_local.py` |
| `AUTH_TOKEN` | Same as `sync_to_local.py` |

**Behavior per queued file:**
1. Read JSON payload
2. If `retry_count >= MAX_RETRIES` → move to `queue/failed/`, log warning
3. If source `file_path` doesn't exist → delete JSON, log skip
4. POST to local server (same as `sync_to_local.py`)
5. On success → delete JSON, log success
6. On failure → increment `retry_count`, update `last_error` and `last_retry`, rewrite JSON

### Monitoring

```bash
# Check queue contents
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/"

# Check retry log
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -20 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/retry_queue.log"

# Check failed items
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/failed/"

# Manually run retry processor
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "/usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py"

# Verify cron is installed
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "sudo crontab -u ec2-user -l | grep process_retry"
```

### Deployment

```bash
# 1. Create queue directories on remote
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo mkdir -p /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/failed
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/failed
  sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue
  sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue/failed
"

# 2. Backup current sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py \
     /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py.bak.pre_retry_queue.\$(date +%Y%m%d)
"

# 3. Upload modified sync_to_local.py
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/sync_to_local.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py

# 4. Upload new process_retry_queue.py
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/process_retry_queue.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py

# 5. Set permissions
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
"

# 6. Add cron entry for ec2-user
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  (sudo crontab -u ec2-user -l 2>/dev/null; echo '*/15 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py > /dev/null 2>&1') | sudo crontab -u ec2-user -
"
```

### Rollback

```bash
# Restore pre-retry-queue sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py.bak.pre_retry_queue.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
"

# Remove cron entry
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo crontab -u ec2-user -l | grep -v process_retry | sudo crontab -u ec2-user -
"

# queue/ directory and process_retry_queue.py can remain — they're inert without the queuing code
```

---

## Inline Retry

**Date Added:** 2026-02-05

`sync_to_local.py` retries up to 3 times with exponential backoff (2s, 4s, 8s) before falling back to the cron-based retry queue. This handles transient network failures between the remote and local servers.

```
sync_to_local.py (with inline retry):
  ├─ Attempt 1 → fail → wait 2s
  ├─ Attempt 2 → fail → wait 4s
  ├─ Attempt 3 → fail → wait 8s (worst case: 14s total)
  └─ All failed → enqueue for cron retry (15 min)
```

**Constants:**
| Constant | Value |
|----------|-------|
| `INLINE_RETRIES` | 3 |
| `RETRY_DELAYS` | [2, 4, 8] seconds |

**Log entries:**
```
2026-02-05 09:15:01 ERROR ERROR photo.jpg -> ConnectTimeout: ...
2026-02-05 09:15:01 INFO RETRY photo.jpg -> attempt 1/3 failed, retrying in 2s
2026-02-05 09:15:03 INFO OK photo.jpg -> HTTP 200 (12345)
```

---

## Related Documentation

| Document | Purpose |
|----------|---------|
| [PHOTO_SYSTEM_DOCUMENTATION.md](../../DOCS/PHOTO_SYSTEM_DOCUMENTATION.md) | Complete system documentation |
| [REMOTE_UPLOAD_CHANGES.md](../CUSTOMER_ID_LOOKUP/REMOTE_UPLOAD_CHANGES.md) | Full upload.php change history |
| [MOBILE_API_RESTORE_GUIDE.md](../../DOCS/MOBILE_API_RESTORE_GUIDE.md) | Restore procedures (updated for async sync) |

---

*Created: 2026-02-05*
