#!/usr/bin/env python3
"""
Generate WebP thumbnail from source image.
Called by upload.php to create WebP versions locally on the remote server.

Usage: python3 generate_webp.py <source_image> <destination.webp> [quality]

Requires: Pillow (pip3 install Pillow)
"""

import os
import sys
from PIL import Image, ImageOps

DEFAULT_QUALITY = 80

def convert_to_webp(source, destination, quality=DEFAULT_QUALITY):
    """Convert an image to WebP format."""
    try:
        with Image.open(source) as img:
            img = ImageOps.exif_transpose(img)
            img.save(destination, 'WEBP', quality=quality)
        return True
    except Exception as e:
        print("Error: {}".format(str(e)), file=sys.stderr)
        return False

if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("Usage: python3 generate_webp.py <source> <destination.webp> [quality]")
        sys.exit(1)

    source = sys.argv[1]
    destination = sys.argv[2]
    quality = int(sys.argv[3]) if len(sys.argv) > 3 else DEFAULT_QUALITY

    if not os.path.isfile(source):
        print("Error: Source file not found: {}".format(source), file=sys.stderr)
        sys.exit(1)

    # Ensure destination directory exists
    dest_dir = os.path.dirname(destination)
    if dest_dir and not os.path.isdir(dest_dir):
        os.makedirs(dest_dir, mode=0o755, exist_ok=True)

    success = convert_to_webp(source, destination, quality)
    sys.exit(0 if success else 1)
