# Remote Server upload.php Changes

**Date:** 2026-01-30
**Purpose:** Add `customer_id` to photo sync payload for improved local folder matching

---

## Summary of Changes

The remote server's `/photoapi/upload.php` has been modified to include `customer_id` when sending photos to the local server. This enables the local server to use database-driven customer folder lookups instead of unreliable name matching.

---

## Files

| File | Purpose |
|------|---------|
| `backup/upload.php.original.20260130` | Original file backup |
| `photoapi/upload.php` | Modified file (ready for deployment) |

---

## Changes Made

### 1. Database Query (Lines 30-35)

**Before:**
```php
$query = "SELECT cs.first_name,cs.last_name,js.job_date,jt.intials,js.job_pid FROM jobs js LEFT JOIN customers cs ON js.customer_id=cs.id LEFT JOIN job_types jt ON js.job_type_id=jt.id
WHERE js.id=".$data['job_id'];
```

**After:**
```php
$query = "SELECT js.customer_id, cs.first_name, cs.last_name, js.job_date, jt.intials, js.job_pid
          FROM jobs js
          LEFT JOIN customers cs ON js.customer_id = cs.id
          LEFT JOIN job_types jt ON js.job_type_id = jt.id
          WHERE js.id = " . intval($data['job_id']);
```

**Changes:**
- Added `js.customer_id` to SELECT
- Added `intval()` for SQL injection protection
- Reformatted for readability

### 2. Extract customer_id (Line 38)

**Added:**
```php
$customer_id = $row['customer_id'];
```

### 3. POST Fields to Local Server (Lines 112-123)

**Before:**
```php
$postFields = array(
    'auth_token' => $remoteAuthToken,
    'file'       => '@' . $filePath,
    'file_name'  => basename($filePath),
    'job_id'     => $job_id,
    'job_type'   => $job_type_text,
    'last_name'  => $last_name,
    'first_name' => $first_name,
    'job_date'   => $job_date_t,
    'photo_type' => $photo_type,
);
```

**After:**
```php
$postFields = array(
    'auth_token'  => $remoteAuthToken,
    'file'        => '@' . $filePath,
    'file_name'   => basename($filePath),
    'job_id'      => $job_id,
    'customer_id' => $customer_id,    // NEW: enables direct customer lookup
    'job_type'    => $job_type_text,
    'last_name'   => $last_name,
    'first_name'  => $first_name,
    'job_date'    => $job_date_t,
    'photo_type'  => $photo_type,
);
```

---

## Deployment Instructions

### File Ownership

**IMPORTANT:** All web files must be owned by `ec2-user:ec2-user`, NOT the SSH user (`Julian`).

### Step 1: Convert SSH Key

```bash
puttygen SSH/schedular_server_private_key.ppk -O private-openssh -o /tmp/aei_key
chmod 600 /tmp/aei_key
```

### Step 2: Backup Current File on Remote

```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
      /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.$(date +%Y%m%d)"
```

### Step 3: Upload Modified File

```bash
# Upload to /tmp
scp -i /tmp/aei_key \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/upload.php \
  Julian@18.225.0.90:/tmp/upload.php.new

# Copy to destination
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "cp /tmp/upload.php.new /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
```

### Step 4: Restore Ownership

```bash
# IMPORTANT: Restore ec2-user ownership (not Julian)
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "sudo chown ec2-user:ec2-user \
      /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
      /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.*"
```

### Step 5: Verify

```bash
# Check syntax and ownership
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "php -l /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php && \
   ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php*"
```

### Step 6: Cleanup

```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 "rm /tmp/upload.php.new"
```

---

## Local Server Requirements

The local server (`uploadlocallat_kuldeep.php`) needs to be updated to:

1. Accept the new `customer_id` field
2. Use `customer_id` to lookup folder path from `unified_customers.remote_customer_id`
3. Use `job_date` year to select the appropriate year folder
4. Fall back to name matching if customer_id lookup fails

### New POST Fields Received

| Field | Type | Description |
|-------|------|-------------|
| `auth_token` | string | Authentication token |
| `file` | file | Image file |
| `file_name` | string | Original filename |
| `job_id` | int | Job PID from remote database |
| `customer_id` | int | **NEW** - Customer ID from remote database |
| `job_type` | string | Job type code (PV, AC, PM, etc.) |
| `first_name` | string | Customer first name |
| `last_name` | string | Customer last name |
| `job_date` | string | Job date (YYYY-MM-DD) |
| `photo_type` | string | 'S' (Survey) or 'I' (Installation) |

---

## Rollback

If issues occur, restore the backup:

```bash
# On remote server
cd /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/
cp upload.php.bak.20260130 upload.php
```

---

## Benefits

| Before | After |
|--------|-------|
| Name matching only | ID-based customer lookup (92%+ match rate) |
| No year awareness | Year-aware folder selection via job_date |
| SQL injection possible | Protected with intval() |

---

## WebP Changes Timeline (2026-02-04)

### Context: Who Uses What

| Consumer | Image Source | Format |
|----------|-------------|--------|
| **Scheduler website** | `scheduler/uploads/thumbnails/` | JPEG (via `thumbnail_helper`) |
| **Mobile app (listing)** | `webp/` folder (via `getimagelisting.php`) | WebP |
| **Mobile app (view)** | `webp/` folder (via `fetch_image.php`) | WebP |

The scheduler website does NOT use WebP. The mobile app DOES use WebP from the `webp/` folder.

### Change 1: WebP in phototab.php (UPDATED 2026-02-05)

`phototab.php` (scheduler web upload) now generates WebP via background `generate_webp.py` (`nohup &`), matching `upload.php`'s pattern. This is required because the mobile app's `getimagelisting.php` scans the `webp/` folder — without WebP, scheduler-uploaded photos are invisible to the mobile app. Also uses dynamic year from `job_date` and correct Python path (`/usr/local/bin/python3.6`).

### Change 2: WebP Removal from upload.php (REVERTED)

Initially removed WebP from `upload.php` (mobile app upload). This caused the mobile app to have no images in its listing, since `getimagelisting.php` and `fetch_image.php` scan the `webp/` folder.

### Change 3: Synchronous generate_webp.py (CAUSED TIMEOUT)

Replaced the local-server-based WebP with a Python script (`generate_webp.py`) running synchronously on the remote server via `exec()`. This caused mobile app timeouts because:
- Python 3.6 interpreter startup on the old AWS server = 1-2 seconds
- Pillow image processing for large photos = 2-3 seconds
- PHP's synchronous `exec()` blocks the entire script, delaying the JSON response to the mobile app

### Change 4: Hybrid Approach (REVERTED — intermediate step)

Briefly restored the original flow where the local server generated WebP and returned binary via `readfile()`. This worked but tightly coupled the two servers for WebP generation.

### Change 5: Async Background WebP (CURRENT - DEPLOYED)

The remote server generates its own WebP files using `generate_webp.py` in a **background process** (`nohup ... &`). The local server responds with JSON immediately and has no WebP responsibility.

**Remote `upload.php`** (mobile app upload):
```
Mobile App → POST base64 image
  1. Decode + save image to Dropbox             (fast)
  2. echo JSON response                          (BUFFERED)
  3. cURL to local server                        (local returns JSON — fast)
  4. Copy to scheduler/uploads                   (fast)
  5. Trigger generate_webp.py in BACKGROUND      (non-blocking — nohup &)
  6. Insert meter_files with expected filename    (fast)
  7. Script ends → JSON sent to mobile app       (FAST)
     ... WebP appears in webp/ folder 1-3s later (background)
```

**Local `uploadlocallat_kuldeep.php`**:
```
Receives file from remote server
  1. Validate + save to customer folder          (fast)
  2. Trigger background Python thumbs            (non-blocking)
  3. Return JSON response                        (immediate)
```

### Why Background WebP Works

The mobile app does NOT request the image listing at the exact moment of upload. By the time the user navigates to view photos, the background `generate_webp.py` process (1-3 seconds) has already completed and the `.webp` file exists in the `webp/` folder.

### Key Implementation Detail

```php
// NON-BLOCKING: nohup + & = runs in background, PHP continues immediately
$cmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($generateScript)
     . ' ' . escapeshellarg($filePath)
     . ' ' . escapeshellarg($webp_dest)
     . ' 80'
     . ' && cp ' . escapeshellarg($webp_dest) . ' ' . escapeshellarg($scheduler_webp_dest)
     . ' > /dev/null 2>&1 &';
exec($cmd);
```

Note: Full path `/usr/local/bin/python3.6` is required because `python3` is not symlinked on the remote server.

### Improvements Over Original

| Aspect | Original | Current |
|--------|----------|---------|
| Mobile app response time | Blocked by local server WebP + cURL | Fast — no synchronous image processing |
| cURL error handling | Silent failure | Logs errors, checks HTTP status |
| cURL timeout | No timeout (hangs if local server down) | `CURLOPT_CONNECTTIMEOUT` = 5s (fail fast if unreachable) |
| Photo type in filename | Hardcoded "-S" | Dynamic `$photo_type` |
| File size in DB | Hardcoded "121" | Dynamic `filesize()` |
| SQL injection | Unprotected | `intval()` + `mysqli_real_escape_string()` |
| Server coupling | Local must return binary WebP | Servers are independent |
| Local thumbnails | None | Background Python (thumbs/ + thumbs/mid/) |
| Customer lookup | Name matching only | ID-based + name fallback + auto-create |

### Files Changed

| File | Server | Change |
|------|--------|--------|
| `upload.php` | Remote | Background `generate_webp.py` via `nohup &`, dynamic year from `job_date`, cURL connect timeout |
| `getimagelisting.php` | Remote | Dynamic year from `job_date` (was hardcoded "2025") |
| `fetch_image.php` | Remote | Dynamic year from `job_date` (was hardcoded "2025") |
| `generate_webp.py` | Remote | Python/Pillow WebP converter (unchanged) |
| `phototab.php` | Remote | Background `generate_webp.py` via `nohup &`, dynamic year from `job_date` (WebP restored — needed for mobile app) |
| `uploadlocallat_kuldeep.php` | Local | Returns JSON, background Python thumbs |

### Backups

| Location | File | Description |
|----------|------|-------------|
| Remote server | `upload.php.bak.pre_year_fix.20260204` | Before dynamic year fix |
| Remote server | `getimagelisting.php.bak.pre_year_fix.20260204` | Before dynamic year fix |
| Remote server | `fetch_image.php.bak.pre_year_fix.20260204` | Before dynamic year fix |
| Remote server | `upload.php.bak.pre_timeout.20260204` | Before cURL connect timeout |
| Remote server | `upload.php.bak.pre_async_webp.20260204` | Before async background approach |
| Remote server | `upload.php.bak.pre_hybrid.20260204` | Before hybrid restore |
| Remote server | `upload.php.bak.pre_webp_removal.20260204` | Before any WebP changes |
| Remote server | `phototab.php.bak.pre_webp_removal.20260204` | Before phototab WebP removal |
| Remote server | `phototab.php.bak.pre_webp_restore.20260205` | Before restoring WebP + dynamic year |
| Local server | `uploadlocallat_kuldeep.php.bak.pre_async_remote.20260204` | Before reverting to JSON-only |
| Local server | `uploadlocallat_kuldeep.php.bak.pre_hybrid.20260204` | Before hybrid restore |
| Project | `REMOTE/photoapi/upload.php.bak.pre_async_webp.20260204` | Before async approach |
| Project | `REMOTE/photoapi/upload.php.bak.pre_hybrid.20260204` | Before hybrid restore |
| Project | `REMOTE/upload.php.pre_webp_fix.20260204` | Original before any changes |
| Project | `LOCAL/uploadlocallat_kuldeep.php.before_async_thumb.20260204` | Original with inline WebP |

### Deployment

```bash
# SSH key setup
puttygen SSH/schedular_server_private_key.ppk -O private-openssh -o /root/.ssh/aei_remote.pem
chmod 600 /root/.ssh/aei_remote.pem

# Deploy remote upload.php
scp -i /root/.ssh/aei_remote.pem REMOTE/photoapi/upload.php Julian@18.225.0.90:/tmp/upload.php.new
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /tmp/upload.php.new /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  rm /tmp/upload.php.new
"

# Local server file is at /var/www/html/upload/uploadlocallat_kuldeep.php (same machine)
```

### Rollback

```bash
# Remote: restore pre-async backup (has synchronous generate_webp.py — will be slow but functional)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_webp.20260204 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"

# Remote: restore to original (requires local server to return binary WebP — NOT current state)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_webp_removal.20260204 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"

# Local: restore to hybrid (inline WebP + readfile)
cp /var/www/html/upload/uploadlocallat_kuldeep.php.bak.pre_async_remote.20260204 \
   /var/www/html/upload/uploadlocallat_kuldeep.php
```

---

## Async Local Server Sync (2026-02-05)

### Problem

The synchronous cURL call to the local server (`upload.aeihawaii.com`) blocked `upload.php` for 2-10+ seconds per upload. Because PHP buffers the `echo` JSON response until the script ends, the mobile app waited for the entire local server round-trip before receiving any response. The local server sync is just an internal copy for office storage — the mobile app doesn't need it.

### Solution

Replaced the synchronous cURL block (lines 105-162) with a background Python script (`sync_to_local.py`) invoked via `nohup ... &`, matching the proven pattern used by `generate_webp.py`.

### New File: `sync_to_local.py`

**Location:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py`

- Python 3.6 background sync script
- Receives 9 positional CLI arguments: `file_path`, `file_name`, `job_id`, `customer_id`, `job_type`, `last_name`, `first_name`, `job_date`, `photo_type`
- Uses `requests.post()` with `files=` for multipart file upload to `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php`
- Logs results to `photoapi/logs/sync_to_local.log`
- Timeouts: connect=10s, read=60s (generous since background)

### Changes to `upload.php`

**Removed:** Synchronous cURL block — `$remoteUploadUrl`, `$postFields`, `curl_init()`, all `curl_setopt()` calls, `curl_exec()`, `curl_close()`, and error logging (old lines 105-162).

**Reordered:** `echo` JSON response moved to the very end of the script (last line before `?>`), so the mobile app gets the response immediately after all fast operations complete.

**New upload flow:**
```
Mobile App → POST base64 image
  1. Decode + save image to /mnt/dropbox/           (fast)
  2. realpath + validate saved file                  (fast)
  3. Copy to scheduler/uploads/                      (fast)
  4. Trigger generate_webp.py in BACKGROUND          (non-blocking — nohup &)
  5. INSERT meter_files record                       (fast)
  6. Trigger sync_to_local.py in BACKGROUND          (NEW — non-blocking — nohup &)
  7. echo JSON response                              (LAST — sent immediately)
  8. Script ends → mobile app gets response           ~0.1-0.5s total
     ... local server sync happens 2-10s later       (background)
     ... WebP appears in webp/ folder 1-3s later     (background)
```

**Background sync invocation:**
```php
$syncScript = __DIR__ . '/sync_to_local.py';
$syncCmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($syncScript)
    . ' ' . escapeshellarg($filePath)
    . ' ' . escapeshellarg(basename($filePath))
    . ' ' . escapeshellarg($job_id)
    . ' ' . escapeshellarg($customer_id)
    . ' ' . escapeshellarg($job_type_text)
    . ' ' . escapeshellarg($last_name)
    . ' ' . escapeshellarg($first_name)
    . ' ' . escapeshellarg($job_date_t)
    . ' ' . escapeshellarg($photo_type)
    . ' > /dev/null 2>&1 &';
exec($syncCmd);
```

### Before/After

| Metric | Before (sync cURL) | After (background Python) |
|--------|-------------------|--------------------------|
| Mobile app wait | 2-10+ seconds (cURL round-trip) | ~0.1-0.5s (fast ops only) |
| Local server down impact | 5s timeout delay per upload | Zero impact on mobile app |
| echo position | Line 99 (buffered, sent at script end) | Last line (sent immediately) |
| Sync mechanism | PHP cURL (synchronous) | Python requests (background nohup) |
| Error visibility | `error_log()` in PHP error log | Dedicated `logs/sync_to_local.log` |

### Deployment

```bash
# SSH key setup
puttygen SSH/schedular_server_private_key.ppk -O private-openssh -o /root/.ssh/aei_remote.pem
chmod 600 /root/.ssh/aei_remote.pem

# 1. Create logs directory on remote (must be 777 — Apache writes as 'apache' user)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo mkdir -p /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
"

# 2. Backup current upload.php
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.$(date +%Y%m%d)
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.*
"

# 3. Upload sync_to_local.py
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/sync_to_local.py \
  Julian@18.225.0.90:/tmp/sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /tmp/sync_to_local.py /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  rm /tmp/sync_to_local.py
"

# 4. Upload modified upload.php
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/upload.php \
  Julian@18.225.0.90:/tmp/upload.php.new
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /tmp/upload.php.new /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  rm /tmp/upload.php.new
"

# 5. Fix /mnt/dropbox/ year directory permissions (Apache runs as 'apache', not ec2-user)
# New year directories are often created by ec2-user with 775 — Apache can't write
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chmod 777 '/mnt/dropbox/2026 Customers/'
  sudo chmod 777 '/mnt/dropbox/2027 Customers/' 2>/dev/null
"

# 6. Ensure AWS IP is whitelisted on LOCAL server firewall
# The local server uses ipset whitelist with DROP default — without this, sync times out
# (Check /var/www/html/security/whitelist.json for persistence config)
sudo ipset test trusted_whitelist 18.225.0.90 2>/dev/null || \
  sudo ipset add trusted_whitelist 18.225.0.90

# 7. Verify sync log appears after a mobile upload
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  tail -f /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log
"

# 8. Run E2E QA test
python3 /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/QA/test_upload_pipeline.py
```

### Rollback

```bash
# Restore pre-async-sync backup (reverts to synchronous cURL — slower but functional)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"
# sync_to_local.py can stay on disk — it's only invoked if upload.php calls it
```

### Files

| File | Server | Action |
|------|--------|--------|
| `photoapi/sync_to_local.py` | Remote | **NEW** — background Python sync script |
| `photoapi/upload.php` | Remote | **MODIFIED** — removed sync cURL, reordered, added nohup sync |
| `photoapi/logs/sync_to_local.log` | Remote | **NEW** — created by sync script on first run |

### Backups

| Location | File | Description |
|----------|------|-------------|
| Remote server | `upload.php.bak.pre_async_sync.20260205` | Before async sync change |
| Project | `REMOTE/photoapi/upload.php.bak.pre_async_webp.20260204` | Before async WebP (has sync cURL) |

### Deployment Issues Discovered (2026-02-05)

These issues were found during E2E testing after initial deployment:

| Issue | Root Cause | Fix |
|-------|-----------|-----|
| `file_put_contents` failed on upload | `/mnt/dropbox/2026 Customers/` had `775` permissions (ec2-user:ec2-user). Apache runs as `apache` user, not in ec2-user group. | `sudo chmod 777 '/mnt/dropbox/2026 Customers/'` |
| `sync_to_local.log` not created | `photoapi/logs/` directory had `755` permissions (ec2-user:ec2-user). Apache `apache` user can't write. | `sudo chmod 777 /var/www/.../photoapi/logs/` |
| sync_to_local.py connection timeout | AWS IP (18.225.0.90) not in `trusted_whitelist` ipset on local server. Firewall INPUT policy is DROP. | `sudo ipset add trusted_whitelist 18.225.0.90` on local server |

**Key Lesson:** The remote server's Apache runs as `apache` user, while files deployed via SSH are owned by `ec2-user`. Any directory that Apache needs to write to must be `777` or have `apache` in the owner/group. This applies to:
- `/mnt/dropbox/{YEAR} Customers/` — new year directories
- `photoapi/logs/` — sync log directory
- `photoapi/logs/sync_to_local.log` — the log file itself (created by `apache`)

**Firewall Note:** The local server's ipset whitelist is volatile (cleared on reboot). The AWS IP is persisted in `/var/www/html/security/whitelist.json` but must be reloaded into the ipset after server restarts. See `/var/www/html/security/README.md` for the persistence mechanism.
