<?php
header('Content-Type: application/json');

// Load database configuration for unified_customers lookup
require_once __DIR__ . '/db_config.php';

// Load environment variables (create .env file with AUTH_TOKEN=your_secure_token_here)
if (file_exists(__DIR__ . '/.env')) {
    $env = parse_ini_file(__DIR__ . '/.env');
    define('AUTH_TOKEN', $env['AUTH_TOKEN'] ?? '');
} else {
    // Fallback - CHANGE THIS IN PRODUCTION
    define('AUTH_TOKEN', 'remote_token');
}

/**
 * Look up customer folder path by remote customer_id and job year
 * This is the preferred method - uses ID-based matching
 * Returns the folder_path if found, null otherwise
 */
function getCustomerFolderByCustomerId($customer_id, $job_year = null) {
    if (empty($customer_id)) {
        return null;
    }

    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("Database connection failed for customer lookup");
        return null;
    }

    // If job_year provided, try to match that year first
    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE remote_customer_id = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ii", $customer_id, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    // Fallback: get most recent folder for this customer
    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE remote_customer_id = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        $conn->close();
        return null;
    }

    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

/**
 * Look up customer folder path by name (fallback method)
 * Returns the folder_path if found, null otherwise
 */
function getCustomerFolderByName($first_name, $last_name, $job_year = null) {
    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("Database connection failed for customer lookup");
        return null;
    }

    // If job_year provided, try to match that year first
    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE first_name = ? AND last_name = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ssi", $first_name, $last_name, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    // Fallback: get most recent folder matching name
    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE first_name = ? AND last_name = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        $conn->close();
        return null;
    }

    $stmt->bind_param("ss", $first_name, $last_name);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

// Validate token
if (!isset($_POST['auth_token']) || $_POST['auth_token'] !== AUTH_TOKEN) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid or missing auth token"
    ]);
    exit;
}

// Check if file was uploaded
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    echo json_encode([
        "status" => "error",
        "message" => "File upload error or file not received"
    ]);
    exit;
}

// Validate file type BEFORE processing
$allowedMimeTypes = [
    'image/jpeg',
    'image/jpg',
    'image/png',
    'image/gif',
    'image/heic',
    'image/heif'
];

$allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'heic', 'heif'];

$finfo = finfo_open(FILEINFO_MIME_TYPE);
$detectedMimeType = finfo_file($finfo, $_FILES['file']['tmp_name']);
finfo_close($finfo);

if (!in_array($detectedMimeType, $allowedMimeTypes)) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid file type. Only images allowed (JPEG, PNG, GIF, HEIC)"
    ]);
    exit;
}

// Sanitize filename
$originalFileName = basename($_FILES['file']['name']);
$pathInfo = pathinfo($originalFileName);
$extension = strtolower($pathInfo['extension'] ?? '');

if (!in_array($extension, $allowedExtensions)) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid file extension. Allowed: jpg, jpeg, png, gif, heic, heif"
    ]);
    exit;
}

$safeBaseName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $pathInfo['filename']);
$safeFileName = $safeBaseName . '.' . $extension;

// Optional: Use passed file_name field if needed
if (isset($_POST['file_name'])) {
    $passedPathInfo = pathinfo(basename($_POST['file_name']));
    $passedExt = strtolower($passedPathInfo['extension'] ?? '');

    if (in_array($passedExt, $allowedExtensions)) {
        $safeBaseName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $passedPathInfo['filename']);
        $safeFileName = $safeBaseName . '.' . $passedExt;
    }
}

// Sanitize and validate all input parameters to prevent path traversal
function sanitize_path_component($input, $fieldName = 'input') {
    if (empty($input)) {
        echo json_encode([
            "status" => "error",
            "message" => "Missing required field: $fieldName"
        ]);
        exit;
    }

    // Remove any path traversal attempts
    $sanitized = str_replace(['..', '/', '\\', "\0"], '', $input);

    // Only allow alphanumeric, spaces, hyphens, underscores
    $sanitized = preg_replace('/[^A-Za-z0-9 _\-]/', '', $sanitized);

    // Trim whitespace
    $sanitized = trim($sanitized);

    if (empty($sanitized)) {
        echo json_encode([
            "status" => "error",
            "message" => "Invalid characters in field: $fieldName"
        ]);
        exit;
    }

    return $sanitized;
}

// Validate and sanitize date
if (!isset($_POST['job_date']) || empty($_POST['job_date'])) {
    echo json_encode([
        "status" => "error",
        "message" => "Missing job_date"
    ]);
    exit;
}

$timestamp = strtotime($_POST['job_date']);
if ($timestamp === false) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid date format"
    ]);
    exit;
}

$job_date = date("Y-m-d", $timestamp);
$job_year = (int)date("Y", $timestamp);

// Get optional customer_id and job_id from remote server
$customer_id = isset($_POST['customer_id']) ? intval($_POST['customer_id']) : null;
$job_id = isset($_POST['job_id']) ? intval($_POST['job_id']) : null;

// Sanitize all path components
$first_name = sanitize_path_component($_POST['first_name'] ?? '', 'first_name');
$last_name = sanitize_path_component($_POST['last_name'] ?? '', 'last_name');
$customer_name = $last_name . ", " . $first_name;
$job_type_text = sanitize_path_component($_POST['job_type'] ?? '', 'job_type');

// Validate photo_type
if (!isset($_POST['photo_type']) || !in_array($_POST['photo_type'], ['I', 'S'])) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid photo_type. Must be 'I' or 'S'"
    ]);
    exit;
}
$photo_type = $_POST['photo_type'];

// Look up customer folder from unified_customers database
// Priority: 1) customer_id + year, 2) customer_id any year, 3) name + year, 4) name any year
$uploadDir_customer = null;
$lookup_method = 'none';

if ($customer_id) {
    // Try customer_id lookup first (most reliable)
    $uploadDir_customer = getCustomerFolderByCustomerId($customer_id, $job_year);
    if ($uploadDir_customer) {
        $lookup_method = 'customer_id';
    }
}

if (!$uploadDir_customer) {
    // Fallback to name-based lookup
    $uploadDir_customer = getCustomerFolderByName($first_name, $last_name, $job_year);
    if ($uploadDir_customer) {
        $lookup_method = 'name';
    }
}

// Verify the database path exists on filesystem
if ($uploadDir_customer && is_dir($uploadDir_customer)) {
    // Customer found in database and folder exists - use normal folder structure
    $uploadDir_customer_survey = $uploadDir_customer . "/Survey";
    if (!is_dir($uploadDir_customer_survey)) {
        mkdir($uploadDir_customer_survey, 0755, true);
    }

    $uploadDir_customer_install = $uploadDir_customer . "/Installation";
    if (!is_dir($uploadDir_customer_install)) {
        mkdir($uploadDir_customer_install, 0755, true);
    }

    if ($photo_type == "I") {
        $final_folder = $uploadDir_customer_install . "/" . $customer_name . ", " . $job_type_text . "-I, " . $job_date . '/';
        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }
    } else {
        $final_folder = $uploadDir_customer_survey . "/" . $customer_name . ", " . $job_type_text . "-S," . $job_date . '/';
        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }
    }
} else {
    // Customer NOT found in database OR folder doesn't exist - redirect to Uploads folder
    $uploads_base = '/mnt/dropbox/Uploads';
    $final_folder = $uploads_base . '/[' . $customer_name . '] [' . $job_type_text . '] [' . $job_date . ']/';

    if (!is_dir($final_folder)) {
        mkdir($final_folder, 0755, true);
    }

    // Log for debugging - helps identify customers needing folder mapping
    error_log("Photo upload: Customer not found - customer_id=$customer_id, name=$first_name $last_name, job_id=$job_id");
}
 $destinationPath = $final_folder . $safeFileName;

if (move_uploaded_file($_FILES['file']['tmp_name'], $destinationPath)) {
	 /* echo json_encode([
        "status" => "success",
        "message" => "File uploaded successfully",
        "path" => $safeFileName
    ]);*/
} else {
    echo json_encode([
        "status" => "error",
        "message" => "Failed to save uploaded file"
    ]);
}
$final_folder=$final_folder.'/webp/';
if (!is_dir($final_folder)) {
    mkdir($final_folder, 0755, true);
}

$customer_namewebp=$last_name."".$first_name;
// Use cryptographically secure random number
$randomId = random_int(100000, 999999);
 if($photo_type=="I"){
$destination= $final_folder.$customer_namewebp."".$job_type_text."-I".$job_date."_IMG".$randomId.'.webp';
 } else {
       $destination= $final_folder.$customer_namewebp."".$job_type_text."-S".$job_date."_IMG".$randomId.'.webp';
 }
convert_to_webp($destinationPath, $destination);
readfile($destination);
exit;
function convert_to_webp($source, $destination, $quality = 80) {
    // Check if file exists
    if (!file_exists($source)) {
        return false;
    }

    $info = @getimagesize($source);
    $mime = $info['mime'] ?? '';

    // Handle HEIC files (requires ImageMagick)
    if (in_array($mime, ['image/heic', 'image/heif']) ||
        preg_match('/\.(heic|heif)$/i', $source)) {

        // Check if ImageMagick is available
        if (extension_loaded('imagick')) {
            try {
                $imagick = new Imagick($source);
                $imagick->setImageFormat('webp');
                $imagick->setImageCompressionQuality($quality);
                $result = $imagick->writeImage($destination);
                $imagick->clear();
                $imagick->destroy();
                return $result;
            } catch (Exception $e) {
                error_log("HEIC conversion failed: " . $e->getMessage());
                return false;
            }
        } else {
            // Fallback: try using system command if available
            $escapedSource = escapeshellarg($source);
            $escapedDest = escapeshellarg($destination);
            exec("convert $escapedSource -quality $quality $escapedDest 2>&1", $output, $returnCode);
            return $returnCode === 0 && file_exists($destination);
        }
    }

    // Handle standard image formats with GD
    switch ($mime) {
        case 'image/jpeg':
        case 'image/jpg':
            $image = @imagecreatefromjpeg($source);
            break;

        case 'image/png':
            $image = @imagecreatefrompng($source);
            if ($image) {
                // Fix transparency issue
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
            }
            break;

        case 'image/gif':
            $image = @imagecreatefromgif($source);
            break;

        default:
            return false; // unsupported
    }

    if (!$image) {
        return false;
    }

    // Save WebP
    $result = imagewebp($image, $destination, $quality);
    imagedestroy($image);
    return $result;
}
function get_uploaded_ext($file)
{
    if (!isset($file['tmp_name']) || !file_exists($file['tmp_name'])) {
        return false;
    }

    $info = getimagesize($file['tmp_name']);
    if (!$info || empty($info['mime'])) {
        return false;
    }

    switch ($info['mime']) {
        case 'image/jpeg': return 'jpg';
        case 'image/png':  return 'png';
        case 'image/gif':  return 'gif';
        default: return false;
    }
}
?>

