# PROPOSED: WebP Photo Conversion & Resize

**Created:** 2026-02-08
**Status:** Research / Planning
**Goal:** Convert existing JPG/JPEG/PNG uploads to WebP format, resized to fit within 800px max dimension, to save disk space on the remote server.

---

## 1. Current State

### Server Environment
| Component | Version/Detail |
|-----------|---------------|
| OS | Amazon Linux AMI 2018.03 |
| PHP | 5.3.29 (no native WebP support — added in PHP 5.4+) |
| GD Library | 2.1.0 — supports JPEG, PNG, GIF. **No WebP support.** |
| ImageMagick | **Not installed** |
| cwebp/dwebp | **Not installed** (but `libwebp-tools 0.3.0` available via yum) |
| Apache | 2.2.34 with mod_rewrite, mod_mime, mod_headers, mod_negotiation |
| WebP MIME type | **Not configured** |

### Disk Usage — Scheduler Uploads

| Path | Size |
|------|------|
| `/httpdocs/scheduler/uploads/` | **205 GB** |
| `/httpdocs/dev/scheduler/uploads/` | 5.9 GB |
| Disk total / available | 985 GB / 642 GB free (35% used) |

### File Breakdown — `scheduler/uploads/` (201,072 image files)

| Extension | Count | Total Size | Avg Size |
|-----------|-------|-----------|----------|
| .JPG | 146,632 | 155.9 GB | 1,115 KB |
| .jpg | 37,472 | 8.9 GB | 250 KB |
| .jpeg | 10,677 | 1.5 GB | 146 KB |
| .png | 6,291 | 1.1 GB | 191 KB |
| .pdf | 22,059 | — | — |
| .webp | 1 | — | — |
| **Image Total** | **201,072** | **~167.4 GB** | — |

### Image Dimension Distribution (sample of 5,000)

| Dimension Range | Count | % | Space Share |
|----------------|-------|---|------------|
| ≤ 800px | 2,755 | 55% | ~4% |
| 801–1600px | 80 | 2% | — |
| 1601–3264px | 1,794 | 36% | } ~96% |
| > 3264px | 371 | 7% | } |

**Key Insight:** 45% of images are oversized (many at 3264x2448 from phone cameras), and these account for **96% of storage space**. Resizing alone would yield massive savings before WebP conversion even factors in.

### Database Tables with Image References

| Table | Total Records | JPG/JPEG | PNG | PDF |
|-------|--------------|----------|-----|-----|
| meter_files | 203,391 | 190,886 | 4,404 | 8,053 |
| permit_files | 59,732 | 477 | 35 | 59,214 |
| sketch_files | 11,955 | 20 | 13 | 11,922 |
| elecphoto_files | 1,085 | 1,077 | 8 | 0 |
| files | 252 | 198 | 1 | 21 |
| genral_files | 8,347 | 19 | 4 | 8,318 |
| presale_files | 8,921 | 0 | 0 | 8,921 |
| predesignsketch_files | 7,500 | 0 | 0 | 7,500 |

**Primary target:** `meter_files` (190,886 JPG images — the vast majority of photos on disk).

### How Photos Are Served

Photos are displayed in views using this pattern:
```php
<!-- Full-size link -->
<a href="<?=base_url()?>uploads/<?=$value['unique_filename']?>">

<!-- Thumbnail display -->
<img src="<?php echo thumbnail($src_pth, 151, 142); ?>" />
```

#### Thumbnail System Architecture (current)

The thumbnail pipeline has three components:

1. **`thumbnail_helper.php`** — CI helper that orchestrates thumbnail generation
   - Receives `(filename, width, height, option)` — option is typically `'crop'`
   - Builds cache path: `uploads/thumbnails/{name}_{option[0]}{w}x{h}.{ext}`
   - Config uses `subfolder` cache mode → thumbnails go to `uploads/thumbnails/`
   - If cached thumbnail exists and is newer than source, returns its URL immediately
   - Otherwise calls `Resize.php` to generate it

2. **`Resize.php`** — Third-party GD wrapper class (`system/application/third_party/`)
   - `openImage()` — uses `getimagesize()` to detect type, calls `imagecreatefromjpeg/png/gif`
   - `resizeImage()` — supports modes: exact, portrait, landscape, auto, crop
   - `saveImage()` — detects extension, calls `imagejpeg/png/gif` to write output
   - **Only handles JPEG, PNG, GIF** — no WebP case in either `openImage()` or `saveImage()`

3. **`thumbnail.php` config** (`system/application/config/`)
   - `thumbnail_base_folder` = `$_SERVER['DOCUMENT_ROOT'] . "/scheduler/uploads/"`
   - `thumbnail_cache_type` = `'subfolder'`
   - `thumbnail_subfolder_name` = `'thumbnails'`
   - `thumbnail_base_url` = `site_url() . 'uploads'`
   - Caching enabled (disabled = false)

---

## 2. Estimated Space Savings

### Conservative Estimates

| Step | Current | After | Savings |
|------|---------|-------|---------|
| Resize >800px images to 800px max | 167.4 GB | ~25 GB | ~142 GB (85%) |
| Convert resized to WebP | ~25 GB | ~10 GB | ~15 GB (60%) |
| **Combined** | **167.4 GB** | **~10 GB** | **~157 GB (94%)** |

Rationale:
- Resizing 3264x2448 → 800x600 reduces pixel count by ~94%, file size by ~85-90%
- WebP is typically 25-34% smaller than JPEG at equivalent quality, often with better visual clarity
- Images already ≤800px still benefit from WebP conversion (~30% savings)

**Projected final uploads size: ~10-15 GB (down from 167.4 GB of images + 205 GB total with PDFs)**

---

## 3. Technical Approach — Recommended

### Strategy: Batch Convert + DB Update + On-the-fly Thumbnail Helper

Two parts: (A) batch conversion of existing files, and (B) a modified `Resize.php` that generates thumbnails on-the-fly from WebP sources using CLI tools.

### Phase 1: Install Prerequisites

```bash
# Install libwebp tools (provides both cwebp and dwebp)
sudo yum install -y libwebp-tools

# Verify both tools
cwebp -version    # Encodes to WebP
dwebp -version    # Decodes WebP to PNG/PPM/etc.

# Add WebP MIME type to Apache
echo "AddType image/webp .webp" | sudo tee -a /etc/httpd/conf/httpd.conf
sudo service httpd restart
```

### Phase 2: Backup

```bash
# CRITICAL: Full backup of uploads before any conversion
# Option A: Tar archive (space needed: ~167 GB)
cd /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/
tar -czf /mnt/backup/scheduler_uploads_backup_$(date +%Y%m%d).tar.gz uploads/

# Option B: rsync to separate location (if another volume is mounted)
rsync -avz uploads/ /mnt/backup/scheduler_uploads_original/

# Also backup the database tables
mysqldump -u AEI_User -p'P@55w02d7777' mandhdesign_schedular \
  meter_files elecphoto_files files genral_files permit_files sketch_files \
  > /mnt/backup/photo_tables_backup_$(date +%Y%m%d).sql
```

### Phase 3: Modify Resize.php for WebP Support (On-the-fly Thumbnails)

The key insight: `libwebp-tools` includes **`dwebp`** (decoder) which converts WebP → PNG. PHP 5.3 GD **can** read PNG. So the pipeline for on-the-fly WebP thumbnails is:

```
WebP source → dwebp → temp PNG → GD resize → temp PNG → cwebp → WebP thumbnail
```

This means thumbnails are generated on demand exactly as they are today — the first request is slightly slower (CLI overhead), but the result is cached and all subsequent requests serve the static cached file.

#### Changes to `Resize.php`

Add WebP handling to `openImage()` and `saveImage()`:

```php
<?php
// In Resize.php — openImage() method
// Add this case to the switch statement:

private function openImage($file) {
    $info = getimagesize($file);

    // Handle WebP files (GD can't read them on PHP 5.3)
    // Use dwebp CLI tool to decode to a temp PNG first
    $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    if ($ext === 'webp') {
        $tmp = tempnam(sys_get_temp_dir(), 'webp_') . '.png';
        exec('dwebp ' . escapeshellarg($file) . ' -o ' . escapeshellarg($tmp) . ' 2>&1', $output, $ret);
        if ($ret === 0 && file_exists($tmp)) {
            $info = getimagesize($tmp);
            $this->width = $info[0];
            $this->height = $info[1];
            $this->_webp_tmp = $tmp;  // Store for cleanup
            return imagecreatefrompng($tmp);
        }
        return false;
    }

    // Existing logic for JPEG/PNG/GIF (unchanged)
    $this->width = $info[0];
    $this->height = $info[1];
    if ($info) {
        switch ($info[2]) {
            case IMAGETYPE_PNG:
                return imagecreatefrompng($file);
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($file);
            case IMAGETYPE_GIF:
                return imagecreatefromgif($file);
        }
    }
    return false;
}
```

```php
// In Resize.php — saveImage() method
// Add this case to the switch statement:

public function saveImage($savePath, $imageQuality="100") {
    $extension = strtolower(strrchr($savePath, '.'));
    $ok = false;

    switch ($extension) {
        case '.webp':
            // GD can't write WebP on PHP 5.3
            // Save as temp PNG, then use cwebp to convert
            $tmpPng = tempnam(sys_get_temp_dir(), 'webp_out_') . '.png';
            if (imagetypes() & IMG_PNG) {
                imagepng($this->imageResized, $tmpPng, 1);
                $quality = min(90, max(1, (int)($imageQuality * 0.9)));
                exec('cwebp -q ' . $quality . ' '
                    . escapeshellarg($tmpPng) . ' -o '
                    . escapeshellarg($savePath) . ' 2>&1', $output, $ret);
                $ok = ($ret === 0 && file_exists($savePath));
                @unlink($tmpPng);
            }
            break;

        case '.jpg':
        case '.jpeg':
            // ... existing code unchanged ...

        case '.gif':
            // ... existing code unchanged ...

        case '.png':
            // ... existing code unchanged ...
    }

    imagedestroy($this->imageResized);

    // Cleanup any temp file from openImage
    if (isset($this->_webp_tmp) && file_exists($this->_webp_tmp)) {
        @unlink($this->_webp_tmp);
    }

    return $ok;
}
```

#### Why This Works

| Step | What Happens | Tool Used |
|------|-------------|-----------|
| 1. Request arrives for thumbnail of `abc123.webp` | `thumbnail_helper.php` checks cache | PHP |
| 2. Cache miss — calls `new Resize('uploads/abc123.webp')` | `openImage()` detects `.webp` extension | PHP |
| 3. Decode WebP to temp PNG | `dwebp abc123.webp -o /tmp/webp_xyz.png` | CLI (dwebp) |
| 4. GD reads the temp PNG, resizes in memory | `imagecreatefrompng()` + `imagecopyresampled()` | PHP GD |
| 5. Save thumbnail as WebP | GD writes temp PNG → `cwebp` converts to `.webp` | CLI (cwebp) |
| 6. Return URL to cached thumbnail | `uploads/thumbnails/abc123_c151x142.webp` | PHP |
| 7. Next request — cache hit | Serve static `.webp` file directly | Apache |

**No changes needed to `thumbnail_helper.php` or `thumbnail.php` config.** The helper already:
- Derives the output filename from the source extension (`.webp` in → `.webp` out)
- Checks cache by file existence and modification time
- Delegates all actual image work to `Resize.php`

#### Performance Characteristics

- **First thumbnail request:** ~200-500ms extra (two exec() calls to dwebp/cwebp)
- **Subsequent requests:** Zero overhead (served as static file from cache by Apache)
- **Temp files:** Created in `/tmp/`, cleaned up immediately after each operation
- **Memory:** Same as current GD usage — the resized image is small (≤800px source)

### Phase 4: Batch Conversion Script

A bash script that converts existing JPG/JPEG/PNG files to WebP:

```bash
#!/bin/bash
# convert_to_webp.sh — Run on the remote server
# REQUIRES: libwebp-tools (cwebp), php-cli (for GD resize)

UPLOADS_DIR="/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads"
LOG_FILE="/tmp/webp_conversion_log.csv"
ERROR_LOG="/tmp/webp_conversion_errors.log"
MAX_DIM=800
WEBP_QUALITY=80

echo "old_filename,new_filename,old_size,new_size" > "$LOG_FILE"

process_file() {
    local file="$1"
    local basename=$(basename "$file")
    local name="${basename%.*}"
    local new_file="${UPLOADS_DIR}/${name}.webp"

    # Skip if already converted
    [ -f "$new_file" ] && return

    # Get dimensions via PHP (since identify/ImageMagick not available)
    local dims=$(php -r "\$i=@getimagesize('$file'); if(\$i) echo \$i[0].'x'.\$i[1];")
    [ -z "$dims" ] && echo "SKIP: Cannot read $basename" >> "$ERROR_LOG" && return

    local w=$(echo "$dims" | cut -dx -f1)
    local h=$(echo "$dims" | cut -dx -f2)
    local old_size=$(stat -c%s "$file")

    # Determine if resize is needed
    local max=$((w > h ? w : h))
    local temp_file="/tmp/resize_temp_$$.jpg"

    if [ "$max" -gt "$MAX_DIM" ]; then
        # Resize using PHP GD, then convert to webp
        php -r "
            \$src = @imagecreatefromjpeg('$file');
            if(!\$src) \$src = @imagecreatefrompng('$file');
            if(!\$src) exit(1);
            \$w = imagesx(\$src); \$h = imagesy(\$src);
            \$ratio = min($MAX_DIM/\$w, $MAX_DIM/\$h);
            \$nw = (int)(\$w * \$ratio); \$nh = (int)(\$h * \$ratio);
            \$dst = imagecreatetruecolor(\$nw, \$nh);
            imagecopyresampled(\$dst, \$src, 0,0,0,0, \$nw,\$nh, \$w,\$h);
            imagejpeg(\$dst, '$temp_file', 92);
            imagedestroy(\$src); imagedestroy(\$dst);
        " 2>/dev/null
        [ $? -ne 0 ] && echo "RESIZE FAIL: $basename" >> "$ERROR_LOG" && return
        cwebp -q $WEBP_QUALITY "$temp_file" -o "$new_file" 2>/dev/null
        rm -f "$temp_file"
    else
        # Already small enough, just convert format
        cwebp -q $WEBP_QUALITY "$file" -o "$new_file" 2>/dev/null
    fi

    if [ -f "$new_file" ]; then
        local new_size=$(stat -c%s "$new_file")
        echo "$basename,${name}.webp,$old_size,$new_size" >> "$LOG_FILE"
        # Remove original only after successful conversion
        rm -f "$file"
        # Also remove any cached thumbnails (they'll regenerate on-the-fly)
        rm -f "${UPLOADS_DIR}/thumbnails/${name}_"*
    else
        echo "CONVERT FAIL: $basename" >> "$ERROR_LOG"
    fi
}

export -f process_file
export UPLOADS_DIR LOG_FILE ERROR_LOG MAX_DIM WEBP_QUALITY

# Process in batches to avoid overwhelming the server
find "$UPLOADS_DIR" -maxdepth 1 -type f \
    \( -iname "*.jpg" -o -iname "*.jpeg" -o -iname "*.png" \) \
    -print0 | xargs -0 -n1 -P4 bash -c 'process_file "$0"'

echo "Conversion complete. See $LOG_FILE for results."
echo "Errors (if any): $ERROR_LOG"
```

### Phase 5: Database Update

After conversion, update all tables that reference filenames:

```sql
-- Update meter_files (largest table — 190,886 JPG records)
UPDATE meter_files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';

-- elecphoto_files
UPDATE elecphoto_files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';

-- files
UPDATE files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';

-- genral_files (image records only — most are PDFs)
UPDATE genral_files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';

-- permit_files (small number of images)
UPDATE permit_files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';

-- sketch_files (small number of images)
UPDATE sketch_files
SET unique_filename = CONCAT(
    LEFT(unique_filename, LENGTH(unique_filename) - LOCATE('.', REVERSE(unique_filename))),
    '.webp'
)
WHERE LOWER(unique_filename) LIKE '%.jpg'
   OR LOWER(unique_filename) LIKE '%.jpeg'
   OR LOWER(unique_filename) LIKE '%.png';
```

### Phase 6: Thumbnail Cache Cleanup

```bash
# Remove all old cached thumbnails (they'll regenerate on-the-fly from WebP sources)
find /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/thumbnails/ \
    -type f \( -iname "*.jpg" -o -iname "*.jpeg" -o -iname "*.png" \) -delete
```

---

## 4. Risks & Mitigations

| Risk | Impact | Mitigation |
|------|--------|-----------|
| PHP 5.3 GD cannot read/write WebP natively | Thumbnails would break | `Resize.php` bridges via `dwebp`/`cwebp` CLI — generates thumbnails on-the-fly |
| `dwebp`/`cwebp` exec() calls blocked | Thumbnails fail | Test exec() permissions first; verify in Phase 1 POC |
| Old browsers don't support WebP | Images don't display | All modern browsers support WebP (Chrome, Firefox, Safari 14+, Edge). IE11 is the only concern — verify no IE11 users |
| Conversion fails for some files | Missing images | Keep conversion log; verify all files before deleting originals |
| DB update misses a table | Broken image links | Audit all tables first; test with small batch on dev |
| Rollback needed | Need originals back | Full backup before starting; keep backups for 90 days |
| Server load during conversion | Site performance | Run during off-hours; use `nice`/`ionice`; limit to 4 parallel workers |
| First thumbnail access slower | User sees brief delay | ~200-500ms one-time cost per image; cached forever after |
| PDF files accidentally affected | PDFs break | Conversion script explicitly targets only .jpg/.jpeg/.png extensions |
| Temp files accumulate in /tmp | Disk fills | Cleanup in Resize.php; also cleared by OS tmpwatch |

### The dwebp/cwebp Bridge — Why This Is Safe

```
PHP 5.3 Limitation:          Our Solution:

  imagecreatefromwebp() ✗     dwebp → temp PNG → imagecreatefrompng() ✓
  imagewebp() ✗               imagepng() → temp PNG → cwebp ✓
```

- `dwebp` and `cwebp` are stable, well-tested tools from Google's libwebp project
- The temp files are tiny (resized thumbnails, not full images)
- exec() is already available on the server (used by other parts of the app)
- Each temp file lives only milliseconds before cleanup

---

## 5. Implementation Order

### Milestone 1: Proof of Concept (10 files)
1. Install `libwebp-tools` on remote (`sudo yum install -y libwebp-tools`)
2. Configure Apache WebP MIME type
3. Pick 10 test images (various sizes) from `dev/scheduler/uploads/`
4. Convert manually with `cwebp`, verify they display in browser
5. Deploy modified `Resize.php` to dev
6. Test that `thumbnail()` generates thumbnails from `.webp` source on-the-fly
7. Verify thumbnail caching works (second request serves cached file)

### Milestone 2: Small Batch (1,000 files)
1. Run conversion script on 1,000 files from `dev/scheduler/uploads/`
2. Update DB records for those files
3. Verify display in scheduler UI — full-size images and thumbnails
4. Measure actual space savings vs estimates
5. Load test: access 50 unconverted thumbnails rapidly to confirm on-the-fly generation

### Milestone 3: Full Conversion — `scheduler/uploads/`
1. Full backup of uploads directory and DB tables
2. Deploy modified `Resize.php` to production scheduler
3. Run conversion script (estimate: 12-24 hours at ~4 files/sec with 4 parallel workers)
4. Update all DB tables
5. Clean up old thumbnail cache
6. Verify random sample of 100 jobs in UI
7. Monitor error logs for 1 week

### Milestone 4: Cleanup
1. Verify backup integrity
2. Remove backup after 90-day retention period
3. Document the new upload workflow for future photos

---

## 6. Future Considerations

### New Photo Uploads
After conversion, new photos uploaded through the scheduler will still be JPG/JPEG (the upload code uses PHP GD). Options:
- **Option A:** Leave new uploads as JPG (they'll be a small % of total over time)
- **Option B:** Add a cron job that periodically converts new uploads to webp
- **Option C:** Modify the upload controller to shell out to `cwebp` after saving — resize to 800px and convert immediately at upload time

### Handling Mixed Formats
During and after migration, both `.jpg` and `.webp` files may coexist. The modified `Resize.php` handles all formats transparently — JPEG/PNG via GD directly, WebP via the dwebp/cwebp bridge.

### PHP Upgrade Path
If/when PHP is upgraded beyond 5.3:
- GD will natively support WebP (PHP 7.0+ with libwebp)
- The dwebp/cwebp bridge in `Resize.php` can be replaced with native `imagecreatefromwebp()` / `imagewebp()`
- No other changes needed — the architecture remains the same

---

## 7. Alternative Approaches Considered

### A. .htaccess Content Negotiation (Rejected)
Serve .webp transparently via Apache rewrite rules while keeping original .jpg files.
- **Rejected because:** Does not save disk space — both files would exist. Defeats the primary goal.

### B. Pre-generate All Thumbnails During Batch Conversion (Rejected)
Generate every possible thumbnail size during the batch conversion step.
- **Rejected because:** Would need to know all thumbnail sizes in advance. The on-the-fly approach with caching is more flexible and handles any size the app requests. Also avoids generating thumbnails that may never be viewed.

### C. Resize Only, Keep JPG (Simpler Alternative)
Just resize oversized images to 800px without changing format.
- **Pro:** No MIME type changes, no DB updates, no Resize.php changes
- **Con:** Misses the additional 25-34% savings from WebP; no visual quality improvement
- **Estimated savings:** ~142 GB (vs ~157 GB with WebP)
- **Could be a good Phase 1** if WebP complexity is concerning

### D. Move to Cloud Storage (S3/CloudFront)
Offload images to S3 with automatic WebP conversion via CloudFront.
- **Pro:** Best long-term solution, infinite storage, CDN delivery
- **Con:** Requires significant code changes, ongoing cost, more complex
- **Consider for:** Future major upgrade

---

## 8. Files That Need Modification

| File | Change | Risk |
|------|--------|------|
| `system/application/third_party/Resize.php` | Add WebP support to `openImage()` and `saveImage()` via dwebp/cwebp CLI bridge | Low — additive change, existing formats untouched |
| `/etc/httpd/conf/httpd.conf` | Add `AddType image/webp .webp` | Low — standard MIME type addition |
| Database (6 tables) | Update `unique_filename` extension from .jpg/.png to .webp | Medium — requires backup; reversible with backup SQL |

**No changes needed to:**
- `thumbnail_helper.php` — already handles `.webp` extension in filename/path logic
- `thumbnail.php` config — cache type and paths work as-is
- View files — they reference `unique_filename` from DB; extension is transparent
- Upload controllers — existing uploads continue to work (future enhancement)

---

## 9. Decision Points

Before proceeding, decisions needed on:

1. **Which approach?** Full WebP conversion (recommended) vs resize-only (simpler)?
2. **Backup storage location?** Need ~167 GB free space for backup archive
3. **Conversion window?** When is lowest-usage time for the scheduler?
4. **IE11 support?** Any users on Internet Explorer? (WebP not supported in IE11)
5. **New uploads policy?** Convert on upload, periodic cron, or leave as JPG?
6. **WebP quality setting?** Recommend `q80` for good quality/size balance — want higher?

---

## Appendix A: Quick Reference Commands

```bash
# Install tools
sudo yum install -y libwebp-tools

# Test single file conversion
cwebp -q 80 input.jpg -o output.webp

# Test WebP decode (for thumbnail pipeline)
dwebp input.webp -o output.png

# Check file size reduction
ls -la input.jpg output.webp

# Check Apache serves correct MIME type
curl -I http://aeihawaii.com/scheduler/uploads/test.webp | grep Content-Type
# Should show: Content-Type: image/webp

# Verify exec() works from PHP
php -r "exec('dwebp -version', \$o, \$r); echo \$r === 0 ? 'OK' : 'FAIL';"
```

## Appendix B: Resize.php Full Diff

The complete modified `Resize.php` with WebP support will be prepared during implementation. The changes are limited to:
- `openImage()`: Add `.webp` detection before the existing switch, decode via `dwebp` to temp PNG
- `saveImage()`: Add `.webp` case to the switch, encode via `cwebp` from temp PNG
- Constructor/destructor: Track and clean up temp files
- All existing JPEG/PNG/GIF handling remains unchanged
