# Folder Year Sync Enhancement

**Date:** 2026-02-05
**Status:** Planned
**Priority:** Medium
**Enhancement Folder:** `ENHANCEMENTS/FOLDER_YEAR_SYNC/`

---

## 1. Problem Statement

The remote and local servers use divergent logic to determine the folder path for uploaded photos. This causes the same photo to live in different year-based directories on each server.

### Remote Server Logic (upload.php)

Strictly year-based — always derives folder year from `job_date`:

```php
$job_year = date("Y", strtotime($job_date_t));
$uploadDir = '/mnt/dropbox/' . $job_year . ' Customers/' . $indexname;
```

A 2026 job **always** goes to `2026 Customers/`.

### Local Server Logic (uploadlocallat_kuldeep.php)

DB-driven with year fallback — queries `unified_customers` table:

```
Priority 1: customer_id + folder_year match       -> exact match, use it
Priority 2: customer_id (any year, most recent)    -> REUSES old year folder
Priority 3: first_name + last_name + folder_year   -> exact match, use it
Priority 4: first_name + last_name (any year)      -> REUSES old year folder
Priority 5: Auto-create new folder + DB record     -> creates for current year
```

A 2026 job for a customer who already exists in 2025 will **reuse the 2025 folder** (Priority 2 or 4).

### Divergence Example

Customer "Mitchell, Linwood" — 2025 Premeasure, then 2026 Installation:

| Upload | Remote Server | Local Server |
|--------|---------------|--------------|
| 2025 PM | `/mnt/dropbox/2025 Customers/M/Mitchell, Linwood/Survey/...` | `/mnt/dropbox/2025 Customers/M/Mitchell, Linwood/Survey/...` |
| 2026 SC | `/mnt/dropbox/2026 Customers/M/Mitchell, Linwood/Install/...` | `/mnt/dropbox/2025 Customers/M/Mitchell, Linwood/Install/...` |

The 2026 Installation photos are in `2026 Customers/` on remote but `2025 Customers/` on local.

---

## 2. Affected Files

| File | Server | Path (Deployed) |
|------|--------|-----------------|
| `uploadlocallat_kuldeep.php` | Local | `/var/www/html/upload/uploadlocallat_kuldeep.php` |

Backup: `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/LOCAL/uploadlocallat_kuldeep.php.current`

The remote `upload.php` does **not** need changes — its year-based routing is correct.

---

## 3. Current Code (Local Server)

### getCustomerFolderByCustomerId() — lines 21-82

```php
function getCustomerFolderByCustomerId($customer_id, $job_year) {
    // Priority 1: Exact customer_id + year match
    $query = "SELECT * FROM unified_customers
              WHERE remote_customer_id='$customer_id' AND folder_year='$job_year'
              ORDER BY id DESC LIMIT 1";
    // If found, return folder_path

    // Priority 2: Any year (most recent)
    $query = "SELECT * FROM unified_customers
              WHERE remote_customer_id='$customer_id'
              ORDER BY folder_year DESC LIMIT 1";
    // If found, return folder_path  <-- THIS IS THE PROBLEM
    // Returns a 2025 path for a 2026 job
}
```

### getCustomerFolderByName() — lines 88-145

```php
function getCustomerFolderByName($first_name, $last_name, $job_year) {
    // Priority 3: Exact name + year match
    // Priority 4: Any year (most recent)  <-- SAME PROBLEM
}
```

---

## 4. Proposed Change

When the lookup falls back to a different year (Priority 2 or 4), instead of returning the old-year folder, use the existing record as a **template** to create a new folder for the correct year.

### Modified Logic

```
Priority 1: customer_id + folder_year match       -> use it (NO CHANGE)
Priority 2: customer_id (different year found)     -> CREATE new year folder from template
Priority 3: first_name + last_name + folder_year   -> use it (NO CHANGE)
Priority 4: first_name + last_name (different year) -> CREATE new year folder from template
Priority 5: No match at all                        -> auto-create (NO CHANGE)
```

### Template-Based Folder Creation

When Priority 2 or 4 finds a record from a different year:

1. Extract the customer name and letter index from the existing `dropbox_folder_path`
   - Example: `/mnt/dropbox/2025 Customers/M/Mitchell, Linwood/` → letter=`M`, name=`Mitchell, Linwood`
2. Construct new path: `/mnt/dropbox/{job_year} Customers/{letter}/{name}/`
3. Create the directory: `mkdir -p /mnt/dropbox/{job_year} Customers/{letter}/{name}/`
4. Insert new `unified_customers` record:
   ```sql
   INSERT INTO unified_customers
   (remote_customer_id, first_name, last_name, folder_year, dropbox_folder_path, ...)
   VALUES ('{customer_id}', '{first_name}', '{last_name}', '{job_year}', '{new_path}', ...)
   ```
5. Return the new path

### Pseudocode

```php
function getCustomerFolderByCustomerId($customer_id, $job_year) {
    // Priority 1: Exact match (unchanged)
    $row = query("... WHERE remote_customer_id='$customer_id' AND folder_year='$job_year'");
    if ($row) return $row['dropbox_folder_path'];

    // Priority 2: Different year — use as template
    $template = query("... WHERE remote_customer_id='$customer_id' ORDER BY folder_year DESC");
    if ($template) {
        // Replace year in path
        $old_path = $template['dropbox_folder_path'];
        $new_path = preg_replace('/\d{4} Customers/', $job_year . ' Customers', $old_path);

        // Create directory
        if (!file_exists($new_path)) {
            mkdir($new_path, 0777, true);
        }

        // Insert new DB record
        insert_unified_customer($customer_id, $first_name, $last_name, $job_year, $new_path);

        return $new_path;
    }

    return null; // fall through to name-based lookup
}
```

---

## 5. Relationship to `folder_path` Column

The `folder_path` column in `meter_files` (added by the PHOTO_FOLDER_PATH enhancement) stores the exact path used on the remote server. This provides the authoritative source of truth.

**Long-term goal:** The local server should eventually use `folder_path` from `meter_files` directly, eliminating the need for its own path computation entirely. The sync payload from `sync_to_local.py` already includes the metadata needed — it could also pass the `folder_path` value.

**This enhancement is a stepping stone:** It ensures the local server's computed path matches the remote server's path, even before the `folder_path` column is used for direct path resolution.

---

## 6. Migration Considerations

- **Existing files stay put:** Files already stored in old-year folders do NOT need to move
- **Multi-year records expected:** The `unified_customers` table will have multiple records per customer (one per year). This is by design.
- **No backward compatibility issues:** The new behavior only affects new uploads. Old uploads continue to be found via their existing DB records.

---

## 7. Testing

### QA Scenario

1. Ensure a customer exists in `unified_customers` for 2025 only
2. Upload a photo for that customer with a 2026 `job_date`
3. Verify:
   - Remote: file lands in `2026 Customers/` (existing behavior)
   - Local: file lands in `2026 Customers/` (new behavior, not 2025)
   - New `unified_customers` record created for 2026
   - Existing 2025 record unchanged

### Edge Cases

| Case | Expected Behavior |
|------|-------------------|
| Customer has both 2025 and 2026 records | Uses 2026 record (Priority 1 match) — no change |
| Customer only has 2024 record, job is 2026 | Creates 2026 folder from 2024 template |
| Brand new customer, no records | Auto-create as before (Priority 5) — no change |
| `folder_year` is NULL in existing record | Treat as no match, fall through to name-based |

---

## 8. Success Criteria

- [ ] New uploads for year N go to `{N} Customers/` on both servers
- [ ] Local server creates new year folder + `unified_customers` record when falling back
- [ ] Existing files in old-year folders remain untouched
- [ ] QA test verifies consistent year-based paths across remote and local
- [ ] No regression for customers with exact year match in DB
