# ENH-F02: API Unification — Eliminate "1" Variant Files

**Status:** Planned (Future)
**Date:** 2026-02-18
**Priority:** Medium
**Risk:** Medium (requires coordinated Flutter app + server changes)
**Effort:** Medium
**Depends On:** PHOTO-012 (Mobile Listing Fix), PHOTO-013 (Security Hardening)

---

## Problem

The photo API has parallel "1" variant files that diverged from the base files:

| Base File | "1" Variant | Used By |
|-----------|-------------|---------|
| `getimagelisting.php` | `getimagelisting1.php` | Mobile (via loginapinew.php) |
| `fetch_image.php` | `fetch_image1.php` | Mobile (image URLs) |

The base files received PHOTO-009 (Unified Storage) improvements:
- DB-only photo lookups (no filesystem scanning)
- Proper metadata JOINs
- Auth token validation
- DB-based image serving with fallbacks

The "1" variants were left behind with bugs (C1: undefined `$info`, no auth on fetch).

Additionally, `loginapinew.php` (used by Flutter) calls `getimagelisting1.php` (line 441), while `loginapi.php` (older controller) calls the correct `getimagelisting.php` (line 370).

A third variant, `getimagelistingnew.php`, is called from one path in loginapinew.php (line 1151). Its purpose and current usage need investigation.

---

## Goal

1. Have `loginapinew.php` call PHOTO-009's `getimagelisting.php` instead of `getimagelisting1.php`
2. Have mobile image URLs use `fetch_image.php` instead of `fetch_image1.php`
3. Delete the "1" variant files
4. Investigate and potentially delete `getimagelistingnew.php`

---

## Approach

### Phase 1: Server-Side Redirect (Zero Flutter Changes)

Update `getimagelisting1.php` to be a thin wrapper that forwards requests to `getimagelisting.php`:

```php
<?php
// getimagelisting1.php — redirect to unified endpoint
$input = file_get_contents("php://input");
$ch = curl_init('http://localhost/photoapi/getimagelisting.php');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $input);
curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
$response = curl_exec($ch);
curl_close($ch);
header('Content-Type: application/json');
echo $response;
```

Similarly, `fetch_image1.php` can redirect to `fetch_image.php`.

**Advantage:** No Flutter app changes needed. Mobile app continues hitting the "1" URLs.
**Risk:** Low — wrapper is transparent.

### Phase 2: Update loginapinew.php

Change line 441 in `loginapinew.php`:
```php
// Before:
$url="https://aeihawaii.com/photoapi/getimagelisting1.php";
// After:
$url="https://aeihawaii.com/photoapi/getimagelisting.php";
```

Update fetch_image URL pattern in the response to use `fetch_image.php`.

### Phase 3: Flutter App Update

Update the Flutter app to construct image URLs using `fetch_image.php` instead of `fetch_image1.php`. This requires a new app build and deployment.

### Phase 4: Cleanup

After confirming no traffic hits the "1" files (check Apache access logs):
- Delete `getimagelisting1.php`
- Delete `fetch_image1.php`
- Investigate `getimagelistingnew.php` usage and delete if unused

---

## Prerequisites

- **PHOTO-012 deployed:** Ensures getimagelisting1.php has correct metadata (in case Phase 1 wrapper approach isn't used and both files coexist during transition)
- **PHOTO-013 deployed:** Ensures security fixes are in place before unification
- **Flutter app image URL format verified:** Confirm the app constructs URLs from the `link` field returned by the listing endpoint (not hardcoded)

---

## Files to Modify

| File | Change |
|------|--------|
| `photoapi/getimagelisting1.php` | Replace with redirect wrapper (Phase 1) or delete (Phase 4) |
| `photoapi/fetch_image1.php` | Replace with redirect wrapper (Phase 1) or delete (Phase 4) |
| `photoappsch/loginapinew.php` | Change URL from `getimagelisting1.php` to `getimagelisting.php` (Phase 2) |
| `photoapi/getimagelistingnew.php` | Investigate and potentially delete (Phase 4) |

---

## Success Criteria

- All mobile photo operations use the PHOTO-009 unified endpoints
- No "1" variant files remain in production
- Zero regression in photo listing, viewing, or metadata display
- Apache access logs show no traffic to deleted endpoints (2-week soak)
