# Enhancement: Folder Year Sync

**Date:** 2026-02-05
**Status:** Planned
**Priority:** Medium

## Problem

The remote and local servers use divergent logic to determine where a photo is stored, leading to inconsistent folder paths across servers.

### Remote Server (upload.php)
- **Strictly year-based:** Derives year from `job_date` always
- `$job_year = date("Y", strtotime($job_date_t))`
- A 2026 job **always** goes to `2026 Customers/` regardless of prior years

### Local Server (uploadlocallat_kuldeep.php)
- **DB-driven with year fallback:** Queries `unified_customers` table
- Lookup priority:
  1. `customer_id` + `folder_year` match
  2. `customer_id` (any year) — uses most recent
  3. `first_name` + `last_name` + `folder_year` match
  4. `first_name` + `last_name` (any year) — uses most recent
  5. Auto-create new folder for current year
- A 2026 job for an existing 2025 customer **reuses the 2025 folder**

### Consequence

A photo for a 2026 job can live in a `2026 Customers/` folder on the remote server but a `2025 Customers/` folder on the local server. This complicates debugging, file retrieval, and cross-server verification.

### Example

Customer "Mitchell, Linwood" — 2025 Premeasure, then 2026 Installation:

| Upload | Remote Path | Local Path |
|--------|-------------|------------|
| 2025 PM | `2025 Customers/M/Mitchell, Linwood/Survey/...` | `2025 Customers/M/Mitchell, Linwood/Survey/...` |
| 2026 SC | `2026 Customers/M/Mitchell, Linwood/Install/...` | `2025 Customers/M/Mitchell, Linwood/Install/...` |

The 2026 Installation photos end up in different year folders on each server.

## Affected Files

| File | Server | Role |
|------|--------|------|
| `LOCAL/uploadlocallat_kuldeep.php` | Local | Receives synced photos — uses DB lookup with year fallback |
| `REMOTE/photoapi/upload.php` | Remote | Uploads photos — uses strict year-based routing |

## Recommended Change

**In `uploadlocallat_kuldeep.php`:** When the DB lookup falls back to a different year (priority 2 or 4), instead of reusing the old-year folder, **create a new customer/job folder for the job's actual year** and insert a new `unified_customers` record.

### Current Behavior (Problematic)

```
Query: SELECT * FROM unified_customers WHERE remote_customer_id='123' AND folder_year='2026'
Result: No match

Query: SELECT * FROM unified_customers WHERE remote_customer_id='123' ORDER BY folder_year DESC
Result: 2025 record found -> REUSE 2025 folder  <-- WRONG
```

### Proposed Behavior

```
Query: SELECT * FROM unified_customers WHERE remote_customer_id='123' AND folder_year='2026'
Result: No match

Query: SELECT * FROM unified_customers WHERE remote_customer_id='123' ORDER BY folder_year DESC
Result: 2025 record found -> USE AS TEMPLATE to create 2026 folder + DB record  <-- CORRECT
```

### Implementation Approach

In `getCustomerFolderByCustomerId()` and `getCustomerFolderByName()`:

1. If exact year match found (priority 1 or 3) — use it as-is (no change)
2. If fallback match found (priority 2 or 4) — use the existing record as a **template**:
   - Extract customer name, letter index from the old path
   - Construct new path with `{job_year} Customers/` instead of old year
   - Create the directory on disk: `mkdir -p /mnt/dropbox/{job_year} Customers/{letter}/{name}/`
   - Insert new `unified_customers` record for the new year
   - Return the new path
3. If no match at all (priority 5) — auto-create as current behavior

### Relationship to `folder_path` Column

The `folder_path` column in `meter_files` (already implemented) stores the absolute path used on the remote server. Long-term, the local server should adopt this value directly rather than computing its own path. This enhancement is a stepping stone toward that goal.

## Architecture

```
CURRENT:
  Remote: 2026 job -> /mnt/dropbox/2026 Customers/M/Mitchell/...
  Local:  2026 job -> DB lookup -> finds 2025 record -> /mnt/dropbox/2025 Customers/M/Mitchell/...
  Result: DIVERGENT paths

PROPOSED:
  Remote: 2026 job -> /mnt/dropbox/2026 Customers/M/Mitchell/...
  Local:  2026 job -> DB lookup -> finds 2025 record -> CREATES 2026 folder + record
                   -> /mnt/dropbox/2026 Customers/M/Mitchell/...
  Result: CONSISTENT paths
```

## Migration Considerations

- Existing files in old-year folders do NOT need to move — only new uploads are affected
- The `unified_customers` table will accumulate multi-year records per customer (expected)
- The `folder_path` column in `meter_files` can serve as the authoritative path reference for historical data

## Success Criteria

- [ ] New uploads for year N go to `{N} Customers/` folder on both servers
- [ ] Local server creates new year folder + DB record when falling back
- [ ] Existing files in old-year folders remain untouched
- [ ] QA test verifies consistent paths across servers

## Full Documentation

See `DOCS/FOLDER_YEAR_SYNC_ENHANCEMENT.md` for detailed specification.
