# AEI Photo System — Unified Process Map (Post-PHOTO-021)

**Date:** 2026-02-21
**Status:** Authoritative — reflects all PHOTO-021 fixes applied
**Supersedes:** PHOTO_SYSTEM_PROCESS_MAP.md (2026-02-21, pre-PHOTO-021)
**Enhancement:** PHOTO-021 (Staging Direct, DB-Only Listing, API Fix)

---

## Table of Contents

1. [Architecture Overview](#1-architecture-overview)
2. [What PHOTO-021 Changed](#2-what-enh-021-changed)
3. [Unified Upload Flow](#3-unified-upload-flow)
   - [Mobile App Upload](#31-mobile-app-upload)
   - [Scheduler Web Upload — Drag & Drop](#32-scheduler-web-upload--drag--drop)
   - [Scheduler Web Upload — Multi-File Picker](#33-scheduler-web-upload--multi-file-picker)
   - [Flow Comparison Table](#34-flow-comparison-table)
4. [WebP Derivative Generation](#4-webp-derivative-generation)
5. [Local Sync & Archival](#5-local-sync--archival)
6. [Photo Display — Unified DB-Only](#6-photo-display--unified-db-only)
   - [Scheduler Web UI](#61-scheduler-web-ui)
   - [Mobile App](#62-mobile-app)
7. [Download Flow](#7-download-flow)
8. [API Endpoints Reference](#8-api-endpoints-reference)
9. [Storage Layout](#9-storage-layout)
10. [Database Schema](#10-database-schema)
11. [Consistency Verification](#11-consistency-verification)
12. [Deprecated Components](#12-deprecated-components)

---

## 1. Architecture Overview

All photo uploads — whether from mobile or scheduler — now follow the same staging-first flow. All photo listing is driven by the `meter_files` database table. No filesystem scanning.

```
┌──────────────────────────────────────────────────────────────────┐
│                  REMOTE SERVER (18.225.0.90)                      │
│                                                                  │
│  UPLOAD ENTRY POINTS                                             │
│    Mobile App ──────► photoapi/upload.php                         │
│    Scheduler Drag ──► controllers/phototab.php                    │
│    Scheduler Multi ─► controllers/preupload.php                   │
│              │                                                   │
│              ▼                                                   │
│    uploads/staging/{filename}         ◄── ALL UPLOADS LAND HERE  │
│              │                                                   │
│              ├──► generate_thumbnails.py (nohup &)               │
│              │      ├─► uploads/thumbs/{webpfilename}  200x200   │
│              │      ├─► uploads/webp/{webpfilename}    1024px    │
│              │      └─► uploads/hi-res/{webpfilename}  2048px    │
│              │                                                   │
│              └──► sync_to_local.py (nohup &)                     │
│                     └─► HTTPS POST to local server               │
│                                                                  │
│  DB INSERT: meter_files (job_id, webpfilename, file_type=99)     │
│                                                                  │
│  SERVING (both platforms read from here)                         │
│    uploads/thumbs/   Grid thumbnails (200x200 Q70)               │
│    uploads/webp/     Standard viewing (1024px Q80)               │
│    uploads/hi-res/   High-res (2048px Q82)        ◄ Future use   │
│                                                                  │
│  LISTING (both platforms use the same source)                    │
│    meter_files DB     ◄── SINGLE SOURCE OF TRUTH                 │
│                                                                  │
│  LEGACY (no longer growing)                                      │
│    uploads/*.jpg     ~200GB old JPEGs              ◄ Frozen      │
│    /mnt/aeiserver/   Old filesystem photos                       │
└──────────────────────┬───────────────────────────────────────────┘
                       │ sync_to_local.py
                       ▼
┌──────────────────────────────────────────────────────────────────┐
│                LOCAL SERVER (upload.aeihawaii.com)                 │
│                                                                  │
│  uploadlocallat_kuldeep.php                                      │
│    → Receives full-size original from staging/                   │
│    → Saves to /mnt/dropbox/{Year} Customers/.../                 │
│    → Background: upload_thumb_generator.py                       │
│       → thumbs/{name}.webp  (200x200 Q70)                       │
│       → thumbs/mid/{name}.webp  (800px Q80)                     │
│    → Tracks in local_photos table                                │
│                                                                  │
│  This is the ARCHIVAL store for full-size originals.             │
└──────────────────────────────────────────────────────────────────┘

┌──────────────────────────────────────────────────────────────────┐
│                    MOBILE APP (Flutter)                            │
│                                                                  │
│  Upload: base64 JSON POST → upload.php → staging/                │
│  Display: uploads/thumbs/ (grid) + uploads/webp/ (detail)        │
│  Future: uploads/hi-res/ for HD viewing & downloads              │
└──────────────────────────────────────────────────────────────────┘

┌──────────────────────────────────────────────────────────────────┐
│                    SCHEDULER WEB UI                                │
│                                                                  │
│  Upload: CI file upload → staging/ (direct)                      │
│  Display: meter_files DB → uploads/thumbs/ + uploads/webp/       │
│  Download: ZIP from uploads/webp/ (or uploads/ JPEG fallback)    │
└──────────────────────────────────────────────────────────────────┘
```

---

## 2. What PHOTO-021 Changed

### Fix 1: Scheduler Upload → Staging Direct

| Aspect | Before (PHOTO-017) | After (PHOTO-021) |
|--------|-------------------|------------------|
| Upload path | `uploads/` root | `uploads/staging/` |
| Copy to staging | Manual `@copy()` after upload | Not needed — already there |
| CI resize (800x1027) | Overwrote `uploads/` copy | Removed entirely |
| JPEG in uploads/ | Created every upload (~200GB waste growing) | No new JPEG ever created |
| Source for WebP | staging/ copy (same) | staging/ (same, just arrived directly) |
| Source for sync | staging/ copy (same) | staging/ (same) |

### Fix 2: DB-Only Photo Listing

| Aspect | Before | After |
|--------|--------|-------|
| Scheduler photo grid | meter_files DB + scandir `/mnt/aeiserver/` | meter_files DB only |
| `$s3files` variable | Array from `getimagelistingm.php` (filesystem scan) | Empty array `array()` |
| Duplicate photos | Common (same photo from DB and filesystem) | Eliminated |
| Download All ZIP | meter_files + s3files (HTTP fetch, slow) | meter_files only |
| `getimagelisting()` method | Called on every page load | Not called (dead code, retained) |

### Fix 3: API full_link Corrected

| Aspect | Before | After |
|--------|--------|-------|
| `getimagelisting.php` full_link | `/webp/` (wrong) | `/hi-res/` (correct) |
| Consistency with other endpoints | Inconsistent | All endpoints return `/hi-res/` |

---

## 3. Unified Upload Flow

All three upload entry points now follow an identical pipeline:

```
Source (browser/app) → staging/{filename} → [background] WebP + Sync → DB INSERT
```

### 3.1 Mobile App Upload

**Entry point:** `photoapi/upload.php`
**Trigger:** Flutter app POST with base64-encoded image + metadata

```
Mobile App sends JSON POST:
  { auth_token, job_id, file_name, image_data (base64) }
         │
         ▼
1. Validate auth token (aei@89806849)
         │
         ▼
2. Decode base64 → raw image bytes
         │
         ▼
3. Save to uploads/staging/{uniqid}_{filename}
   Prefix prevents concurrent upload collisions.
         │
         ▼
4. DB lookup: jobs → customer info, job_date, job_type
         │
         ▼
5. Generate webpfilename:
   {LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-hash}.webp
   PhotoType: S (Survey) for PM/WM/AS/RPM/GCPM, I (Installation) for all others
         │
         ▼
6. DB INSERT meter_files (job_id, unique_filename, webpfilename, file_type=99)
         │
         ├──► Background: generate_thumbnails.py  → thumbs/ + webp/ + hi-res/
         │
         └──► Background: sync_to_local.py        → local server archival
         │
         ▼
7. Return JSON: {"success":1, "status":"ok"}
   Immediate response (~0.5s). WebP appears 1-3s later. Sync 2-10s later.
```

### 3.2 Scheduler Web Upload — Drag & Drop

**Entry point:** `controllers/phototab.php` → `dropImageUpload($job_id)`
**Trigger:** Drag-and-drop or file picker in Photos tab

```
Browser sends multipart file POST
         │
         ▼
1. CI upload library saves to uploads/staging/{encrypted_name}
   Config: upload_path = FCPATH . 'uploads/staging/'
         │
         ▼
2. (Production only) Dev-env path resolution:
   If /dev/ detected in base_url → copy from dev/staging/ to main/staging/
   Otherwise: filePath = /httpdocs/scheduler/uploads/staging/{encrypted_name}
         │
         ▼
3. $stagingPath = $filePath   (already in staging, no copy needed)
         │
         ▼
4. Generate webpfilename:
   {LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-hash}.webp
   Same naming convention as mobile.
         │
         ├──► Background: sync_to_local.py ($stagingPath → local server)
         │
         └──► Background: generate_thumbnails.py ($stagingPath → thumbs/webp/hi-res/)
         │
         ▼
5. filesize($stagingPath) → file_size for DB record
         │
         ▼
6. DB INSERT meter_files (job_id, unique_filename, webpfilename, file_type=99)
```

### 3.3 Scheduler Web Upload — Multi-File Picker

**Entry point:** `controllers/preupload.php` → `submit($job_id)`
**Trigger:** Multi-file upload form

Identical pipeline to §3.2, executed in a loop for each file:

```
FOR EACH file in $_FILES['files']:
  1. CI upload to uploads/staging/{encrypted_name}
  2. Dev-env path resolution
  3. $stagingPath = $filePath
  4. Generate webpfilename
  5. Background: sync_to_local.py + generate_thumbnails.py
  6. filesize($stagingPath) → file_size
  7. DB INSERT meter_files
```

### 3.4 Flow Comparison Table

Every upload path is now consistent:

| Step | Mobile (upload.php) | Scheduler Drag (phototab.php) | Scheduler Multi (preupload.php) |
|------|--------------------|-----------------------------|-------------------------------|
| **Input format** | base64 JSON | multipart form | multipart form (loop) |
| **Landing directory** | `uploads/staging/` | `uploads/staging/` | `uploads/staging/` |
| **Filename prefix** | `uniqid()_` | CI `encrypt_name` | CI `encrypt_name` |
| **JPEG in uploads/ root** | Never | Never | Never |
| **CI resize (800x1027)** | N/A | Removed | Removed |
| **WebP generation** | `generate_thumbnails.py` (nohup &) | `generate_thumbnails.py` (nohup &) | `generate_thumbnails.py` (nohup &) |
| **3 tiers generated** | thumbs + webp + hi-res | thumbs + webp + hi-res | thumbs + webp + hi-res |
| **Local sync** | `sync_to_local.py` (nohup &) | `sync_to_local.py` (nohup &) | `sync_to_local.py` (nohup &) |
| **DB table** | `meter_files` | `meter_files` | `meter_files` |
| **file_type** | 99 | 99 | 99 |
| **webpfilename** | Set | Set | Set |
| **Response** | JSON `{"success":1}` | HTML (view reload) | 302 redirect to gallery |

---

## 4. WebP Derivative Generation

**Script:** `generate_thumbnails.py` v4.0
**Runtime:** Python 3.6 + Pillow (LANCZOS resampling)
**Execution:** Background via `nohup /usr/local/bin/python3.6 ... &`
**Called by:** All three upload entry points identically

### Arguments

```bash
python3.6 generate_thumbnails.py <source_image> <webpfilename> <uploads_dir>
# Example:
python3.6 generate_thumbnails.py \
  /path/uploads/staging/abc123.jpg \
  SmithJohnAC-I02-18-2026_IMG0ec2c8b5682e.webp \
  /path/uploads
```

### 3-Tier Output

| Tier | Directory | Dimensions | Quality | When Generated | Used By |
|------|-----------|-----------|---------|----------------|---------|
| **Thumbnail** | `uploads/thumbs/` | 200x200 crop | Q70 | Phase 1 (priority) | Grid views — scheduler + mobile |
| **Standard** | `uploads/webp/` | 1024px max dim | Q80 | Phase 1 (priority) | Lightbox + mobile detail view |
| **Hi-Res** | `uploads/hi-res/` | 2048px max dim | Q82 | Phase 2 (deferred) | Future HD viewing + downloads |

### Processing Details

- **Phase 1 (immediate):** Thumbnail + Standard — needed for the photo to appear in the UI
- **Phase 2 (deferred):** Hi-Res — only if source > 1024px (no point duplicating standard tier)
- EXIF rotation applied before processing
- RGBA/P/LA modes converted to RGB (white background)
- Atomic saves (temp file + rename) prevent serving partial images
- Idempotent — skips existing output files
- 120-second timeout per image

### Platform Parity

Both mobile and scheduler trigger the exact same script with the same arguments. Output is identical regardless of upload source. A photo uploaded from the mobile app and one uploaded from the scheduler web UI produce the same three WebP tiers.

---

## 5. Local Sync & Archival

### Remote → Local (sync_to_local.py)

**Called by:** All three upload entry points (nohup &)
**Source:** `staging/{filename}` (full-size original)
**Target:** `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php`
**Method:** Multipart POST (file + metadata)

```
Arguments passed (identical from all upload sources):
  staging_path, filename, job_id, customer_id,
  job_type, last_name, first_name, job_date, photo_type
```

### Retry Policy

```
Attempt 1: immediate
Attempt 2: wait 2s
Attempt 3: wait 4s
If all fail: queue/{timestamp}_{random}.json
  → Cron (process_retry_queue.py) retries every 15 min
  → Max 10 retries (~2.5 hours)
  → Logged to photoapi/logs/sync_to_local.log
```

### Local Receipt (uploadlocallat_kuldeep.php)

1. Validate auth token and MIME type
2. Look up customer folder path:
   `/mnt/dropbox/{Year} Customers/{Letter}/{LastName}, {FirstName}/{Survey|Installation}/{Job}/`
3. Move file to customer folder (full-size original preserved permanently)
4. Background: `upload_thumb_generator.py`
   - `thumbs/{name}.webp` (200x200 Q70)
   - `thumbs/mid/{name}.webp` (800px Q80)
5. Track in `local_photos` table

### Platform Parity

Sync is triggered from all three upload entry points. There is no upload path that skips the sync. A photo uploaded from the scheduler web UI is synced to local storage with the same timing and retry guarantees as one uploaded from the mobile app.

### Staging Cleanup

`process_retry_queue.py` (cron) cleans staging files older than 7 days. By then, the file has been synced to local and WebP derivatives generated.

---

## 6. Photo Display — Unified DB-Only

### 6.1 Scheduler Web UI

**Controller:** `phototab.php` → `photo_index($job_id)`
**View:** `views/photo_tab/photos_index.php`

```
1. Query meter_files for unique dates:
   SELECT DISTINCT DATE_FORMAT(created,'%Y-%m-%d') FROM meter_files
   WHERE job_id = ? AND file_type = 99 ORDER BY created DESC

2. For each date, query full records:
   SELECT * FROM meter_files WHERE job_id = ? AND file_type = 99
   AND DATE_FORMAT(created,'%Y-%m-%d') = ? ORDER BY created DESC

3. Pass to view as $meter_files (keyed by date)

4. $s3files = array()  ←── PHOTO-021: no filesystem scanning
```

**View rendering logic (per photo):**

```php
// Thumbnail (grid)
IF webpfilename AND uploads/thumbs/{webpfilename} exists:
    src = base_url() . 'uploads/thumbs/' . rawurlencode(webpfilename)
ELSE:
    src = thumbnail() helper (GD on-the-fly fallback for old photos)

// Lightbox (full view)
IF webpfilename AND uploads/webp/{webpfilename} exists:
    href = base_url() . 'uploads/webp/' . rawurlencode(webpfilename)
ELSE:
    href = thumbnail() helper (GD fallback for old photos)
```

**Photos are grouped by date** with date headers and Select/Deselect buttons per group.

### 6.2 Mobile App

**API chain:**
```
loginapinew.php
  → customerphotostabstatenew($customer_id)
    → getimageforapinew($job_id)
      → POST to getimagelistingnew.php (single most recent photo)
    → POST to getimagelisting.php (all photos for customer's jobs)
    → POST to getimagelisting1.php (with survey photo support)
```

**URLs returned to app (all endpoints consistent post-PHOTO-021):**
```json
{
  "link":       "https://aeihawaii.com/scheduler/uploads/webp/{webpfilename}",
  "thumb_link": "https://aeihawaii.com/scheduler/uploads/thumbs/{webpfilename}",
  "full_link":  "https://aeihawaii.com/scheduler/uploads/hi-res/{webpfilename}"
}
```

**What the mobile app uses today:**
- Grid view: `thumb_link` → `uploads/thumbs/` (200x200 Q70 WebP)
- Detail view: `link` → `uploads/webp/` (1024px Q80 WebP)
- `full_link` → `uploads/hi-res/` (returned, not yet utilized — future HD feature)

### Platform Parity — Display

| Aspect | Scheduler Web | Mobile App |
|--------|--------------|------------|
| **Data source** | `meter_files` DB (direct SQL) | `meter_files` DB (via API) |
| **Filesystem scanning** | None | None |
| **Grid thumbnail** | `uploads/thumbs/{webpfilename}` | `uploads/thumbs/{webpfilename}` |
| **Full view** | `uploads/webp/{webpfilename}` | `uploads/webp/{webpfilename}` |
| **Fallback (old photos)** | GD `thumbnail()` helper | Not applicable (old photos have webpfilename from backfill) |
| **Grouping** | By date (date headers) | By job/customer |

---

## 7. Download Flow

**Controller:** `phototab.php` → `download_all_photos($job_id)`
**Trigger:** "Download All" button on scheduler Photos tab

```
1. Query meter_files for job_id (grouped by date, file_type=99)

2. Create ZIP in memory:
   FOR EACH meter_file:
     IF webpfilename AND uploads/webp/{webpfilename} exists:
       Add uploads/webp/{webpfilename} to ZIP        (1024px Q80 WebP)
     ELSE:
       Add uploads/{unique_filename} to ZIP           (legacy JPEG fallback)

3. Stream ZIP to browser:
   filename = "Job_{id}_{CustomerName}_Photos_{date}.zip"
```

**Post-PHOTO-021 changes:**
- No more `$s3files` in the ZIP (was fetching via HTTP, slow and duplicated DB photos)
- ZIP contains only meter_files-sourced photos
- WebP preferred, JPEG fallback for pre-WebP photos only

**Future:** When hi-res tier is utilized, downloads should pull from `uploads/hi-res/` instead of `uploads/webp/` for maximum quality.

---

## 8. API Endpoints Reference

### Listing Endpoints (All DB-Driven)

| Endpoint | Purpose | Called By | `link` | `thumb_link` | `full_link` |
|----------|---------|-----------|--------|--------------|-------------|
| `getimagelisting.php` | All photos for customer's jobs | Mobile (customer tab) | `webp/` | `thumbs/` | `hi-res/` |
| `getimagelistingnew.php` | Single most recent photo | Mobile (job thumbnail) | `webp/` | `thumbs/` | `hi-res/` |
| `getimagelisting1.php` | All photos + survey proxy | Mobile (customer tab v2) | `webp/` | `thumbs/` | `hi-res/` |

All three listing endpoints now return consistent URL formats. All use `meter_files` DB as their source.

### Upload Endpoints

| Endpoint | Method | Called By | Destination |
|----------|--------|-----------|-------------|
| `photoapi/upload.php` | POST (base64 JSON) | Mobile app | `uploads/staging/` |
| `phototab/dropImageUpload` | POST (multipart) | Scheduler drag-drop | `uploads/staging/` |
| `preupload/submit` | POST (multipart, multi-file) | Scheduler file picker | `uploads/staging/` |

### Serving Endpoints

| Method | URL Pattern | Notes |
|--------|-------------|-------|
| **Direct Apache static** | `uploads/webp/{file}`, `uploads/thumbs/{file}` | Fastest — no PHP |
| `fetch_image.php` | DB lookup → stream file | Fallback: webp/ → uploads/ → /mnt/aeiserver/ |
| `fetch_image1.php` | Filesystem → 302 redirect | Supports `?size=thumb` |

---

## 9. Storage Layout

### Remote Server (Production)

```
/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/
│
├── staging/                            TEMPORARY — 7-day cleanup
│   ├── {uniqid}_{filename}.jpg         Mobile uploads (full-size original)
│   └── {encrypted_name}.jpg            Scheduler uploads (full-size original)
│
├── thumbs/                             ACTIVE — ~196K files
│   └── {webpfilename}.webp             200x200, Q70 (grid thumbnails)
│
├── webp/                               ACTIVE — ~196K files
│   └── {webpfilename}.webp             1024px max dim, Q80 (standard viewing)
│
├── hi-res/                             GENERATED — ~85K files (future use)
│   └── {webpfilename}.webp             2048px max dim, Q82 (HD viewing later)
│
└── *.jpg                               LEGACY — ~200K files, ~200GB
    └── {encrypted_name}.jpg            Old CI-resized 800x1027 JPEGs
                                        FROZEN: no new files added (PHOTO-021)
                                        Read only for pre-WebP photos
```

### Local Server (Archival)

```
/mnt/dropbox/{Year} Customers/
└── {Letter}/
    └── {LastName}, {FirstName}/
        ├── Survey/
        │   └── {CustomerName}, {JobType}-S, {Date}/
        │       ├── {filename}               Full-size original (permanent)
        │       └── thumbs/
        │           ├── {filename}.webp      200x200 Q70
        │           └── mid/
        │               └── {filename}.webp  800px Q80
        └── Installation/
            └── (same structure)
```

---

## 10. Database Schema

### meter_files (Remote DB: mandhdesign_schedular)

| Column | Type | Description |
|--------|------|-------------|
| `id` | INT PK | Auto-increment |
| `job_id` | INT | `jobs.job_pid` (NOT `jobs.id` — use caution) |
| `unique_filename` | VARCHAR | CI encrypted name or uniqid-prefixed name |
| `original_filename` | VARCHAR | User's original filename |
| `file_size` | INT | Bytes at upload time |
| `file_type` | INT | 99 = photo |
| `webpfilename` | VARCHAR | WebP derivative name (e.g., `SmithJohnAC-I02-18-2026_IMGabc123.webp`) |
| `folder_path` | VARCHAR | Deprecated — empty for all new uploads |
| `created` | DATETIME | Upload timestamp |

**Serving logic:**
- If `webpfilename` is set → serve from `uploads/webp/` and `uploads/thumbs/`
- If `webpfilename` is NULL (pre-WebP photos) → fall back to `uploads/{unique_filename}` via GD helper

### local_photos (Local DB)

Tracks files received by `uploadlocallat_kuldeep.php`. Used for reconciliation and local gallery features.

### WebP Filename Convention

```
{LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-random-hash}.webp

Components:
  LastName + FirstName  → customer name (no spaces)
  JobType               → job_types.intials (PV, AC, SWH, SC, etc.)
  PhotoType             → S (Survey) or I (Installation)
  Date                  → MM-DD-YYYY format
  12-char-hash          → substr(md5(uniqid(mt_rand(), true)), 0, 12)

Examples:
  SmithJohnAC-I02-18-2026_IMG0ec2c8b5682e.webp
  YagiLelandSC-I02-25-2026_IMGb66a371aca3f.webp
```

---

## 11. Consistency Verification

### Upload Consistency

| Check | Mobile | Scheduler Drag | Scheduler Multi | Status |
|-------|--------|---------------|-----------------|--------|
| Lands in staging/ | Yes | Yes | Yes | Consistent |
| No JPEG in uploads/ root | Yes | Yes | Yes | Consistent |
| No CI resize | N/A | Removed | Removed | Consistent |
| WebP generation triggered | Yes | Yes | Yes | Consistent |
| All 3 tiers generated | Yes | Yes | Yes | Consistent |
| Local sync triggered | Yes | Yes | Yes | Consistent |
| meter_files INSERT | Yes | Yes | Yes | Consistent |
| webpfilename set | Yes | Yes | Yes | Consistent |
| file_type = 99 | Yes | Yes | Yes | Consistent |

### Display Consistency

| Check | Scheduler Web | Mobile App | Status |
|-------|--------------|------------|--------|
| Source: meter_files DB | Yes | Yes | Consistent |
| No filesystem scanning | Yes | Yes | Consistent |
| Grid: uploads/thumbs/ | Yes | Yes | Consistent |
| View: uploads/webp/ | Yes | Yes | Consistent |
| full_link: uploads/hi-res/ | N/A (view only) | Yes (all endpoints) | Consistent |

### Sync Consistency

| Check | Mobile uploads | Scheduler uploads | Status |
|-------|---------------|-------------------|--------|
| sync_to_local.py called | Yes | Yes | Consistent |
| Source: staging/ file | Yes | Yes | Consistent |
| Retry policy (3 inline + queue) | Yes | Yes | Consistent |
| Local archival path | Yes | Yes | Consistent |
| Local thumbnails generated | Yes | Yes | Consistent |

---

## 12. Deprecated Components

These components are retained for backward compatibility but are no longer actively used by the main photo flow:

### Dead Code (Retained, Not Called)

| Component | Location | Why Kept |
|-----------|----------|----------|
| `getimagelisting()` method | phototab.php lines ~930-1040 | Was called by photo_index, now bypassed. Safe to remove in future cleanup. |
| `getimagelistings3()` method | phototab.php lines ~820-930 | Already unused before PHOTO-021. |
| `getimagelistingm.php` | photoapi/ | Filesystem-scanning endpoint. No longer called by scheduler. Mobile never used it. |

### Legacy Fallbacks (Still Active for Old Photos)

| Component | Purpose | When Used |
|-----------|---------|-----------|
| `uploads/{unique_filename}` | JPEG fallback | Photos where `webpfilename IS NULL` (pre-WebP era) |
| `thumbnail()` GD helper | On-the-fly resize | View fallback when WebP thumb/standard not found |
| `fetch_image.php` chain | Stream individual images | Legacy fallback: webp/ → uploads/ → /mnt/aeiserver/ |
| `/mnt/aeiserver/` legacy scan | getimagelisting.php backward compat | Last resort in getimagelisting.php for photos not in meter_files |

### Future Cleanup Opportunities

1. **Delete legacy JPEGs** — `uploads/*.jpg` where `webpfilename IS NOT NULL` in meter_files (~200GB savings)
2. **Remove dead code** — `getimagelisting()`, `getimagelistings3()` methods in phototab.php
3. **Decommission getimagelistingm.php** — no callers remain
4. **Utilize hi-res tier** — mobile app uses `full_link` for HD viewing; scheduler Download All pulls from hi-res/
5. **S3 deletion handler** — phototab.php lines ~773-794 handle S3 file deletion via hidden inputs that no longer exist in the view

---

## Appendix A: Enhancement History

| ENH | Date | Key Change |
|-----|------|------------|
| PHOTO-002 | 2026-02-04 | Async WebP generation (nohup &), mobile response 2-10s → 0.5s |
| PHOTO-009 | 2026-02-17 | Unified storage: DB-driven reads replace scandir for mobile endpoints |
| PHOTO-011 | 2026-02-09 | Reconciliation, sync source fixes, staging cleanup cron |
| PHOTO-017 | 2026-02-19 | Staging flow, 3-tier WebP (thumbs/webp/hi-res), sync_to_local for scheduler |
| PHOTO-019 | 2026-02-19 | Local photo tracking (local_photos table, unified_customers) |
| PHOTO-020 | 2026-02-20 | Gallery UX: 4-col grid, lightbox improvements |
| **PHOTO-021** | **2026-02-21** | **Staging direct upload, DB-only listing, API full_link fix** |

---

## Appendix B: Credentials

| Component | Host | User | Password |
|-----------|------|------|----------|
| Remote DB | localhost (on 18.225.0.90) | schedular | M1gif9!6 |
| API auth token | — | — | aei@89806849 |
| Local sync auth | — | — | remote_token |
| SSH | 18.225.0.90 | Julian | PPK key |

---

## Appendix C: Verification Test Results (2026-02-21)

### Local Dev Testing

| Test | Result |
|------|--------|
| Photo tab loads (HTTP 200) | Pass |
| No s3files errors in response | Pass |
| No s3files rendering block in HTML | Pass |
| Upload via dropImageUpload → file in staging/ | Pass |
| No file created in uploads/ root | Pass |
| meter_files INSERT with webpfilename | Pass |
| Download All endpoint (no s3files crash) | Pass |
| getimagelisting.php full_link → /hi-res/ | Pass |
| PHP lint (preupload.php) | Pass — no syntax errors |
| PHP lint (getimagelisting.php) | Pass — no syntax errors |
| PHP lint (phototab.php, photos_index.php) | Pre-existing parse issues (CI partial files, not from PHOTO-021) |

### Notes on Local Dev Limitations

- `filesize()` fails on production paths (`/var/www/vhosts/...`) that don't exist locally — pre-existing issue, not related to PHOTO-021
- Background Python scripts (`generate_thumbnails.py`, `sync_to_local.py`) don't execute locally (production paths) — derivatives and sync must be tested on production
- AWS phar stub required at `/var/www/vhosts/aeihawaii.com/httpdocs/scaws/aws.phar` for phototab.php to load locally
