# AEI Photo System — Unified Process Map (Post-PHOTO-021)

**Date:** 2026-02-23
**Status:** Authoritative — reflects PHOTO-021/022/023/024 changes
**Supersedes:** PHOTO_SYSTEM_PROCESS_MAP.md (2026-02-21, pre-PHOTO-021)
**Enhancement:** PHOTO-021 (Staging Direct, DB-Only Listing, API Fix) / PHOTO-022 (Remove Hi-Res Tier) / PHOTO-023 (Error Logging, Cron Fallback) / PHOTO-024 (Remove uploads/ Root Dependency)

> **Provenance:** This document reflects code paths reviewed on 2026-02-21 through 2026-02-23
> across all three upload controllers, all listing endpoints, and the deletion cascade.
> Sections marked **(verified)** are confirmed by code inspection; **(inferred)** indicates
> behavior deduced from surrounding code but not directly tested end-to-end.
>
> **Local dev vs. production caveat:** Background Python scripts (`generate_thumbnails.py`,
> `sync_to_local.py`) do not execute in the local dev environment (paths reference production
> `/var/www/vhosts/...`). WebP derivative generation, archive copies, sync, and cron fallback
> can only be validated on the production server. Local dev testing covers PHP upload flow,
> DB inserts, and view rendering only.

---

## Table of Contents

1. [Architecture Overview](#1-architecture-overview)
2. [What PHOTO-021 Changed](#2-what-photo-021-changed)
3. [Unified Upload Flow](#3-unified-upload-flow)
   - [Mobile App Upload](#31-mobile-app-upload)
   - [Scheduler Web Upload — Drag & Drop](#32-scheduler-web-upload--drag--drop)
   - [Scheduler Web Upload — Multi-File Picker](#33-scheduler-web-upload--multi-file-picker)
   - [Flow Comparison Table](#34-flow-comparison-table)
4. [WebP Derivative Generation](#4-webp-derivative-generation)
5. [Local Sync & Archival](#5-local-sync--archival)
6. [Photo Display](#6-photo-display)
   - [Scheduler Web UI](#61-scheduler-web-ui)
   - [Mobile App](#62-mobile-app)
7. [Download Flow](#7-download-flow)
8. [Photo Deletion Flow](#8-photo-deletion-flow)
9. [API Endpoints Reference](#9-api-endpoints-reference)
10. [Storage Layout](#10-storage-layout)
11. [Database Schema](#11-database-schema)
12. [Consistency Verification](#12-consistency-verification)
13. [Failure Handling & Edge Cases](#13-failure-handling--edge-cases)
14. [Operational Notes](#14-operational-notes)
15. [Deprecated Components](#15-deprecated-components)

---

## 1. Architecture Overview

All photo uploads — whether from mobile or scheduler — now follow the same staging-first flow. The **primary** photo listing path is DB-driven (`meter_files` table). However, several legacy fallback paths still perform `/mnt/aeiserver/` filesystem scanning — see §15 Deprecated Components for the full list and decommission plan.

```
┌──────────────────────────────────────────────────────────────────┐
│                  REMOTE SERVER (18.225.0.90)                      │
│                                                                  │
│  UPLOAD ENTRY POINTS                                             │
│    Mobile App ──────► photoapi/upload.php                         │
│    Scheduler Drag ──► controllers/phototab.php                    │
│    Scheduler Multi ─► controllers/preupload.php                   │
│              │                                                   │
│              ▼                                                   │
│    uploads/staging/{filename}         ◄── ALL UPLOADS LAND HERE  │
│              │                                                   │
│              ├──► generate_thumbnails.py (nohup &, logged)        │
│              │      ├─► uploads/thumbs/{webpfilename}  200x200   │
│              │      └─► uploads/webp/{webpfilename}    1280px    │
│              │                                                   │
│              └──► sync_to_local.py (nohup &)                     │
│                     └─► HTTPS POST to local server               │
│                                                                  │
│  DB INSERT: meter_files (job_id, webpfilename, file_type=99)     │
│                                                                  │
│  SERVING (both platforms read from here)                         │
│    uploads/thumbs/   Grid thumbnails (200x200 Q70)               │
│    uploads/webp/     Standard viewing (1280px Q75)               │
│                                                                  │
│  LISTING (both platforms use the same source)                    │
│    meter_files DB     ◄── SINGLE SOURCE OF TRUTH                 │
│                                                                  │
│  LEGACY (no longer growing, pending cleanup)                     │
│    uploads/*.jpg       ~200GB old CI-resized JPEGs ◄ Frozen      │
│    /mnt/aeiserver/     Old filesystem photos                     │
└──────────────────────┬───────────────────────────────────────────┘
                       │ sync_to_local.py
                       ▼
┌──────────────────────────────────────────────────────────────────┐
│                LOCAL SERVER (upload.aeihawaii.com)                 │
│                                                                  │
│  uploadlocallat_kuldeep.php                                      │
│    → Receives full-size original from staging/                   │
│    → Saves to /mnt/dropbox/{Year} Customers/.../                 │
│    → Background: upload_thumb_generator.py                       │
│       → thumbs/{name}.webp  (200x200 Q70)                       │
│       → thumbs/mid/{name}.webp  (800px Q80)                     │
│    → Tracks in local_photos table                                │
│                                                                  │
│  This is the ARCHIVAL store for full-size originals.             │
└──────────────────────────────────────────────────────────────────┘

┌──────────────────────────────────────────────────────────────────┐
│                    MOBILE APP (Flutter)                            │
│                                                                  │
│  Upload: base64 JSON POST → upload.php → staging/                │
│  Display: uploads/thumbs/ (grid) + uploads/webp/ (detail)        │
└──────────────────────────────────────────────────────────────────┘

┌──────────────────────────────────────────────────────────────────┐
│                    SCHEDULER WEB UI                                │
│                                                                  │
│  Upload: CI file upload → staging/ (direct)                      │
│  Display: meter_files DB → uploads/thumbs/ + uploads/webp/       │
│  Download: ZIP from uploads/webp/ (or uploads/ JPEG fallback)    │
└──────────────────────────────────────────────────────────────────┘
```

---

## 2. What PHOTO-021 Changed

### Fix 1: Scheduler Upload → Staging Direct

| Aspect | Before (PHOTO-017) | After (PHOTO-021) |
|--------|-------------------|------------------|
| Upload path | `uploads/` root | `uploads/staging/` |
| Copy to staging | Manual `@copy()` after upload | Not needed — already there |
| CI resize (800x1027) | Overwrote `uploads/` copy | Removed entirely |
| CI-resized JPEG in uploads/ | Created every upload (~200GB waste growing) | No new files in uploads/ root (PHOTO-021 stopped CI-resize; PHOTO-024 removed archive copy) |
| Source for WebP | staging/ copy (same) | staging/ (same, just arrived directly) |
| Source for sync | staging/ copy (same) | staging/ (same) |

### Fix 2: DB-Only Photo Listing

| Aspect | Before | After |
|--------|--------|-------|
| Scheduler photo grid | meter_files DB + scandir `/mnt/aeiserver/` | meter_files DB only |
| `$s3files` variable | Array from `getimagelistingm.php` (filesystem scan) | Empty array `array()` |
| Duplicate photos | Common (same photo from DB and filesystem) | Eliminated |
| Download All ZIP | meter_files + s3files (HTTP fetch, slow) | meter_files only |
| `getimagelisting()` method | Called on every page load | Not called (dead code, retained) |

### Fix 3: API full_link Corrected (PHOTO-021) / Hi-Res Tier Removed (PHOTO-022)

| Aspect | Before | After |
|--------|--------|-------|
| `getimagelisting.php` full_link | `/webp/` (wrong) | `/webp/` (same as `link` — hi-res tier removed) |
| Hi-res tier (`uploads/hi-res/`) | Generated by `generate_thumbnails.py` | Removed — 2-tier only (thumbs + webp) |
| Consistency with other endpoints | Inconsistent | All endpoints return `link` and `thumb_link` only |

---

## 3. Unified Upload Flow

All three upload entry points now follow an identical pipeline:

```
Source (browser/app) → staging/{filename} → [background] WebP + Sync → DB INSERT
```

### 3.1 Mobile App Upload

**Entry point:** `photoapi/upload.php`
**Trigger:** Flutter app POST with base64-encoded image + metadata

```
Mobile App sends JSON POST:
  { auth_token, job_id, file_name, image_data (base64) }
         │
         ▼
1. Validate auth token (aei@89806849)
         │
         ▼
2. Decode base64 → raw image bytes
         │
         ▼
3. Save to uploads/staging/{uniqid}_{filename}
   Prefix prevents concurrent upload collisions.
         │
         ▼
4. DB lookup: jobs → customer info, job_date, job_type
         │
         ▼
5. Generate webpfilename:
   {LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-hash}.webp
   PhotoType: S (Survey) for PM/WM/AS/RPM/GCPM, I (Installation) for all others
         │
         ▼
6. DB INSERT meter_files (job_id, unique_filename, webpfilename, file_type=99)
         │
         ├──► Background: generate_thumbnails.py  → thumbs/ + webp/
         │
         └──► Background: sync_to_local.py        → local server archival
         │
         ▼
7. Return JSON: {"success":1, "status":"ok"}
   Immediate response (~0.5s). WebP appears 1-3s later. Sync 2-10s later.
```

### 3.2 Scheduler Web Upload — Drag & Drop

**Entry point:** `controllers/phototab.php` → `dropImageUpload($job_id)`
**Trigger:** Drag-and-drop or file picker in Photos tab

```
Browser sends multipart file POST
         │
         ▼
1. CI upload library saves to uploads/staging/{encrypted_name}
   Config: upload_path = FCPATH . 'uploads/staging/'
         │
         ▼
2. (Production only) Dev-env path resolution:
   If /dev/ detected in base_url → copy from dev/staging/ to main/staging/
   Otherwise: filePath = /httpdocs/scheduler/uploads/staging/{encrypted_name}
         │
         ▼
3. $stagingPath = $filePath   (already in staging, no copy needed)
         │
         ▼
4. Generate webpfilename:
   {LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-hash}.webp
   Same naming convention as mobile.
         │
         ├──► Background: sync_to_local.py ($stagingPath → local server)
         │
         └──► Background: generate_thumbnails.py ($stagingPath → thumbs/webp/)
         │
         ▼
5. filesize($stagingPath) → file_size for DB record
         │
         ▼
6. DB INSERT meter_files (job_id, unique_filename, webpfilename, file_type=99)
```

### 3.3 Scheduler Web Upload — Multi-File Picker

**Entry point:** `controllers/preupload.php` → `submit($job_id)`
**Trigger:** Multi-file upload form

Identical pipeline to §3.2, executed in a loop for each file:

```
FOR EACH file in $_FILES['files']:
  1. CI upload to uploads/staging/{encrypted_name}
  2. Dev-env path resolution
  3. $stagingPath = $filePath
  4. Generate webpfilename
  5. Background: sync_to_local.py + generate_thumbnails.py
  6. filesize($stagingPath) → file_size
  7. DB INSERT meter_files
```

### 3.4 Flow Comparison Table

Every upload path is now consistent:

| Step | Mobile (upload.php) | Scheduler Drag (phototab.php) | Scheduler Multi (preupload.php) |
|------|--------------------|-----------------------------|-------------------------------|
| **Input format** | base64 JSON | multipart form | multipart form (loop) |
| **Landing directory** | `uploads/staging/` | `uploads/staging/` | `uploads/staging/` |
| **Filename prefix** | `uniqid()_` | CI `encrypt_name` | CI `encrypt_name` |
| **New files in uploads/ root** | Never | Never | Never |
| **CI resize (800x1027)** | N/A | Removed | Removed |
| **WebP generation** | `generate_thumbnails.py` (nohup &) | `generate_thumbnails.py` (nohup &) | `generate_thumbnails.py` (nohup &) |
| **2 tiers generated** | thumbs + webp | thumbs + webp | thumbs + webp |
| **Local sync** | `sync_to_local.py` (nohup &) | `sync_to_local.py` (nohup &) | `sync_to_local.py` (nohup &) |
| **DB table** | `meter_files` | `meter_files` | `meter_files` |
| **file_type** | 99 | 99 | 99 |
| **webpfilename** | Set | Set | Set |
| **Response** | JSON `{"success":1}` | HTML (view reload) | 302 redirect to gallery |

---

## 4. WebP Derivative Generation

**Script:** `generate_thumbnails.py` v7.0
**Runtime:** Python 3.6 + Pillow (LANCZOS resampling)
**Execution:** Background via `nohup /usr/local/bin/python3.6 ... >> logs/webp_generation.log 2>&1 &`
**Called by:** All three upload entry points identically

### Arguments

```bash
python3.6 generate_thumbnails.py <source_image> <webpfilename> <uploads_dir>
# Example:
python3.6 generate_thumbnails.py \
  /path/uploads/staging/abc123.jpg \
  SmithJohnAC-I02-18-2026_IMG0ec2c8b5682e.webp \
  /path/uploads
```

### Output (2-Tier)

| Tier | Directory | Dimensions | Quality | When Generated | Used By |
|------|-----------|-----------|---------|----------------|---------|
| **Thumbnail** | `uploads/thumbs/` | 200x200 crop | Q70 | Immediate | Grid views — scheduler + mobile |
| **Standard** | `uploads/webp/` | 1280px max dim | Q75 | Immediate | Lightbox, mobile detail view, ZIP download |

Note: Hi-res tier (`uploads/hi-res/`, 2048px Q82) was removed in PHOTO-022. Archive copy to `uploads/` root was removed in PHOTO-024 — full-size archival is handled exclusively by `sync_to_local.py` to the local server.

### Processing Details

- Both tiers generated immediately on upload
- EXIF rotation applied before processing
- RGBA/P/LA modes converted to RGB (white background)
- Atomic saves (temp file + rename) prevent serving partial images
- Idempotent at the file output layer — skips existing thumb and WebP targets. (Note: upload requests and DB inserts are **not** deduplicated by this script; see §13 Duplicate-Upload Policy.)
- 120-second timeout per image

### Error Logging (PHOTO-023)

All three PHP upload entry points redirect `generate_thumbnails.py` output to `photoapi/logs/webp_generation.log` (previously discarded to `/dev/null`). The script writes timestamped entries:

```
[2026-02-23 13:42:15] START source=/path/staging/abc.jpg webp=SmithJohnAC-I...webp
[2026-02-23 13:42:16] OK created=thumb,large,archive skipped=none elapsed=1.2s
```

Similarly, `sync_to_local.py` output is redirected to `photoapi/logs/sync_exec.log`.

### Cron Fallback: fix_missing_webp.py (PHOTO-023)

**Script:** `fix_missing_webp.py` v1.0
**Cron:** `*/30 * * * * /usr/local/bin/python3.6 .../photoapi/fix_missing_webp.py`
**Log:** `photoapi/logs/fix_missing_webp.log`

Catches silent failures from the nohup exec() pattern:
1. Queries `meter_files WHERE file_type=99 AND webpfilename IS NOT NULL AND created > NOW() - 7 DAY`
2. Checks if `thumbs/{webpfilename}` and `webp/{webpfilename}` exist on disk
3. If missing and `staging/{unique_filename}` exists → calls `generate_thumbnails.generate()` (generates thumb + WebP only, no archive copy since PHOTO-024)
4. File lock prevents concurrent runs
5. Logs all actions to `fix_missing_webp.log`

### Platform Parity

Both mobile and scheduler trigger the exact same script with the same arguments. Output is identical regardless of upload source. A photo uploaded from the mobile app and one uploaded from the scheduler web UI produce the same two WebP tiers.

---

## 5. Local Sync & Archival

### Remote → Local (sync_to_local.py)

**Called by:** All three upload entry points (nohup &)
**Source:** `staging/{filename}` (full-size original)
**Target:** `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php`
**Method:** Multipart POST (file + metadata)

```
Arguments passed (identical from all upload sources):
  staging_path, filename, job_id, customer_id,
  job_type, last_name, first_name, job_date, photo_type
```

### Retry Policy

```
Attempt 1: immediate
Attempt 2: wait 2s
Attempt 3: wait 4s
If all fail: queue/{timestamp}_{random}.json
  → Cron (process_retry_queue.py) retries every 15 min
  → Max 10 retries (~2.5 hours)
  → Logged to photoapi/logs/sync_to_local.log
```

### Local Receipt (uploadlocallat_kuldeep.php)

1. Validate auth token and MIME type
2. Look up customer folder path:
   `/mnt/dropbox/{Year} Customers/{Letter}/{LastName}, {FirstName}/{Survey|Installation}/{Job}/`
3. Move file to customer folder (full-size original preserved permanently)
4. Background: `upload_thumb_generator.py`
   - `thumbs/{name}.webp` (200x200 Q70)
   - `thumbs/mid/{name}.webp` (800px Q80)
5. Track in `local_photos` table

### Platform Parity

Sync is triggered from all three upload entry points. There is no upload path that skips the sync. A photo uploaded from the scheduler web UI is synced to local storage with the same timing and retry guarantees as one uploaded from the mobile app.

### Staging Cleanup

`process_retry_queue.py` (cron) cleans staging files older than 7 days. By then, the file has been synced to local and WebP derivatives generated.

---

## 6. Photo Display

### 6.1 Scheduler Web UI

**Controller:** `phototab.php` → `photo_index($job_id)`
**View:** `views/photo_tab/photos_index.php`

**Primary path (DB-driven, meter_files):**

```
1. Query meter_files for unique dates:
   SELECT DISTINCT DATE_FORMAT(created,'%Y-%m-%d') FROM meter_files
   WHERE job_id = ? AND file_type = 99 ORDER BY created DESC

2. For each date, query full records:
   SELECT * FROM meter_files WHERE job_id = ? AND file_type = 99
   AND DATE_FORMAT(created,'%Y-%m-%d') = ? ORDER BY created DESC

3. Pass to view as $meter_files (keyed by date)

4. $s3files = array()  ←── PHOTO-021: eliminated scandir-based listing
```

**View rendering logic (per photo):**

```php
// Thumbnail (grid)
IF webpfilename AND uploads/thumbs/{webpfilename} exists:     ← is_file() check
    src = base_url() . 'uploads/thumbs/' . rawurlencode(webpfilename)
ELSE:
    src = thumbnail() helper (GD on-the-fly, checks uploads/ only — no /mnt/aeiserver/)

// Lightbox (full view)
IF webpfilename AND uploads/webp/{webpfilename} exists:       ← is_file() check
    href = base_url() . 'uploads/webp/' . rawurlencode(webpfilename)
ELSE:
    href = thumbnail() helper (GD fallback, checks uploads/ only)
```

**Note:** The view performs `is_file()` existence checks against `uploads/thumbs/` and `uploads/webp/`. These are targeted file lookups (not directory scans). The `thumbnail()` helper also performs `file_exists()` lookups in `uploads/` for cache hits — but does **not** scan `/mnt/aeiserver/`.

**Legacy "show more" path (filesystem scan — should be decommissioned):**

The `showmore()` method (line 348) calls `getimagelisting()` which POSTs to `getimagelistingm.php` — a **pure /mnt/aeiserver/ filesystem scanner**. This endpoint scans `scandir('/mnt/aeiserver/{job_id}/')` and returns `fullpath` references to `/mnt/aeiserver/`. The caller then does `file_get_contents()` HTTP fetch of each image and `filemtime($fullpath)` for sorting. See §15 Deprecated Components for decommission plan.

**Photos are grouped by date** with date headers and Select/Deselect buttons per group.

### 6.2 Mobile App

**API chain:**
```
loginapinew.php
  → customerphotostabstatenew($customer_id)
    → getimageforapinew($job_id)
      → POST to getimagelistingnew.php (single most recent photo — DB-only)
    → POST to getimagelisting.php (all photos for customer's jobs — DB + /mnt/aeiserver/ fallback)
    → POST to getimagelisting1.php (with survey photo support)
```

**Note on getimagelisting.php:** This endpoint is primarily DB-driven (meter_files query). However, it has a legacy fallback (lines 112-134) that `scandir('/mnt/aeiserver/{job_id}/')` for photos not in meter_files. This fallback should be removed — see §15.

**URLs returned to app (all endpoints consistent post-PHOTO-022):**
```json
{
  "link":       "https://aeihawaii.com/scheduler/uploads/webp/{webpfilename}",
  "thumb_link": "https://aeihawaii.com/scheduler/uploads/thumbs/{webpfilename}",
  "full_link":  "https://aeihawaii.com/scheduler/uploads/webp/{webpfilename}"
}
```

Note: `full_link` now points to `/webp/` (same as `link`) since the hi-res tier was removed in PHOTO-022. Previously pointed to `/hi-res/`.

**What the mobile app uses:**
- Grid view: `thumb_link` → `uploads/thumbs/` (200x200 Q70 WebP)
- Detail view: `link` → `uploads/webp/` (1280px Q75 WebP)
- Full size: `full_link` → `uploads/webp/` (same as `link` — best available remote copy)

### Platform Parity — Display

| Aspect | Scheduler Web | Mobile App |
|--------|--------------|------------|
| **Data source** | `meter_files` DB (direct SQL) | `meter_files` DB (via API) |
| **Filesystem scanning** | None | None |
| **Grid thumbnail** | `uploads/thumbs/{webpfilename}` | `uploads/thumbs/{webpfilename}` |
| **Full view** | `uploads/webp/{webpfilename}` | `uploads/webp/{webpfilename}` |
| **Hi-res tier** | Removed (PHOTO-022) | Removed (PHOTO-022) |
| **Fallback (old photos)** | GD `thumbnail()` helper | Not applicable (old photos have webpfilename from backfill) |
| **Grouping** | By date (date headers) | By job/customer |

---

## 7. Download Flow

**Controller:** `phototab.php` → `download_all_photos($job_id)`
**Trigger:** "Download All" button on scheduler Photos tab

```
1. Query meter_files for job_id (grouped by date, file_type=99)

2. Create ZIP in memory (PHOTO-024: WebP only, no JPEG fallback):
   FOR EACH meter_file:
     IF webpfilename is set:
       Add uploads/webp/{webpfilename} to ZIP        (1280px Q75 WebP)
     ELSE:
       Skip (photos without webpfilename are not included)

3. Stream ZIP to browser:
   filename = "Job_{id}_{CustomerName}_Photos_{date}.zip"
```

**Post-PHOTO-021 changes:**
- No more `$s3files` in the ZIP (was fetching via HTTP, slow and duplicated DB photos)
- ZIP contains only meter_files-sourced photos
- WebP only — JPEG fallback removed (PHOTO-024). Photos without `webpfilename` are skipped (none exist as of 2026-02-23).

Downloads pull from `uploads/webp/` (1280px Q75). The hi-res tier (`uploads/hi-res/`) was removed in PHOTO-022. The `uploads/` root JPEG fallback was removed in PHOTO-024.

---

## 8. Photo Deletion Flow

Photo deletion cascades across three servers: the scheduler app deletes the DB record, then notifies the production server to delete physical files, which then notifies the local server to delete Dropbox-synced copies.

```
Scheduler Web UI (Photos tab)
         │
         ▼
1. User clicks "Delete" → Confirmation modal (Confirm_delete_pic.php)
   User selects photos → clicks "Yes, Do it"
         │
         ▼
2. AJAX POST → phototab.php::delete_image_ajax($image_id, $job_id, $img_date)
   a. Query meter_files for unique_filename + webpfilename (BEFORE delete)
   b. DELETE FROM meter_files WHERE id = ? AND job_id = ?
   c. Check which dates now have zero remaining photos (for UI update)
         │
         ▼
3. cURL POST → https://aeihawaii.com/photoapi/delete_image.php
   Payload: { auth_token, job_id, files: [{unique_filename, webpfilename, ...}] }
         │
         ▼
4. delete_image.php deletes physical files on PRODUCTION (18.225.0.90):
   a. uploads/staging/{unique_filename}   — staging original (if still present)
   b. uploads/thumbs/{webpfilename}       — 200x200 WebP thumbnail
   c. uploads/webp/{webpfilename}         — 1280px WebP standard
   d. /mnt/aeiserver/{job_id}/{unique}    — legacy storage path
         │
         ▼
5. delete_image.php → GET https://upload.aeihawaii.com/delete_local_photo.php
   Params: auth_token, job_id, unique_filename
         │
         ▼
6. delete_local_photo.php on LOCAL server (upload.aeihawaii.com):
   a. Look up local_photos WHERE remote_job_id = ? AND filename = ?
   b. Resolve folder_path (must be under /mnt/dropbox/ — traversal protection)
   c. Delete: {folder_path}/{filename}              — synced full-size photo
              {folder_path}/thumbs/{name}.webp      — local thumbnail
              {folder_path}/thumbs/mid/{name}.webp  — local mid-size thumb
   d. DELETE FROM local_photos WHERE id = ?
   e. Remove empty job subfolder + thumbs dirs (cleanup)
```

### Files Involved

| File | Location | Role |
|------|----------|------|
| `phototab.php` → `delete_image_ajax()` | Scheduler controller | DB delete + trigger cascade |
| `Confirm_delete_pic.php` | Scheduler view | Confirmation modal UI |
| `photos_index.php` | Scheduler view | Delete button + JS handler |
| `photoapi/delete_image.php` | Production server | Delete all remote file tiers |
| `photoapi/delete_local_photo.php` | Local server | Delete Dropbox-synced copies |

### Auth Tokens

| Endpoint | Token |
|----------|-------|
| `delete_image.php` | `aei@89806849` |
| `delete_local_photo.php` | `aei_local_delete_89806849` |

### Safety Features

- **DB rows deleted first** — prevents re-display even if file deletion fails
- **`basename()` sanitization** — all filenames passed through `basename()` to prevent path traversal
- **Dropbox path validation** — `delete_local_photo.php` verifies `realpath()` starts with `/mnt/dropbox/`
- **Empty folder cleanup** — local server removes empty job subfolders after last photo deleted
- **Idempotent** — `is_file()` checks before every `unlink()`, returns success even if file already gone
- **Timeout protection** — cURL timeout 15s (scheduler→production), stream context timeout 5s (production→local)

---

## 9. API Endpoints Reference

### Listing Endpoints (All DB-Driven)

| Endpoint | Purpose | Called By | `link` | `thumb_link` | `full_link` |
|----------|---------|-----------|--------|--------------|-------------|
| `getimagelisting.php` | All photos for customer's jobs | Mobile (customer tab) | `webp/` | `thumbs/` | `webp/` |
| `getimagelistingnew.php` | Single most recent photo | Mobile (job thumbnail) | `webp/` | `thumbs/` | `webp/` |
| `getimagelisting1.php` | All photos + survey proxy | Mobile (customer tab v2) | `webp/` | `thumbs/` | `webp/` (meter_files) or `serve_survey_photo.php` proxy (survey photos) |
| `getimagelistingm.php` | **Legacy** — filesystem scan | Scheduler `showmore()` | fetch_image.php URLs | — | — |

All listing endpoints return `link`, `thumb_link`, and `full_link`. After PHOTO-022, `full_link` points to `/webp/` (same as `link`) for meter_files photos since the hi-res tier was removed. **Exception:** `getimagelisting1.php` sets `full_link` to a `serve_survey_photo.php` proxy URL for survey photos sourced from the local server (these are not pre-converted to WebP on the remote server).

**Data sources:** `getimagelistingnew.php` and `getimagelisting1.php` are DB-only. `getimagelisting.php` is primarily DB but has a `/mnt/aeiserver/` `scandir()` fallback (§15). `getimagelistingm.php` is a **pure** `/mnt/aeiserver/` scanner that should be decommissioned (§15).

### Upload Endpoints

| Endpoint | Method | Called By | Destination |
|----------|--------|-----------|-------------|
| `photoapi/upload.php` | POST (base64 JSON) | Mobile app | `uploads/staging/` |
| `phototab/dropImageUpload` | POST (multipart) | Scheduler drag-drop | `uploads/staging/` |
| `preupload/submit` | POST (multipart, multi-file) | Scheduler file picker | `uploads/staging/` |

### Delete Endpoints

| Endpoint | Method | Called By | Action |
|----------|--------|-----------|--------|
| `phototab/delete_image_ajax` | AJAX POST | Scheduler Photos tab | DB delete + cascade to production |
| `photoapi/delete_image.php` | POST (JSON) | phototab.php (cURL) | Delete all remote file tiers + notify local |
| `delete_local_photo.php` | GET | delete_image.php | Delete Dropbox-synced copies + local_photos row |

### Serving Endpoints

| Method | URL Pattern | Notes |
|--------|-------------|-------|
| **Direct Apache static** | `uploads/webp/{file}`, `uploads/thumbs/{file}` | Fastest — no PHP |
| `fetch_image.php` | DB lookup → stream file | Fallback: webp/ → uploads/ → ~~/mnt/aeiserver/~~ (**decommission**, see §15) |
| `fetch_image1.php` | DB → 302 redirect to static | Supports `?size=thumb`. No /mnt/aeiserver/ reference. |

---

## 10. Storage Layout

### Remote Server (Production)

```
/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/
│
├── staging/                            TEMPORARY — 7-day cleanup
│   ├── {uniqid}_{filename}.jpg         Mobile uploads (full-size original)
│   └── {encrypted_name}.jpg            Scheduler uploads (full-size original)
│
├── thumbs/                             ACTIVE — ~196K files
│   └── {webpfilename}.webp             200x200, Q70 (grid thumbnails)
│
├── webp/                               ACTIVE — ~196K files
│   └── {webpfilename}.webp             1280px max dim, Q75 (standard viewing)
│
└── *.jpg                               LEGACY — ~200K files, ~200GB
    └── {encrypted_name}.jpg            Old CI-resized 800x1027 JPEGs + PHOTO-023 archive copies
                                        FROZEN: no new files (PHOTO-021 stopped CI resize, PHOTO-024 removed archive copy)
                                        Backed up to /mnt/aeiserver_backup/.../photo_backup/ (verified 2026-02-23)
                                        PENDING CLEANUP: safe to delete once confirmed all photos have WebP
```

### Local Server (Archival)

```
/mnt/dropbox/{Year} Customers/
└── {Letter}/
    └── {LastName}, {FirstName}/
        ├── Survey/
        │   └── {CustomerName}, {JobType}-S, {Date}/
        │       ├── {filename}               Full-size original (permanent)
        │       └── thumbs/
        │           ├── {filename}.webp      200x200 Q70
        │           └── mid/
        │               └── {filename}.webp  800px Q80
        └── Installation/
            └── (same structure)
```

---

## 11. Database Schema

### meter_files (Remote DB: mandhdesign_schedular [legacy spelling])

| Column | Type | Description |
|--------|------|-------------|
| `id` | INT PK | Auto-increment |
| `job_id` | INT | `jobs.job_pid` (NOT `jobs.id` — use caution) |
| `unique_filename` | VARCHAR | CI encrypted name or uniqid-prefixed name |
| `original_filename` | VARCHAR | User's original filename |
| `file_size` | INT | Bytes at upload time |
| `file_type` | INT | 99 = photo |
| `webpfilename` | VARCHAR | WebP derivative name (e.g., `SmithJohnAC-I02-18-2026_IMGabc123.webp`) |
| `folder_path` | VARCHAR | Deprecated — empty for all new uploads |
| `created` | DATETIME | Upload timestamp |

**Serving logic:**
- If `webpfilename` is set → serve from `uploads/webp/` and `uploads/thumbs/`
- If `webpfilename` is NULL (pre-WebP photos) → fall back to `uploads/{unique_filename}` via GD helper

### local_photos (Local DB)

Tracks files received by `uploadlocallat_kuldeep.php`. Used for reconciliation and local gallery features.

### WebP Filename Convention

```
{LastName}{FirstName}{JobType}-{PhotoType}{Date}_IMG{12-char-random-hash}.webp

Components:
  LastName + FirstName  → customer name (no spaces)
  JobType               → job_types.intials (PV, AC, SWH, SC, etc.)
  PhotoType             → S (Survey) or I (Installation)
  Date                  → MM-DD-YYYY format
  12-char-hash          → substr(md5(uniqid(mt_rand(), true)), 0, 12)

Examples:
  SmithJohnAC-I02-18-2026_IMG0ec2c8b5682e.webp
  YagiLelandSC-I02-25-2026_IMGb66a371aca3f.webp
```

---

## 12. Consistency Verification

### Upload Consistency

| Check | Mobile | Scheduler Drag | Scheduler Multi | Status |
|-------|--------|---------------|-----------------|--------|
| Lands in staging/ | Yes | Yes | Yes | Consistent |
| No new files in uploads/ root | Yes | Yes | Yes | Consistent |
| No CI resize | N/A | Removed | Removed | Consistent |
| WebP generation triggered | Yes | Yes | Yes | Consistent |
| Both tiers generated (thumbs + webp) | Yes | Yes | Yes | Consistent |
| Local sync triggered | Yes | Yes | Yes | Consistent |
| meter_files INSERT | Yes | Yes | Yes | Consistent |
| webpfilename set | Yes | Yes | Yes | Consistent |
| file_type = 99 | Yes | Yes | Yes | Consistent |

### Display Consistency

| Check | Scheduler Web | Mobile App | Status |
|-------|--------------|------------|--------|
| Primary source: meter_files DB | Yes | Yes | Consistent |
| /mnt/aeiserver/ scanning in primary path | No | No | Consistent |
| /mnt/aeiserver/ scanning in fallback path | Yes (`showmore()→getimagelistingm.php`) | Yes (`getimagelisting.php` lines 112-134) | **Legacy — decommission** |
| Grid: uploads/thumbs/ | Yes | Yes | Consistent |
| View: uploads/webp/ | Yes | Yes | Consistent |
| Full size: uploads/webp/ | Yes (lightbox) | Yes (full_link) | Consistent (PHOTO-022: was hi-res/) |

### Sync Consistency

| Check | Mobile uploads | Scheduler uploads | Status |
|-------|---------------|-------------------|--------|
| sync_to_local.py called | Yes | Yes | Consistent |
| Source: staging/ file | Yes | Yes | Consistent |
| Retry policy (3 inline + queue) | Yes | Yes | Consistent |
| Local archival path | Yes | Yes | Consistent |
| Local thumbnails generated | Yes | Yes | Consistent |

---

## 13. Failure Handling & Edge Cases

### Failure-State Matrix

What happens when individual pipeline steps fail:

| Failure Scenario | DB Row | WebP Derivatives | Local Sync | User Impact | Recovery |
|-----------------|--------|-----------------|------------|-------------|----------|
| **Normal success** | Inserted | Generated | Synced | Photo visible in ~1-3s | — |
| **generate_thumbnails.py crashes** | Exists | Missing | Unaffected | Photo in DB but invisible in UI (no thumb/WebP) | `fix_missing_webp.py` cron (every 30 min) regenerates from staging/ |
| **sync_to_local.py fails** | Exists | Generated | Failed | Photo visible; no Dropbox archival | Retry queue (3 inline + cron every 15 min, max 10 retries) |
| **DB INSERT fails** | Missing | Not triggered | Not triggered | Photo lost (staging file remains as orphan) | Manual: re-upload required. Staging file cleaned after 7 days. |
| **Staging disk full** | Not attempted | — | — | Upload returns error | Alert on disk usage; expand volume |
| **WebP + sync both fail** | Exists | Missing | Failed | Photo in DB but invisible; no archival | Cron fallback for WebP; sync retry queue for local |
| **Cron fallback finds no staging source** | Exists | Missing | N/A | Photo permanently invisible | Manual: check staging/ cleanup timing; re-upload if staging already cleaned |

### Duplicate-Upload Policy

There is **no content-hash deduplication**. If a user uploads the same photo file twice:

- Two separate `meter_files` rows are created (different `id`, different `unique_filename`)
- Two separate sets of WebP derivatives are generated (different `webpfilename` due to random hash)
- Two separate archive copies are created in `uploads/` root
- Two separate copies are synced to the local server
- Both appear in the photo grid as distinct photos
- The user must manually delete the duplicate

This is by design — the system treats each upload as a distinct event. Deduplication would require content hashing at upload time, which adds complexity and latency for minimal benefit (duplicate uploads are infrequent).

---

## 14. Operational Notes

### Monitoring & Alerting Recommendations

| Log File | Location | What to Watch |
|----------|----------|---------------|
| `webp_generation.log` | `photoapi/logs/` | `FAIL` or `ERROR` entries; absence of entries (indicates exec() not firing) |
| `sync_exec.log` | `photoapi/logs/` | Sync errors; timeout patterns |
| `fix_missing_webp.log` | `photoapi/logs/` | `REGEN` entries (indicates primary generation failed); `FATAL` errors |
| `sync_to_local.log` | `photoapi/logs/` | Queue depth growth; repeated failures for same file |

**Suggested health checks:**
- `fix_missing_webp.log` should show `CHECK OK` entries every 30 minutes (proves cron is running)
- `webp_generation.log` should have entries for each upload (absence = exec() not firing)
- Alert if `REGEN` count in `fix_missing_webp.log` exceeds 5 in a single run (systemic failure)

### Log Rotation

The PHOTO-023 log files (`webp_generation.log`, `sync_exec.log`, `fix_missing_webp.log`) are append-only with no built-in rotation. At current upload volume (~50-100 photos/day), these grow at roughly 5-10 KB/day. Consider adding logrotate rules if the system runs for extended periods:

```
# /etc/logrotate.d/photoapi (example)
/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/*.log {
    weekly
    rotate 12
    compress
    missingok
    notifempty
}
```

### Retention Policy

| Storage Tier | Retention | Cleanup Mechanism |
|-------------|-----------|-------------------|
| `uploads/staging/` | 7 days | `process_retry_queue.py` cron (automatic) |
| `uploads/thumbs/` | Permanent | Deleted only via photo deletion cascade |
| `uploads/webp/` | Permanent | Deleted only via photo deletion cascade |
| `/mnt/dropbox/` (local) | Permanent | Deleted only via photo deletion cascade |
| `uploads/*.jpg` (legacy) | Pending cleanup | Backed up locally; safe to delete (§15 Priority 3) |
| `photoapi/logs/` | Unbounded (no rotation) | Manual or logrotate (see above) |

---

## 15. Deprecated Components

These components are retained for backward compatibility. Some are still actively called but should be decommissioned.

### /mnt/aeiserver/ Filesystem Scanning — Active Code (Should Be Decommissioned)

`/mnt/aeiserver/` is a legacy storage path that should NOT be used as a fallback. The following code paths still reference it and need removal:

| Component | Location | What It Does | Called By | Action |
|-----------|----------|-------------|-----------|--------|
| `getimagelistingm.php` | photoapi/ | **Pure `/mnt/aeiserver/` scanner** — `scandir()` to list images, returns `fullpath` to `/mnt/aeiserver/` | `phototab.php::getimagelisting()` line 917 | **Remove** — replace caller with DB-only endpoint |
| `getimagelisting()` method | phototab.php lines 913-1022 | Calls `getimagelistingm.php`, fetches images via HTTP, creates local thumbnails, uses `filemtime($fullpath)` | `showmore()` method line 348 | **Remove** — `showmore()` should use meter_files directly (it already does for the primary view) |
| `getimagelisting.php` fallback | photoapi/ lines 112-134 | After DB query, `scandir('/mnt/aeiserver/{jobId}/')` for photos not in meter_files | Mobile app (via `loginapinew.php`) | **Remove lines 112-134** — DB is authoritative, `/mnt/aeiserver/` should not supplement it |
| `fetch_image.php` fallback | photoapi/ lines 44-50 | If DB lookup fails, tries `'/mnt/aeiserver/' . $jobId . '/' . $fileName` | Mobile app (image serving), `getimagelisting.php` | **Remove** — DB + `uploads/` is authoritative |
| `delete_image.php` cleanup | photoapi/ lines 68-74 | `unlink('/mnt/aeiserver/{jobId}/{unique}')` | Deletion cascade | **Keep for now** — harmless cleanup, no-op if file doesn't exist |

### Dead Code (Retained, Not Called)

| Component | Location | Why Kept |
|-----------|----------|----------|
| `getimagelistings3()` method | phototab.php lines ~820-910 | Already unused before PHOTO-021. Safe to remove. |
| `getimagelistingcnt.php` | photoapi/ | Legacy count endpoint, `/mnt/aeiserver/` scanner. No known callers. |
| S3 deletion handler | phototab.php lines ~773-794 | Handles S3 file deletion via hidden inputs that no longer exist in the view. |

### Legacy Fallbacks (Still Active, Not /mnt/aeiserver/)

| Component | Purpose | When Used |
|-----------|---------|-----------|
| `uploads/{unique_filename}` JPEG | Fallback serving | Photos where `webpfilename IS NULL` (pre-WebP era) |
| `thumbnail()` GD helper | On-the-fly WebP generation | View fallback when pre-generated WebP thumb/standard not found. Checks `uploads/` only — does **not** scan `/mnt/aeiserver/`. |
| `fetch_image1.php` | 302 redirect to static WebP | Size routing (`?size=thumb`). Checks `uploads/thumbs/` and `uploads/webp/` only — no `/mnt/aeiserver/`. |

### Decommission Plan

**Priority 1 — Remove /mnt/aeiserver/ from active code paths:**
1. Remove `getimagelisting.php` lines 112-134 (legacy `/mnt/aeiserver/` scan fallback)
2. Remove `fetch_image.php` lines 44-50 (legacy `/mnt/aeiserver/` file fallback)
3. Rewrite `phototab.php::showmore()` to stop calling `getimagelisting()` → use meter_files DB directly
4. Archive `getimagelistingm.php` — no legitimate callers once step 3 is done

**Priority 2 — Dead code cleanup:**
5. Remove `getimagelistings3()` from phototab.php
6. Remove `getimagelisting()` from phototab.php (after step 3)
7. Archive `getimagelistingcnt.php`
8. Remove S3 deletion handler from phototab.php

**Priority 3 — Storage cleanup (PHOTO-024 enabled):**
9. Delete all `uploads/*.jpg` (~202K files, ~200GB) — all backed up to `/mnt/aeiserver_backup/.../photo_backup/` (verified 2026-02-23, 0 files missing). All photos have `webpfilename` set (verified: 0 rows with NULL `webpfilename`). No code paths reference `uploads/` root for new photos.
10. Prune `uploads/hi-res/` (PHOTO-022: no longer generated, `full_link` points to `/webp/`)

---

## Appendix A: Enhancement History

| ID | Date | Key Change |
|-----|------|------------|
| PHOTO-002 | 2026-02-04 | Async WebP generation (nohup &), mobile response 2-10s → 0.5s |
| PHOTO-009 | 2026-02-17 | Unified storage: DB-driven reads replace scandir for mobile endpoints |
| PHOTO-011 | 2026-02-09 | Reconciliation, sync source fixes, staging cleanup cron |
| PHOTO-017 | 2026-02-19 | Staging flow, WebP generation (thumbs/webp), sync_to_local for scheduler |
| PHOTO-019 | 2026-02-19 | Local photo tracking (local_photos table, unified_customers) |
| PHOTO-020 | 2026-02-20 | Gallery UX: 4-col grid, lightbox improvements |
| **PHOTO-021** | **2026-02-21** | **Staging direct upload, DB-only listing, API full_link fix** |
| **PHOTO-022** | **2026-02-22** | **Remove hi-res tier (uploads/hi-res/, full_link) — 2-tier only (thumbs + webp)** |
| **PHOTO-023** | **2026-02-23** | **Error logging (exec→log), cron fallback (fix_missing_webp.py)** |
| **PHOTO-024** | **2026-02-23** | **Remove uploads/ root dependency: archive copy removed, ZIP uses WebP only, ~200GB cleanup enabled** |

---

## Appendix B: Credentials

| Component | Host | User | Password |
|-----------|------|------|----------|
| Remote DB | localhost (on 18.225.0.90) | schedular | M1gif9!6 |
| API auth token | — | — | aei@89806849 |
| Local sync auth | — | — | remote_token |
| SSH | 18.225.0.90 | Julian | PPK key |

---

## Appendix C: Verification Test Results (2026-02-21)

### Local Dev Testing

| Test | Result |
|------|--------|
| Photo tab loads (HTTP 200) | Pass |
| No s3files errors in response | Pass |
| No s3files rendering block in HTML | Pass |
| Upload via dropImageUpload → file in staging/ | Pass |
| No file created in uploads/ root | Pass |
| meter_files INSERT with webpfilename | Pass |
| Download All endpoint (no s3files crash) | Pass |
| getimagelisting.php full_link → /webp/ (PHOTO-022) | Pass |
| PHP lint (preupload.php) | Pass — no syntax errors |
| PHP lint (getimagelisting.php) | Pass — no syntax errors |
| PHP lint (phototab.php, photos_index.php) | Pre-existing parse issues (CI partial files, not from PHOTO-021) |

### Notes on Local Dev Limitations

- `filesize()` fails on production paths (`/var/www/vhosts/...`) that don't exist locally — pre-existing issue, not related to PHOTO-021
- Background Python scripts (`generate_thumbnails.py`, `sync_to_local.py`) don't execute locally (production paths) — derivatives and sync must be tested on production
- AWS phar stub required at `/var/www/vhosts/aeihawaii.com/httpdocs/scaws/aws.phar` for phototab.php to load locally
