# AEI Photo System Integration Documentation

**Last Updated:** 2026-02-04
**Status:** PARTIALLY OUTDATED — see note below

> **IMPORTANT:** This document predates PHOTO-009, PHOTO-017, and PHOTO-021. Key changes:
>
> - **No /mnt/dropbox/ hierarchy writes** on remote — uploads go to `uploads/staging/` only
> - **3-tier WebP** — `generate_thumbnails.py` v4.0 (thumbs/webp/hi-res) replaces `generate_webp.py`
> - **DB-only listing** — `meter_files` is the single source of truth for all platforms
> - **No scheduler copy to uploads/ root** — JPEG in uploads/ root is frozen
>
> **Canonical reference:** `PHOTO_SYSTEM_PROCESS_MAP.md` (same directory) is the authoritative post-PHOTO-021 document.
> The architecture diagrams, storage sections, and scenario descriptions below are outdated.
> Server addresses, credentials, local server components, and customer mapping sections remain accurate.

Complete documentation of how the remote server Photo API integrates with local server components.

---

## System Overview

The AEI Photo System is a distributed architecture with components on both the remote AWS server (aeihawaii.com) and the local server (upload.aeihawaii.com). Photos flow from mobile apps through the remote server and sync to local storage.

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                           AEI PHOTO SYSTEM ARCHITECTURE                      │
└─────────────────────────────────────────────────────────────────────────────┘

  ┌───────────────────┐
  │   Mobile App      │
  │  (Field Workers)  │
  └─────────┬─────────┘
            │ POST (base64 image + job_id + auth_token)
            ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │                    REMOTE SERVER (18.225.0.90)                           │
  │                         aeihawaii.com                                    │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  /photoapi/upload.php                                            │    │
  │  │  - Validates auth token (see CREDENTIALS.md)                     │    │
  │  │  - Decodes base64 image                                          │    │
  │  │  - Creates folder: REMOTE /mnt/dropbox/{YEAR} Customers/{customer}/ │    │
  │  │  - Saves original, triggers background WebP (generate_webp.py)  │    │
  │  │  - Inserts record into meter_files table (file_type=99)          │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  │                              │                                           │
  │                              │ curl POST to local server                 │
  │                              ▼                                           │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  AWS S3 (aeiimages bucket)     │    Controllers:                 │    │
  │  │  - Legacy cloud backup          │    - phototab.php (S3 + UI)     │    │
  │  │  - Region: eu-north-1          │    - preupload.php (upload form) │    │
  │  │  - Organized by job_id prefix  │    - loginapi.php (mobile API)  │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  └─────────────────────────────────────────────────────────────────────────┘
                              │
                              │ curl POST with file + metadata
                              ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │           LOCAL SERVER (upload.aeihawaii.com / 72.235.242.139)           │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  /var/www/html/upload/uploadlocallat_kuldeep.php                 │    │
  │  │  - Validates auth token (see CREDENTIALS.md)                     │    │
  │  │  - Validates image file type (JPEG, PNG, GIF, HEIC)              │    │
  │  │  - Sanitizes filename and path components                        │    │
  │  │  - Queries unified_customers DB for folder path ✅ (2026-01-30)  │    │
  │  │  - Saves to mapped folder or LOCAL /mnt/dropbox/Uploads/ fallback │    │
  │  │  - Triggers background thumbnail generation (Python)             │    │
  │  │  - Returns JSON response (no WebP responsibility)                │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  │                              │                                           │
  │                              ▼                                           │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  LOCAL STORAGE (internal file system, independent from remote)   │    │
  │  │  LOCAL /mnt/dropbox/{YEAR} Customers/  (2020-2026)              │    │
  │  │  └── {Letter}/                                                   │    │
  │  │      └── {LastName, FirstName}/                                  │    │
  │  │          ├── Survey/                                             │    │
  │  │          │   └── {Customer}, {JobType}-S, {Date}/               │    │
  │  │          │       ├── photo.jpg                                   │    │
  │  │          │       └── webp/                                       │    │
  │  │          │           └── photo.webp                              │    │
  │  │          └── Installation/                                       │    │
  │  │              └── {Customer}, {JobType}-I, {Date}/               │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  │                              │                                           │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  BATCH PROCESSING                                                │    │
  │  │  /var/opt/move_photos/move_photos.py                             │    │
  │  │  - Processes /mnt/dropbox/Uploads/ folder                        │    │
  │  │  - Matches customer names to existing folders                    │    │
  │  │  - Routes to Installation subfolder for PV/AC/SWH jobs          │    │
  │  │  - Uploads to remote API if Job ID present                       │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  │                                                                          │
  │  ┌─────────────────────────────────────────────────────────────────┐    │
  │  │  CUSTOMER MAPPING                                                │    │
  │  │  /var/opt/map_dropbox/                                           │    │
  │  │  - Maps Dropbox folders to AEI database customers                │    │
  │  │  - Maintains unified_customers table                             │    │
  │  │  - 92.4% folder-to-customer match rate                          │    │
  │  └─────────────────────────────────────────────────────────────────┘    │
  └─────────────────────────────────────────────────────────────────────────┘
```

---

## Server Addresses

| Server | Domain | IP Address | Purpose |
|--------|--------|------------|---------|
| **Remote (AWS)** | aeihawaii.com | 18.225.0.90 | Main scheduler, Photo API |
| **Local** | upload.aeihawaii.com | 72.235.242.139 | Photo sync, storage |

### Remote Server Sync Call

The remote server syncs photos to the local server via:

```php
// In /photoapi/upload.php (line 100-101)
$remoteUploadUrl = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php';
$remoteAuthToken = '***';  // See CREDENTIALS.md
```

---

## Active Local Endpoint

**Based on remote server's `/photoapi/upload.php`, only ONE local file is actively called:**

### Active File

| File | URL | Auth Token | Status |
|------|-----|------------|--------|
| `uploadlocallat_kuldeep.php` | `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php` | See CREDENTIALS.md | **ACTIVE** |

**Location:** `/var/www/html/upload/uploadlocallat_kuldeep.php`

### Data Posted from Remote to Local

The remote server sends these fields (from `/photoapi/upload.php` lines 109-119):

```php
$postFields = array(
    'auth_token'  => '***',            // See CREDENTIALS.md
    'file'        => '@' . $filePath,      // The actual image file
    'file_name'   => basename($filePath),
    'job_id'      => $job_id,              // Job PID from database
    'job_type'    => $job_type_text,       // e.g., "PV", "AC", "PM", "WM"
    'last_name'   => $last_name,
    'first_name'  => $first_name,
    'job_date'    => $job_date_t,          // Original date format
    'photo_type'  => $photo_type,          // "S" (Survey) or "I" (Installation)
);
```

### Photo Type Determination (Remote Server Logic)

```php
// Survey photo types (photo_type = 'S')
if ($job_type_text == "PM" || $job_type_text == "WM" ||
    $job_type_text == "AS" || $job_type_text == "RPM" ||
    $job_type_text == "GCPM") {
    $photo_type = 'S';  // Survey
} else {
    $photo_type = 'I';  // Installation
}
```

### Inactive Local Upload Files

| File | Location | Status | Notes |
|------|----------|--------|-------|
| `uploadlocal.php` | `/var/www/html/upload/` | **NOT USED** | Saves to `/mnt/aeiserver/`, no customer routing |
| `uploadlocallat.php` | `/var/www/html/upload/` | **NOT USED** | Less validation than kuldeep version |

These files exist on the local server but are **not called** by the remote server's photo sync process.

---

## Component Details

### Remote Server Components (18.225.0.90)

#### 1. Photo API (`/photoapi/`)

| File | Purpose | Auth Token |
|------|---------|------------|
| `upload.php` | Main upload endpoint (base64) | See CREDENTIALS.md |
| `getimagelistingm.php` | List images for job | See CREDENTIALS.md |
| `getimagelisting.php` | Alternative listing | See CREDENTIALS.md |
| `fetch_image.php` | Serve individual images | - |
| `delete.php` | Delete photos | See CREDENTIALS.md |

**Backup Location:** `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/`

#### 2. Controllers

| File | Purpose |
|------|---------|
| `phototab.php` | Photo tab UI + S3 integration (45 KB) |
| `preupload.php` | Premeasure photo upload form (20 KB) |
| `loginapi.php` | Mobile app API endpoints (13 KB) |

**Backup Location:** `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/controllers/`

#### 3. AWS S3 Integration

```php
$s3 = S3Client::factory(array(
    'key'       => '***',              // See CREDENTIALS.md
    'secret'    => '***',              // See CREDENTIALS.md
    'region'    => 'eu-north-1',
    'signature' => 'v4'
));
$bucketName = 'aeiimages';
```

**SDK Location:** `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/scaws/aws.phar`

---

### Local Server Components

#### 1. Upload Endpoints (`/var/www/html/upload/`)

| File | Purpose | Auth Token | Status |
|------|---------|------------|--------|
| `uploadlocallat_kuldeep.php` | Enhanced upload with HEIC support | See CREDENTIALS.md | **ACTIVE** |
| `uploadlocallat.php` | Upload with customer folder routing | See CREDENTIALS.md | Not used |
| `uploadlocal.php` | Basic upload to `/mnt/aeiserver/` | See CREDENTIALS.md | Not used |

#### 2. Upload Flow Comparison

| Feature | `uploadlocal.php` | `uploadlocallat.php` | `uploadlocallat_kuldeep.php` |
|---------|-------------------|----------------------|------------------------------|
| **Destination** | `/mnt/aeiserver/{customer}-{type}/` | LOCAL `/mnt/dropbox/2025 Customers/` | LOCAL `/mnt/dropbox/{YEAR} Customers/` (via DB) |
| **Customer Folder Check** | No | Yes (filesystem) | Yes (database + filesystem) |
| **Database Integration** | No | No | ✅ Yes (`unified_customers`) |
| **Multi-Year Support** | No | No (2025 only) | ✅ Yes (2020-2026) |
| **Survey/Installation Routing** | No | Yes | Yes |
| **WebP Conversion** | No | Yes | Yes |
| **HEIC Support** | No | No | Yes (ImageMagick) |
| **Input Validation** | Basic | Basic | Comprehensive |
| **Path Sanitization** | Basic | Basic | Full security |

#### 3. Batch Processing (`/var/opt/move_photos/`)

**Script:** `move_photos.py` (83 KB)

**Purpose:** Organizes uploaded folders from `/mnt/dropbox/Uploads/` to customer directories

**Key Features:**
- Parses folder names: `[Customer Name] [Job Type] [Date]`
- Matches to existing customer folders
- Routes Installation jobs (PV, AC, SWH) to `Installation/` subfolder
- Uploads to remote API if `user_info.txt` contains Job ID
- Creates detailed logs

**API Configuration:**
```python
api_config = {
    'url': 'http://aeihawaii.com/upload_api.php',
    'auth_token': '***',            # See CREDENTIALS.md
    'file_type': 99,
    'timeout': 30,
    'max_retries': 3
}
```

#### 4. Customer Mapping (`/var/opt/map_dropbox/`)

**Purpose:** Maps Dropbox customer folders to AEI database records

**Database Tables:**
| Table | Records | Purpose |
|-------|---------|---------|
| `unified_customers` | 17,918 | Combined customer + folder data |
| `local_folder_customers` | 6,320 | Dropbox folder source |
| `remote_customers` | 16,324 | AEI database customers |

**Match Rate:** 92.4% of folders matched to customers

---

## Data Flow Scenarios

### Scenario 1: Mobile App Photo Upload

```
1. Field worker takes photo in mobile app
2. App POSTs to remote server: /photoapi/upload.php
   - Auth: see CREDENTIALS.md
   - Data: base64 image, job_id
3. Remote server:
   a. Decodes image
   b. Saves to REMOTE /mnt/dropbox/{YEAR} Customers/ (AWS EBS)
   c. Echoes JSON response (buffered until script ends)
   d. cURL POSTs to local server: uploadlocallat_kuldeep.php (5s connect timeout)
   e. Copies original to scheduler/uploads/
   f. Triggers background WebP generation (generate_webp.py via nohup &)
   g. Inserts meter_files record
4. Local server:
   a. Validates file type and auth
   b. Looks up customer via unified_customers DB (customer_id → folder path)
   c. Saves to LOCAL /mnt/dropbox/{YEAR} Customers/{customer}/ (internal file system)
   d. Triggers background thumbnail generation (Python)
   e. Returns JSON response
5. Photo available on both servers (independent storage systems)
```

### Scenario 2: Web Upload with Batch Processing

```
1. User uploads photos via web form
2. Photos land in /mnt/dropbox/Uploads/[Customer] [Type] [Date]/
3. Cron or manual trigger runs move_photos.py
4. Script processes each folder:
   a. Parses folder name
   b. Checks for user_info.txt with Job ID
   c. If Job ID: Uploads to remote API
   d. Matches to existing customer folder
   e. Routes to Survey/ or Installation/
   f. Moves folder, cleans up source
5. Updates logs and user_info.txt with status
```

### Scenario 3: S3 Image Retrieval

```
1. User opens Photo Tab in scheduler
2. phototab.php queries S3:
   - getimagelistings3($job_id)
   - Lists objects with job_id prefix
3. Thumbnails generated and cached locally
4. Images served from S3 or local cache
```

---

## Storage Locations

### Remote Server (18.225.0.90)

| Path | Purpose | Type |
|------|---------|------|
| `/mnt/dropbox/{YEAR} Customers/` | Field technician photos + mobile app gallery | AWS EBS volume (local disk) |
| `/mnt/dropbox/.../webp/` | WebP thumbnails for mobile app (via `getimagelisting.php`) | AWS EBS volume |
| `/mnt/aeiserver/` | Job-based folders | Local |
| `/scheduler/uploads/` | Web-accessible copies | Local |
| `/scheduler/uploads/webp/` | Flat WebP copies (meter_files DB reference) | Local |
| `/scheduler/s3uploads/thumbnails/` | Cached S3 thumbnails | Local |
| S3: `aeiimages` | Cloud backup | AWS S3 |

> **Note:** Remote `/mnt/dropbox/` is a directory on the AWS EBS root volume (`/dev/xvda1`), NOT a network mount or shared storage.

### Local Server

| Path | Purpose |
|------|---------|
| `/mnt/dropbox/{YEAR} Customers/` | Customer-organized photos (internal file system) |
| `/mnt/dropbox/Uploads/` | Incoming uploads (pre-processing) |
| `/mnt/aeiserver/` | Job ID-based uploads |

> **Note:** Local `/mnt/dropbox/` is the company's internal file system, completely independent from the remote server's `/mnt/dropbox/`. The cURL sync sends a COPY of each photo from remote to local.

---

## Database Integration

### Remote Database (mandhdesign_schedular)

**Photo Records:** `meter_files` table with `file_type=99`

```sql
INSERT INTO meter_files(
    job_id,
    unique_filename,
    original_filename,
    file_size,
    file_type,
    webpfilename,
    created
) VALUES(
    $job_id,
    "$filename",
    "$filename",
    "121",
    "99",  -- Photo type
    "$webpfilename",
    NOW()
);
```

### Local Database (Schedular)

**Customer Mapping:** `unified_customers` table

```sql
SELECT id, first_name, last_name, email, phone,
       folder_path, folder_year, remote_customer_id
FROM unified_customers
WHERE remote_customer_id = ?
AND folder_path IS NOT NULL;
```

---

## Authentication Tokens

See [CREDENTIALS.md](CREDENTIALS.md) for all authentication tokens.

| Token | Used By | Purpose |
|-------|---------|---------|
| Remote API token | Remote photoapi | Photo API authentication |
| Local sync token | Local upload endpoints | Remote-to-local sync |

**Note:** `uploadlocallat_kuldeep.php` supports `.env` file for token configuration.

---

## File Format Support

### Image Types

| Format | Remote Upload | Local Upload | WebP Conversion |
|--------|---------------|--------------|-----------------|
| JPEG | Yes | Yes | Yes |
| PNG | Yes | Yes | Yes |
| GIF | Yes | Yes | Yes |
| HEIC | Yes | `*_kuldeep.php` only | Yes (ImageMagick) |
| HEIF | Yes | `*_kuldeep.php` only | Yes (ImageMagick) |

### WebP Conversion

All endpoints create WebP thumbnails:
- Quality: 80%
- Location: `{folder}/webp/{filename}.webp`
- Used for: Fast loading in web interface

---

## Error Handling

### Remote Server

- Invalid token: Returns JSON error
- Upload failure: Logs to error_log
- S3 failure: Falls back to local storage

### Local Server

- Invalid token: JSON error response
- File type validation: Rejects non-images
- Path traversal: Sanitized input
- Missing customer: Creates in `/mnt/dropbox/Uploads/`

### Batch Processing

- API timeout: Retries up to 3 times
- Move failure: Logs error, continues processing
- Missing customer: Leaves folder in Uploads

---

## Maintenance

### Log Locations

| Log | Location |
|-----|----------|
| Remote API | `/var/log/httpd/error_log` |
| Local uploads | Apache error log |
| Batch processing | `/var/opt/move_photos/move_photos_*.log` |
| Success moves | `/mnt/dropbox/Uploads/photo_moves_log.txt` |
| API uploads | `/mnt/dropbox/Uploads/api_uploads_log.txt` |

### Cron Jobs

```bash
# Customer mapping daily sync (5:30 AM weekdays)
30 5 * * 1-5 /usr/bin/python3 /var/opt/map_dropbox/scripts/daily_sync.py

# Move photos (as needed)
# python3 /var/opt/move_photos/move_photos.py
```

---

## Customer Folder Mapping Integration

### Implementation Status: ✅ COMPLETE (2026-01-30)

**The upload endpoint `uploadlocallat_kuldeep.php` now uses the `unified_customers` database for folder lookups.**

#### Previous Behavior (Fixed)

```php
// OLD CODE (removed)
$indexname = substr($customer_name, 0, 1);
$uploadDir = '/mnt/dropbox/2025 Customers/' . $indexname;  // HARDCODED YEAR!
$uploadDir_customer = $uploadDir . '/' . $customer_name;
$customer_exists = is_dir($uploadDir_customer);  // Simple filesystem check
```

| Problem (Fixed) | Impact |
|-----------------|--------|
| ~~Hardcoded "2025 Customers"~~ | ~~Won't find customers from 2020-2024~~ |
| ~~No database lookup~~ | ~~Ignores 6,320 mapped folders~~ |

#### Current Implementation

**File:** `/var/www/html/upload/uploadlocallat_kuldeep.php`
**Backup:** `/var/www/html/upload/archive/uploadlocallat_kuldeep.php.bak.20260130`

```php
// Enhanced implementation (2026-01-30)
require_once __DIR__ . '/db_config.php';

// Get customer_id and job_year from remote server
$customer_id = isset($_POST['customer_id']) ? intval($_POST['customer_id']) : null;
$job_id = isset($_POST['job_id']) ? intval($_POST['job_id']) : null;
$job_year = (int)date("Y", strtotime($_POST['job_date']));

// Priority lookup: 1) customer_id + year, 2) customer_id any year, 3) name + year, 4) name any year
$uploadDir_customer = null;

if ($customer_id) {
    // Try customer_id lookup first (most reliable - 92%+ match rate)
    $uploadDir_customer = getCustomerFolderByCustomerId($customer_id, $job_year);
}

if (!$uploadDir_customer) {
    // Fallback to name-based lookup
    $uploadDir_customer = getCustomerFolderByName($first_name, $last_name, $job_year);
}

if ($uploadDir_customer && is_dir($uploadDir_customer)) {
    // Customer found - use mapped folder path with correct year
} else {
    // Not found - falls back to /mnt/dropbox/Uploads/
    error_log("Photo upload: Customer not found - customer_id=$customer_id, name=$first_name $last_name");
}
```

#### Year-Aware Folder Selection (Local Server Only)

The **local server** lookup matches job year to folder year, with fallback to most recent:

| Job Date | customer_id=7824 Result |
|----------|-------------------------|
| 2024-xx-xx | `/mnt/dropbox/2024 Customers/N/Nakahara, Janell & Kenneth` |
| 2025-xx-xx | `/mnt/dropbox/2025 Customers/N/Nakahara, Janell` |
| 2026-xx-xx | `/mnt/dropbox/2025 Customers/N/Nakahara, Janell` (fallback to most recent) |

> **Important — Remote vs Local difference:** The **remote server** (`upload.php`) does NOT
> use database lookup. It always constructs the folder path from `job_date`, so a 2026 job
> always goes into `2026 Customers/`. The local server's DB-driven fallback means cross-year
> customers may have photos in different year folders on each server. See
> [PHOTO_SYSTEM_DOCUMENTATION.md](PHOTO_SYSTEM_DOCUMENTATION.md) "Cross-Year Folder Routing"
> section for the full comparison and mobile app impact.

#### Database Integration Details

**Database:** `Schedular` (local)
**Table:** `unified_customers`

| Field | Purpose |
|-------|---------|
| `folder_path` | Full path to customer folder (e.g., `/mnt/dropbox/2023 Customers/S/Smith, John`) |
| `folder_year` | Year of folder (2020-2026), used for ordering |
| `first_name` | Customer first name for matching |
| `last_name` | Customer last name for matching |

**Query Logic:**
1. Match by exact `first_name` AND `last_name`
2. Only return rows where `folder_path IS NOT NULL`
3. Order by `folder_year DESC` to get most recent
4. Return first match (LIMIT 1)

#### Benefits Achieved

| Before | After |
|--------|-------|
| Only finds 2025 folders | Finds folders from 2020-2026 |
| Hardcoded path lookup | Database-driven dynamic lookup |
| ~30% success rate (estimate) | 92.4% match rate (based on unified_customers data) |
| Silent failures | Error logging for unmatched customers |

#### Fallback Behavior

When customer is NOT found in `unified_customers`:
1. Photos are saved to: `/mnt/dropbox/Uploads/[{customer}] [{type}] [{date}]/`
2. Error is logged: `Photo upload: Customer not found in unified_customers - {name}`
3. Batch processing (`move_photos.py`) can later match and move these files

#### Related Resources

| Resource | Location |
|----------|----------|
| Database Config | `/var/www/html/upload/db_config.php` |
| Backup File | `/var/www/html/upload/uploadlocallat_kuldeep.php.bak.20260130` |
| Customer Mapping Project | `/var/opt/map_dropbox/` |
| Customer API Endpoints | `/var/www/html/upload/api/` |

#### Customer API Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/api/search.php` | Search customers by name |
| `/api/get_customer.php` | Get customer details by ID |
| `/api/get_customer_folders.php` | Get folders for customer |
| `/api/get_directories.php` | List available directories |

---

## Related Documentation

| Document | Location | Purpose |
|----------|----------|---------|
| Photo API Project README | `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/README.md` | Project overview |
| Move Photos Documentation | `/var/www/html/upload/MOVE_PHOTOS_DOCUMENTATION.md` | Batch processing |
| Scheduler Photo Documentation | `/var/www/html/upload/SCHEDULER_PHOTO_DOCUMENTATION.md` | Job-based processing |
| Map Dropbox README | `/var/opt/map_dropbox/README.md` | Customer mapping |
| Remote Server Documentation | `/var/opt/AEI_REMOTE/REMOTE_SERVER.md` | Server architecture |
