# AEI Photo System - Complete Documentation

**Document Version:** 2.2
**Last Updated:** 2026-02-05
**Status:** PARTIALLY OUTDATED — see note below

> **IMPORTANT:** This document predates PHOTO-009 (Unified Storage), PHOTO-017 (3-Tier WebP + Staging),
> and PHOTO-021 (Staging Direct, DB-Only Listing, API Fix). Key changes since this was written:
>
> - **No /mnt/dropbox/ hierarchy writes** — `upload.php` no longer saves to /mnt/dropbox/ on remote
> - **Staging-first** — ALL uploads land in `uploads/staging/` (not uploads/ root)
> - **3-tier WebP** — `generate_thumbnails.py` v4.0 replaces `generate_webp.py` (thumbs + webp + hi-res)
> - **DB-only listing** — scheduler uses `meter_files` only; `getimagelistingm.php`/filesystem scanning eliminated
> - **No CI resize** — the 800x1027 JPEG resize was removed for scheduler uploads
> - **Scheduler thumbnails** — WebP from `uploads/thumbs/` (not GD JPEG `thumbnail_helper`)
>
> **Canonical reference:** `PHOTO_SYSTEM_PROCESS_MAP.md` (same directory) is the authoritative post-PHOTO-021 document.
> The sections below on upload flow, WebP generation, and scheduler display are outdated.
> Sections on local server, credentials, security concerns, and troubleshooting remain accurate.

---

## Table of Contents

1. [System Overview](#system-overview)
2. [Architecture Diagram](#architecture-diagram)
3. [Component Details](#component-details)
   - [Remote Photo API (upload.php)](#1-remote-photo-api-photoapiuploadphp)
   - [Mobile App API Endpoints](#1b-mobile-app-api-endpoints)
   - [Local Upload API](#2-local-upload-api-uploadlocallat_kuldeepphp)
   - [Customer Folder Mapper](#3-customer-folder-mapper-varoptmap_dropbox)
4. [Recent Fixes (2026-02-04)](#recent-fixes-2026-02-04)
5. [Database Integration](#database-integration)
6. [API Security](#api-security)
7. [File Storage Structure](#file-storage-structure)
8. [Maintenance & Operations](#maintenance--operations)
9. [Troubleshooting Guide](#troubleshooting-guide)
10. [Known Issues](#known-issues)
11. [Change History](#change-history)

---

## System Overview

The AEI Photo System is a distributed architecture that handles photo uploads from multiple sources (mobile app, web scheduler) and synchronizes them between a remote AWS server and a local server with Dropbox storage.

### Key Components

| Component | Location | Purpose |
|-----------|----------|---------|
| **Remote Photo API** | `aeihawaii.com/photoapi/` | Receives uploads from mobile app |
| **Scheduler Web Upload** | `aeihawaii.com/scheduler/` | Web-based photo upload interface |
| **Local Upload API** | `upload.aeihawaii.com/uploadlocallat_kuldeep.php` | Receives synced photos, routes to customer folders |
| **Customer Mapper** | `/var/opt/map_dropbox/` | Maps customers to Dropbox folders (daily cron) |

### Server Information

| Server | Domain | IP Address | Purpose |
|--------|--------|------------|---------|
| Remote (AWS) | aeihawaii.com | 18.225.0.90 | Main scheduler, Photo API |
| Local | upload.aeihawaii.com | 72.235.242.139 | Photo storage, Dropbox sync |

---

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                     AEI PHOTO SYSTEM — UPLOAD FLOW                          │
└─────────────────────────────────────────────────────────────────────────────┘

  ┌───────────────────┐     ┌───────────────────┐
  │   Mobile App      │     │  Scheduler Web    │
  │  (Field Workers)  │     │    Interface      │
  └─────────┬─────────┘     └─────────┬─────────┘
            │                         │
            │ POST (base64 + job_id)  │ POST (multipart file)
            ▼                         ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │               REMOTE SERVER (18.225.0.90) — PHP 7.3.30                  │
  │                                                                          │
  │  /photoapi/upload.php              /scheduler/.../phototab.php           │
  │  ├─ Validates auth token           ├─ Validates session                  │
  │  ├─ Decodes base64 image           ├─ Processes file upload              │
  │  ├─ Queries job/customer info      ├─ Queries job/customer info          │
  │  ├─ Saves to REMOTE /mnt/dropbox/ ┤─ Saves to REMOTE /mnt/dropbox/     │
  │  │  (name-based path)              │                                     │
  │  ├─ Creates webp/ folder           │                                     │
  │  ├─ Copies to scheduler/uploads/  │                                     │
  │  ├─ Background generate_webp.py   (nohup &, non-blocking)              │
  │  │   └─ Writes WebP to webp/ in REMOTE /mnt/dropbox/                   │
  │  │   └─ Copies WebP to scheduler/uploads/webp/                         │
  │  ├─ Inserts meter_files record                                          │
  │  ├─ Background sync_to_local.py ──(nohup &, non-blocking)──────┐       │
  │  ├─ echo JSON response (LAST — sent immediately)                │       │
  │  └─ Script ends → mobile app gets response (~0.1-0.5s)          │       │
  │                                                                   │       │
  │     Auth Token: See CREDENTIALS.md                               │       │
  └───────────────────────────────────────────────────────────────────│───────┘
                                                                      │
            Background Python requests.post() with: auth_token,       │
            file, customer_id, job_id, job_type, names, job_date,     │
            photo_type (runs after mobile app already has response)    │
                                                                      ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │              LOCAL SERVER (upload.aeihawaii.com)                         │
  │                                                                          │
  │  /uploadlocallat_kuldeep.php                                             │
  │  ├─ Validates auth token (from .env file)                                │
  │  ├─ Validates file type (JPEG, PNG, GIF, HEIC)                           │
  │  ├─ Sanitizes filename and path components                               │
  │  ├─ Queries unified_customers DB for folder path                         │
  │  │   ├─ Priority 1: customer_id + job_year                               │
  │  │   ├─ Priority 2: customer_id (any year)                               │
  │  │   ├─ Priority 3: name + job_year                                      │
  │  │   ├─ Priority 4: name (any year)                                      │
  │  │   └─ Priority 5: Auto-create folder + insert DB record               │
  │  ├─ Saves to LOCAL /mnt/dropbox/ (DB-driven path)                       │
  │  │   ├─ Found: /mnt/dropbox/{YEAR} Customers/{Letter}/{Customer}/       │
  │  │   ├─ Created: New folder + unified_customers insert                   │
  │  │   └─ Fallback: /mnt/dropbox/Uploads/[Customer] [Type] [Date]/       │
  │  ├─ Returns JSON success immediately                                     │
  │  └─ Triggers background Python for thumbs/ and thumbs/mid/              │
  │                                                                          │
  │     Auth Token: See CREDENTIALS.md (stored in .env)                      │
  └─────────────────────────────────────────────────────────────────────────┘

  IMPORTANT: /mnt/dropbox/ on each server is INDEPENDENT storage.
  - REMOTE /mnt/dropbox/ = local directory on AWS EBS volume (field technician photos)
  - LOCAL  /mnt/dropbox/ = internal company file system (office storage)
  - The cURL sync sends a COPY of each photo from remote to local storage.

  ┌─────────────────────────────────────────────────────────────────────────────┐
  │         CROSS-YEAR FOLDER ROUTING — REMOTE vs LOCAL DIFFERENCE              │
  └─────────────────────────────────────────────────────────────────────────────┘

  The remote and local servers use DIFFERENT logic to determine which year folder
  a photo is stored in. This matters for jobs that span multiple years (e.g.,
  premeasure in 2025, installation in 2026).

  REMOTE (upload.php) — Name-Based, Always Uses Job Year:
    - Derives year directly from job_date: $job_year = date("Y", strtotime($job_date_t))
    - Always creates: /mnt/dropbox/{JOB_YEAR} Customers/{Letter}/{Customer}/
    - No database lookup — folder path is constructed from job metadata
    - A 2026 job ALWAYS goes into "2026 Customers/", even if customer has a 2025 folder

  LOCAL (uploadlocallat_kuldeep.php) — DB-Driven, Falls Back to Most Recent Year:
    - Queries unified_customers DB for existing customer folder
    - Priority: customer_id+year → customer_id(any year) → name+year → name(any year)
    - If no entry for 2026 exists, falls back to most recent year (e.g., 2025 folder)
    - Only creates a new year folder if NO existing DB entry is found at all

  Example — Customer with 2025 Premeasure, 2026 Installation:
  ┌─────────────────────────────────────────────────────────────────────────────┐
  │ Server │ 2025 PM Upload                    │ 2026 SC Upload                │
  │────────│───────────────────────────────────│──────────────────────────────│
  │ REMOTE │ /mnt/dropbox/2025 Customers/M/    │ /mnt/dropbox/2026 Customers/ │
  │        │   Mitchell, Linwood/Survey/...     │   Mitchell, Linwood/Install/ │
  │────────│───────────────────────────────────│──────────────────────────────│
  │ LOCAL  │ /mnt/dropbox/2025 Customers/M/    │ /mnt/dropbox/2025 Customers/ │
  │        │   Mitchell, Lynwood/Survey/...     │   Mitchell, Lynwood/Install/ │
  │        │                                    │   ↑ REUSES 2025 folder       │
  └─────────────────────────────────────────────────────────────────────────────┘

  Result: For cross-year customers, the 2026 installation photos end up in
  DIFFERENT year folders on each server:
    - Remote: 2026 Customers/ (always matches job year)
    - Local:  2025 Customers/ (falls back to existing DB entry)

  Mobile App Impact (FIXED 2026-02-05):
    - getimagelisting.php now queries ALL jobs for a customer by customer_id
    - A 2026 installation job will show photos from ALL years (2025 PM, 2026 SC, etc.)
    - Each image link contains its own job_id, so fetch_image.php resolves the correct folder
    - Response includes per-image metadata: job_id, job_type, job_date, photo_type
    - See section "1b. Mobile App API Endpoints" for full response format


┌─────────────────────────────────────────────────────────────────────────────┐
│               REMOTE STORAGE: /mnt/dropbox/ (AWS EBS volume)                │
│               Purpose: Field technician photos + mobile app gallery         │
└─────────────────────────────────────────────────────────────────────────────┘

  /{YEAR} Customers/                        Written by: upload.php
  └── {Letter}/
      └── {LastName, FirstName}/
          ├── Survey/
          │   └── {Customer}, {JobType}-S, {Date}/
          │       ├── original.jpg                    ← upload.php
          │       └── webp/                           ← generate_webp.py
          │           └── photo.webp                    (mobile app reads)
          └── Installation/
              └── {Customer}, {JobType}-I, {Date}/
                  └── (same structure)

  /scheduler/uploads/                       Written by: upload.php
  ├── {filename}.jpg                        ← copy of original
  ├── thumbnails/                           ← thumbnail_helper (JPEG)
  └── webp/                                 ← copy of WebP from generate_webp.py


┌─────────────────────────────────────────────────────────────────────────────┐
│               LOCAL STORAGE: /mnt/dropbox/ (internal file system)           │
│               Purpose: Office storage with DB-driven routing + thumbnails   │
└─────────────────────────────────────────────────────────────────────────────┘

  /{YEAR} Customers/                        Written by: uploadlocallat_kuldeep.php
  └── {Letter}/
      └── {LastName, FirstName}/
          ├── Survey/
          │   └── {Customer}, {JobType}-S, {Date}/
          │       ├── original.jpg                    ← uploadlocallat_kuldeep.php
          │       └── thumbs/                         ← upload_thumb_generator.py
          │           ├── photo.webp                    (200x200)
          │           └── mid/
          │               └── photo.webp                (800px)
          └── Installation/
              └── (same structure)

  /Uploads/                                 Fallback for unmatched customers
  └── [{Customer}] [{JobType}] [{Date}]/    (local server only)


┌─────────────────────────────────────────────────────────────────────────────┐
│                 AEI PHOTO SYSTEM — MOBILE APP READ FLOW                     │
└─────────────────────────────────────────────────────────────────────────────┘

  ┌───────────────────┐
  │   Mobile App      │
  │  (View Gallery)   │
  └─────────┬─────────┘
            │
            │ 1. POST { auth_token, job_id }
            ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │               REMOTE SERVER (18.225.0.90)                               │
  │                                                                          │
  │  /photoapi/getimagelisting.php  (CROSS-YEAR, 2026-02-05)                 │
  │  ├─ Validates auth token                                                 │
  │  ├─ Queries job to get customer_id                                       │
  │  ├─ Queries ALL jobs for that customer_id                                │
  │  ├─ For each job: derives year/type, scans its webp/ folder              │
  │  └─ Returns JSON array with per-image job_id + metadata                  │
  └─────────────────────────────────────────────────────────────────────────┘
            │
            │ 2. GET ?job_id={id}&img={filename}  (for each image)
            ▼
  ┌─────────────────────────────────────────────────────────────────────────┐
  │  /photoapi/fetch_image.php                                               │
  │  ├─ Queries job/customer info from DB                                    │
  │  ├─ Derives year from job_date                                           │
  │  ├─ Reads WebP file from webp/ on REMOTE /mnt/dropbox/                  │
  │  └─ Returns binary image with Content-Type header                        │
  └─────────────────────────────────────────────────────────────────────────┘
            │
            ▼
  ┌───────────────────┐
  │   Mobile App      │
  │  (Displays Photo) │
  └───────────────────┘

  NOTE: The mobile app NEVER contacts the local server.
  All photo retrieval flows through the remote server's PHP endpoints,
  which read from the REMOTE server's own /mnt/dropbox/ (AWS EBS volume).
```

---

## Component Details

### 1. Remote Photo API (`/photoapi/upload.php`)

**Location:** Remote server `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php`
**PHP Version:** 7.3.30 (upgraded 2026-02-23; still no native WebP GD support in this build)
**Python:** 3.6 at `/usr/local/bin/python3.6` (Pillow 8.4.0)

**Purpose:** Receives photo uploads from the mobile app, saves the original to the remote server's `/mnt/dropbox/` (AWS EBS, name-based path), generates WebP for mobile app gallery (background), syncs the file to local server in background for DB-driven routing and thumbnail generation, and records in the `meter_files` database.

#### Complete Upload Flow

```
Mobile App → POST { auth_token, file_name, image_data (base64), job_id }
  1. Validate auth token (see CREDENTIALS.md)
  2. Decode base64 image data
  3. Query job/customer info from local MySQL
  4. Determine photo_type: Survey (PM, WM, AS, RPM, GCPM) or Installation (all others)
  5. Create folder structure on /mnt/dropbox/{YEAR} Customers/ (year from job_date)
  6. Create webp/ subfolder for mobile app image listing
  7. Save original image to job folder
  8. Validate saved file (realpath + is_readable)
  9. Copy original to scheduler/uploads/ (for thumbnail_helper)
  10. Trigger generate_webp.py in BACKGROUND (nohup &, non-blocking)
      - Converts original → WebP in webp/ folder
      - Copies WebP to scheduler/uploads/webp/
  11. Insert meter_files record (job_id, filenames, filesize, webpfilename)
  12. Trigger sync_to_local.py in BACKGROUND (nohup &, non-blocking)
      - POST file + metadata to local server (upload.aeihawaii.com)
      - On failure: queues payload as JSON in photoapi/queue/ for retry
      - Logs results to photoapi/logs/sync_to_local.log
  13. echo JSON response (LAST — sent immediately to mobile app)
  14. Script ends → mobile app already has response (~0.1-0.5s total)
      ... local server sync happens 2-10s later (background)
      ... WebP appears in webp/ folder 1-3s later (background)
```

**Important:** The `echo` is the last statement before script end, so the mobile app gets an immediate response. Both `generate_webp.py` (step 10) and `sync_to_local.py` (step 12) are non-blocking background processes — the script does not wait for them.

#### Database Connections

| Database | Host | User | Password |
|----------|------|------|----------|
| `mandhdesign_schedular` | localhost | See CREDENTIALS.md | See CREDENTIALS.md |

#### Database Query

```php
$query = "SELECT js.customer_id, cs.first_name, cs.last_name, js.job_date, jt.intials, js.job_pid
          FROM jobs js
          LEFT JOIN customers cs ON js.customer_id = cs.id
          LEFT JOIN job_types jt ON js.job_type_id = jt.id
          WHERE js.id = " . intval($data['job_id']);
```

#### POST Fields Sent to Local Server (via background `sync_to_local.py`)

| Field | Type | Description |
|-------|------|-------------|
| `auth_token` | string | Authentication token (see CREDENTIALS.md) |
| `file` | file | Image file (multipart via Python `requests`) |
| `file_name` | string | Original filename |
| `job_id` | int | Job PID from remote database (`js.job_pid`) |
| `customer_id` | int | Customer ID for direct database lookup |
| `job_type` | string | Job type code (PV, AC, PM, etc.) |
| `first_name` | string | Customer first name (fallback matching) |
| `last_name` | string | Customer last name (fallback matching) |
| `job_date` | string | Job date (YYYY-MM-DD format) |
| `photo_type` | string | 'S' (Survey) or 'I' (Installation) |

#### Local Server Sync (Background)

```php
// sync_to_local.py handles the POST to local server in the background
$syncScript = __DIR__ . '/sync_to_local.py';
$syncCmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($syncScript) . ' ... > /dev/null 2>&1 &';
exec($syncCmd);
```

**Script:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py`
**Target URL:** `https://upload.aeihawaii.com/uploadlocallat_kuldeep.php`
**Log:** `photoapi/logs/sync_to_local.log`
**Timeouts:** connect=10s, read=60s

**Store-and-Forward Retry Queue:** On network/server failure (when source file still exists on disk), `sync_to_local.py` saves the payload as JSON in `photoapi/queue/`. A cron-driven `process_retry_queue.py` retries every 15 minutes (max 10 retries = ~2.5 hours). Failed items are moved to `queue/failed/`. See [ASYNC_SYNC_ENHANCEMENT.md](../ENHANCEMENTS/PHOTO-003_ASYNC_SYNC/ASYNC_SYNC_ENHANCEMENT.md) for full details.

#### WebP Generation (Background)

WebP conversion is done via a Python script running as a background process (GD in the production PHP 7.3 build lacks WebP support):

```php
$cmd = 'nohup /usr/local/bin/python3.6 ' . escapeshellarg($generateScript)
     . ' ' . escapeshellarg($filePath)
     . ' ' . escapeshellarg($webp_dest)
     . ' 80'
     . ' && cp ' . escapeshellarg($webp_dest) . ' ' . escapeshellarg($scheduler_webp_dest)
     . ' > /dev/null 2>&1 &';
exec($cmd);  // Returns immediately — does not block the script
```

**Script:** `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/generate_webp.py`
**Requires:** Python 3.6 + Pillow 8.4.0

#### meter_files Database Insert

```php
INSERT INTO meter_files(job_id, unique_filename, original_filename, file_size, file_type, webpfilename, folder_path, created)
VALUES ({job_pid}, "{filename}", "{filename}", "{filesize}", "99", "{webpfilename}", "{final_folder}", "{datetime}")
```

- `file_type` is always "99"
- `webpfilename` is set to the expected WebP filename before the background process completes
- `folder_path` stores the absolute path to the photo's parent directory (added 2026-02-05). Used by `getimagelisting.php` and `fetch_image.php` to find photos even if the job is later unscheduled, rescheduled, or deleted. NULL for rows created before this enhancement — those use path reconstruction as fallback.

#### Security Enhancements (2026-01-30 / 2026-02-04)

| Enhancement | Before | After |
|-------------|--------|-------|
| SQL Query | No `customer_id` | Added `js.customer_id` to SELECT |
| SQL Security | No protection | Added `intval()` for job_id |
| SQL Insert | No escaping, hardcoded filesize "121" | `mysqli_real_escape_string()`, dynamic `filesize()` |
| Sync Payload | Name-only matching | Added `customer_id` for ID-based lookup |
| Local Server Sync | Synchronous cURL (2-10s blocking) | Background Python `sync_to_local.py` (non-blocking) |
| Sync Errors | `error_log()` in PHP error log | Dedicated `logs/sync_to_local.log` |
| Mobile Response | Delayed by cURL round-trip | Immediate (~0.1-0.5s) |
| WebP photo_type | Hardcoded "-S" in filename | Dynamic `$photo_type` |

---

### 1b. Mobile App API Endpoints (Read Flow)

**Location:** Remote server `/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/`

These endpoints serve the mobile app's photo gallery. They run on the **remote server** and read from the `webp/` folder on the remote server's own `/mnt/dropbox/` (AWS EBS volume). The mobile app never contacts the local server directly — all photo retrieval goes through these remote endpoints.

#### `getimagelisting.php` — List Photos for a Job (Cross-Year)

**Input:** POST JSON `{ auth_token, job_id }`
**Output:** JSON array of image objects with metadata

**Cross-Year Behavior (2026-02-05):** Instead of scanning only the requested job's folder, the endpoint now:
1. Queries the requested `job_id` to get the `customer_id`
2. Queries ALL jobs for that `customer_id` from the `jobs` table
3. Queries `meter_files` for stored `folder_path` values (added 2026-02-05)
4. For each job: uses `folder_path` from `meter_files` when available, falls back to path reconstruction when NULL
5. Deduplicates scanned directories to avoid listing the same image twice
6. Returns a combined array with per-image `job_id`, so `fetch_image.php` resolves the correct folder
7. Maintains backward-compatible legacy scan of `/mnt/aeiserver/` (fixed link format)

```php
// Step 1: Get customer_id from requested job
$query = "SELECT js.customer_id, ... WHERE js.id = " . intval($data['job_id']);

// Step 2: Find ALL jobs for this customer
$query2 = "SELECT js.id, ... WHERE js.customer_id = " . $customerId . " ORDER BY js.job_date DESC";

// Step 3: Scan each job's webp/ folder using helper function
function getJobPhotoInfo($job_row) {
    // Derives photo_type (Survey/Installation), year, folder path from job row
    return array('photo_type' => ..., 'job_type' => ..., 'uploadDir' => ...);
}

// Step 4: Return combined array with per-image metadata
$images[] = array(
    "link" => "https://aeihawaii.com/photoapi/fetch_image.php?job_id=" . $relatedJobId . "&img=" . $img,
    "job_id" => $relatedJobId,        // Each image's own job_id
    "job_type" => $info['job_type'],   // PM, SC, AC, etc.
    "job_date" => $info['job_date'],   // mm-dd-yyyy
    "photo_type" => $info['photo_type'] // Survey or Installation
);
```

**Example Response (cross-year customer):**
```json
[
  {
    "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=105661&img=photo1.webp",
    "job_id": 105661,
    "job_type": "PM",
    "job_date": "07-21-2025",
    "photo_type": "Survey"
  },
  {
    "link": "https://aeihawaii.com/photoapi/fetch_image.php?job_id=108364&img=photo2.webp",
    "job_id": 108364,
    "job_type": "AC",
    "job_date": "01-15-2026",
    "photo_type": "Installation"
  }
]
```

**Security (2026-02-05):**
- Removed hardcoded `$data['auth_token']='aei@89806849'` that bypassed authentication
- Added `intval()` on `job_id` and `customer_id` for SQL injection protection
- Removed duplicate `$jobId` sanitization

#### `fetch_image.php` — Serve Individual Photo

**Input:** GET `?job_id={id}&img={filename}`
**Output:** Binary image data with correct Content-Type header

Works correctly with cross-year listing — each image's `link` contains its own `job_id`, so `fetch_image.php` resolves the correct folder.

**Folder resolution (2026-02-05):**
1. Queries `meter_files` by webp filename and `job_pid` for a stored `folder_path`
2. If `folder_path` is populated and the file exists: serves directly from that path
3. If `folder_path` is NULL (legacy rows): falls back to path reconstruction from `jobs` table

```php
// Try folder_path from meter_files first (resilient to job changes)
$mfQuery = "SELECT mf.folder_path FROM meter_files mf
    INNER JOIN jobs js ON js.job_pid = mf.job_id
    WHERE js.id = $jobId AND mf.webpfilename = '$fileName' AND mf.folder_path IS NOT NULL";

// Fallback: reconstruct path from jobs table (legacy behavior)
$uploadDir = '/mnt/dropbox/' . $job_year . ' Customers/'.$indexname.'/'.$customername
           .'/'.$photo_type."/".$customer_name.", ".$job_type_text
           ."-".$photo_s.', '.$job_date.'/webp/';
readfile($filePath);
```

**Security (2026-02-05):**
- `intval()` on `$jobId` (was `basename()` — insufficient for SQL)
- Input validation (`!$jobId || !$fileName`) moved before DB query
- Removed dead code (`exit;die;`)

#### Other API Endpoints

| File | Purpose |
|------|---------|
| `getimagelistingcnt.php` | Image count for a job |
| `getimagelistingnew.php` | Alternate image listing |
| `getimagelistingm.php` | Mobile-specific listing variant |
| `delete.php` | Delete photo (with job lookup) |
| `delete_image.php` | Direct image deletion |

#### Backup Location

Local backups of remote upload.php versions:
```
/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/
├── upload.php.original.20260130                       # Before customer_id enhancement
├── upload.php.pre_webp_fix.20260204                   # Before any WebP changes
├── upload.php.remote.20260204                         # After mobile API fix
├── photoapi/upload.php.bak.pre_async_webp.20260204    # Before async background approach
├── photoapi/upload.php.bak.pre_async_sync.20260205    # Before async local server sync
├── photoapi/generate_webp.py                          # Python WebP converter
├── photoapi/sync_to_local.py                          # Python background local server sync (with retry queue)
├── photoapi/process_retry_queue.py                    # Cron-driven retry processor for failed syncs
├── photoapi/getimagelisting.php                       # Cross-year photo listing
├── photoapi/fetch_image.php                           # Individual photo serving (security fixes)
```

---

### 2. Local Upload API (`uploadlocallat_kuldeep.php`)

**Location:** `/var/www/html/upload/uploadlocallat_kuldeep.php`

**Purpose:** Receives photo uploads from the remote server via cURL, validates them, looks up customer folder paths from the `unified_customers` database, saves files to the DB-driven path on the local server's `/mnt/dropbox/` (internal file system), auto-creates missing customer folders, and generates background thumbnails. This provides database-driven routing that the remote server's name-based path construction does not have.

#### Key Features

| Feature | Description |
|---------|-------------|
| **Token Authentication** | Uses `.env` file for secure token storage |
| **File Type Validation** | MIME type detection + extension validation |
| **Path Sanitization** | Prevents directory traversal attacks |
| **Database Folder Lookup** | Uses `unified_customers` table for dynamic folder routing |
| **Multi-Year Support** | Finds customer folders across 2020-2026 |
| **Auto-Create Folders** | Creates new customer folders if not found in database |
| **Database Auto-Update** | Inserts new customers into `unified_customers` table |
| **Background Thumbnails** | Gallery thumbs (200x200 WebP) and mid-sized (800px WebP) via Python background script |
| **HEIC Support** | Converts iPhone HEIC images via ImageMagick |

#### Configuration Files

| File | Purpose |
|------|---------|
| `/var/www/html/upload/.env` | Auth token storage |
| `/var/www/html/upload/db_config.php` | Database connection settings |

#### Customer Folder Resolution (Priority Order)

```php
// 1. Try customer_id + job_year (most specific)
$folder = getCustomerFolderByCustomerId($customer_id, $job_year);

// 2. Fallback to customer_id (any year)
if (!$folder) {
    $folder = getCustomerFolderByCustomerId($customer_id, null);
}

// 3. Fallback to name + job_year
if (!$folder) {
    $folder = getCustomerFolderByName($first_name, $last_name, $job_year);
}

// 4. AUTO-CREATE: Create new customer folder for current year
if (!$folder) {
    $folder = createCustomerFolder($last_name, $first_name, $job_year);
    // Also inserts into unified_customers database
    insertNewCustomer($customer_id, $first_name, $last_name, $folder, $job_year);
}

// 5. Final fallback to Uploads folder (only if folder creation fails)
if (!$folder) {
    $folder = "/mnt/dropbox/Uploads/[{$customer}] [{$type}] [{$date}]/";
}
```

#### Auto-Created Folder Structure

When a new customer is created, the following structure is generated:

```
/mnt/dropbox/{YEAR} Customers/
└── {Letter}/                          # First letter of last name
    └── {LastName, FirstName}/         # New customer folder
        ├── Survey/                    # Created automatically
        └── Installation/              # Created automatically
            └── {Name}, {Type}-I, {Date}/  # Job folder with photos
```

The new customer is also added to the `unified_customers` database:

| Field | Value |
|-------|-------|
| `remote_customer_id` | From POST data |
| `first_name` | From POST data |
| `last_name` | From POST data |
| `folder_name` | "LastName, FirstName" |
| `folder_path` | Full path to customer folder |
| `folder_year` | Current year from job_date |
| `match_type` | "api_created" |
| `data_source` | "matched" |

#### Image Processing Architecture

Image processing is handled differently on each server, with both using non-blocking background processes:

**Remote Server — Background WebP for Mobile App (asynchronous):**
- Script: `/var/www/vhosts/.../photoapi/generate_webp.py` (Python 3.6 / Pillow 8.4.0)
- Triggered via `nohup /usr/local/bin/python3.6 ... > /dev/null 2>&1 &` after cURL sync
- Production PHP 7.3 GD build lacks WebP support, so Python/Pillow handles WebP conversion
- Generates WebP images in `webp/` subfolder within each job folder
- Also copies WebP to `scheduler/uploads/webp/` for database reference
- Mobile app reads images via `getimagelisting.php` → `fetch_image.php` (both scan `webp/` folder)
- Non-blocking: upload.php does NOT wait for WebP generation to complete

**Remote Server — Scheduler Website Thumbnails (on-demand):**
- CodeIgniter's `thumbnail_helper` generates JPEG thumbnails on the fly
- Thumbnails served from `scheduler/uploads/thumbnails/` (e.g., `photo_c207x207.jpeg`)
- No WebP needed — the scheduler website uses JPEG only

**Local Server — Background Python Thumbnails (asynchronous):**
- Script: `/var/www/html/upload/upload_thumb_generator.py` (Python 3 / Pillow 10.2.0)
- Triggered via `nohup` immediately after saving the original file
- API returns JSON success response instantly — no image processing blocks the response
- Generates thumbnails in the background:
  - `thumbs/`: Gallery thumbnails (200x200 WebP)
  - `thumbs/mid/`: Gallery mid-sized images (800px max WebP)
- Uses Pillow with LANCZOS resampling
- Logs to `/var/log/upload_thumb_generator.log`

```
Remote Server Flow (upload.php):
  1. Decode base64, save to REMOTE /mnt/dropbox   ← name-based path (AWS EBS)
  2. Create webp/ folder in REMOTE /mnt/dropbox    ← immediate
  3. Validate saved file (realpath)                 ← immediate
  4. Copy original to scheduler/uploads/            ← on remote server
  5. Trigger generate_webp.py                       ← BACKGROUND (nohup &)
     └─ Writes WebP to REMOTE /mnt/dropbox/.../webp/  ← mobile app reads this
     └─ Copies WebP to scheduler/uploads/webp/        ← DB reference
  6. Insert meter_files record                      ← immediate
  7. Trigger sync_to_local.py                       ← BACKGROUND (nohup &)
     └─ POST file + metadata to local server          ← runs after response sent
     └─ Logs to photoapi/logs/sync_to_local.log       ← dedicated log file
  8. echo JSON response                             ← LAST (sent immediately)
  9. Script ends → mobile app already has response  ← ~0.1-0.5s total
     ... local server sync 2-10s later              ← background process
     ... WebP appears in webp/ 1-3s later           ← background process

Local Server Flow (uploadlocallat_kuldeep.php):
  1. Validate + save to LOCAL /mnt/dropbox    ← DB-driven path (internal storage)
  2. Trigger upload_thumb_generator.py        ← BACKGROUND (nohup &)
  3. Return JSON success                      ← API response sent immediately
     --- Background processing ---
  4. Generate thumbs/ (200x200) in LOCAL /mnt/dropbox   ← local gallery use
  5. Generate thumbs/mid/ (800px) in LOCAL /mnt/dropbox  ← local gallery use
```

**Storage Architecture:** Each server has its own independent `/mnt/dropbox/`:
- **Remote** `/mnt/dropbox/` = local directory on AWS EBS volume. Used for field
  technician photos and the mobile app gallery. `upload.php` writes originals here,
  `generate_webp.py` writes WebP here, and `getimagelisting.php`/`fetch_image.php`
  read from here to serve the mobile app.
- **Local** `/mnt/dropbox/` = internal company file system. Used for office storage
  with DB-driven customer folder routing. `uploadlocallat_kuldeep.php` writes originals
  here and `upload_thumb_generator.py` generates thumbnails here.
- The cURL sync sends a **copy** of each photo from remote to local — they are
  separate files on separate storage systems.

**Who Uses What:**

| Consumer | Read Path | Served Via | Format | Generated By |
|----------|-----------|-----------|--------|-------------|
| Mobile app (gallery) | `/mnt/dropbox/.../webp/` | `getimagelisting.php` → `fetch_image.php` (remote server) | WebP | `generate_webp.py` (remote, background) |
| Scheduler website | `scheduler/uploads/thumbnails/` | CodeIgniter `thumbnail_helper` (remote server) | JPEG | `thumbnail_helper` (on-demand) |
| Local gallery | `/mnt/dropbox/.../thumbs/` and `thumbs/mid/` | Direct filesystem (local server) | WebP | `upload_thumb_generator.py` (local, background) |

**Scheduler Website Thumbnail Display:**
```html
<!-- Example: scheduler phototab display -->
<img src="scheduler/uploads/thumbnails/photo_c207x207.jpeg">   <!-- 207x207 crop -->
<a href="scheduler/uploads/thumbnails/photo_l750x0.jpeg">      <!-- 750px lightbox -->
```

**Python Script Usage:**
```bash
# Remote server: WebP for mobile app (called by upload.php in background)
/usr/local/bin/python3.6 /var/www/.../photoapi/generate_webp.py source.jpg dest.webp 80

# Local server: Gallery thumbnails (called by uploadlocallat_kuldeep.php in background)
python3 /var/www/html/upload/upload_thumb_generator.py /path/to/image.jpg /path/to/job_folder/

# Local server: Batch process all images in a folder
python3 /var/www/html/upload/upload_thumb_generator.py --folder /path/to/job_folder/
```

---

### 3. Customer Folder Mapper (`/var/opt/map_dropbox/`)

**Purpose:** Maps Dropbox customer directories to AEI database records, creating a unified lookup table for photo routing.

#### Database Schema

| Table | Records | Purpose |
|-------|---------|---------|
| `unified_customers` | 17,918 | Combined customer + folder data |
| `local_folder_customers` | 6,320 | Dropbox folder source data |
| `remote_customers` | 16,324 | Clean AEI customer data |

#### Key Metrics

- **92.4%** folder-to-remote match rate
- **5,839** customers with both folder AND remote data
- **481** folder-only (in Dropbox but not in AEI database)
- **11,598** remote-only (in AEI database but no Dropbox folder)

#### Daily Sync Process

**Cron Schedule:** `30 5 * * 1-5` (5:30 AM weekdays)

```bash
/usr/bin/python3 /var/opt/map_dropbox/scripts/daily_sync.py
```

**Sync Operations:**
1. Fetch new customers from remote AEI database
2. Scan for new Dropbox folders
3. Create customer-folder mappings
4. Generate sync report

#### Project Structure

```
/var/opt/map_dropbox/
├── src/                    # Core production scripts
│   ├── config.py           # Database & path configuration
│   ├── customer_mapper.py  # Main mapping script
│   ├── database_manager.py # Database rebuild tool
│   └── matching_engine.py  # Name matching algorithm
├── scripts/                # Utility scripts
│   ├── daily_sync.py       # Daily cron job
│   └── rebuild_database.py # Full rebuild
├── tests/                  # Validation scripts
├── docs/                   # Documentation
└── logs/                   # Runtime logs
```

---

## Recent Fixes (2026-02-04)

### Issue 1: `.env` File Permission Denied

**Symptom:** API returning authentication errors
**Root Cause:** `.env` file had `root:root` ownership with `600` permissions, Apache (www-data) couldn't read it
**Impact:** AUTH_TOKEN became empty string, all requests failed authentication

**Fix Applied:**
```bash
chown root:www-data /var/www/html/upload/.env
chmod 640 /var/www/html/upload/.env
```

**Verification:**
```bash
ls -la /var/www/html/upload/.env
# Should show: -rw-r----- 1 root www-data
```

### Issue 2: MySQL Root Access Denied

**Symptom:** HTTP 500 errors with "Access denied for user 'root'@'localhost'"
**Root Cause:** MySQL root user configured with `unix_socket` auth, www-data couldn't connect
**Impact:** Database lookups failed, all photo uploads returned 500 errors

**Fix Applied:**
```sql
-- Created dedicated database user (see CREDENTIALS.md for credentials)
CREATE USER 'upload_user'@'localhost' IDENTIFIED BY '***';
GRANT SELECT ON Schedular.unified_customers TO 'upload_user'@'localhost';
FLUSH PRIVILEGES;
```

```php
// Updated db_config.php
define('SCHEDULAR_USER', 'upload_user');  // Changed from 'root'
```

**Verification:**
```bash
sudo -u www-data php -r "
require '/var/www/html/upload/db_config.php';
\$conn = getSchedularConnection();
echo \$conn ? 'SUCCESS' : 'FAILED';
"
```

### Issue 3: Mobile App Photos Not Syncing to Local Server

**Symptom:** Photos uploaded via mobile app appeared on remote scheduler but did not sync to local server. Web uploads via scheduler site worked correctly.

**Root Cause:** The remote server's `/photoapi/upload.php` had the sync URL overwritten by a test URL:
```php
$remoteUploadUrl = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php';
$remoteUploadUrl = 'https://yislms.com/up.php';  // <-- This line overwrote the correct URL
```

**Impact:** All mobile app photo uploads were being sent to `yislms.com` instead of `upload.aeihawaii.com`, causing complete sync failure for mobile uploads.

**Fix Applied:**
```bash
# On remote server (18.225.0.90)
# Removed line 104 containing the incorrect URL
sed -i '104d' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
```

**Backups Created:**
| Location | File |
|----------|------|
| Local | `/var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/upload.php.remote.20260204` |
| Remote | `/var/www/vhosts/.../photoapi/upload.php.bak.20260204` |

**Verification:**
```bash
# Check that only correct URL remains
# Via SSH to remote server (see CREDENTIALS.md for access)
grep 'remoteUploadUrl' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
# Should show only: $remoteUploadUrl = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php';

# Monitor local server for incoming mobile uploads
tail -f /var/log/apache2/upload_access.log | grep "18.225.0.90"
```

### Before/After Comparison

**Local Server Fixes (Issues 1 & 2):**

| Metric | Before Fix | After Fix |
|--------|------------|-----------|
| HTTP Status | 500 (error) | 200 (success) |
| Response Size | ~3KB (error JSON) | ~3KB (success JSON) |
| Auth Token | Empty string | Configured (see CREDENTIALS.md) |
| DB Connection | Failed | Working |

**Mobile API Fix (Issue 3):**

| Metric | Before Fix | After Fix |
|--------|------------|-----------|
| Mobile Sync URL | `yislms.com` (wrong) | `upload.aeihawaii.com` (correct) |
| Mobile Photos Syncing | No | Yes |
| Web Photos Syncing | Yes | Yes |

---

## Database Integration

### Connection Configuration

**File:** `/var/www/html/upload/db_config.php`

| Database | Host | User | Purpose |
|----------|------|------|---------|
| Remote (AEI) | 18.225.0.90 | AEI_User | Job/customer data |
| Local Auth | localhost | aei_web_user | Token storage |
| Local Schedular | localhost | upload_user | Customer folder mappings |

See [CREDENTIALS.md](CREDENTIALS.md) for passwords.

### Key Queries

**Find Customer Folder by ID:**
```sql
SELECT folder_path FROM unified_customers
WHERE remote_customer_id = ?
AND folder_year = ?
AND folder_path IS NOT NULL
LIMIT 1
```

**Find Customer Folder by Name:**
```sql
SELECT folder_path FROM unified_customers
WHERE first_name = ? AND last_name = ?
AND folder_path IS NOT NULL
ORDER BY folder_year DESC
LIMIT 1
```

---

## API Security

### Authentication Tokens

See [CREDENTIALS.md](CREDENTIALS.md) for all authentication tokens.

| Endpoint | Storage |
|----------|---------|
| Remote `/photoapi/upload.php` | Hardcoded in PHP |
| Local `uploadlocallat_kuldeep.php` | `.env` file |

### Security Features

| Feature | Server | Implementation |
|---------|--------|---------------|
| Token Authentication | Both | Required `auth_token` parameter |
| SQL Injection Protection | Remote | `intval()` on job_id in `upload.php`, `getimagelisting.php`, `fetch_image.php`; `mysqli_real_escape_string()` on strings |
| SQL Injection Protection | Local | Prepared statements with `bind_param()` |
| MIME Type Validation | Local | `finfo_file()` detection |
| Extension Whitelist | Local | jpg, jpeg, png, gif, heic, heif |
| Path Sanitization | Local | Removes `..`, `/`, `\`, null bytes |
| Character Whitelist | Local | Alphanumeric, space, hyphen, underscore |
| Command Injection Protection | Both | `escapeshellarg()` on all exec() parameters |

### File Permissions

**Local Server:**

| File/Directory | Owner | Permissions |
|----------------|-------|-------------|
| `.env` | root:www-data | 640 |
| `uploadlocallat_kuldeep.php` | root:root | 755 |
| `/mnt/dropbox/` | www-data:www-data | 755 |

**Remote Server:**

| File/Directory | Owner | Permissions | Notes |
|----------------|-------|-------------|-------|
| `upload.php` | ec2-user:ec2-user | 755 | Apache runs as `apache` user |
| `getimagelisting.php` | ec2-user:ec2-user | 777 | Cross-year photo listing |
| `fetch_image.php` | ec2-user:ec2-user | 777 | Individual photo serving |
| `sync_to_local.py` | ec2-user:ec2-user | 755 | Executed by Apache via `exec()` |
| `generate_webp.py` | ec2-user:ec2-user | 755 | Executed by Apache via `exec()` |
| `process_retry_queue.py` | ec2-user:ec2-user | 755 | Cron-driven retry processor |
| `photoapi/logs/` | ec2-user:ec2-user | **777** | Must be writable by `apache` user |
| `photoapi/queue/` | ec2-user:ec2-user | **777** | Retry queue (Apache writes via sync_to_local.py) |
| `photoapi/queue/failed/` | ec2-user:ec2-user | **777** | Expired retry items |
| `/mnt/dropbox/{YEAR} Customers/` | varies | **777** | Must be writable by `apache` — new year dirs often created as 775 by ec2-user |

**Important:** On the remote server, Apache runs as `apache` (not `ec2-user`). Directories that Apache writes to — `/mnt/dropbox/{YEAR} Customers/`, `photoapi/logs/` — must have `777` permissions or `apache` in the owner/group. New year directories (e.g., `2027 Customers/`) may need `chmod 777` when first created.

### Firewall Requirements (Local Server)

The local server uses a layered firewall (ipset + fail2ban + UFW + nftables) with a DROP default INPUT policy. The AWS remote server IP must be whitelisted for `sync_to_local.py` to connect:

```bash
# Check if AWS IP is whitelisted
sudo ipset test trusted_whitelist 18.225.0.90

# Add if missing (required after local server reboot)
sudo ipset add trusted_whitelist 18.225.0.90
```

The whitelist is volatile (lost on reboot). The IP is persisted in `/var/www/html/security/whitelist.json` — the persistence mechanism reloads it on boot. See `/var/www/html/security/README.md` for details.

---

## File Storage Structure

### Customer Folders (Matched)

```
/mnt/dropbox/{YEAR} Customers/
└── {Letter}/                          # A-Z
    └── {LastName, FirstName}/         # Customer folder
        ├── Survey/
        │   └── {Name}, {Type}-S, {Date}/
        │       ├── photo1.jpg             # Original image
        │       ├── webp/                  # Mobile app gallery (remote generate_webp.py)
        │       │   └── {Name}{Type}-S{Date}_IMG{random}.webp
        │       └── thumbs/                # Local gallery (upload_thumb_generator.py)
        │           ├── photo1.webp        # 200x200
        │           └── mid/
        │               └── photo1.webp    # 800px max
        └── Installation/
            └── {Name}, {Type}-I, {Date}/
                ├── photo1.jpg
                ├── webp/
                │   └── {Name}{Type}-I{Date}_IMG{random}.webp
                └── thumbs/
                    ├── photo1.webp
                    └── mid/
                        └── photo1.webp
```

**Storage Notes:**
- Each server has its own **independent** `/mnt/dropbox/` — they are NOT shared
- Remote `/mnt/dropbox/` (AWS EBS): `upload.php` writes originals, `generate_webp.py` writes WebP to `webp/`
- Remote `webp/` is read by `getimagelisting.php` and `fetch_image.php` to serve the **mobile app gallery**
- Local `/mnt/dropbox/` (internal): `uploadlocallat_kuldeep.php` writes originals (DB-driven path)
- Local `thumbs/` is created by `upload_thumb_generator.py` for internal gallery use
- The cURL sync copies each photo from remote to local — separate files on separate storage

### Remote Server Additional Storage

```
/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads/
├── {filename}.jpg                     # Original (copied from /mnt/dropbox)
├── thumbnails/                        # Auto-generated by thumbnail_helper (JPEG)
│   ├── {filename}_c207x207.jpeg       # 207x207 crop
│   └── {filename}_l750x0.jpeg         # 750px lightbox
└── webp/                              # Copied from job's webp/ folder
    └── {Name}{Type}-{PhotoType}{Date}_IMG{random}.webp
```

### Uploads Folder (Unmatched)

```
/mnt/dropbox/Uploads/
└── [{Customer}] [{JobType}] [{Date}]/
    ├── photo1.jpg
    └── thumbs/
        ├── photo1.webp
        └── mid/
            └── photo1.webp
```

### Year Directories

| Year | Path | Status |
|------|------|--------|
| 2020 | `/mnt/dropbox/2020 Customers/` | Archive |
| 2021 | `/mnt/dropbox/2021 Customers/` | Archive |
| 2022 | `/mnt/dropbox/2022 Customers/` | Archive |
| 2023 | `/mnt/dropbox/2023 Customers/` | Archive |
| 2024 | `/mnt/dropbox/2024 Customers/` | Active |
| 2025 | `/mnt/dropbox/2025 Customers/` | Active |
| 2026 | `/mnt/dropbox/2026 Customers/` | Current |

---

## Maintenance & Operations

### Log Locations

| Component | Log Location |
|-----------|--------------|
| Apache Upload | `/var/log/apache2/upload_access.log` |
| Apache Errors | `/var/log/apache2/upload_error.log` |
| Local Server Sync | `photoapi/logs/sync_to_local.log` (remote server) |
| Retry Queue | `photoapi/logs/retry_queue.log` (remote server) |
| Customer Mapper | `/var/opt/map_dropbox/logs/daily_sync.log` |

### Monitoring Commands

```bash
# Watch for incoming API requests
tail -f /var/log/apache2/upload_access.log | grep "18.225.0.90"

# Check recent errors
tail -50 /var/log/apache2/upload_error.log | grep -v "cache_disk"

# Check upload statistics
grep "200" /var/log/apache2/upload_access.log | grep "uploadlocallat" | wc -l

# List recent uploads
ls -lt /mnt/dropbox/Uploads/ | head -10
```

### Database Maintenance

```bash
# Rebuild customer mappings (takes ~19 minutes)
cd /var/opt/map_dropbox
PYTHONPATH=/var/opt/map_dropbox python3 src/database_manager.py --force-rebuild

# Run daily sync manually
python3 /var/opt/map_dropbox/scripts/daily_sync.py

# Check mapping statistics
mysql -u upload_user -p'***' Schedular -e "  # See CREDENTIALS.md for password
SELECT
    COUNT(*) as total,
    SUM(CASE WHEN folder_path IS NOT NULL THEN 1 ELSE 0 END) as with_folders,
    SUM(CASE WHEN remote_customer_id IS NOT NULL THEN 1 ELSE 0 END) as with_remote
FROM unified_customers;
"
```

---

## Troubleshooting Guide

### Issue: Photos Not Syncing from Remote Server

**Check:**
1. Verify remote server is sending requests:
   ```bash
   grep "18.225.0.90" /var/log/apache2/upload_access.log | tail -10
   ```

2. Check HTTP status codes (should be 200):
   ```bash
   grep "18.225.0.90" /var/log/apache2/upload_access.log | grep "uploadlocallat" | tail -5
   ```

3. Check error log for PHP errors:
   ```bash
   grep "18.225.0.90" /var/log/apache2/upload_error.log | tail -10
   ```

### Issue: HTTP 500 Errors

**Common Causes:**
- Database connection failure
- File permission issues
- PHP fatal errors

**Check:**
```bash
# Check for PHP errors
tail -20 /var/log/apache2/upload_error.log | grep -i "fatal\|error"

# Test database connection
sudo -u www-data php -r "
require '/var/www/html/upload/db_config.php';
echo getSchedularConnection() ? 'DB OK' : 'DB FAILED';
"

# Verify .env permissions
ls -la /var/www/html/upload/.env
```

### Issue: Photos Going to Uploads Instead of Customer Folder

**Check:**
1. Verify customer exists in unified_customers:
   ```sql
   SELECT * FROM unified_customers
   WHERE first_name LIKE '%CustomerName%'
   AND folder_path IS NOT NULL;
   ```

2. Check if customer_id is being sent from remote:
   ```bash
   # Customer lookup logs are in error_log
   grep "Photo upload: Customer not found" /var/log/apache2/upload_error.log | tail -5
   ```

### Issue: Authentication Failures

**Check:**
```bash
# Verify .env file readable by www-data
sudo -u www-data cat /var/www/html/upload/.env

# Test token parsing
sudo -u www-data php -r "
\$env = parse_ini_file('/var/www/html/upload/.env');
echo 'AUTH_TOKEN: ' . (\$env['AUTH_TOKEN'] ?? 'NOT SET');
"
```

### Issue: Mobile App Timeout When Uploading Photos

**Symptom:** Mobile app shows timeout exception during photo upload.

**Root Cause History:** The remote server's `upload.php` must complete ALL processing before PHP sends the buffered JSON response to the mobile app. If any step blocks synchronously for too long, the app times out.

**Common causes:**
1. `generate_webp.py` running synchronously (was fixed by adding `nohup ... &`)
2. `sync_to_local.py` not running in background (must use `nohup ... &`)
3. A new synchronous operation was added to upload.php

**Check (via SSH to remote server — see CREDENTIALS.md for access):**
```bash
# On remote server: check for errors
tail -20 /var/log/httpd/error_log | grep -i 'webp\|generate\|sync'

# Check sync log for local server connectivity
tail -20 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log

# Verify generate_webp.py runs correctly
/usr/local/bin/python3.6 -c 'from PIL import Image; print("Pillow OK")'

# Verify requests module available for sync_to_local.py
/usr/local/bin/python3.6 -c 'import requests; print("requests OK")'
```

**Important:** Both `generate_webp.py` and `sync_to_local.py` MUST use `nohup ... > /dev/null 2>&1 &` to run in background. If the `nohup` or `&` is missing, it blocks the script. The local server sync was previously synchronous cURL, which caused 2-10s delays; it was moved to background Python in 2026-02-05.

### Issue: Mobile App Shows No Photos in Gallery

**Symptom:** Photos upload successfully but the mobile app gallery is empty.

**Root Cause:** The mobile app's `getimagelisting.php` scans the `webp/` folder. If WebP files are missing, the gallery appears empty.

**Check (via SSH to remote server — see CREDENTIALS.md for access):**
```bash
# On remote server: verify webp/ folder exists and has files for a specific job
ls -la '/mnt/dropbox/2025 Customers/J/Jain, Ankur/Installation/Jain, Ankur, SWH-I, 09-24-2025/webp/'

# Check if generate_webp.py is running (should complete within seconds)
ps aux | grep generate_webp

# Verify the webp/ folder was created by upload.php
find /mnt/dropbox/2025\ Customers/ -name 'webp' -type d | head -5
```

**Note:** As of 2026-02-05, `getimagelisting.php` scans ALL jobs for the customer (cross-year). If the gallery is empty, it means no `webp/` folders exist for any of the customer's jobs. Check if `generate_webp.py` ran successfully during upload.

### Issue: SSH Access to Remote Server

See `/var/opt/AEI_REMOTE/REMOTE_SERVER.md` for SSH connection details, key conversion, and troubleshooting.

---

## Security & Infrastructure Concerns

**Documented:** 2026-02-05

These are known infrastructure-level risks that exist independently of the application-level fixes applied in recent changes. They require planning and coordination to resolve.

### 1. PHP Version (Remote Server) — RESOLVED

| Attribute | Detail |
|-----------|--------|
| **Server** | Remote (18.225.0.90) |
| **Current Version** | PHP 7.3.30 (upgraded 2026-02-23 from PHP 5.3.29) |
| **EOL Date** | PHP 7.3 EOL: December 2021 (still unsupported, but much better than 5.3) |
| **Severity** | Medium (was High) |

PHP was upgraded from 5.3.29 to 7.3.30 on 2026-02-23 along with Apache 2.2→2.4. The DB driver was migrated from `mysql` to `mysqli`. Python/Pillow is still used for WebP generation as the production GD build lacks WebP support.

**Recommendation:** Plan further migration to PHP 8.1+ when possible. The current PHP 7.3 is EOL but significantly reduces the vulnerability surface compared to 5.3.

### 2. Permissive File Permissions (777) on Remote Server

| Directory | Current | Why |
|-----------|---------|-----|
| `photoapi/logs/` | `777` | Apache runs as `apache`, files owned by `ec2-user` |
| `photoapi/queue/` | `777` | Same — `sync_to_local.py` runs as `apache` via `exec()` |
| `photoapi/queue/failed/` | `777` | Same |
| `/mnt/dropbox/{YEAR} Customers/` | `777` | Same — `upload.php` writes via Apache |

Apache runs as the `apache` user while deployed files are owned by `ec2-user`. Since `apache` is not in the `ec2-user` group, directories that Apache writes to must be world-writable (`777`). This allows **any** user on the system to read, modify, or delete these files.

**Recommendations (in order of preference):**

1. **ACL-based (best):** Use POSIX ACLs to grant `apache` write access without opening to all users:
   ```bash
   # Grant apache user write access to specific directories
   sudo setfacl -m u:apache:rwx /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
   sudo setfacl -m u:apache:rwx /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue
   sudo setfacl -d -m u:apache:rwx /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/queue  # default for new files
   # Then tighten permissions
   sudo chmod 750 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
   ```
   Requires `acl` package and filesystem ACL support (check with `mount | grep acl`).

2. **Group-based:** Add `apache` to the `ec2-user` group (or create a shared group):
   ```bash
   sudo usermod -aG ec2-user apache
   # Then use 775 instead of 777
   sudo chmod 775 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
   ```
   Requires Apache restart. May have side effects if `ec2-user` group has other permissions.

3. **Ownership-based:** Change directory ownership to `apache:ec2-user` with `770`:
   ```bash
   sudo chown apache:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
   sudo chmod 770 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
   ```
   Apache can write (owner), ec2-user can read/write (group), others blocked.

### 3. Hardcoded Credentials in PHP Source

| File | Credential | Storage |
|------|-----------|---------|
| Remote `upload.php` | API auth token | Hardcoded in PHP source |
| Remote `sync_to_local.py` | Local server auth token | Hardcoded in Python source |
| Remote `process_retry_queue.py` | Local server auth token | Hardcoded in Python source |
| Local `uploadlocallat_kuldeep.php` | API auth token | `.env` file (correct approach) |

The local server correctly uses a `.env` file for token storage. The remote server hardcodes tokens directly in source files, which means:
- Tokens are visible in version control history
- Tokens are readable by anyone with filesystem access to the web root
- Token rotation requires editing multiple files

**Recommendation:** Move remote server tokens to a configuration file outside the web root:
```bash
# Create config outside web root (not accessible via HTTP)
sudo mkdir -p /var/www/vhosts/aeihawaii.com/config
echo "AUTH_TOKEN=aei@..." > /var/www/vhosts/aeihawaii.com/config/.env
sudo chown ec2-user:apache /var/www/vhosts/aeihawaii.com/config/.env
sudo chmod 640 /var/www/vhosts/aeihawaii.com/config/.env
```
Then load via `parse_ini_file()` in PHP or `dotenv` equivalent in Python. This matches the pattern already established on the local server.

---

## Known Issues

| Issue | Location | Impact | Workaround |
|-------|----------|--------|------------|
| **PHP 7.3 EOL** | Remote server (18.225.0.90) | PHP 7.3.30 (upgraded from 5.3.29 on 2026-02-23); still EOL but much improved | Python/Pillow still used for WebP; plan PHP 8.1+ upgrade |
| **777 permissions** | Remote `photoapi/logs/`, `queue/`, `/mnt/dropbox/` | World-writable directories due to apache/ec2-user split | Use `setfacl` or group membership — see Security Concerns section |
| **Hardcoded tokens** | Remote `upload.php`, `sync_to_local.py`, `process_retry_queue.py` | Tokens in source files, visible in VCS | Move to `.env` outside web root — see Security Concerns section |
| ~~**Year "2025" hardcoded**~~ | ~~Remote `upload.php`, `getimagelisting.php`, `fetch_image.php`~~ | **FIXED 2026-02-04** — all three now derive year from `job_date` via `date('Y', strtotime($job_date))` | N/A |
| ~~**`phototab.php` no WebP**~~ | ~~Remote `phototab.php` controller~~ | **FIXED 2026-02-05** — restored background WebP via `generate_webp.py` (`nohup &`), dynamic year, correct Python path | N/A |
| ~~**Cross-year photos not visible**~~ | ~~Remote `getimagelisting.php`~~ | **FIXED 2026-02-05** — now queries all jobs by `customer_id`, scans each job's `webp/` folder, returns combined listing with per-image `job_id` and metadata | N/A |
| ~~**`getimagelisting.php` auth bypass**~~ | ~~Remote `getimagelisting.php`~~ | **FIXED 2026-02-05** — removed hardcoded `$data['auth_token']='aei@89806849'` that bypassed auth validation | N/A |
| ~~**SQL injection in read endpoints**~~ | ~~Remote `getimagelisting.php`, `fetch_image.php`~~ | **FIXED 2026-02-05** — added `intval()` on `job_id` (was `preg_replace` / `basename()`) | N/A |
| **Date format mismatch** | Local `Y-m-d` vs Remote `m-d-Y` | Photos on local vs remote are in differently-named date folders | `folder_path` stores the correct path at write time; migration of existing local folders deferred |
| ~~**Photos lost on job reschedule/delete**~~ | ~~Remote `getimagelisting.php`, `fetch_image.php`~~ | **FIXED 2026-02-05** — `meter_files.folder_path` now stores the absolute path at upload time, so photos are found even if the job is later unscheduled/rescheduled/deleted. Falls back to path reconstruction for pre-enhancement rows. | N/A |
| ~~**Survey path spacing bug (local)**~~ | ~~Local `uploadlocallat_kuldeep.php`~~ | **FIXED 2026-02-05** — Local server wrote `-S,{date}` (no space), remote wrote `-S, {date}` (with space). All 4 instances on local server corrected to include the space. | N/A |
| **`getimagelisting.php` legacy path** | Remote `getimagelisting.php` | Also scans `/mnt/aeiserver/{jobId}/` which appears unused (link format fixed 2026-02-05) | No impact — folder doesn't exist, scan is silently skipped |

---

## Related Documentation

| Document | Location |
|----------|----------|
| **General Documentation** | |
| Credentials | `DOCS/CREDENTIALS.md` |
| Photo System Integration | `DOCS/PHOTO_SYSTEM_INTEGRATION.md` |
| Mobile API Restore Guide | `DOCS/MOBILE_API_RESTORE_GUIDE.md` |
| QA Test Suite | `QA/README.md` |
| **Enhancement Documentation** | |
| Async Sync | `ENHANCEMENTS/PHOTO-003_ASYNC_SYNC/ASYNC_SYNC_ENHANCEMENT.md` |
| Customer ID Lookup | `ENHANCEMENTS/PHOTO-001_CUSTOMER_ID_LOOKUP/REMOTE_UPLOAD_CHANGES.md` |
| Photo Folder Path | `ENHANCEMENTS/PHOTO-005_PHOTO_FOLDER_PATH/PHOTO_FOLDER_PATH_ENHANCEMENT.md` |
| Cross-Year Photo Listing | `ENHANCEMENTS/PHOTO-008_CROSS_YEAR_PHOTO_LISTING/CROSS_YEAR_PHOTO_LISTING_ENHANCEMENT.md` |
| Stuck Process Detection | `ENHANCEMENTS/PHOTO-004_STUCK_PROCESS_DETECTION/STUCK_PROCESS_DETECTION_ENHANCEMENT.md` |
| Chunked Upload (archived) | `ARCHIVE/CHUNKED_UPLOAD/CHUNKED_UPLOAD_ENHANCEMENT.md` |
| **External References** | |
| Remote Server / SSH | `/var/opt/AEI_REMOTE/REMOTE_SERVER.md` |
| Map Dropbox README | `/var/opt/map_dropbox/README.md` |
| Local Server Security | `/var/www/html/security/README.md` |
| Project README | `README.md` |

---

## Change History

| Date | Change | Author |
|------|--------|--------|
| 2026-02-05 | **Photo folder path persistence**: Added `folder_path VARCHAR(512)` to `meter_files` table. `upload.php` stores the absolute path at upload time. `getimagelisting.php` and `fetch_image.php` use stored path when available, fall back to path reconstruction for legacy rows. Photos now survive job reschedule/unschedule/delete. See `ENHANCEMENTS/PHOTO-005_PHOTO_FOLDER_PATH/PHOTO_FOLDER_PATH_ENHANCEMENT.md` | System |
| 2026-02-05 | **Survey path spacing fix**: Fixed 4 instances in `uploadlocallat_kuldeep.php` where Survey folder name was `-S,{date}` (missing space after comma). Now matches remote server format `-S, {date}`. | System |
| 2026-02-04 | **Async background WebP**: Remote `upload.php` generates WebP via background `generate_webp.py` (`nohup &`), eliminating mobile app timeout | System |
| 2026-02-04 | Local server reverted to JSON-only response; no WebP responsibility for remote server | System |
| 2026-02-04 | Fixed Python path on remote server: `python3` not symlinked, must use `/usr/local/bin/python3.6` | System |
| 2026-02-04 | Fixed photo_type bug: WebP filenames were hardcoded "-S", now use dynamic `$photo_type` | System |
| 2026-02-04 | Added SQL security to `meter_files` insert: `intval()`, `mysqli_real_escape_string()`, dynamic `filesize()` | System |
| 2026-02-04 | Documentation: corrected `/mnt/dropbox/` architecture — two independent storage systems (remote AWS EBS + local internal), mobile app read flow, and each server's storage responsibilities | System |
| 2026-02-04 | Fixed hardcoded "2025" year in `upload.php`, `getimagelisting.php`, `fetch_image.php` — now derives year from `job_date` | System |
| 2026-02-04 | Added `CURLOPT_CONNECTTIMEOUT` (5s) to local server sync — prevents local server outage from blocking mobile app response | System |
| 2026-02-04 | Added cURL error logging for local server sync failures | System |
| 2026-02-05 | **Cross-year photo listing**: `getimagelisting.php` rewritten to query all jobs for a customer by `customer_id`, scan each job's `webp/` folder across years, and return combined listing with per-image `job_id`, `job_type`, `job_date`, `photo_type` metadata. Fixed auth bypass (hardcoded token), SQL injection (`intval()`), and broken `/mnt/aeiserver/` link format. `fetch_image.php` updated with `intval()` on job_id, early input validation, dead code removal. Backward-compatible response format. | System |
| 2026-02-05 | **Security & infrastructure concerns documented**: PHP 5.3 EOL risk, 777 permissions (with ACL/group remediation options), hardcoded credentials (with `.env` migration path). Added to Known Issues table. | System |
| 2026-02-05 | **Store-and-forward retry queue**: `sync_to_local.py` now queues failed syncs as JSON in `photoapi/queue/`. New `process_retry_queue.py` (cron every 15 min) retries up to 10 times (~2.5 hrs), then moves to `queue/failed/`. Source file must exist to queue (no retry for deleted photos). Log at `photoapi/logs/retry_queue.log` | System |
| 2026-02-05 | **Chunked upload reverted**: 3-phase chunked upload protocol was implemented then reverted due to fail2ban banning the AWS IP during multi-request sequences. Archived to `ARCHIVE/CHUNKED_UPLOAD/`. | System |
| 2026-02-05 | **Async local server sync**: Replaced synchronous cURL with background `sync_to_local.py` (Python/requests via `nohup &`). Mobile app response time reduced from 2-10s to ~0.5s. Echo JSON moved to last line. Dedicated log at `photoapi/logs/sync_to_local.log` | System |
| 2026-02-05 | **Deployment fixes**: `photoapi/logs/` and `/mnt/dropbox/2026 Customers/` set to `chmod 777` (Apache runs as `apache` user, not `ec2-user`). AWS IP `18.225.0.90` added to local server `trusted_whitelist` ipset (required for background sync). Created QA test suite (`QA/test_upload_pipeline.py` — 10-step E2E test, 10/10 PASS). | System |
| 2026-02-05 | `phototab.php` WebP generation updated to background `generate_webp.py` via `nohup &`, dynamic year, `/usr/local/bin/python3.6` — required for mobile app `webp/` folder listing | System |
| 2026-02-04 | Moved local thumbnail generation to background Python (`upload_thumb_generator.py`); local API returns JSON immediately | System |
| 2026-02-04 | Added auto-create customer folder feature: creates new folders and updates database when customer not found | System |
| 2026-02-04 | Fixed mobile API sync: removed incorrect `yislms.com` URL from remote `/photoapi/upload.php` | System |
| 2026-02-04 | Fixed .env permissions, created upload_user for MySQL | System |
| 2026-02-04 | Documented mobile app API endpoints (`getimagelisting.php`, `fetch_image.php`) and their `webp/` dependency | System |
| 2026-01-30 | Added customer_id to remote server sync, integrated unified_customers DB | Developer |
| 2026-01-30 | Reorganized map_dropbox project structure | Developer |

---

*Document generated: 2026-02-05*
