# Mobile API (photoapi/upload.php) Restore Guide

**Date:** 2026-02-04 (Original)
**Updated:** 2026-02-05 (Async sync — see critical note below)
**Issue:** Mobile app photo uploads not syncing to local server

---

## CRITICAL: Sync Architecture Change (2026-02-05)

> **The synchronous cURL sync code was intentionally removed from `upload.php` on 2026-02-05.**
>
> The current `upload.php` does NOT contain `curl_init()`, `curl_exec()`, or `$remoteUploadUrl`.
> This is **correct and expected** — local server sync is now handled by `sync_to_local.py`
> running as a background process via `nohup ... &`.
>
> **DO NOT** restore an older version of `upload.php` that contains synchronous cURL code.
> Doing so will reintroduce 2-10 second blocking delays on every mobile app upload and
> may cause mobile app timeouts.
>
> See [ASYNC_SYNC_ENHANCEMENT.md](../ENHANCEMENTS/PHOTO-003_ASYNC_SYNC/ASYNC_SYNC_ENHANCEMENT.md) for full details on the change.

---

## Summary

Photos uploaded via the mobile app appear on the remote scheduler but do NOT sync to the local server. Web uploads via the scheduler site work correctly.

---

## Current Architecture (2026-02-05+)

```
upload.php (fast operations only):
  1. Decode + save image                    → /mnt/dropbox/{YEAR} Customers/
  2. Copy to scheduler/uploads/             → for thumbnail_helper
  3. generate_webp.py in BACKGROUND         → nohup & (WebP for mobile gallery)
  4. INSERT meter_files                     → database record
  5. sync_to_local.py in BACKGROUND         → nohup & (local server copy)
  6. echo JSON response                     → LAST LINE (sent immediately)
```

**Key files on remote server:**
| File | Purpose |
|------|---------|
| `upload.php` | Main upload handler — no cURL code |
| `sync_to_local.py` | Background Python script for local server sync |
| `generate_webp.py` | Background Python script for WebP generation |
| `logs/sync_to_local.log` | Sync log (created by sync_to_local.py) |

---

## Step 1: Diagnose the Issue

### Check Sync Log (First — Most Informative)

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -20 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log"
```

**Interpret results:**
- `INFO OK photo.jpg -> HTTP 200` = sync working
- `WARNING FAIL ... HTTP 500` = local server error (check local Apache logs)
- `ERROR ... ConnectionError` = network/firewall issue (see Step 2)
- No log file / empty = sync_to_local.py not running (see Step 3)

### Check Local Server for Incoming Requests

```bash
# On local server — look for requests from AWS IP
grep "18.225.0.90" /var/log/apache2/upload_access.log | tail -10
```

### Check PHP Error Logs on Remote

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -50 /var/log/httpd/error_log | grep -i 'upload.php\|photoapi\|sync'"
```

---

## Step 2: Fix Network/Firewall Issues

The most common cause of sync failure is the local server firewall blocking the AWS IP.

```bash
# Check if AWS IP is whitelisted on LOCAL server
sudo ipset test trusted_whitelist 18.225.0.90

# Add if missing (required after local server reboot)
sudo ipset add trusted_whitelist 18.225.0.90

# Verify connectivity from remote to local
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "curl -sk -o /dev/null -w '%{http_code}' https://upload.aeihawaii.com/"
# Should return 200 or 405 (any response = reachable)
```

**Note:** The ipset whitelist is volatile — lost on reboot. The IP is persisted in `/var/www/html/security/whitelist.json` and reloaded on boot.

---

## Step 3: Fix sync_to_local.py Issues

### Verify sync_to_local.py Exists and Is Executable

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py"
# Should be: -rwxr-xr-x ec2-user:ec2-user
```

### Verify Python and requests Module

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "/usr/local/bin/python3.6 -c 'import requests; print(\"OK\")'"
```

### Verify logs/ Directory Is Writable

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/"
# Must be drwxrwxrwx (777) — Apache runs as 'apache' user, not ec2-user
```

If missing or wrong permissions:
```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo mkdir -p /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chmod 777 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs
"
```

### Verify upload.php Invokes sync_to_local.py

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "grep 'sync_to_local' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
# Should show: $syncScript = __DIR__ . '/sync_to_local.py';
```

### Redeploy sync_to_local.py if Missing

```bash
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/sync_to_local.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
"
```

---

## Step 4: Fix upload.php Issues

### Verify Current upload.php Is the Async Version

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "grep -c 'curl_init\|curl_exec' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
# Should return: 0 (no cURL code)

ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "grep -c 'sync_to_local.py' /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
# Should return: 2+ (script reference + command construction)
```

**If upload.php has `curl_init` but no `sync_to_local.py`:** Someone restored an old version. Redeploy the current version:

```bash
# Backup whatever is on the server
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
      /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.$(date +%Y%m%d%H%M)"

# Deploy current version
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/upload.php \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  php -l /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"
```

### Check File Permissions

```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
# Should be owned by ec2-user:ec2-user
```

---

## Step 5: Test Mobile Upload

After fixing, test by uploading a photo via the mobile app and check:

1. **Remote server** — Photo appears in scheduler
2. **Sync log** — Check for successful sync:
   ```bash
   ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
     "tail -5 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log"
   ```
3. **Local server logs** — Check for incoming request:
   ```bash
   tail -f /var/log/apache2/upload_access.log | grep "18.225.0.90"
   ```
4. **Local storage** — Photo appears in customer folder or Uploads
5. **E2E QA test:**
   ```bash
   python3 /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/QA/test_upload_pipeline.py
   ```

---

## Emergency Rollback (Use With Caution)

> **Warning:** Rolling back to the synchronous cURL version will reintroduce 2-10 second delays
> on every mobile app upload. Only use this if the async sync is completely broken AND you cannot
> fix the underlying issue (missing Python module, etc.).

```bash
# Check if pre-async backup exists
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.20260205"

# Rollback (SLOWER but functional — has synchronous cURL)
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php \
     /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_rollback.$(date +%Y%m%d%H%M)
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php.bak.pre_async_sync.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
"
```

**After rollback:** The `sync_to_local.py` file can remain on disk — it's only invoked if `upload.php` calls it. To restore the async version later, redeploy from the project:

```bash
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/upload.php \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/upload.php"
```

---

## Local Server Compatibility

The local server (`uploadlocallat_kuldeep.php`) is backwards compatible with both sync methods:

| Field | Required | Behavior if Missing |
|-------|----------|---------------------|
| `auth_token` | Yes | Rejects request |
| `file` | Yes | Rejects request |
| `customer_id` | No | Falls back to name matching |
| `job_type` | No | Uses default |
| `first_name` | No | Uses fallback |
| `last_name` | No | Uses fallback |

---

## Quick Reference

| Version | Location | Sync Method | Has customer_id? |
|---------|----------|-------------|------------------|
| **Current (async)** | `REMOTE/photoapi/upload.php` | Background `sync_to_local.py` | YES |
| Pre-async backup | Remote: `upload.php.bak.pre_async_sync.20260205` | Synchronous cURL (SLOW) | YES |
| Original | `backup/upload.php.original.20260130` | Synchronous cURL (SLOW) | NO |

### How to Identify Which Version Is Deployed

| Check | Current (async) | Old (sync cURL) |
|-------|----------------|-----------------|
| `grep 'curl_init' upload.php` | No matches | Found |
| `grep 'sync_to_local' upload.php` | Found | No matches |
| `echo` position | Last line before `?>` | Middle of script |

---

## Version History

| Date | Change |
|------|--------|
| 2026-02-04 | Original guide — cURL-based sync diagnostics |
| 2026-02-05 | **Major update** — rewrote for async `sync_to_local.py` architecture. Removed cURL verification steps. Added critical warning against restoring old cURL-based versions. |

---

*Updated: 2026-02-05*
