#!/usr/bin/env python3
"""
Process the retry queue for failed local server syncs.
Reads JSON files from queue/, retries POST to local server,
moves to queue/failed/ on max retries exceeded.

Designed to run via cron every 15 minutes:
  */15 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py > /dev/null 2>&1

Requires: requests (pip3 install requests)
"""

import os
import sys
import json
import glob
import shutil
import logging
from datetime import datetime

UPLOAD_URL = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php'
AUTH_TOKEN = 'remote_token'
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'retry_queue.log')
QUEUE_DIR = os.path.join(SCRIPT_DIR, 'queue')
FAILED_DIR = os.path.join(QUEUE_DIR, 'failed')
CONNECT_TIMEOUT = 10
READ_TIMEOUT = 60
MAX_RETRIES = 10


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def sync_file(file_path, file_name, job_id, customer_id, job_type,
              last_name, first_name, job_date, photo_type):
    """POST file and metadata to local server.

    Returns (success, error_reason) tuple.
    """
    import requests

    try:
        with open(file_path, 'rb') as f:
            files = {'file': (file_name, f)}
            data = {
                'auth_token':  AUTH_TOKEN,
                'file_name':   file_name,
                'job_id':      job_id,
                'customer_id': customer_id,
                'job_type':    job_type,
                'last_name':   last_name,
                'first_name':  first_name,
                'job_date':    job_date,
                'photo_type':  photo_type,
            }
            resp = requests.post(
                UPLOAD_URL,
                files=files,
                data=data,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )

        if resp.status_code == 200:
            return True, None
        else:
            return False, "HTTP %d body=%s" % (resp.status_code, resp.text[:200])

    except Exception as e:
        return False, "%s: %s" % (type(e).__name__, str(e))


def process_queue():
    """Scan queue/ for JSON files and retry each one."""
    if not os.path.isdir(QUEUE_DIR):
        return

    pattern = os.path.join(QUEUE_DIR, '*.json')
    queue_files = sorted(glob.glob(pattern))

    if not queue_files:
        return

    logging.info("Processing %d queued item(s)", len(queue_files))

    for queue_path in queue_files:
        filename = os.path.basename(queue_path)

        try:
            with open(queue_path, 'r') as f:
                payload = json.load(f)
        except (ValueError, IOError) as e:
            logging.error("SKIP %s -> cannot read JSON: %s", filename, str(e))
            continue

        retry_count = payload.get('retry_count', 0)
        file_path = payload.get('file_path', '')
        file_name = payload.get('file_name', '')

        # Max retries exceeded -> move to failed/
        if retry_count >= MAX_RETRIES:
            if not os.path.isdir(FAILED_DIR):
                os.makedirs(FAILED_DIR, mode=0o777, exist_ok=True)
            failed_path = os.path.join(FAILED_DIR, filename)
            shutil.move(queue_path, failed_path)
            logging.warning("EXPIRED %s -> moved to failed/ after %d retries (last error: %s)",
                            file_name, retry_count, payload.get('last_error', 'unknown'))
            continue

        # Source file gone -> skip (nothing to send)
        if not os.path.isfile(file_path):
            os.remove(queue_path)
            logging.info("SKIP %s -> source file no longer exists, removing from queue", file_name)
            continue

        # Attempt retry
        success, error_reason = sync_file(
            file_path=file_path,
            file_name=file_name,
            job_id=payload.get('job_id', ''),
            customer_id=payload.get('customer_id', ''),
            job_type=payload.get('job_type', ''),
            last_name=payload.get('last_name', ''),
            first_name=payload.get('first_name', ''),
            job_date=payload.get('job_date', ''),
            photo_type=payload.get('photo_type', ''),
        )

        if success:
            os.remove(queue_path)
            logging.info("RETRY OK %s -> succeeded on retry %d (queued %s)",
                         file_name, retry_count + 1, payload.get('queued_at', ''))
        else:
            # Update retry count and error, rewrite JSON
            payload['retry_count'] = retry_count + 1
            payload['last_error'] = error_reason or 'Unknown error'
            payload['last_retry'] = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            with open(queue_path, 'w') as f:
                json.dump(payload, f, indent=2)
            logging.warning("RETRY FAIL %s -> attempt %d/%d: %s",
                            file_name, retry_count + 1, MAX_RETRIES, error_reason)

    logging.info("Queue processing complete")


if __name__ == "__main__":
    setup_logging()
    process_queue()
