<?php
header('Content-Type: application/json');

// Load database configuration for unified_customers lookup
require_once __DIR__ . '/db_config.php';

// Load environment variables (create .env file with AUTH_TOKEN=your_secure_token_here)
if (file_exists(__DIR__ . '/.env')) {
    $env = parse_ini_file(__DIR__ . '/.env');
    define('AUTH_TOKEN', $env['AUTH_TOKEN'] ?? '');
} else {
    // Fallback - CHANGE THIS IN PRODUCTION
    define('AUTH_TOKEN', 'remote_token');
}

// --- Chunked upload constants ---
define('CHUNKS_DIR', __DIR__ . '/.chunks');
define('CHUNK_MAX_AGE', 48 * 3600);  // 48 hours

// --- Allowed MIME types and extensions (shared by single and chunked paths) ---
$allowedMimeTypes = [
    'image/jpeg',
    'image/jpg',
    'image/png',
    'image/gif',
    'image/heic',
    'image/heif'
];
$allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'heic', 'heif'];

/**
 * Look up customer folder path by remote customer_id and job year
 * This is the preferred method - uses ID-based matching
 * Returns the folder_path if found, null otherwise
 */
function getCustomerFolderByCustomerId($customer_id, $job_year = null) {
    if (empty($customer_id)) {
        return null;
    }

    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("Database connection failed for customer lookup");
        return null;
    }

    // If job_year provided, try to match that year first
    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE remote_customer_id = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ii", $customer_id, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    // Fallback: get most recent folder for this customer
    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE remote_customer_id = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        $conn->close();
        return null;
    }

    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

/**
 * Look up customer folder path by name (fallback method)
 * Returns the folder_path if found, null otherwise
 */
function getCustomerFolderByName($first_name, $last_name, $job_year = null) {
    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("Database connection failed for customer lookup");
        return null;
    }

    // If job_year provided, try to match that year first
    if ($job_year) {
        $stmt = $conn->prepare(
            "SELECT folder_path FROM unified_customers
             WHERE first_name = ? AND last_name = ?
             AND folder_year = ?
             AND folder_path IS NOT NULL
             LIMIT 1"
        );
        if ($stmt) {
            $stmt->bind_param("ssi", $first_name, $last_name, $job_year);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $stmt->close();
                $conn->close();
                return $row['folder_path'];
            }
            $stmt->close();
        }
    }

    // Fallback: get most recent folder matching name
    $stmt = $conn->prepare(
        "SELECT folder_path FROM unified_customers
         WHERE first_name = ? AND last_name = ?
         AND folder_path IS NOT NULL
         ORDER BY folder_year DESC
         LIMIT 1"
    );

    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        $conn->close();
        return null;
    }

    $stmt->bind_param("ss", $first_name, $last_name);
    $stmt->execute();
    $result = $stmt->get_result();

    $folder_path = null;
    if ($row = $result->fetch_assoc()) {
        $folder_path = $row['folder_path'];
    }

    $stmt->close();
    $conn->close();

    return $folder_path;
}

/**
 * Create a new customer folder structure for the current year
 * Returns the customer folder path if successful, null otherwise
 */
function createCustomerFolder($last_name, $first_name, $job_year) {
    $base_path = '/mnt/dropbox';

    // Get first letter of last name for letter folder
    $letter = strtoupper(substr($last_name, 0, 1));
    if (!preg_match('/[A-Z]/', $letter)) {
        error_log("Invalid last name for folder creation: $last_name");
        return null;
    }

    // Build folder paths
    $year_folder = $base_path . '/' . $job_year . ' Customers';
    $letter_folder = $year_folder . '/' . $letter;
    $customer_folder = $letter_folder . '/' . $last_name . ', ' . $first_name;

    // Create year folder if needed
    if (!is_dir($year_folder)) {
        if (!mkdir($year_folder, 0755, true)) {
            error_log("Failed to create year folder: $year_folder");
            return null;
        }
    }

    // Create letter folder if needed
    if (!is_dir($letter_folder)) {
        if (!mkdir($letter_folder, 0755, true)) {
            error_log("Failed to create letter folder: $letter_folder");
            return null;
        }
    }

    // Create customer folder if needed
    if (!is_dir($customer_folder)) {
        if (!mkdir($customer_folder, 0755, true)) {
            error_log("Failed to create customer folder: $customer_folder");
            return null;
        }
        error_log("Created new customer folder: $customer_folder");
    }

    return $customer_folder;
}

/**
 * Insert a new customer record into unified_customers database
 * Returns true on success, false on failure
 */
function insertNewCustomer($customer_id, $first_name, $last_name, $folder_path, $folder_year) {
    $conn = getSchedularConnection();
    if (!$conn) {
        error_log("Database connection failed for customer insert");
        return false;
    }

    $folder_name = $last_name . ', ' . $first_name;

    $stmt = $conn->prepare(
        "INSERT INTO unified_customers
         (remote_customer_id, first_name, last_name, folder_name, folder_path, folder_year, match_type, data_source, created_date)
         VALUES (?, ?, ?, ?, ?, ?, 'api_created', 'matched', NOW())"
    );

    if (!$stmt) {
        error_log("Prepare failed for customer insert: " . $conn->error);
        $conn->close();
        return false;
    }

    $stmt->bind_param("isssis", $customer_id, $first_name, $last_name, $folder_name, $folder_path, $folder_year);
    $result = $stmt->execute();

    if (!$result) {
        error_log("Failed to insert customer: " . $stmt->error);
    } else {
        error_log("Inserted new customer into unified_customers: $folder_name (ID: $customer_id, Year: $folder_year)");
    }

    $stmt->close();
    $conn->close();

    return $result;
}

// Sanitize and validate all input parameters to prevent path traversal
function sanitize_path_component($input, $fieldName = 'input') {
    if (empty($input)) {
        echo json_encode([
            "status" => "error",
            "message" => "Missing required field: $fieldName"
        ]);
        exit;
    }

    // Remove any path traversal attempts
    $sanitized = str_replace(['..', '/', '\\', "\0"], '', $input);

    // Only allow alphanumeric, spaces, hyphens, underscores
    $sanitized = preg_replace('/[^A-Za-z0-9 _\-]/', '', $sanitized);

    // Trim whitespace
    $sanitized = trim($sanitized);

    if (empty($sanitized)) {
        echo json_encode([
            "status" => "error",
            "message" => "Invalid characters in field: $fieldName"
        ]);
        exit;
    }

    return $sanitized;
}

/**
 * Resolve the final destination folder for a photo.
 * Extracted from the main flow so it can be called by both single-upload and chunked finalize.
 *
 * Returns array with keys: 'final_folder', 'lookup_method', 'safe_file_name'
 * or echoes JSON error and exits on validation failure.
 */
function resolvePhotoDestination($post_data) {
    global $allowedExtensions;

    // Validate and sanitize date
    if (empty($post_data['job_date'])) {
        echo json_encode(["status" => "error", "message" => "Missing job_date"]);
        exit;
    }
    $timestamp = strtotime($post_data['job_date']);
    if ($timestamp === false) {
        echo json_encode(["status" => "error", "message" => "Invalid date format"]);
        exit;
    }
    $job_date = date("Y-m-d", $timestamp);
    $job_year = (int)date("Y", $timestamp);

    $customer_id = isset($post_data['customer_id']) ? intval($post_data['customer_id']) : null;

    $first_name = sanitize_path_component($post_data['first_name'] ?? '', 'first_name');
    $last_name = sanitize_path_component($post_data['last_name'] ?? '', 'last_name');
    $customer_name = $last_name . ", " . $first_name;
    $job_type_text = sanitize_path_component($post_data['job_type'] ?? '', 'job_type');

    if (empty($post_data['photo_type']) || !in_array($post_data['photo_type'], ['I', 'S'])) {
        echo json_encode(["status" => "error", "message" => "Invalid photo_type. Must be 'I' or 'S'"]);
        exit;
    }
    $photo_type = $post_data['photo_type'];

    // Sanitize file_name for safe filesystem use
    $safe_file_name = null;
    if (!empty($post_data['file_name'])) {
        $passedPathInfo = pathinfo(basename($post_data['file_name']));
        $passedExt = strtolower($passedPathInfo['extension'] ?? '');
        if (in_array($passedExt, $allowedExtensions)) {
            $safeBase = preg_replace('/[^A-Za-z0-9_\-]/', '_', $passedPathInfo['filename']);
            $safe_file_name = $safeBase . '.' . $passedExt;
        }
    }

    // Look up customer folder from unified_customers database
    $uploadDir_customer = null;
    $lookup_method = 'none';

    if ($customer_id) {
        $uploadDir_customer = getCustomerFolderByCustomerId($customer_id, $job_year);
        if ($uploadDir_customer) {
            $lookup_method = 'customer_id';
        }
    }

    if (!$uploadDir_customer) {
        $uploadDir_customer = getCustomerFolderByName($first_name, $last_name, $job_year);
        if ($uploadDir_customer) {
            $lookup_method = 'name';
        }
    }

    if ($uploadDir_customer && is_dir($uploadDir_customer)) {
        $uploadDir_customer_survey = $uploadDir_customer . "/Survey";
        if (!is_dir($uploadDir_customer_survey)) {
            mkdir($uploadDir_customer_survey, 0755, true);
        }
        $uploadDir_customer_install = $uploadDir_customer . "/Installation";
        if (!is_dir($uploadDir_customer_install)) {
            mkdir($uploadDir_customer_install, 0755, true);
        }

        if ($photo_type == "I") {
            $final_folder = $uploadDir_customer_install . "/" . $customer_name . ", " . $job_type_text . "-I, " . $job_date . '/';
        } else {
            $final_folder = $uploadDir_customer_survey . "/" . $customer_name . ", " . $job_type_text . "-S," . $job_date . '/';
        }
        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }
    } else {
        $uploadDir_customer = createCustomerFolder($last_name, $first_name, $job_year);

        if ($uploadDir_customer) {
            insertNewCustomer($customer_id, $first_name, $last_name, $uploadDir_customer, $job_year);
            $lookup_method = 'api_created';

            $uploadDir_customer_survey = $uploadDir_customer . "/Survey";
            if (!is_dir($uploadDir_customer_survey)) {
                mkdir($uploadDir_customer_survey, 0755, true);
            }
            $uploadDir_customer_install = $uploadDir_customer . "/Installation";
            if (!is_dir($uploadDir_customer_install)) {
                mkdir($uploadDir_customer_install, 0755, true);
            }

            if ($photo_type == "I") {
                $final_folder = $uploadDir_customer_install . "/" . $customer_name . ", " . $job_type_text . "-I, " . $job_date . '/';
            } else {
                $final_folder = $uploadDir_customer_survey . "/" . $customer_name . ", " . $job_type_text . "-S," . $job_date . '/';
            }
            if (!is_dir($final_folder)) {
                mkdir($final_folder, 0755, true);
            }

            error_log("Photo upload: Created new customer folder - customer_id=$customer_id, name=$first_name $last_name, path=$uploadDir_customer");
        } else {
            $uploads_base = '/mnt/dropbox/Uploads';
            $final_folder = $uploads_base . '/[' . $customer_name . '] [' . $job_type_text . '] [' . $job_date . ']/';
            if (!is_dir($final_folder)) {
                mkdir($final_folder, 0755, true);
            }
            error_log("Photo upload: Failed to create customer folder, using Uploads - customer_id=$customer_id, name=$first_name $last_name");
        }
    }

    return [
        'final_folder' => $final_folder,
        'lookup_method' => $lookup_method,
        'safe_file_name' => $safe_file_name,
    ];
}

// ============================================================================
// Chunked upload helper functions
// ============================================================================

function ensureChunksDir() {
    if (!is_dir(CHUNKS_DIR)) {
        if (!mkdir(CHUNKS_DIR, 0755, true)) {
            error_log("Failed to create chunks directory: " . CHUNKS_DIR);
            return false;
        }
    }
    return true;
}

function isValidFileId($file_id) {
    return preg_match('/^[0-9a-f]{40}$/', $file_id) === 1;
}

function getChunkMetaPath($file_id) {
    return CHUNKS_DIR . '/' . $file_id . '.meta';
}

function getChunkPartPath($file_id, $index) {
    return CHUNKS_DIR . '/' . $file_id . '.part.' . intval($index);
}

function readChunkMeta($file_id) {
    $path = getChunkMetaPath($file_id);
    if (!file_exists($path)) {
        return null;
    }
    $json = file_get_contents($path);
    if ($json === false) {
        return null;
    }
    return json_decode($json, true);
}

function writeChunkMeta($file_id, $meta) {
    $path = getChunkMetaPath($file_id);
    return file_put_contents($path, json_encode($meta, JSON_PRETTY_PRINT)) !== false;
}

/**
 * Probabilistic cleanup of abandoned chunk sessions (older than CHUNK_MAX_AGE).
 * Called with ~1% chance on each init request to avoid dedicated cron.
 */
function cleanupAbandonedPhotoChunks() {
    if (!is_dir(CHUNKS_DIR)) {
        return;
    }
    $now = time();
    $metaFiles = glob(CHUNKS_DIR . '/*.meta');
    if (!$metaFiles) {
        return;
    }
    foreach ($metaFiles as $metaPath) {
        if (($now - filemtime($metaPath)) > CHUNK_MAX_AGE) {
            $meta = json_decode(file_get_contents($metaPath), true);
            $fileId = basename($metaPath, '.meta');
            if (isValidFileId($fileId)) {
                $totalChunks = isset($meta['total_chunks']) ? intval($meta['total_chunks']) : 0;
                for ($i = 0; $i < $totalChunks; $i++) {
                    $partPath = getChunkPartPath($fileId, $i);
                    if (file_exists($partPath)) {
                        unlink($partPath);
                    }
                }
                unlink($metaPath);
                error_log("Chunked upload cleanup: removed abandoned session $fileId");
            }
        }
    }
}

// ============================================================================
// Chunked upload handlers
// ============================================================================

function handlePhotoChunkInit() {
    // Probabilistic cleanup (~1% chance)
    if (mt_rand(1, 100) === 1) {
        cleanupAbandonedPhotoChunks();
    }

    if (!ensureChunksDir()) {
        echo json_encode(["status" => "error", "message" => "Server error: cannot create chunks directory"]);
        exit;
    }

    $file_id = $_POST['file_id'] ?? '';
    if (!isValidFileId($file_id)) {
        echo json_encode(["status" => "error", "message" => "Invalid file_id (must be 40 hex chars)"]);
        exit;
    }

    $file_name = $_POST['file_name'] ?? '';
    $file_size = isset($_POST['file_size']) ? intval($_POST['file_size']) : 0;
    $total_chunks = isset($_POST['total_chunks']) ? intval($_POST['total_chunks']) : 0;
    $chunk_size = isset($_POST['chunk_size']) ? intval($_POST['chunk_size']) : 0;

    if (empty($file_name) || $file_size <= 0 || $total_chunks <= 0 || $chunk_size <= 0) {
        echo json_encode(["status" => "error", "message" => "Missing or invalid init parameters"]);
        exit;
    }

    // Check for existing session (resume support)
    $existing = readChunkMeta($file_id);
    if ($existing) {
        // Return which chunks are already received
        $chunks_received = [];
        for ($i = 0; $i < $existing['total_chunks']; $i++) {
            if (file_exists(getChunkPartPath($file_id, $i))) {
                $chunks_received[] = $i;
            }
        }
        echo json_encode([
            "status" => "ok",
            "message" => "Resuming chunked upload",
            "file_id" => $file_id,
            "chunks_received" => $chunks_received
        ]);
        exit;
    }

    // Create new session — store all metadata needed for finalize
    $meta = [
        'file_id' => $file_id,
        'file_name' => $file_name,
        'file_size' => $file_size,
        'total_chunks' => $total_chunks,
        'chunk_size' => $chunk_size,
        'job_id' => $_POST['job_id'] ?? '',
        'customer_id' => $_POST['customer_id'] ?? '',
        'job_type' => $_POST['job_type'] ?? '',
        'last_name' => $_POST['last_name'] ?? '',
        'first_name' => $_POST['first_name'] ?? '',
        'job_date' => $_POST['job_date'] ?? '',
        'photo_type' => $_POST['photo_type'] ?? '',
        'created_at' => date('Y-m-d H:i:s'),
    ];

    if (!writeChunkMeta($file_id, $meta)) {
        echo json_encode(["status" => "error", "message" => "Failed to create upload session"]);
        exit;
    }

    echo json_encode([
        "status" => "ok",
        "message" => "Chunked upload session created",
        "file_id" => $file_id,
        "chunks_received" => []
    ]);
    exit;
}

function handlePhotoChunkUpload() {
    $file_id = $_POST['file_id'] ?? '';
    if (!isValidFileId($file_id)) {
        echo json_encode(["status" => "error", "message" => "Invalid file_id"]);
        exit;
    }

    $meta = readChunkMeta($file_id);
    if (!$meta) {
        echo json_encode(["status" => "error", "message" => "No upload session found for file_id"]);
        exit;
    }

    $chunk_index = isset($_POST['chunk_index']) ? intval($_POST['chunk_index']) : -1;
    if ($chunk_index < 0 || $chunk_index >= $meta['total_chunks']) {
        echo json_encode(["status" => "error", "message" => "Invalid chunk_index"]);
        exit;
    }

    if (!isset($_FILES['chunk']) || $_FILES['chunk']['error'] !== UPLOAD_ERR_OK) {
        echo json_encode(["status" => "error", "message" => "Chunk file not received"]);
        exit;
    }

    $partPath = getChunkPartPath($file_id, $chunk_index);
    if (!move_uploaded_file($_FILES['chunk']['tmp_name'], $partPath)) {
        echo json_encode(["status" => "error", "message" => "Failed to save chunk"]);
        exit;
    }

    echo json_encode([
        "status" => "ok",
        "message" => "Chunk $chunk_index received",
        "chunk_index" => $chunk_index
    ]);
    exit;
}

function handlePhotoChunkFinalize() {
    global $allowedMimeTypes, $allowedExtensions;

    $file_id = $_POST['file_id'] ?? '';
    if (!isValidFileId($file_id)) {
        echo json_encode(["status" => "error", "message" => "Invalid file_id"]);
        exit;
    }

    $meta = readChunkMeta($file_id);
    if (!$meta) {
        echo json_encode(["status" => "error", "message" => "No upload session found for file_id"]);
        exit;
    }

    // Verify all chunks present
    $missing = [];
    for ($i = 0; $i < $meta['total_chunks']; $i++) {
        if (!file_exists(getChunkPartPath($file_id, $i))) {
            $missing[] = $i;
        }
    }
    if (!empty($missing)) {
        echo json_encode([
            "status" => "error",
            "message" => "Missing chunks: " . implode(', ', $missing),
            "missing_chunks" => $missing
        ]);
        exit;
    }

    // Reassemble chunks into a temporary file
    $assembledPath = CHUNKS_DIR . '/' . $file_id . '.assembled';
    $out = fopen($assembledPath, 'wb');
    if (!$out) {
        echo json_encode(["status" => "error", "message" => "Failed to create assembled file"]);
        exit;
    }

    for ($i = 0; $i < $meta['total_chunks']; $i++) {
        $partPath = getChunkPartPath($file_id, $i);
        $in = fopen($partPath, 'rb');
        if (!$in) {
            fclose($out);
            unlink($assembledPath);
            echo json_encode(["status" => "error", "message" => "Failed to read chunk $i"]);
            exit;
        }
        while (!feof($in)) {
            fwrite($out, fread($in, 8192));
        }
        fclose($in);
    }
    fclose($out);

    // Verify assembled file size
    $actualSize = filesize($assembledPath);
    if ($actualSize != $meta['file_size']) {
        unlink($assembledPath);
        echo json_encode([
            "status" => "error",
            "message" => "File size mismatch: expected {$meta['file_size']}, got $actualSize"
        ]);
        exit;
    }

    // MIME type validation on assembled file (same as single-upload path)
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $detectedMimeType = finfo_file($finfo, $assembledPath);
    finfo_close($finfo);

    if (!in_array($detectedMimeType, $allowedMimeTypes)) {
        unlink($assembledPath);
        echo json_encode([
            "status" => "error",
            "message" => "Invalid file type. Only images allowed (JPEG, PNG, GIF, HEIC)"
        ]);
        exit;
    }

    // Extension validation
    $pathInfo = pathinfo(basename($meta['file_name']));
    $extension = strtolower($pathInfo['extension'] ?? '');
    if (!in_array($extension, $allowedExtensions)) {
        unlink($assembledPath);
        echo json_encode([
            "status" => "error",
            "message" => "Invalid file extension. Allowed: jpg, jpeg, png, gif, heic, heif"
        ]);
        exit;
    }

    // Resolve destination folder (same logic as single-upload)
    $dest = resolvePhotoDestination($meta);
    $final_folder = $dest['final_folder'];
    $lookup_method = $dest['lookup_method'];

    // Use the safe file name from resolvePhotoDestination, or derive from file_name
    if ($dest['safe_file_name']) {
        $safeFileName = $dest['safe_file_name'];
    } else {
        $safeBase = preg_replace('/[^A-Za-z0-9_\-]/', '_', $pathInfo['filename']);
        $safeFileName = $safeBase . '.' . $extension;
    }

    $destinationPath = $final_folder . $safeFileName;

    // copy() + unlink() — .chunks/ is local SSD, /mnt/dropbox/ is CIFS mount (cross-filesystem)
    if (!copy($assembledPath, $destinationPath)) {
        unlink($assembledPath);
        echo json_encode(["status" => "error", "message" => "Failed to save assembled file to destination"]);
        exit;
    }
    unlink($assembledPath);

    // Cleanup chunk parts and metadata
    for ($i = 0; $i < $meta['total_chunks']; $i++) {
        $partPath = getChunkPartPath($file_id, $i);
        if (file_exists($partPath)) {
            unlink($partPath);
        }
    }
    unlink(getChunkMetaPath($file_id));

    // Trigger background thumbnail generation (same as single-upload path)
    $thumbScript = __DIR__ . '/upload_thumb_generator.py';
    $escapedSource = escapeshellarg($destinationPath);
    $escapedFolder = escapeshellarg($final_folder);
    $thumbCommand = "nohup python3 " . escapeshellarg($thumbScript) . " $escapedSource $escapedFolder > /dev/null 2>&1 &";
    exec($thumbCommand);

    error_log("Chunked upload success: $safeFileName -> $final_folder (lookup: $lookup_method, file_id: $file_id, chunks: {$meta['total_chunks']})");

    echo json_encode([
        "status" => "success",
        "message" => "Chunked upload assembled and saved",
        "path" => $safeFileName,
        "folder" => $final_folder,
        "lookup_method" => $lookup_method
    ]);
    exit;
}

// ============================================================================
// Auth validation (applies to ALL requests — single upload and chunked)
// ============================================================================

// Validate token
if (!isset($_POST['auth_token']) || $_POST['auth_token'] !== AUTH_TOKEN) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid or missing auth token"
    ]);
    exit;
}

// ============================================================================
// Routing: chunked upload handlers (action field present) vs single upload
// ============================================================================

if (isset($_POST['action'])) {
    switch ($_POST['action']) {
        case 'photo_chunk_init':
            handlePhotoChunkInit();
            break;
        case 'photo_chunk_upload':
            handlePhotoChunkUpload();
            break;
        case 'photo_chunk_finalize':
            handlePhotoChunkFinalize();
            break;
        default:
            echo json_encode([
                "status" => "error",
                "message" => "Unknown action: " . $_POST['action']
            ]);
            exit;
    }
    // Handlers call exit; this is a safety net
    exit;
}

// ============================================================================
// Existing single-upload flow (unchanged — no 'action' field in POST)
// ============================================================================

// Check if file was uploaded
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    echo json_encode([
        "status" => "error",
        "message" => "File upload error or file not received"
    ]);
    exit;
}

// Validate file type BEFORE processing
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$detectedMimeType = finfo_file($finfo, $_FILES['file']['tmp_name']);
finfo_close($finfo);

if (!in_array($detectedMimeType, $allowedMimeTypes)) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid file type. Only images allowed (JPEG, PNG, GIF, HEIC)"
    ]);
    exit;
}

// Sanitize filename
$originalFileName = basename($_FILES['file']['name']);
$pathInfo = pathinfo($originalFileName);
$extension = strtolower($pathInfo['extension'] ?? '');

if (!in_array($extension, $allowedExtensions)) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid file extension. Allowed: jpg, jpeg, png, gif, heic, heif"
    ]);
    exit;
}

$safeBaseName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $pathInfo['filename']);
$safeFileName = $safeBaseName . '.' . $extension;

// Optional: Use passed file_name field if needed
if (isset($_POST['file_name'])) {
    $passedPathInfo = pathinfo(basename($_POST['file_name']));
    $passedExt = strtolower($passedPathInfo['extension'] ?? '');

    if (in_array($passedExt, $allowedExtensions)) {
        $safeBaseName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $passedPathInfo['filename']);
        $safeFileName = $safeBaseName . '.' . $passedExt;
    }
}

// Validate and sanitize date
if (!isset($_POST['job_date']) || empty($_POST['job_date'])) {
    echo json_encode([
        "status" => "error",
        "message" => "Missing job_date"
    ]);
    exit;
}

$timestamp = strtotime($_POST['job_date']);
if ($timestamp === false) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid date format"
    ]);
    exit;
}

$job_date = date("Y-m-d", $timestamp);
$job_year = (int)date("Y", $timestamp);

// Get optional customer_id and job_id from remote server
$customer_id = isset($_POST['customer_id']) ? intval($_POST['customer_id']) : null;
$job_id = isset($_POST['job_id']) ? intval($_POST['job_id']) : null;

// Sanitize all path components
$first_name = sanitize_path_component($_POST['first_name'] ?? '', 'first_name');
$last_name = sanitize_path_component($_POST['last_name'] ?? '', 'last_name');
$customer_name = $last_name . ", " . $first_name;
$job_type_text = sanitize_path_component($_POST['job_type'] ?? '', 'job_type');

// Validate photo_type
if (!isset($_POST['photo_type']) || !in_array($_POST['photo_type'], ['I', 'S'])) {
    echo json_encode([
        "status" => "error",
        "message" => "Invalid photo_type. Must be 'I' or 'S'"
    ]);
    exit;
}
$photo_type = $_POST['photo_type'];

// Look up customer folder from unified_customers database
// Priority: 1) customer_id + year, 2) customer_id any year, 3) name + year, 4) name any year
$uploadDir_customer = null;
$lookup_method = 'none';

if ($customer_id) {
    // Try customer_id lookup first (most reliable)
    $uploadDir_customer = getCustomerFolderByCustomerId($customer_id, $job_year);
    if ($uploadDir_customer) {
        $lookup_method = 'customer_id';
    }
}

if (!$uploadDir_customer) {
    // Fallback to name-based lookup
    $uploadDir_customer = getCustomerFolderByName($first_name, $last_name, $job_year);
    if ($uploadDir_customer) {
        $lookup_method = 'name';
    }
}

// Verify the database path exists on filesystem
if ($uploadDir_customer && is_dir($uploadDir_customer)) {
    // Customer found in database and folder exists - use normal folder structure
    $uploadDir_customer_survey = $uploadDir_customer . "/Survey";
    if (!is_dir($uploadDir_customer_survey)) {
        mkdir($uploadDir_customer_survey, 0755, true);
    }

    $uploadDir_customer_install = $uploadDir_customer . "/Installation";
    if (!is_dir($uploadDir_customer_install)) {
        mkdir($uploadDir_customer_install, 0755, true);
    }

    if ($photo_type == "I") {
        $final_folder = $uploadDir_customer_install . "/" . $customer_name . ", " . $job_type_text . "-I, " . $job_date . '/';
        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }
    } else {
        $final_folder = $uploadDir_customer_survey . "/" . $customer_name . ", " . $job_type_text . "-S," . $job_date . '/';
        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }
    }
} else {
    // Customer NOT found in database OR folder doesn't exist
    // Create new customer folder for current year and update database
    $uploadDir_customer = createCustomerFolder($last_name, $first_name, $job_year);

    if ($uploadDir_customer) {
        // Successfully created customer folder - add to database
        insertNewCustomer($customer_id, $first_name, $last_name, $uploadDir_customer, $job_year);
        $lookup_method = 'api_created';

        // Create Survey and Installation subfolders
        $uploadDir_customer_survey = $uploadDir_customer . "/Survey";
        if (!is_dir($uploadDir_customer_survey)) {
            mkdir($uploadDir_customer_survey, 0755, true);
        }

        $uploadDir_customer_install = $uploadDir_customer . "/Installation";
        if (!is_dir($uploadDir_customer_install)) {
            mkdir($uploadDir_customer_install, 0755, true);
        }

        // Determine final folder based on photo type
        if ($photo_type == "I") {
            $final_folder = $uploadDir_customer_install . "/" . $customer_name . ", " . $job_type_text . "-I, " . $job_date . '/';
        } else {
            $final_folder = $uploadDir_customer_survey . "/" . $customer_name . ", " . $job_type_text . "-S," . $job_date . '/';
        }

        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }

        error_log("Photo upload: Created new customer folder - customer_id=$customer_id, name=$first_name $last_name, path=$uploadDir_customer");
    } else {
        // Fallback to Uploads folder if folder creation fails
        $uploads_base = '/mnt/dropbox/Uploads';
        $final_folder = $uploads_base . '/[' . $customer_name . '] [' . $job_type_text . '] [' . $job_date . ']/';

        if (!is_dir($final_folder)) {
            mkdir($final_folder, 0755, true);
        }

        error_log("Photo upload: Failed to create customer folder, using Uploads - customer_id=$customer_id, name=$first_name $last_name, job_id=$job_id");
    }
}
 $destinationPath = $final_folder . $safeFileName;

if (move_uploaded_file($_FILES['file']['tmp_name'], $destinationPath)) {
    // Trigger background thumbnail generation via Python script
    // Generates webp/, thumbs/ (200x200), and thumbs/mid/ (800px) asynchronously
    // Remote server now generates its own WebP - no need to return binary data
    $thumbScript = __DIR__ . '/upload_thumb_generator.py';
    $escapedSource = escapeshellarg($destinationPath);
    $escapedFolder = escapeshellarg($final_folder);
    $thumbCommand = "nohup python3 " . escapeshellarg($thumbScript) . " $escapedSource $escapedFolder > /dev/null 2>&1 &";
    exec($thumbCommand);

    error_log("Photo upload success: $safeFileName -> $final_folder (lookup: $lookup_method, thumb generation triggered)");

    echo json_encode([
        "status" => "success",
        "message" => "File uploaded successfully",
        "path" => $safeFileName,
        "folder" => $final_folder,
        "lookup_method" => $lookup_method
    ]);
} else {
    echo json_encode([
        "status" => "error",
        "message" => "Failed to save uploaded file"
    ]);
}
?>
