#!/usr/bin/env python3
"""
AEI Photo Upload Pipeline — End-to-End QA Test

Tests the complete upload flow:
  Phase 1 — Upload & Response:
    1. POST test image to remote API (upload.php)
    2. Verify JSON response + timing
  Phase 2 — Remote Artifacts:
    3. Verify file saved to /mnt/dropbox/ on remote
    4. Verify WebP generated in webp/ folder
    5. Verify copy to scheduler/uploads/
    6. Verify WebP copy to scheduler/uploads/webp/
    7. Verify meter_files DB insert
    8. Verify sync_to_local.py triggered (log entry)
  Phase 3 — Local Server:
    9. Verify file received on local server
    10. Verify thumbnails generated on local server
  Phase 4 — Async Sync Enhancement (ASYNC_SYNC_ENHANCEMENT.md):
    11. No synchronous cURL in upload.php (curl_init/exec/close removed)
    12. Echo JSON is last PHP statement before ?>
    13. sync_to_local.py exists with correct permissions (755)
    14. process_retry_queue.py exists and is executable (755)
    15. queue/ and queue/failed/ directories exist with correct permissions
    16. Cron entry for process_retry_queue.py (ec2-user, */15)
    17. logs/ directory exists and is writable (777)
    18. Python 3.6 requests module available
    19. Queue empty after successful upload (no false queuing)
  Phase 5 — Chunked Upload Path (CHUNKED_UPLOAD_ENHANCEMENT.md):
    20. Upload large (>= 3MB) test image to remote API
    21. Large file saved to /mnt/dropbox/ on remote
    22. Sync used chunked mode (CHUNKED START + CHUNKED OK in log)
    23. Large file received on local server
    24. Local thumbnails generated for large file
    25. .chunks directory configured on local server
  Phase 6 — Photo Folder Path Enhancement (PHOTO_FOLDER_PATH_ENHANCEMENT.md):
    26. folder_path column exists in meter_files
    27. folder_path populated for test upload
    28. folder_path matches actual file location
    29. getimagelisting.php returns test image
    30. fetch_image.php serves image via folder_path
    31. Survey path spacing fix deployed (local)

Usage:
  python3 test_upload_pipeline.py                    # Full test (default job)
  python3 test_upload_pipeline.py --job-id 108946    # Test with specific job
  python3 test_upload_pipeline.py --skip-cleanup      # Leave test data for inspection
  python3 test_upload_pipeline.py --remote-only       # Skip local server checks

Prerequisites:
  - pip3 install requests Pillow paramiko
  - SSH key at /root/.ssh/aei_remote.pem
  - Network access to aeihawaii.com (remote) and upload.aeihawaii.com (local)
"""

import argparse
import base64
import io
import json
import os
import re
import sys
import time
import subprocess

try:
    import requests
    from PIL import Image
except ImportError:
    print("ERROR: pip3 install requests Pillow")
    sys.exit(1)

# ── Configuration ──────────────────────────────────────────────────

REMOTE_API_URL = "https://aeihawaii.com/photoapi/upload.php"
REMOTE_AUTH_TOKEN = "aei@89806849"
SSH_KEY = "/root/.ssh/aei_remote.pem"
SSH_USER = "Julian"
SSH_HOST = "18.225.0.90"
DB_USER = "schedular"
DB_PASS = "M1gif9!6"
DB_NAME = "mandhdesign_schedular"
TEST_FILENAME = "qa_pipeline_test.jpg"
TEST_FILENAME_CHUNKED = "qa_chunked_test.jpg"
DEFAULT_JOB_ID = "108946"
REMOTE_PHOTOAPI = "/var/www/vhosts/aeihawaii.com/httpdocs/photoapi"
REMOTE_SCHEDULER = "/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/uploads"
CHUNK_THRESHOLD = 3 * 1024 * 1024  # 3MB — must match sync_to_local.py

# ── Helpers ────────────────────────────────────────────────────────

class Colors:
    PASS = "\033[92m"
    FAIL = "\033[91m"
    WARN = "\033[93m"
    INFO = "\033[94m"
    RESET = "\033[0m"
    BOLD = "\033[1m"

def ssh_cmd(cmd, timeout=30):
    """Run command on remote server via SSH."""
    result = subprocess.run(
        ["ssh", "-i", SSH_KEY, "-o", "ConnectTimeout=10",
         "-o", "StrictHostKeyChecking=no",
         f"{SSH_USER}@{SSH_HOST}", cmd],
        capture_output=True, text=True, timeout=timeout
    )
    return result.stdout.strip(), result.stderr.strip(), result.returncode

def result_line(step_num, name, passed, detail=""):
    """Print a formatted test result line."""
    status = f"{Colors.PASS}PASS{Colors.RESET}" if passed else f"{Colors.FAIL}FAIL{Colors.RESET}"
    detail_str = f" — {detail}" if detail else ""
    print(f"  [{status}] {step_num:>2}. {name}{detail_str}")
    return passed

def create_test_image():
    """Generate a small test JPEG and return base64."""
    img = Image.new("RGB", (200, 200), color="blue")
    buf = io.BytesIO()
    img.save(buf, format="JPEG", quality=80)
    return base64.b64encode(buf.getvalue()).decode()

def create_large_test_image():
    """Generate a >= 3MB test JPEG for chunked upload testing.

    Uses random pixel data to prevent JPEG compression from reducing
    the file size below the 3MB chunked upload threshold.
    Returns (base64_data, file_size_bytes).
    """
    width, height = 2200, 2200
    data = os.urandom(width * height * 3)
    img = Image.frombytes("RGB", (width, height), data)
    buf = io.BytesIO()
    img.save(buf, format="JPEG", quality=95)
    raw = buf.getvalue()
    return base64.b64encode(raw).decode(), len(raw)

# ── Test Steps ─────────────────────────────────────────────────────

def test_upload(job_id, filename=None, image_data=None):
    """POST test image to remote API and return response + timing."""
    if filename is None:
        filename = TEST_FILENAME
    if image_data is None:
        image_data = create_test_image()

    payload = {
        "auth_token": REMOTE_AUTH_TOKEN,
        "file_name": filename,
        "image_data": image_data,
        "job_id": str(job_id),
    }

    start = time.time()
    try:
        resp = requests.post(REMOTE_API_URL, json=payload, timeout=60, verify=True)
    except Exception as e:
        return None, 0, str(e)
    elapsed = time.time() - start

    try:
        data = resp.json()
    except Exception:
        data = {"raw": resp.text[:200]}

    return data, elapsed, None


def test_remote_checks(job_id):
    """Steps 3-8: Verify all remote server artifacts."""
    results = {}

    # Build single SSH command that checks everything
    cmd = f"""
    echo '===FILES==='
    find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' 2>/dev/null | head -1

    echo '===WEBP==='
    JOBDIR=$(find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' -exec dirname {{}} \\; 2>/dev/null | head -1)
    [ -n "$JOBDIR" ] && find "$JOBDIR/webp/" -name '*.webp' 2>/dev/null | head -1 || echo ''

    echo '===SCHEDULER==='
    ls '{REMOTE_SCHEDULER}/{TEST_FILENAME}' 2>/dev/null || echo ''

    echo '===SCHEDULER_WEBP==='
    ls -t {REMOTE_SCHEDULER}/webp/*.webp 2>/dev/null | head -1

    echo '===METER_FILES==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SELECT id,job_id,file_size,webpfilename FROM meter_files WHERE unique_filename='{TEST_FILENAME}' ORDER BY id DESC LIMIT 1;" 2>/dev/null

    echo '===SYNC_LOG==='
    cat {REMOTE_PHOTOAPI}/logs/sync_to_local.log 2>/dev/null | grep '{TEST_FILENAME}' | tail -1
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    # Parse sections
    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["file_saved"] = section("FILES")
    results["webp_generated"] = section("WEBP")
    results["scheduler_copy"] = section("SCHEDULER")
    results["scheduler_webp"] = section("SCHEDULER_WEBP")
    results["meter_files"] = section("METER_FILES")
    results["sync_log"] = section("SYNC_LOG")

    return results


def test_chunked_remote_checks(job_id):
    """Steps 21-22: Verify large file on remote and chunked sync mode."""
    cmd = f"""
    echo '===LARGE_FILE==='
    find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME_CHUNKED}' 2>/dev/null | head -1

    echo '===CHUNKED_LOG==='
    grep '{TEST_FILENAME_CHUNKED}' {REMOTE_PHOTOAPI}/logs/sync_to_local.log 2>/dev/null | tail -5
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    return {
        "file_saved": section("LARGE_FILE"),
        "chunked_log": section("CHUNKED_LOG"),
    }


def test_local_checks(customer_id, last_name, first_name, job_date, test_filename=None):
    """Steps 9-10 (and 23-24): Verify file received on local server with thumbnails.

    Uses the local unified_customers database (/var/opt/map_dropbox) to
    find the customer folder path, then checks that specific directory
    instead of scanning the entire /mnt/dropbox/ filesystem.
    """
    if test_filename is None:
        test_filename = TEST_FILENAME

    results = {}

    # Query unified_customers DB for the customer's folder path
    # This mirrors the lookup that uploadlocallat_kuldeep.php performs
    LOCAL_DB_USER = "upload_user"
    LOCAL_DB_PASS = "P@55w02d778899"
    LOCAL_DB_NAME = "Schedular"
    folder_paths = []

    # Priority 1: customer_id lookup (ordered by year DESC, matching upload handler)
    try:
        db_result = subprocess.run(
            ["mysql", "-u", LOCAL_DB_USER, f"-p{LOCAL_DB_PASS}", LOCAL_DB_NAME, "-N", "-e",
             f"SELECT folder_path FROM unified_customers "
             f"WHERE remote_customer_id = {customer_id} AND folder_path IS NOT NULL "
             f"ORDER BY folder_year DESC;"],
            capture_output=True, text=True, timeout=5
        )
        for line in db_result.stdout.strip().split("\n"):
            if line.strip():
                folder_paths.append(line.strip())
    except Exception:
        pass

    # Priority 2: name lookup fallback
    if not folder_paths:
        try:
            db_result = subprocess.run(
                ["mysql", "-u", LOCAL_DB_USER, f"-p{LOCAL_DB_PASS}", LOCAL_DB_NAME, "-N", "-e",
                 f"SELECT folder_path FROM unified_customers "
                 f"WHERE first_name = '{first_name}' AND last_name = '{last_name}' "
                 f"AND folder_path IS NOT NULL ORDER BY folder_year DESC;"],
                capture_output=True, text=True, timeout=5
            )
            for line in db_result.stdout.strip().split("\n"):
                if line.strip():
                    folder_paths.append(line.strip())
        except Exception:
            pass

    # Also check auto-created year folder and Uploads fallback
    job_year = job_date[:4] if job_date else "2026"
    idx = last_name[0].upper() if last_name else "?"
    folder_paths.append(f"/mnt/dropbox/{job_year} Customers/{idx}")
    folder_paths.append("/mnt/dropbox/Uploads")

    results["db_lookup"] = folder_paths[0] if folder_paths else "NO DB MATCH"

    # Search each candidate folder (fast — targeted directories only)
    results["file_found"] = ""
    for folder in folder_paths:
        if not os.path.isdir(folder):
            continue
        try:
            find_result = subprocess.run(
                ["find", folder, "-maxdepth", "6", "-name", test_filename],
                capture_output=True, text=True, timeout=10
            )
            if find_result.stdout.strip():
                results["file_found"] = find_result.stdout.strip().split("\n")[0]
                break
        except Exception:
            continue

    # Check for thumbnails near the found file
    if results["file_found"]:
        parent = os.path.dirname(results["file_found"])
        webp_name = test_filename.replace(".jpg", ".webp")
        thumb_path = os.path.join(parent, "thumbs", webp_name)
        mid_path = os.path.join(parent, "thumbs", "mid", webp_name)
        results["thumbnail"] = thumb_path if os.path.isfile(thumb_path) else ""
        results["mid_thumb"] = mid_path if os.path.isfile(mid_path) else ""
    else:
        results["thumbnail"] = ""
        results["mid_thumb"] = ""

    return results


def test_async_enhancement_checks():
    """Steps 11-19: Verify async sync enhancement deployment (ASYNC_SYNC_ENHANCEMENT.md)."""
    results = {}

    cmd = f"""
    echo '===NO_CURL==='
    grep -c 'curl_init\\|curl_exec\\|curl_close' {REMOTE_PHOTOAPI}/upload.php 2>/dev/null || true

    echo '===ECHO_LAST==='
    tac {REMOTE_PHOTOAPI}/upload.php | grep -m1 -n '[a-zA-Z]' | head -1

    echo '===SYNC_PERMS==='
    stat -c '%a %U:%G' {REMOTE_PHOTOAPI}/sync_to_local.py 2>/dev/null || echo 'NOT_FOUND'

    echo '===RETRY_PERMS==='
    stat -c '%a %U:%G' {REMOTE_PHOTOAPI}/process_retry_queue.py 2>/dev/null || echo 'NOT_FOUND'

    echo '===QUEUE_DIR==='
    stat -c '%a' {REMOTE_PHOTOAPI}/queue/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===QUEUE_FAILED==='
    stat -c '%a' {REMOTE_PHOTOAPI}/queue/failed/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===CRON==='
    sudo crontab -u ec2-user -l 2>/dev/null | grep process_retry || echo 'NOT_FOUND'

    echo '===LOGS_DIR==='
    stat -c '%a' {REMOTE_PHOTOAPI}/logs/ 2>/dev/null || echo 'NOT_FOUND'

    echo '===PYTHON_REQUESTS==='
    /usr/local/bin/python3.6 -c 'import requests; print("OK")' 2>/dev/null || echo 'MISSING'

    echo '===QUEUE_COUNT==='
    grep -l '{TEST_FILENAME}' {REMOTE_PHOTOAPI}/queue/*.json 2>/dev/null | wc -l
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["no_curl"] = section("NO_CURL")
    results["echo_last"] = section("ECHO_LAST")
    results["sync_perms"] = section("SYNC_PERMS")
    results["retry_perms"] = section("RETRY_PERMS")
    results["queue_dir"] = section("QUEUE_DIR")
    results["queue_failed"] = section("QUEUE_FAILED")
    results["cron"] = section("CRON")
    results["logs_dir"] = section("LOGS_DIR")
    results["python_requests"] = section("PYTHON_REQUESTS")
    results["queue_count"] = section("QUEUE_COUNT")

    return results


def test_chunked_infra_checks():
    """Step 25: Verify .chunks directory on local server."""
    results = {}
    chunks_dir = "/var/www/html/upload/.chunks"

    results["chunks_exists"] = os.path.isdir(chunks_dir)

    if results["chunks_exists"]:
        stat_info = os.stat(chunks_dir)
        results["chunks_perms"] = oct(stat_info.st_mode)[-3:]
        # Check .htaccess exists for deny rule
        htaccess = os.path.join(chunks_dir, ".htaccess")
        results["htaccess"] = os.path.isfile(htaccess)
    else:
        results["chunks_perms"] = ""
        results["htaccess"] = False

    return results


def test_folder_path_checks():
    """Steps 26-28: Verify folder_path column and population in meter_files."""
    results = {}

    cmd = f"""
    echo '===COLUMN==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SHOW COLUMNS FROM meter_files LIKE 'folder_path';" 2>/dev/null

    echo '===FOLDER_PATH==='
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e "SELECT folder_path FROM meter_files WHERE unique_filename='{TEST_FILENAME}' ORDER BY id DESC LIMIT 1;" 2>/dev/null
    """

    out, err, rc = ssh_cmd(cmd, timeout=15)
    sections = out.split("===")

    def section(name):
        for i, s in enumerate(sections):
            if s.strip() == name and i + 1 < len(sections):
                return sections[i + 1].strip()
        return ""

    results["column_info"] = section("COLUMN")
    results["folder_path"] = section("FOLDER_PATH")

    return results


def test_image_api_checks(job_id):
    """Steps 29-30: Verify getimagelisting.php and fetch_image.php."""
    results = {}

    # Step 29: POST to getimagelisting.php
    try:
        listing_url = "https://aeihawaii.com/photoapi/getimagelisting.php"
        payload = {
            "auth_token": REMOTE_AUTH_TOKEN,
            "job_id": str(job_id),
        }
        resp = requests.post(listing_url, json=payload, timeout=30, verify=True)
        try:
            data = resp.json()
        except Exception:
            data = None
        results["listing_status"] = resp.status_code
        results["listing_data"] = data
    except Exception as e:
        results["listing_status"] = 0
        results["listing_data"] = None
        results["listing_error"] = str(e)

    # Step 30: GET one image link from the listing
    results["fetch_status"] = 0
    results["fetch_content_type"] = ""
    results["fetch_size"] = 0

    if results["listing_data"] and isinstance(results["listing_data"], list):
        # Find a link URL in the response
        link_url = None
        for entry in results["listing_data"]:
            if isinstance(entry, dict) and entry.get("link"):
                link_url = entry["link"]
                break

        if link_url:
            try:
                img_resp = requests.get(link_url, timeout=30, verify=True)
                results["fetch_status"] = img_resp.status_code
                results["fetch_content_type"] = img_resp.headers.get("Content-Type", "")
                results["fetch_size"] = len(img_resp.content)
                results["fetch_url"] = link_url
            except Exception as e:
                results["fetch_error"] = str(e)
        else:
            results["fetch_error"] = "no link URL in listing response"
    else:
        results["fetch_error"] = "no valid listing data"

    return results


def test_survey_spacing_fix():
    """Step 31: Verify survey path spacing fix on local server."""
    results = {}
    local_file = "/var/www/html/upload/uploadlocallat_kuldeep.php"

    if not os.path.isfile(local_file):
        results["file_exists"] = False
        return results

    results["file_exists"] = True

    try:
        with open(local_file, "r") as f:
            content = f.read()

        # Count instances of '-S,' without space after comma (the bug)
        # We need to match '-S,' NOT followed by a space
        bad_matches = re.findall(r'-S,(?! )', content)
        results["bad_count"] = len(bad_matches)

        # Count instances of '-S, ' (the fix — with space)
        good_matches = re.findall(r'-S, ', content)
        results["good_count"] = len(good_matches)
    except Exception as e:
        results["bad_count"] = -1
        results["good_count"] = -1
        results["error"] = str(e)

    return results


def cleanup_remote():
    """Remove test artifacts from remote server."""
    cmd = f"""
    # Clean small test file
    JOBDIR=$(find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME}' -exec dirname {{}} \\; 2>/dev/null | head -1)
    [ -n "$JOBDIR" ] && sudo rm -rf "$JOBDIR"

    # Clean large (chunked) test file
    JOBDIR2=$(find '/mnt/dropbox/2026 Customers/' '/mnt/dropbox/2025 Customers/' -name '{TEST_FILENAME_CHUNKED}' -exec dirname {{}} \\; 2>/dev/null | head -1)
    [ -n "$JOBDIR2" ] && sudo rm -rf "$JOBDIR2"

    sudo rm -f '{REMOTE_SCHEDULER}/{TEST_FILENAME}'
    sudo rm -f '{REMOTE_SCHEDULER}/{TEST_FILENAME_CHUNKED}'
    sudo rm -f {REMOTE_SCHEDULER}/webp/*$(date +%m-%d-%Y)*.webp
    mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -e "DELETE FROM meter_files WHERE unique_filename IN ('{TEST_FILENAME}', '{TEST_FILENAME_CHUNKED}');" 2>/dev/null
    sudo truncate -s 0 {REMOTE_PHOTOAPI}/logs/sync_to_local.log 2>/dev/null
    for qf in {REMOTE_PHOTOAPI}/queue/*.json; do
        [ -f "$qf" ] && (grep -q '{TEST_FILENAME}' "$qf" 2>/dev/null || grep -q '{TEST_FILENAME_CHUNKED}' "$qf" 2>/dev/null) && sudo rm -f "$qf"
    done
    echo 'Remote cleanup done'
    """
    ssh_cmd(cmd, timeout=15)


def cleanup_local():
    """Remove test artifacts from local server."""
    for fname in [TEST_FILENAME, TEST_FILENAME_CHUNKED]:
        try:
            find_result = subprocess.run(
                ["find", "/mnt/dropbox/2025 Customers/", "/mnt/dropbox/2026 Customers/",
                 "/mnt/dropbox/Uploads/", "-maxdepth", "8",
                 "-name", fname],
                capture_output=True, text=True, timeout=15
            )
            for path in find_result.stdout.strip().split("\n"):
                if path:
                    parent = os.path.dirname(path)
                    # Remove the job folder (contains test file + thumbs)
                    subprocess.run(["rm", "-rf", parent], timeout=5)
        except Exception:
            pass


# ── Main ───────────────────────────────────────────────────────────

def main():
    parser = argparse.ArgumentParser(description="AEI Photo Upload Pipeline E2E Test")
    parser.add_argument("--job-id", default=DEFAULT_JOB_ID, help="Job ID to test with")
    parser.add_argument("--skip-cleanup", action="store_true", help="Leave test data for inspection")
    parser.add_argument("--remote-only", action="store_true", help="Skip local server checks")
    parser.add_argument("--wait", type=int, default=60,
                        help="Seconds to wait for background processes (default 60 — accounts for chunked + inline retry)")
    args = parser.parse_args()

    print(f"\n{Colors.BOLD}AEI Photo Upload Pipeline — E2E Test{Colors.RESET}")
    print(f"{'=' * 50}")
    print(f"  Job ID:    {args.job_id}")
    print(f"  Files:     {TEST_FILENAME} (single), {TEST_FILENAME_CHUNKED} (chunked)")
    print(f"  Remote:    {REMOTE_API_URL}")
    print(f"  Timestamp: {time.strftime('%Y-%m-%d %H:%M:%S')}")
    print(f"{'=' * 50}\n")

    passed = 0
    failed = 0
    # Total steps: 31 full, or 25 remote-only (skip steps 9-10, 23-25, 31)
    total = 31 if not args.remote_only else 25

    # ── Phase 1: Upload small file (Steps 1-2) ──
    print(f"{Colors.INFO}Phase 1 — Upload (single path)...{Colors.RESET}")
    resp_data, elapsed, error = test_upload(args.job_id)

    if error:
        result_line(1, "API POST", False, error)
        result_line(2, "Response time", False, "upload failed")
        failed += 2
        print(f"\n{Colors.FAIL}Upload failed — cannot continue.{Colors.RESET}")
        return 1
    else:
        ok = resp_data.get("success") == 1 and resp_data.get("status") == "ok"
        if result_line(1, "API response", ok, json.dumps(resp_data)):
            passed += 1
        else:
            failed += 1

        fast = elapsed < 2.0
        if result_line(2, "Response time", fast, f"{elapsed:.3f}s {'(< 2s)' if fast else '(SLOW > 2s)'}"):
            passed += 1
        else:
            failed += 1

    # ── Upload large file for chunked testing (results displayed in Phase 5) ──
    print(f"\n{Colors.INFO}Generating large test image for chunked path...{Colors.RESET}")
    large_b64, large_size = create_large_test_image()
    print(f"  Generated {large_size:,} bytes ({large_size / 1024 / 1024:.1f} MB)")
    if large_size < CHUNK_THRESHOLD:
        print(f"  {Colors.WARN}WARNING: Image only {large_size} bytes — may not trigger chunked path (threshold {CHUNK_THRESHOLD}){Colors.RESET}")

    print(f"{Colors.INFO}Uploading large test image...{Colors.RESET}")
    chunked_resp, chunked_elapsed, chunked_error = test_upload(args.job_id, TEST_FILENAME_CHUNKED, large_b64)
    if chunked_error:
        print(f"  {Colors.WARN}Large upload error: {chunked_error}{Colors.RESET}")
    else:
        c_ok = chunked_resp.get("success") == 1 and chunked_resp.get("status") == "ok"
        print(f"  {'OK' if c_ok else 'FAIL'} — {chunked_elapsed:.1f}s")

    # ── Wait for background processes (both small + large syncs) ──
    print(f"\n{Colors.INFO}Waiting {args.wait}s for background processes (both uploads)...{Colors.RESET}")
    time.sleep(args.wait)

    # ── Phase 2: Remote checks for small file (Steps 3-8) ──
    print(f"\n{Colors.INFO}Phase 2 — Remote artifacts (single path)...{Colors.RESET}")
    remote = test_remote_checks(args.job_id)

    if result_line(3, "File saved to /mnt/dropbox/", bool(remote["file_saved"]),
                   os.path.basename(os.path.dirname(remote["file_saved"])) if remote["file_saved"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    if result_line(4, "WebP generated in webp/", bool(remote["webp_generated"]),
                   os.path.basename(remote["webp_generated"]) if remote["webp_generated"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    if result_line(5, "Copied to scheduler/uploads/", bool(remote["scheduler_copy"]),
                   "present" if remote["scheduler_copy"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    if result_line(6, "WebP in scheduler/uploads/webp/", bool(remote["scheduler_webp"]),
                   os.path.basename(remote["scheduler_webp"]) if remote["scheduler_webp"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    if result_line(7, "meter_files DB insert", bool(remote["meter_files"]),
                   remote["meter_files"][:80] if remote["meter_files"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    sync_ok = "OK" in remote.get("sync_log", "") or "200" in remote.get("sync_log", "")
    sync_err = "ERROR" in remote.get("sync_log", "")
    if result_line(8, "sync_to_local.py executed", bool(remote["sync_log"]),
                   remote["sync_log"][:80] if remote["sync_log"] else "NO LOG ENTRY"):
        passed += 1
    else:
        failed += 1

    # ── Fetch job details for local lookup ──
    job_info = {"customer_id": "0", "last_name": "", "first_name": "", "job_date": ""}
    try:
        info_cmd = (f"mysql -u {DB_USER} -p'{DB_PASS}' {DB_NAME} -N -e \""
                    f"SELECT js.customer_id, cs.last_name, cs.first_name, js.job_date "
                    f"FROM jobs js LEFT JOIN customers cs ON js.customer_id=cs.id "
                    f"WHERE js.id={args.job_id} LIMIT 1;\"")
        info_out, _, _ = ssh_cmd(info_cmd, timeout=10)
        if info_out:
            parts = info_out.split("\t")
            if len(parts) >= 4:
                job_info = {"customer_id": parts[0], "last_name": parts[1],
                            "first_name": parts[2], "job_date": parts[3]}
    except Exception:
        pass

    # ── Phase 3: Local checks for small file (Steps 9-10) ──
    if not args.remote_only:
        print(f"\n{Colors.INFO}Phase 3 — Local server (single path)...{Colors.RESET}")
        local = test_local_checks(
            customer_id=job_info["customer_id"],
            last_name=job_info["last_name"],
            first_name=job_info["first_name"],
            job_date=job_info["job_date"],
        )

        if result_line(9, "File received on local server", bool(local["file_found"]),
                       local["file_found"].split("/mnt/dropbox/")[-1] if local["file_found"] else "NOT FOUND"):
            passed += 1
        else:
            failed += 1

        thumbs_ok = bool(local["thumbnail"]) and bool(local["mid_thumb"])
        if result_line(10, "Local thumbnails generated", thumbs_ok,
                       "thumbs/ + thumbs/mid/" if thumbs_ok else "MISSING"):
            passed += 1
        else:
            failed += 1

    # ── Phase 4: Async sync enhancement checks (Steps 11-19) ──
    print(f"\n{Colors.INFO}Phase 4 — Async sync enhancement deployment...{Colors.RESET}")
    async_checks = test_async_enhancement_checks()

    # Step 11: No synchronous cURL in upload.php
    curl_count = async_checks["no_curl"].strip().split("\n")[0] if async_checks["no_curl"] else "0"
    no_curl = curl_count == "0"
    if result_line(11, "No synchronous cURL in upload.php", no_curl,
                   "clean (no curl_init/exec/close)" if no_curl else f"{curl_count} cURL references found"):
        passed += 1
    else:
        failed += 1

    # Step 12: Echo JSON is last PHP statement before ?>
    echo_last_line = async_checks["echo_last"]
    echo_ok = "echo" in echo_last_line.lower() and "json_encode" in echo_last_line.lower()
    if result_line(12, "Echo JSON is last statement", echo_ok,
                   "echo json_encode() is final statement" if echo_ok else f"last line: {echo_last_line[:60]}"):
        passed += 1
    else:
        failed += 1

    # Step 13: sync_to_local.py permissions
    sync_perms = async_checks["sync_perms"]
    sync_ok = sync_perms != "NOT_FOUND" and sync_perms.startswith("755")
    if result_line(13, "sync_to_local.py permissions", sync_ok,
                   sync_perms if sync_perms != "NOT_FOUND" else "FILE NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 14: process_retry_queue.py permissions
    retry_perms = async_checks["retry_perms"]
    retry_ok = retry_perms != "NOT_FOUND" and retry_perms.startswith("755")
    if result_line(14, "process_retry_queue.py permissions", retry_ok,
                   retry_perms if retry_perms != "NOT_FOUND" else "FILE NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 15: queue/ and queue/failed/ directories
    queue_dir = async_checks["queue_dir"]
    queue_failed = async_checks["queue_failed"]
    dirs_ok = queue_dir == "777" and queue_failed != "NOT_FOUND"
    detail_parts = []
    detail_parts.append(f"queue/={'ok' if queue_dir == '777' else queue_dir}")
    detail_parts.append(f"failed/={'ok' if queue_failed != 'NOT_FOUND' else 'MISSING'}")
    if result_line(15, "queue/ and queue/failed/ directories", dirs_ok,
                   ", ".join(detail_parts)):
        passed += 1
    else:
        failed += 1

    # Step 16: Cron entry for process_retry_queue.py
    cron = async_checks["cron"]
    cron_ok = "process_retry" in cron and "NOT_FOUND" not in cron
    if result_line(16, "Cron entry for retry processor", cron_ok,
                   cron[:70] if cron_ok else "NOT FOUND in ec2-user crontab"):
        passed += 1
    else:
        failed += 1

    # Step 17: logs/ directory writable
    logs_dir = async_checks["logs_dir"]
    logs_ok = logs_dir == "777"
    if result_line(17, "logs/ directory writable", logs_ok,
                   f"permissions={logs_dir}" if logs_dir != "NOT_FOUND" else "DIRECTORY NOT FOUND"):
        passed += 1
    else:
        failed += 1

    # Step 18: Python requests module
    py_req = async_checks["python_requests"]
    py_ok = py_req == "OK"
    if result_line(18, "Python 3.6 requests module", py_ok,
                   "available" if py_ok else "MISSING — pip3.6 install requests"):
        passed += 1
    else:
        failed += 1

    # Step 19: Queue empty after successful upload
    queue_count = async_checks["queue_count"]
    queue_empty = queue_count == "0"
    sync_was_queued = "QUEUED" in remote.get("sync_log", "")
    if sync_was_queued and not queue_empty:
        # Sync failed and was correctly queued — queue is working as designed
        result_line(19, "Queue captured failed sync", True,
                    f"{queue_count} item(s) queued (sync failed — check connectivity)")
        passed += 1
    elif result_line(19, "Queue empty (no false queuing)", queue_empty,
                     "0 items in queue/" if queue_empty else f"{queue_count} items in queue/ (unexpected — sync showed OK)"):
        passed += 1
    else:
        failed += 1

    # ── Phase 5: Chunked upload path (Steps 20-25) ──
    print(f"\n{Colors.INFO}Phase 5 — Chunked upload path (>= 3MB)...{Colors.RESET}")

    # Step 20: Large file upload
    if chunked_error:
        result_line(20, "Large file upload (>= 3MB)", False, chunked_error)
        failed += 1
    else:
        c_ok = chunked_resp.get("success") == 1 and chunked_resp.get("status") == "ok"
        if result_line(20, "Large file upload (>= 3MB)", c_ok,
                       f"{large_size / 1024 / 1024:.1f}MB, {chunked_elapsed:.1f}s" if c_ok else json.dumps(chunked_resp)):
            passed += 1
        else:
            failed += 1

    # Steps 21-22: Remote checks for chunked file
    chunked_remote = test_chunked_remote_checks(args.job_id)

    if result_line(21, "Large file saved to /mnt/dropbox/", bool(chunked_remote["file_saved"]),
                   os.path.basename(os.path.dirname(chunked_remote["file_saved"])) if chunked_remote["file_saved"] else "NOT FOUND"):
        passed += 1
    else:
        failed += 1

    chunked_log = chunked_remote.get("chunked_log", "")
    # Check for CHUNKED or ROUTE entries indicating chunked path was used
    chunked_mode_used = "CHUNKED" in chunked_log or "ROUTE" in chunked_log
    chunked_ok_log = "CHUNKED OK" in chunked_log
    chunked_queued = "QUEUED" in chunked_log
    if chunked_ok_log:
        detail = "CHUNKED OK"
    elif chunked_queued:
        detail = "chunked sync queued (check connectivity)"
    elif chunked_mode_used:
        detail = chunked_log.split("\n")[-1][:80]
    else:
        detail = "NO CHUNKED LOG — sync may have used single mode"
    if result_line(22, "Sync used chunked mode", chunked_mode_used,
                   detail):
        passed += 1
    else:
        failed += 1

    # Steps 23-24: Local checks for chunked file
    # The chunked sync may still be running (WSL2 networking causes slow chunk
    # delivery). Poll the remote log until sync completes or is queued, up to
    # 120s extra, then check the local server.
    if not args.remote_only:
        # If chunked sync hasn't resolved yet, poll for completion
        if not chunked_ok_log and not chunked_queued:
            print(f"  {Colors.WARN}Chunked sync still running — polling up to 120s...{Colors.RESET}")
            poll_deadline = time.time() + 120
            while time.time() < poll_deadline:
                time.sleep(10)
                cr = test_chunked_remote_checks(args.job_id)
                chunked_log = cr.get("chunked_log", "")
                chunked_ok_log = "CHUNKED OK" in chunked_log
                chunked_queued = "QUEUED" in chunked_log
                if chunked_ok_log or chunked_queued:
                    status = "completed" if chunked_ok_log else "queued"
                    print(f"  Chunked sync {status} after {int(time.time() - poll_deadline + 120)}s")
                    break
            else:
                print(f"  Chunked sync still running after 120s — checking local anyway")

        # Additional wait for thumbnail generation after chunked OK
        if chunked_ok_log:
            time.sleep(5)

        local_chunked = test_local_checks(
            customer_id=job_info["customer_id"],
            last_name=job_info["last_name"],
            first_name=job_info["first_name"],
            job_date=job_info["job_date"],
            test_filename=TEST_FILENAME_CHUNKED,
        )

        # Determine chunked sync state for context-aware reporting:
        # - CHUNKED OK: completed, check file + thumbs normally
        # - QUEUED: deferred to retry queue (expected with WSL2 networking)
        # - Still running (CHUNKED in log, no OK/QUEUED): in progress, accept as valid
        chunked_deferred = (chunked_queued or chunked_mode_used) and not chunked_ok_log

        if local_chunked["file_found"]:
            # File arrived — check normally regardless of sync state
            if result_line(23, "Large file received on local server", True,
                           local_chunked["file_found"].split("/mnt/dropbox/")[-1]):
                passed += 1

            c_thumbs_ok = bool(local_chunked["thumbnail"]) and bool(local_chunked["mid_thumb"])
            if result_line(24, "Local thumbnails for large file", c_thumbs_ok,
                           "thumbs/ + thumbs/mid/" if c_thumbs_ok else "MISSING"):
                passed += 1
            else:
                failed += 1
        elif chunked_deferred:
            # Sync in progress or queued — WSL2 networking causes slow chunked delivery
            if chunked_queued:
                note = "queued for retry — will deliver within 15min"
            else:
                note = "still in progress — WSL2 networking causes slow chunk delivery"
            result_line(23, "Large file delivery deferred", True, note)
            passed += 1
            result_line(24, "Thumbnails deferred", True,
                        "will generate after chunked sync completes")
            passed += 1
        else:
            # Neither found nor deferred — unexpected
            result_line(23, "Large file received on local server", False, "NOT FOUND")
            failed += 1
            result_line(24, "Local thumbnails for large file", False, "MISSING")
            failed += 1

    # Step 25: .chunks directory configured
    if not args.remote_only:
        infra = test_chunked_infra_checks()

        chunks_ok = infra["chunks_exists"] and infra["htaccess"]
        detail_parts = []
        if infra["chunks_exists"]:
            detail_parts.append(f"perms={infra['chunks_perms']}")
            detail_parts.append(f".htaccess={'ok' if infra['htaccess'] else 'MISSING'}")
        else:
            detail_parts.append("DIRECTORY NOT FOUND")
        if result_line(25, ".chunks directory configured", chunks_ok,
                       ", ".join(detail_parts)):
            passed += 1
        else:
            failed += 1

    # ── Phase 6: Photo folder path enhancement (Steps 26-31) ──
    print(f"\n{Colors.INFO}Phase 6 — Photo folder path enhancement...{Colors.RESET}")

    # Steps 26-28: folder_path column checks
    fp_checks = test_folder_path_checks()

    # Step 26: folder_path column exists
    col_info = fp_checks["column_info"]
    col_exists = bool(col_info) and "folder_path" in col_info.lower()
    col_detail = col_info[:80] if col_info else "COLUMN NOT FOUND"
    if result_line(26, "folder_path column in meter_files", col_exists, col_detail):
        passed += 1
    else:
        failed += 1

    # Step 27: folder_path populated for test upload
    folder_path_val = fp_checks["folder_path"]
    fp_populated = bool(folder_path_val) and folder_path_val != "NULL"
    if result_line(27, "folder_path populated for test upload", fp_populated,
                   folder_path_val if fp_populated else "NULL or empty"):
        passed += 1
    else:
        failed += 1

    # Step 28: folder_path matches actual file location
    file_saved_path = remote.get("file_saved", "")
    if fp_populated and file_saved_path:
        # folder_path should be the directory portion of the saved file
        fp_matches = file_saved_path.startswith(folder_path_val)
        if result_line(28, "folder_path matches file location", fp_matches,
                       f"path={folder_path_val}" if fp_matches else
                       f"mismatch: folder_path={folder_path_val}, file={file_saved_path}"):
            passed += 1
        else:
            failed += 1
    else:
        if result_line(28, "folder_path matches file location", False,
                       "cannot compare — folder_path or file_saved missing"):
            passed += 1
        else:
            failed += 1

    # Steps 29-30: Image API checks
    api_checks = test_image_api_checks(args.job_id)

    # Step 29: getimagelisting.php returns test image
    listing_data = api_checks.get("listing_data")
    if isinstance(listing_data, list):
        # If listing returned images, consider it a pass
        # (webp filename may differ from original upload name)
        listing_ok = len(listing_data) > 0
        if result_line(29, "getimagelisting.php returns images", listing_ok,
                       f"{len(listing_data)} image(s) returned"):
            passed += 1
        else:
            failed += 1
    else:
        err = api_checks.get("listing_error", "invalid response")
        if result_line(29, "getimagelisting.php returns images", False,
                       f"HTTP {api_checks.get('listing_status', '?')} — {err}"):
            passed += 1
        else:
            failed += 1

    # Step 30: fetch_image.php serves image
    fetch_status = api_checks.get("fetch_status", 0)
    fetch_ct = api_checks.get("fetch_content_type", "")
    fetch_size = api_checks.get("fetch_size", 0)
    fetch_ok = fetch_status == 200 and fetch_ct.startswith("image/")
    if fetch_ok:
        if result_line(30, "fetch_image.php serves image", True,
                       f"{fetch_ct}, {fetch_size:,} bytes"):
            passed += 1
        else:
            failed += 1
    else:
        fetch_err = api_checks.get("fetch_error", f"HTTP {fetch_status}, {fetch_ct}")
        if result_line(30, "fetch_image.php serves image", False, fetch_err):
            passed += 1
        else:
            failed += 1

    # Step 31: Survey path spacing fix (local only)
    if not args.remote_only:
        spacing = test_survey_spacing_fix()

        if not spacing.get("file_exists"):
            if result_line(31, "Survey path spacing fix", False,
                           "uploadlocallat_kuldeep.php not found"):
                passed += 1
            else:
                failed += 1
        else:
            bad = spacing.get("bad_count", -1)
            good = spacing.get("good_count", 0)
            spacing_ok = bad == 0 and good == 4
            if spacing_ok:
                detail = f"0 bad, {good} correct '-S, ' instances"
            else:
                detail = f"{bad} bad '-S,' (no space), {good} correct '-S, ' (expected 4)"
            if result_line(31, "Survey path spacing fix", spacing_ok, detail):
                passed += 1
            else:
                failed += 1

    # ── Summary ──
    print(f"\n{'=' * 50}")
    all_pass = failed == 0
    color = Colors.PASS if all_pass else Colors.FAIL
    print(f"  {color}{Colors.BOLD}Result: {passed}/{total} passed, {failed} failed{Colors.RESET}")
    if sync_err and not sync_ok:
        print(f"  {Colors.WARN}Note: sync_to_local.py ran but got an error (local server may be unreachable){Colors.RESET}")
    print(f"{'=' * 50}")

    # ── Cleanup ──
    if not args.skip_cleanup:
        print(f"\n{Colors.INFO}Cleaning up test data...{Colors.RESET}")
        cleanup_remote()
        if not args.remote_only:
            cleanup_local()
        print("  Done.")
    else:
        print(f"\n{Colors.WARN}Skipping cleanup (--skip-cleanup). Test data left for inspection.{Colors.RESET}")

    print()
    return 0 if all_pass else 1


if __name__ == "__main__":
    sys.exit(main())
