#!/usr/bin/env python3
"""
Sync uploaded photo to local server (background process).
Called by upload.php via nohup to avoid blocking the mobile app response.

Retries up to INLINE_RETRIES times with exponential backoff before
falling back to the queue/ directory for cron-based retry via
process_retry_queue.py (every 15 minutes).

Files >= 3MB use a 3-phase chunked protocol (init -> chunk x N -> finalize)
with per-chunk retry and resume support. Files < 3MB use a single POST.
See DOCS/CHUNKED_UPLOAD_ENHANCEMENT.md for protocol details.

The inline retry handles intermittent TCP packet loss caused by the
local server's WSL2 mirrored networking mode (see DOCS/WSL2_NETWORKING_INVESTIGATION.md).

Usage: python3 sync_to_local.py <file_path> <file_name> <job_id> <customer_id> \
           <job_type> <last_name> <first_name> <job_date> <photo_type>

Requires: requests (pip3 install requests)
"""

import os
import sys
import json
import time
import math
import random
import hashlib
import logging
from datetime import datetime

UPLOAD_URL = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php'
AUTH_TOKEN = 'remote_token'
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'sync_to_local.log')
QUEUE_DIR = os.path.join(SCRIPT_DIR, 'queue')
CONNECT_TIMEOUT = 10
READ_TIMEOUT = 60
INLINE_RETRIES = 3
RETRY_DELAYS = [2, 4, 8]  # seconds between attempts

# Chunked upload constants
CHUNK_THRESHOLD = 3 * 1024 * 1024  # 3MB — files >= this use chunked upload
CHUNK_SIZE = 1 * 1024 * 1024       # 1MB per chunk


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def generate_file_id(file_name, job_id, customer_id, file_size):
    """Deterministic SHA-1 file ID for chunked upload resume support.

    Same file + job + customer + size always produces the same ID,
    allowing resume across process restarts (e.g. retry queue picks up
    where sync_to_local.py left off).
    """
    key = '%s:%s:%s:%s' % (file_name, job_id, customer_id, file_size)
    return hashlib.sha1(key.encode()).hexdigest()


def _chunked_request(data, files=None):
    """Single HTTP POST with per-request retry (reuses INLINE_RETRIES/RETRY_DELAYS).

    Returns (success, response_json_or_none, error_reason).
    """
    import requests

    for attempt in range(1, INLINE_RETRIES + 1):
        try:
            resp = requests.post(
                UPLOAD_URL,
                data=data,
                files=files,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )
            if resp.status_code == 200:
                try:
                    body = resp.json()
                except ValueError:
                    body = None
                if body and body.get('status') in ('ok', 'success'):
                    return True, body, None
                else:
                    msg = "Server error: %s" % (resp.text[:200])
                    if attempt < INLINE_RETRIES:
                        delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                        time.sleep(delay)
                        continue
                    return False, body, msg
            else:
                msg = "HTTP %d body=%s" % (resp.status_code, resp.text[:200])
                if attempt < INLINE_RETRIES:
                    delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                    time.sleep(delay)
                    continue
                return False, None, msg
        except Exception as e:
            msg = "%s: %s" % (type(e).__name__, str(e))
            if attempt < INLINE_RETRIES:
                delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                time.sleep(delay)
                continue
            return False, None, msg

    return False, None, "All retries exhausted"


def sync_file_chunked(file_path, file_name, job_id, customer_id, job_type,
                      last_name, first_name, job_date, photo_type):
    """3-phase chunked upload: init -> chunk x N -> finalize.

    Returns (success, error_reason) tuple matching sync_file() signature.
    """
    if not os.path.isfile(file_path):
        msg = "File not found: %s" % file_path
        logging.error(msg)
        return False, msg

    file_size = os.path.getsize(file_path)
    total_chunks = int(math.ceil(file_size / float(CHUNK_SIZE)))
    file_id = generate_file_id(file_name, job_id, customer_id, file_size)

    logging.info("CHUNKED START %s -> file_id=%s size=%d chunks=%d",
                 file_name, file_id[:12], file_size, total_chunks)

    # Phase 1: Init
    init_data = {
        'action':       'photo_chunk_init',
        'auth_token':   AUTH_TOKEN,
        'file_id':      file_id,
        'file_name':    file_name,
        'file_size':    str(file_size),
        'total_chunks': str(total_chunks),
        'chunk_size':   str(CHUNK_SIZE),
        'job_id':       job_id,
        'customer_id':  customer_id,
        'job_type':     job_type,
        'last_name':    last_name,
        'first_name':   first_name,
        'job_date':     job_date,
        'photo_type':   photo_type,
    }

    ok, body, err = _chunked_request(init_data)
    if not ok:
        msg = "Chunked init failed: %s" % err
        logging.error("CHUNKED INIT FAIL %s -> %s", file_name, msg)
        return False, msg

    # Determine which chunks to skip (already uploaded — resume support)
    chunks_received = set(body.get('chunks_received', []))
    if chunks_received:
        logging.info("CHUNKED RESUME %s -> %d/%d chunks already on server",
                     file_name, len(chunks_received), total_chunks)

    # Phase 2: Upload each chunk
    with open(file_path, 'rb') as f:
        for chunk_index in range(total_chunks):
            if chunk_index in chunks_received:
                # Skip — already uploaded in a previous attempt
                f.seek(CHUNK_SIZE, 1)  # advance read position
                continue

            f.seek(chunk_index * CHUNK_SIZE)
            chunk_data = f.read(CHUNK_SIZE)

            chunk_post = {
                'action':      'photo_chunk_upload',
                'auth_token':  AUTH_TOKEN,
                'file_id':     file_id,
                'chunk_index': str(chunk_index),
            }
            chunk_files = {'chunk': ('chunk_%d' % chunk_index, chunk_data)}

            ok, body, err = _chunked_request(chunk_post, files=chunk_files)
            if not ok:
                msg = "Chunk %d/%d failed: %s" % (chunk_index, total_chunks, err)
                logging.error("CHUNKED CHUNK FAIL %s -> %s", file_name, msg)
                return False, msg

            logging.info("CHUNKED CHUNK OK %s -> %d/%d",
                         file_name, chunk_index + 1, total_chunks)

    # Phase 3: Finalize
    finalize_data = {
        'action':     'photo_chunk_finalize',
        'auth_token': AUTH_TOKEN,
        'file_id':    file_id,
    }

    ok, body, err = _chunked_request(finalize_data)
    if not ok:
        msg = "Chunked finalize failed: %s" % err
        logging.error("CHUNKED FINALIZE FAIL %s -> %s", file_name, msg)
        return False, msg

    logging.info("CHUNKED OK %s -> file_id=%s (%s)", file_name, file_id[:12], job_id)
    return True, None


def sync_file(file_path, file_name, job_id, customer_id, job_type,
              last_name, first_name, job_date, photo_type):
    """POST file and metadata to local server (single-upload path).

    Returns (success, error_reason) tuple.
    error_reason is None on success, or a string describing the failure.
    """
    import requests

    if not os.path.isfile(file_path):
        msg = "File not found: %s" % file_path
        logging.error(msg)
        return False, msg

    try:
        with open(file_path, 'rb') as f:
            files = {'file': (file_name, f)}
            data = {
                'auth_token':  AUTH_TOKEN,
                'file_name':   file_name,
                'job_id':      job_id,
                'customer_id': customer_id,
                'job_type':    job_type,
                'last_name':   last_name,
                'first_name':  first_name,
                'job_date':    job_date,
                'photo_type':  photo_type,
            }
            resp = requests.post(
                UPLOAD_URL,
                files=files,
                data=data,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )

        if resp.status_code == 200:
            logging.info("OK %s -> HTTP %d (%s)", file_name, resp.status_code, job_id)
            return True, None
        else:
            msg = "HTTP %d body=%s" % (resp.status_code, resp.text[:200])
            logging.warning("FAIL %s -> %s", file_name, msg)
            return False, msg

    except Exception as e:
        msg = "%s: %s" % (type(e).__name__, str(e))
        logging.error("ERROR %s -> %s", file_name, msg)
        return False, msg


def enqueue_failed(file_path, file_name, job_id, customer_id, job_type,
                   last_name, first_name, job_date, photo_type, error_reason,
                   upload_mode='single'):
    """Save failed sync payload to queue/ directory as JSON for later retry."""
    if not os.path.isdir(QUEUE_DIR):
        os.makedirs(QUEUE_DIR, mode=0o777, exist_ok=True)

    now = datetime.now()
    rand = '%06d' % random.randint(0, 999999)
    filename = 'sync_%s_%s.json' % (now.strftime('%Y%m%d_%H%M%S'), rand)

    payload = {
        'file_path':    file_path,
        'file_name':    file_name,
        'job_id':       job_id,
        'customer_id':  customer_id,
        'job_type':     job_type,
        'last_name':    last_name,
        'first_name':   first_name,
        'job_date':     job_date,
        'photo_type':   photo_type,
        'upload_mode':  upload_mode,
        'queued_at':    now.strftime('%Y-%m-%d %H:%M:%S'),
        'retry_count':  0,
        'last_error':   error_reason or 'Unknown error',
    }

    queue_path = os.path.join(QUEUE_DIR, filename)
    with open(queue_path, 'w') as f:
        json.dump(payload, f, indent=2)

    logging.info("QUEUED %s -> %s (mode: %s, reason: %s)",
                 file_name, filename, upload_mode, error_reason)


if __name__ == "__main__":
    if len(sys.argv) != 10:
        print("Usage: sync_to_local.py <file_path> <file_name> <job_id> <customer_id>"
              " <job_type> <last_name> <first_name> <job_date> <photo_type>")
        sys.exit(1)

    setup_logging()

    file_path = sys.argv[1]
    file_name = sys.argv[2]
    job_id = sys.argv[3]
    customer_id = sys.argv[4]
    job_type = sys.argv[5]
    last_name = sys.argv[6]
    first_name = sys.argv[7]
    job_date = sys.argv[8]
    photo_type = sys.argv[9]

    kwargs = dict(
        file_path=file_path, file_name=file_name, job_id=job_id,
        customer_id=customer_id, job_type=job_type, last_name=last_name,
        first_name=first_name, job_date=job_date, photo_type=photo_type,
    )

    # Route by file size: >= 3MB uses chunked, < 3MB uses single POST
    use_chunked = os.path.isfile(file_path) and os.path.getsize(file_path) >= CHUNK_THRESHOLD
    upload_mode = 'chunked' if use_chunked else 'single'

    if use_chunked:
        logging.info("ROUTE %s -> chunked (size=%d, threshold=%d)",
                     file_name, os.path.getsize(file_path), CHUNK_THRESHOLD)

    # Inline retry with exponential backoff.
    # Handles intermittent TCP drops from WSL2 mirrored networking.
    # For chunked uploads, each phase has its own per-request retry;
    # this outer loop retries the entire chunked sequence if it fails.
    success = False
    error_reason = None
    for attempt in range(1, INLINE_RETRIES + 1):
        if use_chunked:
            success, error_reason = sync_file_chunked(**kwargs)
        else:
            success, error_reason = sync_file(**kwargs)
        if success:
            break
        if attempt < INLINE_RETRIES:
            delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
            logging.info("RETRY %s -> attempt %d/%d failed, retrying in %ds",
                         file_name, attempt, INLINE_RETRIES, delay)
            time.sleep(delay)

    if not success and os.path.isfile(file_path):
        # All inline retries exhausted — queue for cron-based retry.
        # Only queue if source file still exists (network/server failure).
        # If file is gone, there's nothing to retry.
        enqueue_failed(error_reason=error_reason, upload_mode=upload_mode, **kwargs)

    sys.exit(0 if success else 1)
