#!/usr/bin/env python3
"""
Process the retry queue for failed local server syncs.
Reads JSON files from queue/, retries POST to local server,
moves to queue/failed/ on max retries exceeded.

Supports both single-upload and chunked upload modes.
Files >= 3MB use chunked upload (resume-capable); files < 3MB use single POST.
Queue items with upload_mode='chunked' use the 3-phase chunked protocol.
Old queue items without upload_mode default to single, auto-upgraded to chunked
if file >= 3MB threshold. See DOCS/CHUNKED_UPLOAD_ENHANCEMENT.md.

Designed to run via cron every 15 minutes:
  */15 * * * * /usr/local/bin/python3.6 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py > /dev/null 2>&1

Requires: requests (pip3 install requests)
"""

import os
import sys
import json
import glob
import math
import time
import shutil
import hashlib
import logging
from datetime import datetime

UPLOAD_URL = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php'
AUTH_TOKEN = 'remote_token'
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'retry_queue.log')
QUEUE_DIR = os.path.join(SCRIPT_DIR, 'queue')
FAILED_DIR = os.path.join(QUEUE_DIR, 'failed')
CONNECT_TIMEOUT = 10
READ_TIMEOUT = 60
MAX_RETRIES = 10
INLINE_RETRIES = 3
RETRY_DELAYS = [2, 4, 8]  # seconds between per-request retry attempts

# Chunked upload constants (duplicated from sync_to_local.py per established pattern)
CHUNK_THRESHOLD = 3 * 1024 * 1024  # 3MB — files >= this use chunked upload
CHUNK_SIZE = 1 * 1024 * 1024       # 1MB per chunk


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def generate_file_id(file_name, job_id, customer_id, file_size):
    """Deterministic SHA-1 file ID for chunked upload resume support."""
    key = '%s:%s:%s:%s' % (file_name, job_id, customer_id, file_size)
    return hashlib.sha1(key.encode()).hexdigest()


def _chunked_request(data, files=None):
    """Single HTTP POST with per-request retry.

    Returns (success, response_json_or_none, error_reason).
    """
    import requests

    for attempt in range(1, INLINE_RETRIES + 1):
        try:
            resp = requests.post(
                UPLOAD_URL,
                data=data,
                files=files,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )
            if resp.status_code == 200:
                try:
                    body = resp.json()
                except ValueError:
                    body = None
                if body and body.get('status') in ('ok', 'success'):
                    return True, body, None
                else:
                    msg = "Server error: %s" % (resp.text[:200])
                    if attempt < INLINE_RETRIES:
                        delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                        time.sleep(delay)
                        continue
                    return False, body, msg
            else:
                msg = "HTTP %d body=%s" % (resp.status_code, resp.text[:200])
                if attempt < INLINE_RETRIES:
                    delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                    time.sleep(delay)
                    continue
                return False, None, msg
        except Exception as e:
            msg = "%s: %s" % (type(e).__name__, str(e))
            if attempt < INLINE_RETRIES:
                delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
                time.sleep(delay)
                continue
            return False, None, msg

    return False, None, "All retries exhausted"


def sync_file_chunked(file_path, file_name, job_id, customer_id, job_type,
                      last_name, first_name, job_date, photo_type):
    """3-phase chunked upload: init -> chunk x N -> finalize.

    Returns (success, error_reason) tuple matching sync_file() signature.
    """
    if not os.path.isfile(file_path):
        return False, "File not found: %s" % file_path

    file_size = os.path.getsize(file_path)
    total_chunks = int(math.ceil(file_size / float(CHUNK_SIZE)))
    file_id = generate_file_id(file_name, job_id, customer_id, file_size)

    logging.info("CHUNKED START %s -> file_id=%s size=%d chunks=%d",
                 file_name, file_id[:12], file_size, total_chunks)

    # Phase 1: Init
    init_data = {
        'action':       'photo_chunk_init',
        'auth_token':   AUTH_TOKEN,
        'file_id':      file_id,
        'file_name':    file_name,
        'file_size':    str(file_size),
        'total_chunks': str(total_chunks),
        'chunk_size':   str(CHUNK_SIZE),
        'job_id':       job_id,
        'customer_id':  customer_id,
        'job_type':     job_type,
        'last_name':    last_name,
        'first_name':   first_name,
        'job_date':     job_date,
        'photo_type':   photo_type,
    }

    ok, body, err = _chunked_request(init_data)
    if not ok:
        msg = "Chunked init failed: %s" % err
        logging.error("CHUNKED INIT FAIL %s -> %s", file_name, msg)
        return False, msg

    chunks_received = set(body.get('chunks_received', []))
    if chunks_received:
        logging.info("CHUNKED RESUME %s -> %d/%d chunks already on server",
                     file_name, len(chunks_received), total_chunks)

    # Phase 2: Upload each chunk
    with open(file_path, 'rb') as f:
        for chunk_index in range(total_chunks):
            if chunk_index in chunks_received:
                f.seek(CHUNK_SIZE, 1)
                continue

            f.seek(chunk_index * CHUNK_SIZE)
            chunk_data = f.read(CHUNK_SIZE)

            chunk_post = {
                'action':      'photo_chunk_upload',
                'auth_token':  AUTH_TOKEN,
                'file_id':     file_id,
                'chunk_index': str(chunk_index),
            }
            chunk_files = {'chunk': ('chunk_%d' % chunk_index, chunk_data)}

            ok, body, err = _chunked_request(chunk_post, files=chunk_files)
            if not ok:
                msg = "Chunk %d/%d failed: %s" % (chunk_index, total_chunks, err)
                logging.error("CHUNKED CHUNK FAIL %s -> %s", file_name, msg)
                return False, msg

    # Phase 3: Finalize
    finalize_data = {
        'action':     'photo_chunk_finalize',
        'auth_token': AUTH_TOKEN,
        'file_id':    file_id,
    }

    ok, body, err = _chunked_request(finalize_data)
    if not ok:
        msg = "Chunked finalize failed: %s" % err
        logging.error("CHUNKED FINALIZE FAIL %s -> %s", file_name, msg)
        return False, msg

    logging.info("CHUNKED OK %s -> file_id=%s (%s)", file_name, file_id[:12], job_id)
    return True, None


def sync_file(file_path, file_name, job_id, customer_id, job_type,
              last_name, first_name, job_date, photo_type):
    """POST file and metadata to local server.

    Returns (success, error_reason) tuple.
    """
    import requests

    try:
        with open(file_path, 'rb') as f:
            files = {'file': (file_name, f)}
            data = {
                'auth_token':  AUTH_TOKEN,
                'file_name':   file_name,
                'job_id':      job_id,
                'customer_id': customer_id,
                'job_type':    job_type,
                'last_name':   last_name,
                'first_name':  first_name,
                'job_date':    job_date,
                'photo_type':  photo_type,
            }
            resp = requests.post(
                UPLOAD_URL,
                files=files,
                data=data,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )

        if resp.status_code == 200:
            return True, None
        else:
            return False, "HTTP %d body=%s" % (resp.status_code, resp.text[:200])

    except Exception as e:
        return False, "%s: %s" % (type(e).__name__, str(e))


def process_queue():
    """Scan queue/ for JSON files and retry each one."""
    if not os.path.isdir(QUEUE_DIR):
        return

    pattern = os.path.join(QUEUE_DIR, '*.json')
    queue_files = sorted(glob.glob(pattern))

    if not queue_files:
        return

    logging.info("Processing %d queued item(s)", len(queue_files))

    for queue_path in queue_files:
        filename = os.path.basename(queue_path)

        try:
            with open(queue_path, 'r') as f:
                payload = json.load(f)
        except (ValueError, IOError) as e:
            logging.error("SKIP %s -> cannot read JSON: %s", filename, str(e))
            continue

        retry_count = payload.get('retry_count', 0)
        file_path = payload.get('file_path', '')
        file_name = payload.get('file_name', '')

        # Max retries exceeded -> move to failed/
        if retry_count >= MAX_RETRIES:
            if not os.path.isdir(FAILED_DIR):
                os.makedirs(FAILED_DIR, mode=0o777, exist_ok=True)
            failed_path = os.path.join(FAILED_DIR, filename)
            shutil.move(queue_path, failed_path)
            logging.warning("EXPIRED %s -> moved to failed/ after %d retries (last error: %s)",
                            file_name, retry_count, payload.get('last_error', 'unknown'))
            continue

        # Source file gone -> skip (nothing to send)
        if not os.path.isfile(file_path):
            os.remove(queue_path)
            logging.info("SKIP %s -> source file no longer exists, removing from queue", file_name)
            continue

        # Determine upload mode: explicit from queue JSON, or auto-detect by file size
        upload_mode = payload.get('upload_mode', '')
        if not upload_mode:
            # Old queue items without upload_mode: auto-upgrade to chunked if file >= threshold
            file_size = os.path.getsize(file_path)
            upload_mode = 'chunked' if file_size >= CHUNK_THRESHOLD else 'single'

        # Attempt retry
        sync_kwargs = dict(
            file_path=file_path,
            file_name=file_name,
            job_id=payload.get('job_id', ''),
            customer_id=payload.get('customer_id', ''),
            job_type=payload.get('job_type', ''),
            last_name=payload.get('last_name', ''),
            first_name=payload.get('first_name', ''),
            job_date=payload.get('job_date', ''),
            photo_type=payload.get('photo_type', ''),
        )

        if upload_mode == 'chunked':
            success, error_reason = sync_file_chunked(**sync_kwargs)
        else:
            success, error_reason = sync_file(**sync_kwargs)

        if success:
            os.remove(queue_path)
            logging.info("RETRY OK %s -> succeeded on retry %d (mode: %s, queued %s)",
                         file_name, retry_count + 1, upload_mode, payload.get('queued_at', ''))
        else:
            # Update retry count and error, rewrite JSON
            payload['retry_count'] = retry_count + 1
            payload['last_error'] = error_reason or 'Unknown error'
            payload['last_retry'] = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            payload['upload_mode'] = upload_mode  # persist auto-detected mode
            with open(queue_path, 'w') as f:
                json.dump(payload, f, indent=2)
            logging.warning("RETRY FAIL %s -> attempt %d/%d (mode: %s): %s",
                            file_name, retry_count + 1, MAX_RETRIES, upload_mode, error_reason)

    logging.info("Queue processing complete")


if __name__ == "__main__":
    setup_logging()
    process_queue()
