# Chunked File Transfer Enhancement

**Date:** 2026-02-05
**Status:** Implemented (pending deployment)
**Files Changed:** `uploadlocallat_kuldeep.php` (local), `sync_to_local.py` (remote), `process_retry_queue.py` (remote)
**Servers:** Remote (18.225.0.90) and Local (upload.aeihawaii.com)

---

## Problem Statement

The current photo sync pipeline uses a single multipart POST to transfer files from the remote server to the local server via `sync_to_local.py`. This works reliably for small files but has two problems:

1. **No partial progress recovery** — if a 5MB photo transfer fails at 90%, the entire file must be re-uploaded from scratch on retry
2. **WSL2 networking unreliability** — the local server runs WSL2 with mirrored networking mode, which causes ~40-60% per-attempt TCP SYN-ACK packet loss (see [WSL2_NETWORKING_INVESTIGATION.md](WSL2_NETWORKING_INVESTIGATION.md))

For large files, repeated full re-uploads waste bandwidth and delay delivery to the local office server.

---

## Feasibility Analysis (TCP SYN-ACK Trade-off)

The WSL2 mirrored networking causes ~40-60% per-attempt TCP connection failure. Each HTTP request requires a new TCP handshake. More requests = more handshakes = more chances for individual failures. However, chunked transfer preserves partial progress.

**Probability model** (assuming 40% per-attempt failure, 3 retries per request = 93.6% per-request success):

| Scenario | Requests | P(all succeed) first attempt | Resume benefit |
|----------|----------|------------------------------|----------------|
| 1MB file, single POST | 1 | 93.6% | None — full re-upload |
| 1MB file, chunked | 3 (init+chunk+finalize) | 82.0% | Minimal — small file |
| 5MB file, single POST | 1 | 93.6% | None — full 5MB re-upload |
| 5MB file, chunked | 7 (init+5 chunks+finalize) | 63.6% | **Yes — only unsent chunks** |
| 5MB file, chunked resume (3 remaining) | 4 | 76.7% | Preserved 60% progress |

**Conclusion:** For files < 3MB, single POST is more reliable (fewer TCP handshakes). For files >= 3MB, the first attempt is less likely to fully complete, but partial progress is preserved — reducing total bytes transferred over multiple retries. The 3MB threshold balances these competing factors.

---

## Threshold Decision

| File size | Upload mode | Rationale |
|-----------|-------------|-----------|
| < 3MB | Single POST (existing) | Fewer handshakes (1 vs 3+), higher first-attempt success rate |
| >= 3MB | Chunked (3-phase) | Resume capability preserves partial progress for large files |

**Chunk size: 1MB** — balances per-chunk retry granularity against TCP handshake overhead.

---

## Protocol Specification

### Three-Phase Chunked Upload Protocol

```
Phase 1: INIT
  POST action=photo_chunk_init
  → Creates session metadata, returns existing progress for resume

Phase 2: CHUNK × N
  POST action=photo_chunk_upload  (with file data)
  → Uploads one chunk at a time, server validates and stores

Phase 3: FINALIZE
  POST action=photo_chunk_finalize
  → Server reassembles chunks, validates, moves to customer folder
```

### Phase 1: Init

**Request:**
```
POST /uploadlocallat_kuldeep.php
Content-Type: application/x-www-form-urlencoded

action=photo_chunk_init
auth_token=<token>
file_id=<sha1 hex>
file_name=<original filename>
file_size=<total bytes>
total_chunks=<N>
chunk_size=<bytes per chunk>
job_id=<int>
customer_id=<int>
job_type=<string>
last_name=<string>
first_name=<string>
job_date=<YYYY-MM-DD>
photo_type=<S|I>
```

**Response (new session):**
```json
{
  "status": "ok",
  "message": "Chunked upload session created",
  "file_id": "<sha1>",
  "chunks_received": []
}
```

**Response (resume — chunks already exist):**
```json
{
  "status": "ok",
  "message": "Resuming chunked upload",
  "file_id": "<sha1>",
  "chunks_received": [0, 1, 2]
}
```

### Phase 2: Chunk Upload

**Request:**
```
POST /uploadlocallat_kuldeep.php
Content-Type: multipart/form-data

action=photo_chunk_upload
auth_token=<token>
file_id=<sha1>
chunk_index=<0-based>
chunk (file field): binary chunk data
```

**Response:**
```json
{
  "status": "ok",
  "message": "Chunk 2 received",
  "chunk_index": 2
}
```

### Phase 3: Finalize

**Request:**
```
POST /uploadlocallat_kuldeep.php
Content-Type: application/x-www-form-urlencoded

action=photo_chunk_finalize
auth_token=<token>
file_id=<sha1>
```

**Response:**
```json
{
  "status": "success",
  "message": "Chunked upload assembled and saved",
  "path": "safe_filename.jpg",
  "folder": "/mnt/dropbox/2026 Customers/...",
  "lookup_method": "customer_id"
}
```

### file_id Generation

Deterministic SHA-1 hash of `file_name:job_id:customer_id:file_size`:

```python
import hashlib
file_id = hashlib.sha1(
    '{}:{}:{}:{}'.format(file_name, job_id, customer_id, file_size).encode()
).hexdigest()
```

The deterministic ID means the same file produces the same session ID across process restarts, enabling resume after `sync_to_local.py` crashes and is re-invoked by `process_retry_queue.py`.

---

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                   CHUNKED UPLOAD — FILE SIZE ROUTING                        │
└─────────────────────────────────────────────────────────────────────────────┘

  sync_to_local.py (remote server, background nohup &)
  │
  ├─ file_size < 3MB ──────────────── SINGLE POST PATH (unchanged)
  │   │
  │   └─ POST file + metadata to uploadlocallat_kuldeep.php
  │       ├─ No 'action' field in POST → existing flow
  │       ├─ Validate, save, trigger thumbnails
  │       └─ Return JSON success/error
  │
  └─ file_size >= 3MB ─────────────── CHUNKED PATH (new)
      │
      ├─ 1. INIT: POST action=photo_chunk_init
      │     └─ Server creates .chunks/{file_id}.meta + returns chunks_received[]
      │
      ├─ 2. CHUNK × N: POST action=photo_chunk_upload (skip already-received)
      │     └─ For each 1MB chunk not in chunks_received[]:
      │         ├─ Read chunk from file
      │         ├─ POST with chunk data
      │         └─ Server saves .chunks/{file_id}.part.{N}
      │
      └─ 3. FINALIZE: POST action=photo_chunk_finalize
            └─ Server:
                ├─ Reads metadata from .chunks/{file_id}.meta
                ├─ Reassembles chunks into temp file
                ├─ Validates MIME type (same as single-upload)
                ├─ Resolves customer folder (same logic as single-upload)
                ├─ copy() + unlink() to /mnt/dropbox/ (cross-filesystem)
                ├─ Triggers background thumbnail generation
                ├─ Cleans up .chunks/{file_id}.*
                └─ Returns JSON success


  RESUME SCENARIO (process restarts, retry queue picks up):

  process_retry_queue.py
  │
  └─ file_size >= 3MB, upload_mode='chunked'
      │
      ├─ 1. INIT: POST action=photo_chunk_init (same file_id)
      │     └─ Server returns chunks_received: [0, 1, 2]  ← already uploaded
      │
      ├─ 2. CHUNK: Only sends chunks [3, 4] (skips 0-2)
      │
      └─ 3. FINALIZE: Same as above
```

---

## Implementation Details

### File Changes

#### 1. `uploadlocallat_kuldeep.php` (local server receiver)

**New constants:**
```php
define('CHUNKS_DIR', __DIR__ . '/.chunks');
define('CHUNK_MAX_AGE', 48 * 3600);  // 48 hours
```

**New helper functions:**
- `ensureChunksDir()` — creates `.chunks/` directory if needed
- `isValidFileId($file_id)` — validates 40-char hex string
- `getChunkMetaPath($file_id)` — returns `.chunks/{file_id}.meta`
- `getChunkPartPath($file_id, $index)` — returns `.chunks/{file_id}.part.{index}`
- `readChunkMeta($file_id)` — reads JSON metadata file
- `writeChunkMeta($file_id, $meta)` — writes JSON metadata file

**New handlers:**
- `handlePhotoChunkInit()` — validates params, creates/reads session, returns progress
- `handlePhotoChunkUpload()` — receives single chunk, stores as `.part.N` file
- `handlePhotoChunkFinalize()` — reassembles chunks, validates MIME, resolves destination, saves file, triggers thumbnails, cleans up session

**Refactored function:**
- `resolvePhotoDestination(...)` — extracted from lines 374-463, called by both finalize and existing single-upload path

**Routing:**
```php
// After function definitions, before existing validation flow:
if (isset($_POST['action'])) {
    switch ($_POST['action']) {
        case 'photo_chunk_init':     handlePhotoChunkInit(); break;
        case 'photo_chunk_upload':   handlePhotoChunkUpload(); break;
        case 'photo_chunk_finalize': handlePhotoChunkFinalize(); break;
        default: /* fall through to existing flow */
    }
}
// Existing single-upload flow continues unchanged below
```

**Critical detail:** Finalize uses `copy()` + `unlink()`, NOT `rename()`. The `.chunks/` directory is on the local SSD (`/var/www/html/upload/.chunks/`) while the destination is on CIFS mount (`/mnt/dropbox/`). `rename()` fails across filesystems; `copy()` + `unlink()` works correctly.

#### 2. `sync_to_local.py` (remote server sender)

**New constants:**
```python
CHUNK_THRESHOLD = 3 * 1024 * 1024  # 3MB
CHUNK_SIZE = 1 * 1024 * 1024       # 1MB
```

**New functions:**
- `generate_file_id(file_name, job_id, customer_id, file_size)` — SHA-1 of `name:job_id:cust_id:size`
- `_chunked_request(data, files=None)` — single HTTP POST with per-request retry (reuses `INLINE_RETRIES` / `RETRY_DELAYS`)
- `sync_file_chunked(file_path, file_name, job_id, customer_id, job_type, last_name, first_name, job_date, photo_type)` — 3-phase chunked upload, returns `(success, error_reason)`

**Modified `__main__`:** Routes to `sync_file_chunked()` if file >= 3MB, else existing single-POST `sync_file()`.

**Modified `enqueue_failed()`:** Adds `upload_mode` field (`'single'` or `'chunked'`) to queue JSON.

#### 3. `process_retry_queue.py` (remote server retry processor)

**New constants and functions:** Duplicated from `sync_to_local.py` (matches established pattern — both files are standalone scripts in the same directory).

**Modified `process_queue()`:** Detects `upload_mode` from queue JSON:
- `upload_mode == 'chunked'` → calls `sync_file_chunked()`
- `upload_mode == 'single'` or missing → calls `sync_file()`
- Auto-upgrade: if `upload_mode` is missing and file >= 3MB, uses chunked

---

## Security Considerations

| Concern | Mitigation |
|---------|------------|
| Auth bypass on chunk endpoints | Every request (init, chunk, finalize) validates `auth_token` |
| Malicious file content | MIME type validation at finalize (same `finfo_file()` check as single-upload) |
| Path traversal via file_id | `isValidFileId()` requires exactly 40 hex chars — no slashes, dots, or special chars |
| .chunks directory web access | Dot-prefix is hidden by default Apache config; add `.htaccess` deny rule as defense-in-depth |
| Chunk index injection | Validated as non-negative integer < `total_chunks` |
| Abandoned chunks consuming disk | Probabilistic cleanup: 1% chance on each init call, removes sessions > 48 hours old |
| Cross-filesystem copy | Uses `copy()` + `unlink()` instead of `rename()` (fails across FS boundaries) |

---

## Edge Cases

| Scenario | Handling |
|----------|----------|
| **Cross-filesystem copy** | `copy()` + `unlink()` — `.chunks/` is local SSD, `/mnt/dropbox/` is CIFS mount |
| **Concurrent uploads, same file** | Deterministic `file_id` means same session — second process resumes from first's progress |
| **Abandoned chunks** | 48-hour expiry, probabilistic cleanup (1% on each init) |
| **Backward compatibility** | Existing single-upload path is unchanged — no `action` field → old flow |
| **Old queue items** | Queue JSON without `upload_mode` defaults to `'single'`; auto-upgraded to chunked if file >= 3MB |
| **Chunk out of order** | Each chunk is stored independently as `.part.N` — order doesn't matter |
| **Duplicate chunk** | Overwrites existing `.part.N` — idempotent |
| **Finalize with missing chunks** | Server validates all chunks present before assembly; returns error listing missing indices |
| **File size mismatch** | Finalize verifies assembled size matches declared `file_size` |

---

## Deployment Sequence

```bash
# 1. Create .chunks directory on local server
sudo mkdir -p /var/www/html/upload/.chunks
sudo chown www-data:www-data /var/www/html/upload/.chunks
sudo chmod 755 /var/www/html/upload/.chunks

# 2. Add .htaccess to deny web access to .chunks/
echo "Deny from all" | sudo tee /var/www/html/upload/.chunks/.htaccess

# 3. Deploy receiver FIRST (backward compatible — still accepts single POST)
sudo cp /var/www/html/upload/uploadlocallat_kuldeep.php \
        /var/www/html/upload/uploadlocallat_kuldeep.php.bak.pre_chunked.$(date +%Y%m%d)
sudo cp /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/LOCAL/uploadlocallat_kuldeep.php.current \
        /var/www/html/upload/uploadlocallat_kuldeep.php
sudo chown root:root /var/www/html/upload/uploadlocallat_kuldeep.php
sudo chmod 755 /var/www/html/upload/uploadlocallat_kuldeep.php

# 4. Test chunked endpoint with manual curl from remote (see Verification section)

# 5. Deploy sender (sync_to_local.py) on remote
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py \
     /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py.bak.pre_chunked.\$(date +%Y%m%d)
"
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/sync_to_local.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
"

# 6. Deploy retry processor (process_retry_queue.py) on remote
scp -i /root/.ssh/aei_remote.pem \
  /var/opt/AEI_REMOTE/AEI_PHOTO_API_PROJECT/REMOTE/photoapi/process_retry_queue.py \
  Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
  sudo chmod 755 /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
"

# 7. Verify with a real >3MB photo upload through mobile app
```

---

## Verification

```bash
# 1. Test init endpoint
curl -sk -X POST \
  -d "action=photo_chunk_init&auth_token=remote_token&file_id=$(python3 -c "import hashlib; print(hashlib.sha1(b'test:1:1:3000000').hexdigest())")&file_name=test.jpg&file_size=3000000&total_chunks=3&chunk_size=1048576&job_id=1&customer_id=1&job_type=PM&last_name=Test&first_name=User&job_date=2026-02-05&photo_type=S" \
  https://upload.aeihawaii.com/uploadlocallat_kuldeep.php

# 2. Verify single-upload still works (no action field)
# Upload a test photo through mobile app — should work identically

# 3. Check .chunks directory for session files during upload
ls -la /var/www/html/upload/.chunks/

# 4. Monitor sender logs for CHUNKED entries
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 \
  "tail -f /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/logs/sync_to_local.log"

# 5. Monitor local server Apache logs
tail -f /var/log/apache2/upload_access.log | grep "18.225.0.90"
```

---

## Rollback

### Sender Rollback (reverts to single POST only)
```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py.bak.pre_chunked.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/sync_to_local.py
"
```

### Retry Processor Rollback
```bash
ssh -i /root/.ssh/aei_remote.pem Julian@18.225.0.90 "
  sudo cp /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py.bak.pre_chunked.20260205 \
          /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
  sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/photoapi/process_retry_queue.py
"
```

### Receiver
No rollback needed — chunked handlers are inert without a chunked sender. The existing single-upload path is completely unchanged. The `.chunks/` directory can remain.

### Queue Items
- Old queue items (without `upload_mode`) continue working with the rolled-back sender
- Chunked queue items would fall back to single POST if the sender is rolled back (file >= 3MB just uses single POST again)

---

## Related Documentation

| Document | Purpose |
|----------|---------|
| [ASYNC_SYNC_ENHANCEMENT.md](ASYNC_SYNC_ENHANCEMENT.md) | Background sync and retry queue (foundation for this enhancement) |
| [WSL2_NETWORKING_INVESTIGATION.md](WSL2_NETWORKING_INVESTIGATION.md) | WSL2 TCP SYN-ACK issue (motivation for chunked approach) |
| [PHOTO_SYSTEM_DOCUMENTATION.md](PHOTO_SYSTEM_DOCUMENTATION.md) | Complete system documentation (updated with chunked upload) |

---

*Created: 2026-02-05*
