# AEI Database Schema Documentation

## Overview

This document describes the structure of the AEI (Alternate Energy, Inc.) scheduling and job management database. The database is located at `18.225.0.90` in the `mandhdesign_schedular` schema.

**Database Statistics:**
- ~17,000 customers
- ~84,000 jobs
- ~191,000 notes
- ~44,600 proposals
- ~446 total tables (many are backups/archives)

---

## Core Entity Relationships

```
                                    ┌─────────────┐
                                    │   users     │
                                    │ (employees) │
                                    └──────┬──────┘
                                           │
    ┌──────────────┐      ┌────────────────┼────────────────┐
    │    lead      │      │                │                │
    │ (prospects)  │      │                ▼                │
    └──────┬───────┘      │        ┌──────────────┐         │
           │              │        │  customers   │         │
           │ converts to  │        └──────┬───────┘         │
           └──────────────┼───────────────┤                 │
                          │               │                 │
                          │    ┌──────────┼──────────┐      │
                          │    │          │          │      │
                          ▼    ▼          ▼          ▼      │
                    ┌──────────┐    ┌──────────┐  ┌─────────┴───┐
                    │ proposal │    │   jobs   │  │ ost_ticket  │
                    └──────────┘    └────┬─────┘  │  (support)  │
                                         │        └─────────────┘
                         ┌───────────────┼───────────────┐
                         │               │               │
                         ▼               ▼               ▼
                    ┌─────────┐    ┌──────────┐    ┌──────────┐
                    │  notes  │    │  files   │    │ referrals│
                    └─────────┘    └──────────┘    └──────────┘
```

---

## Primary Tables

### 1. customers

The central entity - stores all customer information.

| Column | Type | Description |
|--------|------|-------------|
| id | int(11) | Primary key |
| first_name | varchar(255) | Customer first name |
| last_name | varchar(255) | Customer last name |
| first_name1, last_name1 | varchar(255) | Secondary contact (spouse) |
| home_phone | varchar(255) | Primary phone |
| secondary_phone | varchar(255) | Alt phone |
| email | varchar(255) | Email address |
| address | varchar(255) | Street address |
| city | varchar(255) | City |
| state | varchar(255) | State (usually HI) |
| zip_code | varchar(255) | ZIP code |
| neighborhood | varchar(255) | Neighborhood name |
| association | varchar(255) | HOA name if applicable |
| customer_type_id | int(11) | FK to customer_types (1=Residential, 2=Commercial) |
| household_type_id | int(11) | FK to household_types |
| avg_electric_bill | varchar(255) | Average monthly bill |
| sun_zone | varchar(255) | Solar zone for production estimates |
| heco_account_number | varchar(255) | Hawaiian Electric account |
| heco_meter_number | varchar(255) | Meter number |
| comments | text | General notes |
| cust_tmk | varchar(200) | Tax Map Key |
| created | datetime | Record creation date |
| customer_status | int(11) | Status flag (0=active) |

**Common Queries:**
```sql
-- Find customer by name
SELECT * FROM customers
WHERE first_name LIKE '%John%' OR last_name LIKE '%Smith%';

-- Find customer by address
SELECT * FROM customers
WHERE address LIKE '%Kailua%' OR city LIKE '%Kailua%';

-- Find customer by phone
SELECT * FROM customers
WHERE home_phone LIKE '%808555%' OR secondary_phone LIKE '%808555%';

-- Get customer with job count
SELECT c.*, COUNT(j.id) as total_jobs
FROM customers c
LEFT JOIN jobs j ON c.id = j.customer_id
WHERE c.id = ?
GROUP BY c.id;
```

---

### 2. jobs

All scheduled work - installations, service calls, inspections, etc.

| Column | Type | Description |
|--------|------|-------------|
| id | int(11) | Primary key |
| customer_id | int(11) | FK to customers |
| job_type_id | int(11) | FK to job_types |
| user_id | int(11) | Sales rep who created |
| job_date | date | Scheduled date |
| current_status | varchar(100) | Status text |
| job_phase_status | varchar(50) | Phase status |

**PV System Details (for Photovoltaic jobs):**
| Column | Type | Description |
|--------|------|-------------|
| module_brand_id | int(11) | FK to module_brands |
| module_count | varchar(255) | Number of panels |
| system_size | varchar(255) | System size in kW |
| module_power_id | int(11) | FK to module_powers |
| inverter_type_id | int(11) | FK to inverter_types |
| inverter_brand_id | int(11) | FK to inverter_brands |
| inverter_count | varchar(255) | Number of inverters |
| roof_type_id | int(11) | FK to roof_types |

**SWH System Details (for Solar Water Heating):**
| Column | Type | Description |
|--------|------|-------------|
| tank_brand_id | int(11) | FK to tank_brands |
| tank_size_id | int(11) | FK to tank_sizes |
| panel_type_id | int(11) | FK to panel_types (SWH collector) |
| panel_size_id | int(255) | FK to panel_sizes |
| panel_count | int(11) | Number of collectors |

**Status/Permit Fields:**
| Column | Type | Description |
|--------|------|-------------|
| electrical_permit | int(11) | Electrical permit status |
| building_permit | int(11) | Building permit status |
| nemstatus_permit | int(11) | Net Energy Metering status |
| installed_date | datetime | Installation completed date |
| job_status | tinyint(2) | 0=active, 1=cancelled |

**Common Queries:**
```sql
-- Get all jobs for a customer with type names
SELECT j.*, jt.type as job_type_name, jt.intials as job_code
FROM jobs j
JOIN job_types jt ON j.job_type_id = jt.id
WHERE j.customer_id = ?
ORDER BY j.job_date DESC;

-- Get PV jobs with equipment details
SELECT j.*,
    mb.brand as module_brand,
    j.module_count,
    j.system_size,
    ib.brand as inverter_brand
FROM jobs j
LEFT JOIN module_brands mb ON j.module_brand_id = mb.id
LEFT JOIN inverter_brands ib ON j.inverter_brand_id = ib.id
WHERE j.job_type_id = 2  -- Photovoltaic
AND j.customer_id = ?;

-- Get SWH jobs with equipment
SELECT j.*,
    tb.brand as tank_brand,
    ts.size as tank_size,
    pt.type as panel_type
FROM jobs j
LEFT JOIN tank_brands tb ON j.tank_brand_id = tb.id
LEFT JOIN tank_sizes ts ON j.tank_size_id = ts.id
LEFT JOIN panel_types pt ON j.panel_type_id = pt.id
WHERE j.job_type_id = 1  -- Solar Water Heating
AND j.customer_id = ?;
```

---

### 3. job_types

Lookup table for job categories.

| id | type | intials |
|----|------|---------|
| 1 | Solar Water Heating | SWH |
| 2 | Photovoltaic | PV |
| 3 | Solar Attic Fan | SAF |
| 5 | Service Call | SC |
| 6 | Prelag | PL |
| 7 | Premeasure | PM |
| 8 | Electrical Upgrades & Service | EL |
| 9 | SWH Premeasure | WM |
| 16 | AC | AC |
| 21 | EV | EV |
| 25 | Service Plans | SP |
| 29 | Roofing | RF |
| 30 | Storage | ST |

---

### 4. notes

Comments and activity log for jobs.

| Column | Type | Description |
|--------|------|-------------|
| id | int(11) | Primary key |
| job_id | int(11) | FK to jobs |
| service_call_id | int(11) | FK to service_calls (if applicable) |
| user_id | int(11) | User who created note |
| note | text | Note content (may contain HTML) |
| created | datetime | Timestamp |
| comment_type | tinyint(2) | FK to comment_type |

**comment_type values:**
| id | comment_type |
|----|--------------|
| 1 | Site Survey |
| 2 | Installation (Roof & Electrical) |
| 3 | Administrative & Accounting |
| 4 | Rescheduling Activity Comment |
| 5 | Technician Notes |
| 6 | Design & Engineering |

**Common Queries:**
```sql
-- Get all notes for a customer's jobs
SELECT n.*, j.id as job_id, jt.type as job_type, u.first_name as author
FROM notes n
JOIN jobs j ON n.job_id = j.id
JOIN job_types jt ON j.job_type_id = jt.id
LEFT JOIN users u ON n.user_id = u.id
WHERE j.customer_id = ?
ORDER BY n.created DESC;

-- Get recent notes for a specific job
SELECT n.*, ct.comment_type as category
FROM notes n
LEFT JOIN comment_type ct ON n.comment_type = ct.id
WHERE n.job_id = ?
ORDER BY n.created DESC
LIMIT 20;
```

---

### 5. proposal

Sales proposals/quotes for customers.

| Column | Type | Description |
|--------|------|-------------|
| id | int(10) | Primary key |
| customer_id | int(11) | FK to customers (if existing customer) |
| dpp_firstname | varchar(150) | First name on proposal |
| dpp_lastname | varchar(150) | Last name on proposal |
| address | varchar(150) | Installation address |
| city | varchar(100) | City |
| phone | varchar(100) | Phone |
| email | varchar(100) | Email |
| job_type | tinyint(4) | Type of system |
| system_size | varchar(50) | System size in kW |
| nom | int(11) | Number of modules |
| price | varchar(50) | Quoted price |
| status | varchar(100) | Proposal status (new, sold, etc.) |
| user_id | int(11) | Sales rep |
| created | datetime | Creation date |
| lead_source_id | int(11) | FK to lead_origin |

**Common Queries:**
```sql
-- Get proposals for a customer
SELECT p.*, u.first_name as sales_rep
FROM proposal p
LEFT JOIN users u ON p.user_id = u.id
WHERE p.customer_id = ?
ORDER BY p.created DESC;

-- Search proposals by name
SELECT * FROM proposal
WHERE dpp_firstname LIKE '%name%'
   OR dpp_lastname LIKE '%name%'
   OR address LIKE '%address%'
ORDER BY created DESC;
```

---

### 6. lead

Prospective customers before conversion.

| Column | Type | Description |
|--------|------|-------------|
| id | int(10) | Primary key |
| firstname | varchar(150) | First name |
| lastname | varchar(150) | Last name |
| email | varchar(150) | Email |
| phone | varchar(150) | Phone |
| address | varchar(150) | Address |
| city | varchar(150) | City |
| lead_source_id | int(11) | FK to lead_origin |
| sales_user_id | varchar(150) | Assigned sales rep |
| status | varchar(50) | Lead status |
| proposal_id | int(11) | FK to proposal if created |
| comment | text | Notes |
| create_date | datetime | Creation date |
| is_delete | int(1) | 1=active, 0=deleted |

---

### 7. ost_ticket (osTicket Support Tickets)

Customer service/support tickets.

| Column | Type | Description |
|--------|------|-------------|
| ticket_id | int(11) | Primary key |
| number | varchar(20) | Ticket number |
| customer_id | int(11) | FK to customers |
| job_id | int(11) | FK to jobs |
| status | enum | 'open','closed' |
| source | enum | 'Web','Email','Phone','API','BBB','RICO','AEI-Website','Other' |
| job_type | varchar(50) | Related job type |
| created | datetime | Creation date |
| closed | datetime | Closure date |
| staff_id | int(10) | Assigned staff |

**Related tables:**
- `ost_ticket_thread` - Ticket messages/replies
- `ost_user` - Support users
- `ost_user_email` - User email addresses

---

### 8. files

Attached files for jobs.

| Column | Type | Description |
|--------|------|-------------|
| id | int(11) | Primary key |
| job_id | int(11) | FK to jobs |
| service_call_id | int(11) | FK to service_calls |
| unique_filename | varchar(255) | Stored filename |
| original_filename | varchar(255) | Original upload name |
| mime | varchar(255) | MIME type |
| created | datetime | Upload date |

---

### 9. users

System users (employees, sales reps, installers).

| Column | Type | Description |
|--------|------|-------------|
| id | int(11) | Primary key |
| uname | varchar(255) | Username |
| first_name | varchar(255) | First name |
| last_name | varchar(255) | Last name |
| email | varchar(255) | Email |
| type | varchar(255) | Role: admin, sales, office, installer, etc. |
| user_status | varchar(11) | 'active' or 'inactive' |

---

## Equipment Lookup Tables

### PV Equipment

| Table | Description |
|-------|-------------|
| module_brands | Solar panel manufacturers (Panasonic, REC, etc.) |
| module_powers | Panel wattages |
| inverter_brands | Inverter manufacturers (SMA, Enphase, SolarEdge) |
| inverter_types | Inverter models |
| inverter_power | Inverter sizes |

### SWH Equipment

| Table | Description |
|-------|-------------|
| tank_brands | Water tank manufacturers |
| tank_sizes | Tank capacities |
| panel_types | Collector types (ThermoRay, Solene, Sun Earth) |
| panel_sizes | Collector sizes |

### Site Details

| Table | Description |
|-------|-------------|
| roof_types | Roof materials (Metal, Tile, Shingle, etc.) |
| rafter_types | Rafter/truss types |
| service_panel_amperages | Electrical panel sizes |
| wire_run_types | Wire run methods |

---

## Permit & HOA Tables

### permit_details
Stores permit information from county systems.

### hoa_details
HOA/association information including:
- Association name and contact
- Design review requirements
- Fees and turnaround times
- Special restrictions

---

## Common Search Patterns

### Find Customer by Any Criteria
```sql
SELECT c.*,
    (SELECT COUNT(*) FROM jobs WHERE customer_id = c.id) as job_count
FROM customers c
WHERE
    c.first_name LIKE CONCAT('%', ?, '%')
    OR c.last_name LIKE CONCAT('%', ?, '%')
    OR c.address LIKE CONCAT('%', ?, '%')
    OR c.email LIKE CONCAT('%', ?, '%')
    OR c.home_phone LIKE CONCAT('%', ?, '%')
    OR c.heco_account_number = ?
ORDER BY c.last_name, c.first_name;
```

### Get Complete Customer Profile
```sql
-- Customer details
SELECT * FROM customers WHERE id = ?;

-- All jobs with types
SELECT j.*, jt.type, jt.intials
FROM jobs j
JOIN job_types jt ON j.job_type_id = jt.id
WHERE j.customer_id = ?
ORDER BY j.job_date DESC;

-- All notes across jobs
SELECT n.*, j.id as job_id, jt.type as job_type
FROM notes n
JOIN jobs j ON n.job_id = j.id
JOIN job_types jt ON j.job_type_id = jt.id
WHERE j.customer_id = ?
ORDER BY n.created DESC;

-- Proposals
SELECT * FROM proposal WHERE customer_id = ?;

-- Support tickets
SELECT * FROM ost_ticket WHERE customer_id = ?;
```

### Find Jobs by Status/Date
```sql
-- Active PV installations this month
SELECT j.*, c.first_name, c.last_name, c.address
FROM jobs j
JOIN customers c ON j.customer_id = c.id
WHERE j.job_type_id = 2
AND j.job_date >= DATE_FORMAT(NOW(), '%Y-%m-01')
AND j.job_status = 0;

-- Pending permit jobs
SELECT j.*, c.first_name, c.last_name
FROM jobs j
JOIN customers c ON j.customer_id = c.id
WHERE j.electrical_permit = 0 OR j.building_permit = 0
AND j.job_status = 0;
```

---

## Notes on Data Quality

1. **Many backup tables**: Tables with dates in names (e.g., `system_price_13jan2015`) are historical backups - use current tables only.

2. **HTML in notes**: The `notes.note` field often contains HTML formatting - strip tags for plain text display.

3. **Varchar numbers**: Many numeric fields are stored as varchar (system_size, prices) - cast when needed.

4. **Timestamps**: Created dates may be NULL on older records.

5. **Soft deletes**: Some tables use is_delete or status flags instead of hard deletes.

---

## AI Query Integration Notes

For the chat system to query this database effectively:

1. **Customer lookup**: Search by name, address, phone, email, or HECO account number
2. **Job history**: Get all jobs for a customer with equipment details
3. **Notes/Activity**: Retrieve conversation history and job notes
4. **Equipment details**: Look up what was installed (panels, inverters, tanks)
5. **Status tracking**: Check permit status, installation status, service history

The database is **READ-ONLY** for the chat system - no modifications allowed.
