<?php  if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Creates a WEBP derivative next to $src.
 * Returns the public URL path to the generated file, or original URL on failure.
 *
 * $srcFs: absolute filesystem path to original
 * $srcUrl: public URL to original
 * $suffix: '', '.thumb' to distinguish sizes
 * $maxW,$maxH: resize box (keeps aspect)
 * $quality: webp quality (0-100)
 */
if (!function_exists('pcos_make_webp')) {
    function pcos_make_webp($srcFs, $srcUrl, $suffix = '', $maxW = null, $maxH = null, $quality = 80)
    {
        $pi = pathinfo($srcFs);
        $dstFs = $pi['dirname'] . '/' . $pi['filename'] . $suffix . '.webp';
        $dstUrl = preg_replace('~\.' . preg_quote($pi['extension'], '~') . '$~i', $suffix . '.webp', $srcUrl);

        // If already exists, return it
        if (is_file($dstFs) && filesize($dstFs) > 0) return $dstUrl;

        // Try GD → imagewebp (PHP/GD must support webp)
        $ext = strtolower($pi['extension']);
        $im = null;
        if (in_array($ext, array('jpg','jpeg'))) {
            $im = @imagecreatefromjpeg($srcFs);
        } elseif ($ext === 'png') {
            $im = @imagecreatefrompng($srcFs);
            if ($im && function_exists('imagesavealpha')) {
                imagesavealpha($im, true);
            }
        } elseif ($ext === 'gif') {
            $im = @imagecreatefromgif($srcFs);
        } elseif ($ext === 'webp') {
            // source already webp
            return $srcUrl;
        }

        // Resize if requested
        if ($im && $maxW && $maxH) {
            $w = imagesx($im); $h = imagesy($im);
            $ratio = min($maxW / max(1,$w), $maxH / max(1,$h));
            if ($ratio < 1) {
                $nw = max(1, (int)floor($w * $ratio));
                $nh = max(1, (int)floor($h * $ratio));
                $tmp = imagecreatetruecolor($nw, $nh);
                // Preserve transparency for PNG/GIF
                imagealphablending($tmp, false);
                imagesavealpha($tmp, true);
                imagecopyresampled($tmp, $im, 0,0,0,0, $nw,$nh, $w,$h);
                imagedestroy($im);
                $im = $tmp;
            }
        }

        // Prefer GD imagewebp
        if ($im && function_exists('imagewebp')) {
            @imagewebp($im, $dstFs, $quality);
            imagedestroy($im);
            if (is_file($dstFs) && filesize($dstFs) > 0) return $dstUrl;
        } elseif ($im) {
            imagedestroy($im);
        }

        // Fallback: try cwebp CLI (requires webp package)
        // Build a resize param for cwebp if requested
        $resize = '';
        if ($maxW && $maxH) $resize = " -resize " . (int)$maxW . " " . (int)$maxH . " ";
        $cmd = "cwebp -q " . (int)$quality . $resize . " " . escapeshellarg($srcFs) . " -o " . escapeshellarg($dstFs) . " 2>&1";
        @exec($cmd, $out, $rc);
        if (is_file($dstFs) && filesize($dstFs) > 0) return $dstUrl;

        // If all fails, return original URL
        return $srcUrl;
    }
}
