<?php

class Timesheet extends Controller {

    function Timesheet() {
        parent::Controller();
        if (!$this->session->userdata('logged_in')) {
            $tempurl = explode("/index.php/", $_SERVER['PHP_SELF']);
            redirect("login?ref=" . $tempurl[1]);
        }
        // Must be superadmin or have time_sheet permission
        if (!$this->session->userdata('superadmin') && !$this->session->userdata('time_sheet')) {
            redirect("admin/jobschedule");
        }
        $this->load->helper(array('form', 'url'));
    }

    /**
     * Check if current user is superadmin
     */
    function _is_admin() {
        return ($this->session->userdata('superadmin') || $this->session->userdata('superuser')) ? true : false;
    }

    /**
     * Get current user ID
     */
    function _user_id() {
        return intval($this->session->userdata('user_id'));
    }

    // ========================================
    // Hardening Helpers (Phase 3)
    // ========================================

    /**
     * Safe integer cast with default fallback
     */
    function _safe_int($value, $default = 0) {
        $val = intval($value);
        return ($val > 0) ? $val : intval($default);
    }

    /**
     * Validate YYYY-MM-DD date format + checkdate
     */
    function _safe_date($value) {
        $value = trim($value);
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return false;
        }
        $parts = explode('-', $value);
        if (!checkdate(intval($parts[1]), intval($parts[2]), intval($parts[0]))) {
            return false;
        }
        return $value;
    }

    /**
     * Validate YYYY-MM-DD HH:MM:SS datetime format
     */
    function _safe_datetime($value) {
        $value = trim($value);
        if (!preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/', $value)) {
            return false;
        }
        if (strtotime($value) === false) {
            return false;
        }
        $parts = explode(' ', $value);
        $date_parts = explode('-', $parts[0]);
        if (!checkdate(intval($date_parts[1]), intval($date_parts[2]), intval($date_parts[0]))) {
            return false;
        }
        return $value;
    }

    /**
     * JSON success response with Content-Type header
     */
    function _json_ok($data = array()) {
        header('Content-Type: application/json');
        $data['success'] = true;
        echo json_encode($data);
    }

    /**
     * JSON error response with Content-Type header
     */
    function _json_error($message) {
        header('Content-Type: application/json');
        echo json_encode(array('success' => false, 'message' => $message));
    }

    /**
     * Check admin + send json_error if not. Returns true if admin.
     */
    function _require_admin() {
        if (!$this->_is_admin()) {
            $this->_json_error('Access denied.');
            return false;
        }
        return true;
    }

    /**
     * Escape a LIKE search term (prevents wildcard injection)
     */
    function _escape_like($term) {
        return '%' . $this->db->escape_like_str($term) . '%';
    }

    /**
     * Insert audit trail row
     */
    function _audit_log($entry_id, $action, $before = null, $after = null, $reason = null) {
        $sql = "INSERT INTO ts_entry_audit (entry_id, action, changed_by, changed_at, before_json, after_json, reason)
                VALUES (?, ?, ?, ?, ?, ?, ?)";
        $this->db->query($sql, array(
            intval($entry_id),
            $action,
            $this->_user_id(),
            date('Y-m-d H:i:s'),
            $before !== null ? json_encode($before) : null,
            $after !== null ? json_encode($after) : null,
            $reason
        ));
    }

    // ========================================
    // Page Controllers
    // ========================================

    /**
     * Main page - clock button + current week grid
     */
    function index() {
        $data = array();
        $data['is_admin'] = $this->_is_admin();
        $data['user_id'] = $this->_user_id();

        // Get current clock-in status
        $data['active_entry'] = $this->_get_active_entry($data['user_id']);

        // Get current week entries (Mon-Sun)
        $today = date('Y-m-d');
        $day_of_week = date('N'); // 1=Mon, 7=Sun
        $monday = date('Y-m-d', strtotime("-" . ($day_of_week - 1) . " days", strtotime($today)));
        $sunday = date('Y-m-d', strtotime("+" . (7 - $day_of_week) . " days", strtotime($today)));

        $data['week_start'] = $monday;
        $data['week_end'] = $sunday;
        $data['week_entries'] = $this->_get_entries_range($data['user_id'], $monday, $sunday);
        $data['today_total'] = $this->_get_day_total($data['user_id'], $today);
        $data['week_total'] = $this->_get_range_total($data['user_id'], $monday, $sunday);

        $this->load->view("common/header");
        $this->load->view("timesheet/index", $data);
        $this->load->view("common/footer");
    }

    /**
     * AJAX: Clock in (optionally to a specific customer)
     */
    function clock_in() {
        $user_id = $this->_user_id();
        $now = date('Y-m-d H:i:s');
        $customer_id = $this->input->post('customer_id') ? intval($this->input->post('customer_id')) : 0;

        // Check for existing active entry
        $active = $this->_get_active_entry($user_id);
        if ($active) {
            $this->_json_error('You are already clocked in. Clock out first before starting a new entry.');
            return;
        }

        // Validate customer_id exists if provided
        if ($customer_id > 0) {
            $cust_check = $this->db->query("SELECT id FROM customers WHERE id = ?", array($customer_id))->row_array();
            if (!$cust_check) {
                $this->_json_error('Invalid customer selected.');
                return;
            }
        }

        if ($customer_id > 0) {
            $sql = "INSERT INTO ts_time_entries (user_id, customer_id, clock_in, entry_type, status, created_by, created_at)
                    VALUES (?, ?, ?, 'clock', 'active', ?, ?)";
            $this->db->query($sql, array($user_id, $customer_id, $now, $user_id, $now));
        } else {
            $sql = "INSERT INTO ts_time_entries (user_id, clock_in, entry_type, status, created_by, created_at)
                    VALUES (?, ?, 'clock', 'active', ?, ?)";
            $this->db->query($sql, array($user_id, $now, $user_id, $now));
        }

        $new_id = $this->db->insert_id();

        // Audit log
        $after_data = array('user_id' => $user_id, 'clock_in' => $now, 'customer_id' => $customer_id, 'entry_type' => 'clock');
        $this->_audit_log($new_id, 'create', null, $after_data);

        $msg = 'Clocked in at ' . date('g:i A');
        if ($customer_id) {
            $label = $this->_get_customer_label($customer_id);
            $msg .= ' - ' . ($label ? $label : 'Customer #' . $customer_id);
        }

        $this->_json_ok(array(
            'message' => $msg,
            'clock_in' => date('g:i A'),
            'clock_in_raw' => $now,
            'customer_id' => $customer_id
        ));
    }

    /**
     * AJAX: Clock out
     */
    function clock_out() {
        $user_id = $this->_user_id();
        $now = date('Y-m-d H:i:s');

        $active = $this->_get_active_entry($user_id);
        if (!$active) {
            $this->_json_error('You are not clocked in.');
            return;
        }

        // Calculate duration
        $clock_in_time = strtotime($active['clock_in']);
        $clock_out_time = strtotime($now);
        $duration_minutes = round(($clock_out_time - $clock_in_time) / 60);

        // Prevent negative/zero duration
        if ($duration_minutes <= 0) {
            $this->_json_error('Clock out time must be after clock in time.');
            return;
        }

        // Auto break: 30 min if shift >= 6 hours (360 min)
        $break_minutes = 0;
        if ($duration_minutes >= 360) {
            $break_minutes = 30;
        }

        // Audit: capture before state
        $before_data = array(
            'id' => $active['id'],
            'clock_in' => $active['clock_in'],
            'clock_out' => null,
            'duration_minutes' => null,
            'break_minutes' => null
        );

        $sql = "UPDATE ts_time_entries SET
                clock_out = ?,
                duration_minutes = ?,
                break_minutes = ?,
                updated_by = ?,
                updated_at = ?
                WHERE id = ?";
        $this->db->query($sql, array($now, $duration_minutes, $break_minutes, $user_id, $now, intval($active['id'])));

        // Audit log
        $after_data = array(
            'clock_out' => $now,
            'duration_minutes' => $duration_minutes,
            'break_minutes' => $break_minutes
        );
        $this->_audit_log($active['id'], 'edit', $before_data, $after_data);

        $net_minutes = $duration_minutes - $break_minutes;

        $this->_json_ok(array(
            'message' => 'Clocked out at ' . date('g:i A'),
            'duration' => $this->_format_minutes($net_minutes),
            'clock_out' => date('g:i A')
        ));
    }

    /**
     * AJAX: Get current clock status
     */
    function get_status() {
        $user_id = $this->_user_id();
        $active = $this->_get_active_entry($user_id);
        $today_total = $this->_get_day_total($user_id, date('Y-m-d'));

        $this->_json_ok(array(
            'clocked_in' => $active ? true : false,
            'clock_in_time' => $active ? date('g:i A', strtotime($active['clock_in'])) : null,
            'entry_id' => $active ? $active['id'] : null,
            'today_total' => $this->_format_minutes($today_total)
        ));
    }

    /**
     * Weekly detail view
     */
    function weekly($y = 0, $m = 0, $d = 0) {
        $data = array();
        $data['is_admin'] = $this->_is_admin();

        // Role filter (admin only, session-persistent)
        $role_filter = array();
        if ($data['is_admin']) {
            $role_filter = $this->_get_role_filter();
            $data['roles'] = $this->_get_roles();
        }
        $data['role_filter'] = $role_filter;

        // Group filter
        $group_filter = 0;
        if ($data['is_admin']) {
            $group_filter = $this->_get_group_filter();
            $data['group_filter'] = $group_filter;
            $data['groups'] = $this->_get_all_groups();
        }

        // Allow admin to view other users
        $view_user_id = $this->_user_id();
        if ($data['is_admin'] && isset($_GET['user_id'])) {
            $view_user_id = intval($_GET['user_id']);
        }
        $data['view_user_id'] = $view_user_id;
        $data['user_id'] = $this->_user_id();

        // Get user name for display
        $data['view_user_name'] = $this->_get_user_name($view_user_id);

        // Determine week start
        if ($y && $m && $d) {
            $target_date = $y . '-' . str_pad($m, 2, '0', STR_PAD_LEFT) . '-' . str_pad($d, 2, '0', STR_PAD_LEFT);
        } else {
            $target_date = date('Y-m-d');
        }

        $day_of_week = date('N', strtotime($target_date));
        $monday = date('Y-m-d', strtotime("-" . ($day_of_week - 1) . " days", strtotime($target_date)));
        $sunday = date('Y-m-d', strtotime("+6 days", strtotime($monday)));

        $data['week_start'] = $monday;
        $data['week_end'] = $sunday;
        $data['prev_week'] = date('Y/m/d', strtotime("-7 days", strtotime($monday)));
        $data['next_week'] = date('Y/m/d', strtotime("+7 days", strtotime($monday)));

        // Build day-by-day data
        $data['days'] = array();
        for ($i = 0; $i < 7; $i++) {
            $day_date = date('Y-m-d', strtotime("+$i days", strtotime($monday)));
            $entries = $this->_get_entries_for_day($view_user_id, $day_date);
            $day_total = $this->_get_day_total($view_user_id, $day_date);
            $data['days'][] = array(
                'date' => $day_date,
                'day_name' => date('l', strtotime($day_date)),
                'entries' => $entries,
                'total_minutes' => $day_total
            );
        }

        $data['week_total'] = $this->_get_range_total($view_user_id, $monday, $sunday);

        // Get employees list for admin dropdown (group filter takes precedence)
        if ($data['is_admin']) {
            if ($group_filter > 0) {
                $data['employees'] = $this->_get_employees_by_group($group_filter);
            } else {
                $data['employees'] = $this->_get_employees($role_filter);
            }
        }

        $this->load->view("common/header");
        $this->load->view("timesheet/weekly", $data);
        $this->load->view("common/footer");
    }

    /**
     * Monthly summary view
     */
    function monthly($y = 0, $m = 0) {
        $data = array();
        $data['is_admin'] = $this->_is_admin();

        // Role filter (admin only, session-persistent)
        $role_filter = array();
        if ($data['is_admin']) {
            $role_filter = $this->_get_role_filter();
            $data['roles'] = $this->_get_roles();
        }
        $data['role_filter'] = $role_filter;

        // Group filter
        $group_filter = 0;
        if ($data['is_admin']) {
            $group_filter = $this->_get_group_filter();
            $data['group_filter'] = $group_filter;
            $data['groups'] = $this->_get_all_groups();
        }

        $view_user_id = $this->_user_id();
        if ($data['is_admin'] && isset($_GET['user_id'])) {
            $view_user_id = intval($_GET['user_id']);
        }
        $data['view_user_id'] = $view_user_id;
        $data['user_id'] = $this->_user_id();
        $data['view_user_name'] = $this->_get_user_name($view_user_id);

        if ($y && $m) {
            $year = intval($y);
            $month = intval($m);
        } else {
            $year = intval(date('Y'));
            $month = intval(date('m'));
        }

        $data['year'] = $year;
        $data['month'] = $month;
        $data['month_name'] = date('F', mktime(0, 0, 0, $month, 1, $year));
        $days_in_month = date('t', mktime(0, 0, 0, $month, 1, $year));

        // Prev/next month
        $prev = mktime(0, 0, 0, $month - 1, 1, $year);
        $next = mktime(0, 0, 0, $month + 1, 1, $year);
        $data['prev_month'] = date('Y/m', $prev);
        $data['next_month'] = date('Y/m', $next);

        // Get all entries for the month in one query
        $month_start = sprintf('%04d-%02d-01', $year, $month);
        $month_end = sprintf('%04d-%02d-%02d', $year, $month, $days_in_month);
        $all_entries = $this->_get_entries_range($view_user_id, $month_start, $month_end);

        // Group entries by date
        $entries_by_date = array();
        foreach ($all_entries as $entry) {
            $d = date('Y-m-d', strtotime($entry['clock_in']));
            if (!isset($entries_by_date[$d])) $entries_by_date[$d] = array();
            $entries_by_date[$d][] = $entry;
        }

        // Build day-by-day data with entries
        $data['days'] = array();
        $grand_total = 0;
        $customer_totals = array(); // Track per-customer totals for summary
        for ($d = 1; $d <= $days_in_month; $d++) {
            $day_date = sprintf('%04d-%02d-%02d', $year, $month, $d);
            $day_total = $this->_get_day_total($view_user_id, $day_date);
            $grand_total += $day_total;
            $day_entries = isset($entries_by_date[$day_date]) ? $entries_by_date[$day_date] : array();

            // Accumulate per-customer totals
            foreach ($day_entries as $ent) {
                if (!empty($ent['customer_id']) && $ent['clock_out']) {
                    $cid = intval($ent['customer_id']);
                    if (!isset($customer_totals[$cid])) {
                        $customer_totals[$cid] = array(
                            'customer_id' => $cid,
                            'customer_name' => isset($ent['customer_name']) ? trim($ent['customer_name']) : '',
                            'customer_address' => isset($ent['customer_address']) ? trim($ent['customer_address']) : '',
                            'minutes' => 0
                        );
                    }
                    $customer_totals[$cid]['minutes'] += (intval($ent['duration_minutes']) - intval($ent['break_minutes']));
                }
            }

            $data['days'][] = array(
                'date' => $day_date,
                'day_name' => date('D', strtotime($day_date)),
                'day_num' => $d,
                'total_minutes' => $day_total,
                'is_weekend' => (date('N', strtotime($day_date)) >= 6),
                'entries' => $day_entries
            );
        }
        $data['grand_total'] = $grand_total;
        $data['customer_totals'] = $customer_totals;

        // OT threshold
        $ot_row = $this->db->query("SELECT setting_value FROM ts_settings WHERE setting_key = ?", array('overtime_daily_threshold'))->row_array();
        $data['ot_daily'] = $ot_row ? intval($ot_row['setting_value']) : 480;

        if ($data['is_admin']) {
            if ($group_filter > 0) {
                $data['employees'] = $this->_get_employees_by_group($group_filter);
            } else {
                $data['employees'] = $this->_get_employees($role_filter);
            }
        }

        $this->load->view("common/header");
        $this->load->view("timesheet/monthly", $data);
        $this->load->view("common/footer");
    }

    /**
     * Admin dashboard - all employees
     */
    function admin_view() {
        if (!$this->_is_admin()) {
            redirect("timesheet");
            return;
        }

        $data = array();
        $data['is_admin'] = true;
        $data['user_id'] = $this->_user_id();

        // Role filter (session-persistent, defaults to 'installer')
        $role_filter = $this->_get_role_filter();
        $data['role_filter'] = $role_filter;
        $data['roles'] = $this->_get_roles();

        // Group filter
        $group_filter = $this->_get_group_filter();
        $data['group_filter'] = $group_filter;
        $data['groups'] = $this->_get_all_groups();

        // Get employees: group filter takes precedence over role filter
        if ($group_filter > 0) {
            $employees = $this->_get_employees_by_group($group_filter);
        } else {
            $employees = $this->_get_employees($role_filter);
        }
        $today = date('Y-m-d');

        foreach ($employees as &$emp) {
            $emp['active_entry'] = $this->_get_active_entry($emp['id']);
            $emp['today_total'] = $this->_get_day_total($emp['id'], $today);

            // This week total
            $day_of_week = date('N');
            $monday = date('Y-m-d', strtotime("-" . ($day_of_week - 1) . " days"));
            $sunday = date('Y-m-d', strtotime("+" . (7 - $day_of_week) . " days"));
            $emp['week_total'] = $this->_get_range_total($emp['id'], $monday, $sunday);
        }

        // Exception flags
        $data['exception_flags'] = $this->_get_exception_flags($employees);
        $data['employees'] = $employees;

        $this->load->view("common/header");
        $this->load->view("timesheet/admin", $data);
        $this->load->view("common/footer");
    }

    /**
     * AJAX: Save manual entry (admin only)
     */
    function save_entry() {
        if (!$this->_require_admin()) return;

        $user_id = intval($this->input->post('user_id'));
        $customer_id = $this->input->post('customer_id') ? intval($this->input->post('customer_id')) : 0;
        $clock_in = trim($this->input->post('clock_in'));
        $clock_out = trim($this->input->post('clock_out'));
        $break_minutes = intval($this->input->post('break_minutes'));
        $notes = trim($this->input->post('notes'));
        $now = date('Y-m-d H:i:s');

        // Required fields
        if (!$user_id || !$clock_in || !$clock_out) {
            $this->_json_error('Required fields missing.');
            return;
        }

        // Validate datetime format
        $clock_in = $this->_safe_datetime($clock_in);
        $clock_out = $this->_safe_datetime($clock_out);
        if ($clock_in === false || $clock_out === false) {
            $this->_json_error('Invalid date/time format. Use YYYY-MM-DD HH:MM:SS.');
            return;
        }

        // Calculate duration
        $duration_minutes = round((strtotime($clock_out) - strtotime($clock_in)) / 60);
        if ($duration_minutes <= 0) {
            $this->_json_error('Clock out must be after clock in.');
            return;
        }

        // Max shift 24h (1440 min)
        if ($duration_minutes > 1440) {
            $this->_json_error('Shift cannot exceed 24 hours.');
            return;
        }

        // Break validation
        if ($break_minutes < 0) {
            $break_minutes = 0;
        }
        if ($break_minutes >= $duration_minutes) {
            $this->_json_error('Break time must be less than total shift duration.');
            return;
        }

        // Validate customer exists if provided
        if ($customer_id > 0) {
            $cust_check = $this->db->query("SELECT id FROM customers WHERE id = ?", array($customer_id))->row_array();
            if (!$cust_check) {
                $this->_json_error('Invalid customer selected.');
                return;
            }
        }

        // Overlap check
        $overlap_sql = "SELECT id FROM ts_time_entries
                        WHERE user_id = ? AND status = 'active' AND clock_in < ? AND clock_out > ?
                        LIMIT 1";
        $overlap = $this->db->query($overlap_sql, array($user_id, $clock_out, $clock_in))->row_array();
        if ($overlap) {
            $this->_json_error('This entry overlaps with an existing time entry.');
            return;
        }

        if ($customer_id > 0) {
            $sql = "INSERT INTO ts_time_entries (user_id, customer_id, clock_in, clock_out, duration_minutes, break_minutes, notes, entry_type, status, created_by, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, 'manual', 'active', ?, ?)";
            $this->db->query($sql, array($user_id, $customer_id, $clock_in, $clock_out, $duration_minutes, $break_minutes, $notes, $this->_user_id(), $now));
        } else {
            $sql = "INSERT INTO ts_time_entries (user_id, clock_in, clock_out, duration_minutes, break_minutes, notes, entry_type, status, created_by, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, 'manual', 'active', ?, ?)";
            $this->db->query($sql, array($user_id, $clock_in, $clock_out, $duration_minutes, $break_minutes, $notes, $this->_user_id(), $now));
        }

        $new_id = $this->db->insert_id();

        // Audit log
        $after_data = array(
            'user_id' => $user_id, 'customer_id' => $customer_id,
            'clock_in' => $clock_in, 'clock_out' => $clock_out,
            'duration_minutes' => $duration_minutes, 'break_minutes' => $break_minutes,
            'notes' => $notes, 'entry_type' => 'manual'
        );
        $this->_audit_log($new_id, 'create', null, $after_data);

        $this->_json_ok(array('message' => 'Entry saved.'));
    }

    /**
     * AJAX: Edit existing entry (admin only)
     */
    function edit_entry() {
        if (!$this->_require_admin()) return;

        $entry_id = intval($this->input->post('entry_id'));
        $customer_id = $this->input->post('customer_id');
        $clock_in = trim($this->input->post('clock_in'));
        $clock_out = trim($this->input->post('clock_out'));
        $break_minutes = intval($this->input->post('break_minutes'));
        $notes = trim($this->input->post('notes'));
        $now = date('Y-m-d H:i:s');

        if (!$entry_id || !$clock_in || !$clock_out) {
            $this->_json_error('Required fields missing.');
            return;
        }

        // Fetch existing row for audit + user_id
        $existing = $this->db->query("SELECT * FROM ts_time_entries WHERE id = ? AND status = 'active'", array($entry_id))->row_array();
        if (!$existing) {
            $this->_json_error('Entry not found or already deleted.');
            return;
        }

        // Validate datetime format
        $clock_in = $this->_safe_datetime($clock_in);
        $clock_out = $this->_safe_datetime($clock_out);
        if ($clock_in === false || $clock_out === false) {
            $this->_json_error('Invalid date/time format. Use YYYY-MM-DD HH:MM:SS.');
            return;
        }

        $duration_minutes = round((strtotime($clock_out) - strtotime($clock_in)) / 60);
        if ($duration_minutes <= 0) {
            $this->_json_error('Clock out must be after clock in.');
            return;
        }

        // Max shift 24h
        if ($duration_minutes > 1440) {
            $this->_json_error('Shift cannot exceed 24 hours.');
            return;
        }

        // Break validation
        if ($break_minutes < 0) {
            $break_minutes = 0;
        }
        if ($break_minutes >= $duration_minutes) {
            $this->_json_error('Break time must be less than total shift duration.');
            return;
        }

        // Validate customer exists if provided
        $cust_id_val = ($customer_id !== null && $customer_id !== '') ? intval($customer_id) : 0;
        if ($cust_id_val > 0) {
            $cust_check = $this->db->query("SELECT id FROM customers WHERE id = ?", array($cust_id_val))->row_array();
            if (!$cust_check) {
                $this->_json_error('Invalid customer selected.');
                return;
            }
        }

        // Overlap check (excludes self)
        $overlap_sql = "SELECT id FROM ts_time_entries
                        WHERE user_id = ? AND status = 'active' AND clock_in < ? AND clock_out > ? AND id != ?
                        LIMIT 1";
        $overlap = $this->db->query($overlap_sql, array(intval($existing['user_id']), $clock_out, $clock_in, $entry_id))->row_array();
        if ($overlap) {
            $this->_json_error('This entry overlaps with an existing time entry.');
            return;
        }

        // Build update
        if ($cust_id_val > 0) {
            $sql = "UPDATE ts_time_entries SET
                    customer_id = ?,
                    clock_in = ?,
                    clock_out = ?,
                    duration_minutes = ?,
                    break_minutes = ?,
                    notes = ?,
                    updated_by = ?,
                    updated_at = ?
                    WHERE id = ? AND status = 'active'";
            $this->db->query($sql, array($cust_id_val, $clock_in, $clock_out, $duration_minutes, $break_minutes, $notes, $this->_user_id(), $now, $entry_id));
        } else {
            $sql = "UPDATE ts_time_entries SET
                    clock_in = ?,
                    clock_out = ?,
                    duration_minutes = ?,
                    break_minutes = ?,
                    notes = ?,
                    updated_by = ?,
                    updated_at = ?
                    WHERE id = ? AND status = 'active'";
            $this->db->query($sql, array($clock_in, $clock_out, $duration_minutes, $break_minutes, $notes, $this->_user_id(), $now, $entry_id));
        }

        // Audit log
        $before_data = array(
            'clock_in' => $existing['clock_in'], 'clock_out' => $existing['clock_out'],
            'duration_minutes' => $existing['duration_minutes'], 'break_minutes' => $existing['break_minutes'],
            'notes' => $existing['notes'], 'customer_id' => $existing['customer_id']
        );
        $after_data = array(
            'clock_in' => $clock_in, 'clock_out' => $clock_out,
            'duration_minutes' => $duration_minutes, 'break_minutes' => $break_minutes,
            'notes' => $notes, 'customer_id' => $cust_id_val
        );
        $this->_audit_log($entry_id, 'edit', $before_data, $after_data);

        $this->_json_ok(array('message' => 'Entry updated.'));
    }

    /**
     * AJAX: Soft-delete entry (admin only)
     */
    function delete_entry() {
        if (!$this->_require_admin()) return;

        $entry_id = intval($this->input->post('entry_id'));
        $now = date('Y-m-d H:i:s');

        if (!$entry_id) {
            $this->_json_error('Entry ID required.');
            return;
        }

        // Fetch existing for audit
        $existing = $this->db->query("SELECT * FROM ts_time_entries WHERE id = ? AND status = 'active'", array($entry_id))->row_array();
        if (!$existing) {
            $this->_json_error('Entry not found or already deleted.');
            return;
        }

        $sql = "UPDATE ts_time_entries SET status = 'deleted', updated_by = ?, updated_at = ? WHERE id = ?";
        $this->db->query($sql, array($this->_user_id(), $now, $entry_id));

        // Audit log
        $before_data = array(
            'id' => $existing['id'], 'user_id' => $existing['user_id'],
            'clock_in' => $existing['clock_in'], 'clock_out' => $existing['clock_out'],
            'duration_minutes' => $existing['duration_minutes'], 'break_minutes' => $existing['break_minutes'],
            'notes' => $existing['notes'], 'customer_id' => $existing['customer_id'],
            'status' => 'active'
        );
        $this->_audit_log($entry_id, 'delete', $before_data, null);

        $this->_json_ok(array('message' => 'Entry deleted.'));
    }

    /**
     * CSV export (admin only)
     */
    function export_csv() {
        if (!$this->_is_admin()) {
            redirect("timesheet");
            return;
        }

        $from = isset($_GET['from']) ? $this->_safe_date(trim($_GET['from'])) : false;
        $to = isset($_GET['to']) ? $this->_safe_date(trim($_GET['to'])) : false;
        $emp_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : 0;

        // Default dates if invalid
        if ($from === false) $from = date('Y-m-01');
        if ($to === false) $to = date('Y-m-d');

        $bindings = array($from, $to);
        $where = "WHERE e.status = 'active' AND DATE(e.clock_in) >= ? AND DATE(e.clock_in) <= ?";
        if ($emp_id) {
            $where .= " AND e.user_id = ?";
            $bindings[] = $emp_id;
        }

        $sql = "SELECT e.*, CONCAT(u.first_name, ' ', u.last_name) AS employee_name,
                    CONCAT(c.first_name, ' ', c.last_name) AS customer_name, c.address AS customer_address
                FROM ts_time_entries e
                LEFT JOIN users u ON u.id = e.user_id
                LEFT JOIN customers c ON c.id = e.customer_id
                " . $where . "
                ORDER BY e.clock_in ASC";
        $rows = $this->db->query($sql, $bindings)->result_array();

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="timesheet_' . $from . '_to_' . $to . '.csv"');

        $out = fopen('php://output', 'w');
        fputcsv($out, array('Employee', 'Date', 'Clock In', 'Clock Out', 'Duration (hrs)', 'Break (min)', 'Net Hours', 'Customer', 'Address', 'Type', 'Notes'));

        foreach ($rows as $row) {
            $net_min = intval($row['duration_minutes']) - intval($row['break_minutes']);
            fputcsv($out, array(
                $row['employee_name'],
                date('Y-m-d', strtotime($row['clock_in'])),
                date('g:i A', strtotime($row['clock_in'])),
                $row['clock_out'] ? date('g:i A', strtotime($row['clock_out'])) : 'Active',
                $row['duration_minutes'] ? round($row['duration_minutes'] / 60, 2) : '',
                $row['break_minutes'],
                $net_min > 0 ? round($net_min / 60, 2) : '',
                $row['customer_name'] ? trim($row['customer_name']) : '',
                $row['customer_address'] ? trim($row['customer_address']) : '',
                $row['entry_type'],
                $row['notes']
            ));
        }
        fclose($out);
        exit;
    }

    /**
     * AJAX: Search customers by name or address (for entry form)
     */
    function search_customers() {
        $q = trim($this->input->post('q'));
        if (strlen($q) < 2) {
            $this->_json_error('Enter at least 2 characters.');
            return;
        }

        $like_term = $this->_escape_like($q);

        $sql = "SELECT c.id, c.first_name, c.last_name, c.address, c.city
                FROM customers c
                WHERE (CONCAT(c.first_name, ' ', c.last_name) LIKE ?
                   OR c.address LIKE ?
                   OR c.last_name LIKE ?)
                ORDER BY c.last_name, c.first_name
                LIMIT 20";
        $rows = $this->db->query($sql, array($like_term, $like_term, $like_term))->result_array();

        $results = array();
        foreach ($rows as $r) {
            $results[] = array(
                'id' => intval($r['id']),
                'name' => trim($r['first_name'] . ' ' . $r['last_name']),
                'address' => trim($r['address']),
                'city' => trim($r['city'])
            );
        }

        $this->_json_ok(array('results' => $results));
    }

    /**
     * AJAX: Get a user's time entries for a day or week (admin only).
     * POST params: user_id, date, mode ('day' or 'week')
     */
    function get_user_day_entries() {
        if (!$this->_require_admin()) return;

        $user_id = intval($this->input->post('user_id'));
        $date = trim($this->input->post('date'));
        $mode = trim($this->input->post('mode'));

        if (!$user_id || !$date) {
            $this->_json_error('Missing user_id or date.');
            return;
        }

        // Validate date
        $date = $this->_safe_date($date);
        if ($date === false) {
            $this->_json_error('Invalid date format.');
            return;
        }

        // Validate mode enum
        if ($mode !== 'day' && $mode !== 'week') {
            $mode = 'day';
        }

        if ($mode == 'week') {
            $day_of_week = date('N', strtotime($date));
            $from = date('Y-m-d', strtotime("-" . ($day_of_week - 1) . " days", strtotime($date)));
            $to = date('Y-m-d', strtotime("+" . (7 - $day_of_week) . " days", strtotime($date)));
        } else {
            $from = $date;
            $to = $date;
        }

        $sql = "SELECT e.*, CONCAT(c.first_name, ' ', c.last_name) AS customer_name, c.address AS customer_address
                FROM ts_time_entries e
                LEFT JOIN customers c ON c.id = e.customer_id
                WHERE e.user_id = ? AND e.status = 'active'
                AND DATE(e.clock_in) >= ? AND DATE(e.clock_in) <= ?
                ORDER BY e.clock_in ASC";
        $entries = $this->db->query($sql, array($user_id, $from, $to))->result_array();

        $user_name = $this->_get_user_name($user_id);

        $this->_json_ok(array(
            'entries' => $entries,
            'user_name' => $user_name,
            'user_id' => $user_id,
            'from' => $from,
            'to' => $to,
            'mode' => $mode
        ));
    }

    /**
     * Group management page (admin only)
     */
    function group_admin() {
        if (!$this->_is_admin()) {
            redirect("timesheet");
            return;
        }

        $data = array();
        $data['is_admin'] = true;
        $data['user_id'] = $this->_user_id();
        $data['groups'] = $this->_get_all_groups();
        $data['all_employees'] = $this->_get_employees(array());

        $this->load->view("common/header");
        $this->load->view("timesheet/group_admin", $data);
        $this->load->view("common/footer");
    }

    /**
     * AJAX: Create or rename a group (admin only)
     */
    function save_group() {
        if (!$this->_require_admin()) return;

        $group_id = intval($this->input->post('group_id'));
        $group_name = trim($this->input->post('group_name'));
        $description = trim($this->input->post('description'));
        $now = date('Y-m-d H:i:s');

        if ($group_name == '') {
            $this->_json_error('Group name is required.');
            return;
        }

        if ($group_id) {
            $sql = "UPDATE ts_groups SET group_name = ?, description = ?, updated_at = ? WHERE id = ? AND status = 'active'";
            $this->db->query($sql, array($group_name, $description, $now, $group_id));
            $this->_json_ok(array('message' => 'Group updated.'));
        } else {
            $sql = "INSERT INTO ts_groups (group_name, description, created_by, created_at) VALUES (?, ?, ?, ?)";
            $this->db->query($sql, array($group_name, $description, $this->_user_id(), $now));
            $this->_json_ok(array('message' => 'Group created.'));
        }
    }

    /**
     * AJAX: Soft-delete group + remove memberships (admin only)
     */
    function delete_group() {
        if (!$this->_require_admin()) return;

        $group_id = intval($this->input->post('group_id'));
        if (!$group_id) {
            $this->_json_error('Group ID required.');
            return;
        }

        $now = date('Y-m-d H:i:s');
        $this->db->query("UPDATE ts_groups SET status = 'deleted', updated_at = ? WHERE id = ?", array($now, $group_id));
        $this->db->query("DELETE FROM ts_group_members WHERE group_id = ?", array($group_id));

        $this->_json_ok(array('message' => 'Group deleted.'));
    }

    /**
     * AJAX: Replace-all membership for a group (admin only)
     */
    function save_group_members() {
        if (!$this->_require_admin()) return;

        $group_id = intval($this->input->post('group_id'));
        $user_ids_str = trim($this->input->post('user_ids'));
        $now = date('Y-m-d H:i:s');

        if (!$group_id) {
            $this->_json_error('Group ID required.');
            return;
        }

        $this->db->query("DELETE FROM ts_group_members WHERE group_id = ?", array($group_id));

        if ($user_ids_str != '') {
            $user_ids = array_map('intval', explode(',', $user_ids_str));
            foreach ($user_ids as $uid) {
                if ($uid > 0) {
                    $this->db->query("INSERT INTO ts_group_members (group_id, user_id, added_by, added_at) VALUES (?, ?, ?, ?)",
                        array($group_id, $uid, $this->_user_id(), $now));
                }
            }
        }

        $this->_json_ok(array('message' => 'Members saved.'));
    }

    /**
     * AJAX: Return members of a group (admin only)
     */
    function get_group_members() {
        if (!$this->_require_admin()) return;

        $group_id = intval($this->input->post('group_id'));
        if (!$group_id) {
            $this->_json_error('Group ID required.');
            return;
        }

        $sql = "SELECT gm.user_id FROM ts_group_members gm WHERE gm.group_id = ?";
        $rows = $this->db->query($sql, array($group_id))->result_array();
        $member_ids = array();
        foreach ($rows as $r) {
            $member_ids[] = intval($r['user_id']);
        }

        $this->_json_ok(array('member_ids' => $member_ids));
    }

    // ========================================
    // Private helper methods
    // ========================================

    function _get_all_groups() {
        $sql = "SELECT g.*, COUNT(gm.id) AS member_count
                FROM ts_groups g
                LEFT JOIN ts_group_members gm ON gm.group_id = g.id
                WHERE g.status = 'active'
                GROUP BY g.id
                ORDER BY g.group_name";
        return $this->db->query($sql)->result_array();
    }

    function _get_employees_by_group($group_id) {
        $sql = "SELECT u.id, u.first_name, u.last_name, u.email, u.type as role
                FROM users u
                INNER JOIN ts_group_members gm ON gm.user_id = u.id
                WHERE gm.group_id = ? AND u.user_status = 'active'
                ORDER BY u.first_name, u.last_name";
        return $this->db->query($sql, array(intval($group_id)))->result_array();
    }

    /**
     * Get group filter from URL/session. Returns integer group_id (0 = no filter).
     */
    function _get_group_filter() {
        if (isset($_GET['group_id'])) {
            $gid = intval($_GET['group_id']);
            $this->session->set_userdata('ts_group_filter', $gid);
            return $gid;
        }
        $saved = $this->session->userdata('ts_group_filter');
        if ($saved !== false && $saved !== null) {
            return intval($saved);
        }
        return 0;
    }

    function _get_active_entry($user_id) {
        $sql = "SELECT e.*, CONCAT(c.first_name, ' ', c.last_name) AS customer_name, c.address AS customer_address
                FROM ts_time_entries e
                LEFT JOIN customers c ON c.id = e.customer_id
                WHERE e.user_id = ? AND e.clock_out IS NULL AND e.status = 'active' ORDER BY e.clock_in DESC LIMIT 1";
        $result = $this->db->query($sql, array(intval($user_id)))->result_array();
        return count($result) ? $result[0] : null;
    }

    function _get_entries_range($user_id, $from, $to) {
        $sql = "SELECT e.*, CONCAT(c.first_name, ' ', c.last_name) AS customer_name, c.address AS customer_address
                FROM ts_time_entries e
                LEFT JOIN customers c ON c.id = e.customer_id
                WHERE e.user_id = ? AND e.status = 'active' AND DATE(e.clock_in) >= ? AND DATE(e.clock_in) <= ? ORDER BY e.clock_in ASC";
        return $this->db->query($sql, array(intval($user_id), $from, $to))->result_array();
    }

    function _get_entries_for_day($user_id, $date) {
        $sql = "SELECT e.*, CONCAT(c.first_name, ' ', c.last_name) AS customer_name, c.address AS customer_address
                FROM ts_time_entries e
                LEFT JOIN customers c ON c.id = e.customer_id
                WHERE e.user_id = ? AND e.status = 'active' AND DATE(e.clock_in) = ? ORDER BY e.clock_in ASC";
        return $this->db->query($sql, array(intval($user_id), $date))->result_array();
    }

    function _get_day_total($user_id, $date) {
        $sql = "SELECT SUM(COALESCE(duration_minutes, 0) - break_minutes) AS total FROM ts_time_entries WHERE user_id = ? AND status = 'active' AND DATE(clock_in) = ? AND clock_out IS NOT NULL";
        $row = $this->db->query($sql, array(intval($user_id), $date))->row_array();
        return $row && $row['total'] ? intval($row['total']) : 0;
    }

    function _get_range_total($user_id, $from, $to) {
        $sql = "SELECT SUM(COALESCE(duration_minutes, 0) - break_minutes) AS total FROM ts_time_entries WHERE user_id = ? AND status = 'active' AND DATE(clock_in) >= ? AND DATE(clock_in) <= ? AND clock_out IS NOT NULL";
        $row = $this->db->query($sql, array(intval($user_id), $from, $to))->row_array();
        return $row && $row['total'] ? intval($row['total']) : 0;
    }

    /**
     * Get the active role filter as an array.
     */
    function _get_role_filter() {
        if (isset($_GET['roles'])) {
            $raw = trim($_GET['roles']);
            if ($raw == 'all' || $raw == '') {
                $this->session->set_userdata('ts_role_filter', '');
                return array();
            }
            $roles = array_map('trim', explode(',', $raw));
            $roles = array_filter($roles, 'strlen');
            $this->session->set_userdata('ts_role_filter', implode(',', $roles));
            return $roles;
        }
        if (isset($_GET['role'])) {
            $role = trim($_GET['role']);
            if ($role == 'all' || $role == '') {
                $this->session->set_userdata('ts_role_filter', '');
                return array();
            }
            $this->session->set_userdata('ts_role_filter', $role);
            return array($role);
        }
        $saved = $this->session->userdata('ts_role_filter');
        if ($saved !== false && $saved !== null) {
            if ($saved === '') return array();
            return array_map('trim', explode(',', $saved));
        }
        $this->session->set_userdata('ts_role_filter', 'installer');
        return array('installer');
    }

    function _get_employees($roles = array()) {
        if (!empty($roles)) {
            $placeholders = array();
            $bindings = array();
            foreach ($roles as $r) {
                $placeholders[] = '?';
                $bindings[] = $r;
            }
            $sql = "SELECT id, first_name, last_name, email, type as role FROM users WHERE user_status = 'active' AND type IN (" . implode(',', $placeholders) . ") ORDER BY first_name, last_name";
            return $this->db->query($sql, $bindings)->result_array();
        }
        $sql = "SELECT id, first_name, last_name, email, type as role FROM users WHERE user_status = 'active' ORDER BY first_name, last_name";
        return $this->db->query($sql)->result_array();
    }

    function _get_roles() {
        $sql = "SELECT DISTINCT type FROM users WHERE user_status = 'active' AND type IS NOT NULL AND type != '' ORDER BY type";
        $rows = $this->db->query($sql)->result_array();
        $roles = array();
        foreach ($rows as $r) {
            $roles[] = $r['type'];
        }
        return $roles;
    }

    function _get_user_name($user_id) {
        $sql = "SELECT CONCAT(first_name, ' ', last_name) AS name FROM users WHERE id = ?";
        $row = $this->db->query($sql, array(intval($user_id)))->row_array();
        return $row ? $row['name'] : 'Unknown';
    }

    function _format_minutes($minutes) {
        if ($minutes <= 0) return '0h 0m';
        $h = floor($minutes / 60);
        $m = $minutes % 60;
        return $h . 'h ' . $m . 'm';
    }

    function _get_customer_label($customer_id) {
        $sql = "SELECT id, first_name, last_name, address FROM customers WHERE id = ?";
        $row = $this->db->query($sql, array(intval($customer_id)))->row_array();
        if (!$row) return null;
        $parts = array();
        if ($row['first_name'] || $row['last_name']) {
            $parts[] = trim($row['first_name'] . ' ' . $row['last_name']);
        }
        if ($row['address']) {
            $parts[] = $row['address'];
        }
        return implode(' - ', $parts);
    }

    /**
     * Get exception flags for admin dashboard employees
     */
    function _get_exception_flags($employees) {
        $flags = array();
        $today = date('Y-m-d');
        $now_ts = time();

        foreach ($employees as $emp) {
            $emp_flags = array();
            $emp_id = intval($emp['id']);

            // Active > 12h warning
            if (!empty($emp['active_entry'])) {
                $clock_in_ts = strtotime($emp['active_entry']['clock_in']);
                $elapsed_hours = ($now_ts - $clock_in_ts) / 3600;
                if ($elapsed_hours > 12) {
                    $emp_flags[] = array('type' => 'warning', 'label' => 'Active >' . floor($elapsed_hours) . 'h');
                }
            }

            // Missing clock-out from prior day
            $missing_sql = "SELECT id FROM ts_time_entries WHERE user_id = ? AND status = 'active' AND clock_out IS NULL AND DATE(clock_in) < ? LIMIT 1";
            $missing = $this->db->query($missing_sql, array($emp_id, $today))->row_array();
            if ($missing) {
                $emp_flags[] = array('type' => 'danger', 'label' => 'Missing clock-out');
            }

            // Manual entries today
            $manual_sql = "SELECT COUNT(*) AS cnt FROM ts_time_entries WHERE user_id = ? AND status = 'active' AND entry_type = 'manual' AND DATE(created_at) = ?";
            $manual = $this->db->query($manual_sql, array($emp_id, $today))->row_array();
            if ($manual && intval($manual['cnt']) > 0) {
                $emp_flags[] = array('type' => 'info', 'label' => intval($manual['cnt']) . ' manual');
            }

            // Edits today
            $edit_sql = "SELECT COUNT(*) AS cnt FROM ts_entry_audit WHERE changed_by != 0 AND action = 'edit' AND DATE(changed_at) = ? AND entry_id IN (SELECT id FROM ts_time_entries WHERE user_id = ?)";
            $edits = $this->db->query($edit_sql, array($today, $emp_id))->row_array();
            if ($edits && intval($edits['cnt']) > 0) {
                $emp_flags[] = array('type' => 'info', 'label' => intval($edits['cnt']) . ' edit(s)');
            }

            if (!empty($emp_flags)) {
                $flags[$emp_id] = $emp_flags;
            }
        }

        return $flags;
    }
}
