#!/usr/bin/env python3
"""
RDF Status - Show current workflow state

Displays status of active and recent workflows.

Usage:
    rdf status                    # Show all active workflows
    rdf status --last <id>        # Show specific workflow
    rdf status --type book        # Show only book workflows
"""

import argparse
import json
import sys
import os
from pathlib import Path
from datetime import datetime

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes


def find_workflows(workflow_type: str = None) -> list:
    """Find all workflow state files."""
    workflows = []

    # Check book_projects
    book_dir = Path("book_projects")
    if book_dir.exists():
        for project_dir in book_dir.iterdir():
            if project_dir.is_dir():
                state_file = project_dir / "workflow_state.json"
                if state_file.exists():
                    try:
                        with open(state_file, 'r') as f:
                            state = json.load(f)
                            state["workflow_type"] = "book"
                            state["project_dir"] = str(project_dir)
                            workflows.append(state)
                    except json.JSONDecodeError:
                        pass

    # Check essays
    essay_dir = Path("essays")
    if essay_dir.exists():
        for project_dir in essay_dir.iterdir():
            if project_dir.is_dir():
                state_file = project_dir / "workflow_state.json"
                if state_file.exists():
                    try:
                        with open(state_file, 'r') as f:
                            state = json.load(f)
                            state["workflow_type"] = "essay"
                            state["project_dir"] = str(project_dir)
                            workflows.append(state)
                    except json.JSONDecodeError:
                        pass

    # Filter by type if specified
    if workflow_type:
        workflows = [w for w in workflows if w.get("workflow_type") == workflow_type]

    # Sort by creation date (newest first)
    workflows.sort(
        key=lambda x: x.get("created_at", ""),
        reverse=True
    )

    return workflows


def get_workflow_by_id(workflow_id: str) -> dict:
    """Get a specific workflow by ID."""
    # Try book_projects first
    if workflow_id.startswith("BOOK_"):
        state_file = Path("book_projects") / workflow_id / "workflow_state.json"
        if state_file.exists():
            with open(state_file, 'r') as f:
                state = json.load(f)
                state["workflow_type"] = "book"
                state["project_dir"] = str(state_file.parent)
                return state

    # Try essays
    if workflow_id.startswith("ESSAY_"):
        state_file = Path("essays") / workflow_id / "workflow_state.json"
        if state_file.exists():
            with open(state_file, 'r') as f:
                state = json.load(f)
                state["workflow_type"] = "essay"
                state["project_dir"] = str(state_file.parent)
                return state

    # Search all workflows
    for workflow in find_workflows():
        if workflow.get("book_id") == workflow_id or workflow.get("essay_id") == workflow_id:
            return workflow

    return None


def list_artifacts(project_dir: str) -> list:
    """List artifacts in a project directory."""
    artifacts = []
    project_path = Path(project_dir)

    if not project_path.exists():
        return artifacts

    for item in project_path.iterdir():
        if item.is_file():
            artifacts.append({
                "name": item.name,
                "path": str(item),
                "size_bytes": item.stat().st_size,
                "modified": datetime.fromtimestamp(item.stat().st_mtime).isoformat()
            })

    return artifacts


def main():
    parser = argparse.ArgumentParser(
        description="Show workflow status",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("--last", metavar="ID",
                        help="Show specific workflow by ID")
    parser.add_argument("--type", choices=["book", "essay"],
                        help="Filter by workflow type")
    parser.add_argument("--limit", type=int, default=10,
                        help="Maximum workflows to show")
    parser.add_argument("--format", choices=["json", "text"], default="json",
                        help="Output format")

    args = parser.parse_args()

    try:
        if args.last:
            # Get specific workflow
            workflow = get_workflow_by_id(args.last)

            if not workflow:
                resp = error_response(
                    ErrorCodes.PROJECT_NOT_FOUND,
                    f"Workflow not found: {args.last}"
                )
                if args.format == "json":
                    resp.print_json()
                else:
                    print(f"Error: {resp.message}")
                return 1

            workflow["artifacts"] = list_artifacts(workflow.get("project_dir", ""))

            if args.format == "json":
                response = success_response(
                    f"Workflow status: {args.last}",
                    data=workflow
                )
                response.print_json()
            else:
                print(f"Workflow: {args.last}")
                print(f"Type: {workflow.get('workflow_type')}")
                print(f"Status: {workflow.get('status', workflow.get('current_phase'))}")
                print(f"Created: {workflow.get('created_at')}")
                print(f"Directory: {workflow.get('project_dir')}")
                print(f"\nArtifacts:")
                for artifact in workflow.get("artifacts", []):
                    print(f"  - {artifact['name']} ({artifact['size_bytes']} bytes)")

        else:
            # List all workflows
            workflows = find_workflows(args.type)[:args.limit]

            if not workflows:
                if args.format == "json":
                    response = success_response(
                        "No active workflows found",
                        data={"workflows": []}
                    )
                    response.print_json()
                else:
                    print("No active workflows found.")
                return 0

            # Summarize each workflow
            summaries = []
            for w in workflows:
                summary = {
                    "id": w.get("book_id") or w.get("essay_id"),
                    "type": w.get("workflow_type"),
                    "status": w.get("status", w.get("current_phase")),
                    "created_at": w.get("created_at"),
                    "project_dir": w.get("project_dir")
                }
                if w.get("topic"):
                    summary["topic"] = w.get("topic")
                summaries.append(summary)

            if args.format == "json":
                response = success_response(
                    f"Found {len(summaries)} workflow(s)",
                    data={"workflows": summaries}
                )
                response.print_json()
            else:
                print(f"Active Workflows ({len(summaries)}):\n")
                for s in summaries:
                    print(f"  {s['id']} ({s['type']})")
                    print(f"    Status: {s['status']}")
                    print(f"    Created: {s['created_at']}")
                    print()

        return 0

    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
