#!/usr/bin/env python3
"""
Semantic Graph Traversal - Query knowledge graph by relationship types.

Enables sophisticated concept graph queries like:
- "Find concepts that support X"
- "Trace the origins of Y"
- "What concepts are influenced by Z"
- "Find paths between concept A and B"

Relationship Types (from V3 migration):
- related: Generic relationship
- supports: A provides evidence/foundation for B
- contradicts: A conflicts with B
- influences: A has influence on B
- derived_from: A is derived from/based on B
- part_of: A is a component of B
- precedes: A comes before B (temporal/logical)
- similar_to: A is similar/analogous to B

Usage:
    from semantic_graph import SemanticGraph

    graph = SemanticGraph()

    # Find supporting concepts
    supporters = graph.find_related("Anthroposophy", relationship="supports")

    # Trace origins
    origins = graph.trace_path("Anthroposophy", direction="backward",
                               relationship="derived_from")

    # Find all paths between two concepts
    paths = graph.find_paths("Theosophy", "Anthroposophy", max_depth=4)
"""

import sys
import logging
from pathlib import Path
from typing import List, Dict, Any, Optional, Set, Tuple
from dataclasses import dataclass, field
from collections import defaultdict

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent))

from db_utils import execute_query

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


# =============================================================================
# RELATIONSHIP TYPES
# =============================================================================

RELATIONSHIP_TYPES = {
    'related': {
        'description': 'Generic relationship between concepts',
        'bidirectional': True,
        'inverse': 'related'
    },
    'supports': {
        'description': 'A provides evidence or foundation for B',
        'bidirectional': False,
        'inverse': 'supported_by'
    },
    'contradicts': {
        'description': 'A conflicts with or opposes B',
        'bidirectional': True,
        'inverse': 'contradicts'
    },
    'influences': {
        'description': 'A has influence on the development of B',
        'bidirectional': False,
        'inverse': 'influenced_by'
    },
    'derived_from': {
        'description': 'A is derived from or based on B',
        'bidirectional': False,
        'inverse': 'source_of'
    },
    'part_of': {
        'description': 'A is a component or aspect of B',
        'bidirectional': False,
        'inverse': 'contains'
    },
    'precedes': {
        'description': 'A comes before B (temporal or logical)',
        'bidirectional': False,
        'inverse': 'follows'
    },
    'similar_to': {
        'description': 'A is similar or analogous to B',
        'bidirectional': True,
        'inverse': 'similar_to'
    }
}


# =============================================================================
# DATA STRUCTURES
# =============================================================================

@dataclass
class ConceptNode:
    """A concept in the knowledge graph."""
    concept_id: int
    name: str
    category: Optional[str] = None
    aliases: Optional[List[str]] = None
    document_count: int = 0


@dataclass
class ConceptEdge:
    """A relationship between concepts."""
    source_id: int
    target_id: int
    relationship_type: str
    weight: float = 1.0
    confidence: float = 1.0
    document_id: Optional[str] = None
    chunk_id: Optional[str] = None
    bidirectional: bool = False
    verified: bool = False


@dataclass
class GraphPath:
    """A path through the concept graph."""
    nodes: List[ConceptNode]
    edges: List[ConceptEdge]
    total_weight: float = 0.0

    @property
    def length(self) -> int:
        return len(self.edges)

    def to_string(self) -> str:
        """Format path as readable string."""
        if not self.nodes:
            return "(empty path)"

        parts = []
        for i, node in enumerate(self.nodes):
            parts.append(node.name)
            if i < len(self.edges):
                edge = self.edges[i]
                arrow = "<-->" if edge.bidirectional else "-->"
                parts.append(f" -[{edge.relationship_type}]{arrow} ")

        return "".join(parts)


@dataclass
class TraversalResult:
    """Result of a graph traversal query."""
    query_concept: ConceptNode
    related_concepts: List[ConceptNode]
    paths: List[GraphPath]
    relationship_filter: Optional[str]
    direction: str  # 'forward', 'backward', 'both'
    depth: int
    stats: Dict[str, Any] = field(default_factory=dict)


# =============================================================================
# SEMANTIC GRAPH CLASS
# =============================================================================

class SemanticGraph:
    """
    Query and traverse the concept knowledge graph.

    Provides semantic queries over concept relationships extracted
    from the document corpus.
    """

    def __init__(self):
        """Initialize the semantic graph."""
        self._concept_cache: Dict[int, ConceptNode] = {}
        self._name_to_id: Dict[str, int] = {}

    # -------------------------------------------------------------------------
    # Concept Lookup
    # -------------------------------------------------------------------------

    def get_concept(self, identifier: str | int) -> Optional[ConceptNode]:
        """
        Get a concept by ID or name.

        Args:
            identifier: Concept ID (int) or name (str)

        Returns:
            ConceptNode or None if not found
        """
        # Check cache
        if isinstance(identifier, int) and identifier in self._concept_cache:
            return self._concept_cache[identifier]

        if isinstance(identifier, str) and identifier.lower() in self._name_to_id:
            concept_id = self._name_to_id[identifier.lower()]
            return self._concept_cache.get(concept_id)

        # Query database
        if isinstance(identifier, int):
            result = execute_query(
                """
                SELECT c.concept_id, c.name, c.category, c.aliases,
                       COUNT(dc.document_id) as document_count
                FROM concepts c
                LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
                WHERE c.concept_id = %s
                GROUP BY c.concept_id
                """,
                (identifier,),
                fetch='one'
            )
        else:
            # Search by name or alias
            result = execute_query(
                """
                SELECT c.concept_id, c.name, c.category, c.aliases,
                       COUNT(dc.document_id) as document_count
                FROM concepts c
                LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
                WHERE LOWER(c.name) = LOWER(%s)
                   OR LOWER(%s) = ANY(SELECT LOWER(unnest(c.aliases)))
                GROUP BY c.concept_id
                LIMIT 1
                """,
                (identifier, identifier),
                fetch='one'
            )

        if not result:
            return None

        node = ConceptNode(
            concept_id=result['concept_id'],
            name=result['name'],
            category=result['category'],
            aliases=result['aliases'] or [],
            document_count=result['document_count']
        )

        # Cache it
        self._concept_cache[node.concept_id] = node
        self._name_to_id[node.name.lower()] = node.concept_id

        return node

    def search_concepts(
        self,
        query: str,
        limit: int = 10,
        category: str = None
    ) -> List[ConceptNode]:
        """
        Search for concepts by name pattern.

        Args:
            query: Search pattern (supports ILIKE)
            limit: Maximum results
            category: Optional category filter

        Returns:
            List of matching ConceptNodes
        """
        if category:
            results = execute_query(
                """
                SELECT c.concept_id, c.name, c.category, c.aliases,
                       COUNT(dc.document_id) as document_count
                FROM concepts c
                LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
                WHERE (c.name ILIKE %s OR %s ILIKE ANY(c.aliases))
                  AND c.category = %s
                GROUP BY c.concept_id
                ORDER BY document_count DESC
                LIMIT %s
                """,
                (f"%{query}%", f"%{query}%", category, limit),
                fetch='all'
            )
        else:
            results = execute_query(
                """
                SELECT c.concept_id, c.name, c.category, c.aliases,
                       COUNT(dc.document_id) as document_count
                FROM concepts c
                LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
                WHERE c.name ILIKE %s OR %s ILIKE ANY(c.aliases)
                GROUP BY c.concept_id
                ORDER BY document_count DESC
                LIMIT %s
                """,
                (f"%{query}%", f"%{query}%", limit),
                fetch='all'
            )

        nodes = []
        for r in results or []:
            node = ConceptNode(
                concept_id=r['concept_id'],
                name=r['name'],
                category=r['category'],
                aliases=r['aliases'] or [],
                document_count=r['document_count']
            )
            self._concept_cache[node.concept_id] = node
            nodes.append(node)

        return nodes

    # -------------------------------------------------------------------------
    # Relationship Queries
    # -------------------------------------------------------------------------

    def find_related(
        self,
        concept: str | int,
        relationship: str = None,
        direction: str = 'both',
        min_confidence: float = 0.0,
        limit: int = 50
    ) -> List[Tuple[ConceptNode, ConceptEdge]]:
        """
        Find concepts related to a given concept.

        Args:
            concept: Concept ID or name
            relationship: Filter by relationship type (None for all)
            direction: 'forward' (outgoing), 'backward' (incoming), or 'both'
            min_confidence: Minimum confidence threshold
            limit: Maximum results

        Returns:
            List of (ConceptNode, ConceptEdge) tuples
        """
        source_node = self.get_concept(concept)
        if not source_node:
            logger.warning(f"Concept not found: {concept}")
            return []

        results = []

        # Forward relationships (source -> target)
        if direction in ('forward', 'both'):
            query = """
                SELECT cr.*, c.name, c.category, c.aliases
                FROM concept_relationships cr
                JOIN concepts c ON cr.target_concept_id = c.concept_id
                WHERE cr.source_concept_id = %s
                  AND cr.confidence >= %s
            """
            params = [source_node.concept_id, min_confidence]

            if relationship:
                query += " AND cr.relationship_type = %s"
                params.append(relationship)

            query += " ORDER BY cr.weight DESC, cr.confidence DESC LIMIT %s"
            params.append(limit)

            forward_results = execute_query(query, tuple(params), fetch='all')

            for r in forward_results or []:
                target_node = ConceptNode(
                    concept_id=r['target_concept_id'],
                    name=r['name'],
                    category=r['category'],
                    aliases=r['aliases'] or []
                )
                edge = ConceptEdge(
                    source_id=r['source_concept_id'],
                    target_id=r['target_concept_id'],
                    relationship_type=r['relationship_type'],
                    weight=r['weight'] or 1.0,
                    confidence=r['confidence'] or 1.0,
                    document_id=r.get('document_id'),
                    chunk_id=r.get('chunk_id'),
                    bidirectional=r.get('bidirectional', False),
                    verified=r.get('verified', False)
                )
                results.append((target_node, edge))

        # Backward relationships (target <- source)
        if direction in ('backward', 'both'):
            query = """
                SELECT cr.*, c.name, c.category, c.aliases
                FROM concept_relationships cr
                JOIN concepts c ON cr.source_concept_id = c.concept_id
                WHERE cr.target_concept_id = %s
                  AND cr.confidence >= %s
            """
            params = [source_node.concept_id, min_confidence]

            if relationship:
                # Use inverse relationship type for backward search
                inv_rel = RELATIONSHIP_TYPES.get(relationship, {}).get('inverse', relationship)
                query += " AND cr.relationship_type = %s"
                params.append(inv_rel if inv_rel != relationship else relationship)

            query += " ORDER BY cr.weight DESC, cr.confidence DESC LIMIT %s"
            params.append(limit)

            backward_results = execute_query(query, tuple(params), fetch='all')

            for r in backward_results or []:
                source_concept = ConceptNode(
                    concept_id=r['source_concept_id'],
                    name=r['name'],
                    category=r['category'],
                    aliases=r['aliases'] or []
                )
                edge = ConceptEdge(
                    source_id=r['source_concept_id'],
                    target_id=r['target_concept_id'],
                    relationship_type=r['relationship_type'],
                    weight=r['weight'] or 1.0,
                    confidence=r['confidence'] or 1.0,
                    document_id=r.get('document_id'),
                    chunk_id=r.get('chunk_id'),
                    bidirectional=r.get('bidirectional', False),
                    verified=r.get('verified', False)
                )
                results.append((source_concept, edge))

        return results[:limit]

    def get_supporters(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts that support the given concept."""
        results = self.find_related(concept, relationship='supports', direction='backward', limit=limit)
        return [node for node, edge in results]

    def get_supported(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts supported by the given concept."""
        results = self.find_related(concept, relationship='supports', direction='forward', limit=limit)
        return [node for node, edge in results]

    def get_influences(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts influenced by the given concept."""
        results = self.find_related(concept, relationship='influences', direction='forward', limit=limit)
        return [node for node, edge in results]

    def get_influenced_by(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts that influenced the given concept."""
        results = self.find_related(concept, relationship='influences', direction='backward', limit=limit)
        return [node for node, edge in results]

    def get_origins(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Trace origins - concepts this was derived from."""
        results = self.find_related(concept, relationship='derived_from', direction='forward', limit=limit)
        return [node for node, edge in results]

    def get_derivatives(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts derived from this one."""
        results = self.find_related(concept, relationship='derived_from', direction='backward', limit=limit)
        return [node for node, edge in results]

    def get_contradictions(self, concept: str | int, limit: int = 20) -> List[ConceptNode]:
        """Find concepts that contradict this one."""
        results = self.find_related(concept, relationship='contradicts', direction='both', limit=limit)
        return [node for node, edge in results]

    # -------------------------------------------------------------------------
    # Path Finding
    # -------------------------------------------------------------------------

    def find_paths(
        self,
        source: str | int,
        target: str | int,
        max_depth: int = 4,
        relationship_filter: str = None,
        limit: int = 10
    ) -> List[GraphPath]:
        """
        Find paths between two concepts.

        Uses BFS to find shortest paths first.

        Args:
            source: Source concept ID or name
            target: Target concept ID or name
            max_depth: Maximum path length
            relationship_filter: Only follow this relationship type
            limit: Maximum paths to return

        Returns:
            List of GraphPath objects
        """
        source_node = self.get_concept(source)
        target_node = self.get_concept(target)

        if not source_node or not target_node:
            logger.warning(f"Source or target concept not found")
            return []

        if source_node.concept_id == target_node.concept_id:
            return [GraphPath(nodes=[source_node], edges=[], total_weight=0)]

        # BFS for shortest paths
        paths = []
        visited = set()
        queue = [(source_node, [source_node], [])]  # (current, path_nodes, path_edges)

        while queue and len(paths) < limit:
            current, path_nodes, path_edges = queue.pop(0)

            if len(path_edges) >= max_depth:
                continue

            # Get neighbors
            neighbors = self.find_related(
                current.concept_id,
                relationship=relationship_filter,
                direction='both',
                limit=50
            )

            for neighbor_node, edge in neighbors:
                if neighbor_node.concept_id in visited:
                    continue

                new_path_nodes = path_nodes + [neighbor_node]
                new_path_edges = path_edges + [edge]

                if neighbor_node.concept_id == target_node.concept_id:
                    # Found a path
                    total_weight = sum(e.weight for e in new_path_edges)
                    paths.append(GraphPath(
                        nodes=new_path_nodes,
                        edges=new_path_edges,
                        total_weight=total_weight
                    ))
                else:
                    queue.append((neighbor_node, new_path_nodes, new_path_edges))

            visited.add(current.concept_id)

        # Sort by path length, then by total weight
        paths.sort(key=lambda p: (p.length, -p.total_weight))

        return paths[:limit]

    def trace_path(
        self,
        concept: str | int,
        direction: str = 'backward',
        relationship: str = 'derived_from',
        max_depth: int = 5
    ) -> TraversalResult:
        """
        Trace a chain of relationships from a concept.

        Useful for questions like "trace the origins of X" or
        "what influenced Y and what did those influence?"

        Args:
            concept: Starting concept
            direction: 'forward' or 'backward'
            relationship: Relationship type to follow
            max_depth: Maximum traversal depth

        Returns:
            TraversalResult with discovered concepts and paths
        """
        start_node = self.get_concept(concept)
        if not start_node:
            raise ValueError(f"Concept not found: {concept}")

        discovered = []
        paths = []
        visited = {start_node.concept_id}

        def traverse(node: ConceptNode, current_path: GraphPath, depth: int):
            if depth >= max_depth:
                return

            related = self.find_related(
                node.concept_id,
                relationship=relationship,
                direction=direction,
                limit=20
            )

            for next_node, edge in related:
                if next_node.concept_id in visited:
                    continue

                visited.add(next_node.concept_id)
                discovered.append(next_node)

                new_path = GraphPath(
                    nodes=current_path.nodes + [next_node],
                    edges=current_path.edges + [edge],
                    total_weight=current_path.total_weight + edge.weight
                )
                paths.append(new_path)

                traverse(next_node, new_path, depth + 1)

        initial_path = GraphPath(nodes=[start_node], edges=[], total_weight=0)
        traverse(start_node, initial_path, 0)

        return TraversalResult(
            query_concept=start_node,
            related_concepts=discovered,
            paths=paths,
            relationship_filter=relationship,
            direction=direction,
            depth=max_depth,
            stats={
                'concepts_discovered': len(discovered),
                'paths_found': len(paths),
                'max_path_length': max(p.length for p in paths) if paths else 0
            }
        )

    # -------------------------------------------------------------------------
    # Graph Statistics
    # -------------------------------------------------------------------------

    def get_statistics(self) -> Dict[str, Any]:
        """Get overall graph statistics."""
        concept_count = execute_query(
            "SELECT COUNT(*) as count FROM concepts",
            fetch='one'
        )

        relationship_count = execute_query(
            "SELECT COUNT(*) as count FROM concept_relationships",
            fetch='one'
        )

        by_type = execute_query(
            """
            SELECT relationship_type, COUNT(*) as count
            FROM concept_relationships
            GROUP BY relationship_type
            ORDER BY count DESC
            """,
            fetch='all'
        )

        by_category = execute_query(
            """
            SELECT category, COUNT(*) as count
            FROM concepts
            WHERE category IS NOT NULL
            GROUP BY category
            ORDER BY count DESC
            """,
            fetch='all'
        )

        avg_connections = execute_query(
            """
            SELECT AVG(connection_count) as avg_connections
            FROM (
                SELECT source_concept_id, COUNT(*) as connection_count
                FROM concept_relationships
                GROUP BY source_concept_id
            ) subq
            """,
            fetch='one'
        )

        return {
            'total_concepts': concept_count['count'] if concept_count else 0,
            'total_relationships': relationship_count['count'] if relationship_count else 0,
            'relationships_by_type': {r['relationship_type']: r['count'] for r in (by_type or [])},
            'concepts_by_category': {r['category']: r['count'] for r in (by_category or [])},
            'avg_connections_per_concept': round(avg_connections['avg_connections'] or 0, 2)
        }

    def get_most_connected(self, limit: int = 20) -> List[Tuple[ConceptNode, int]]:
        """Get concepts with most relationships."""
        results = execute_query(
            """
            SELECT c.concept_id, c.name, c.category, c.aliases,
                   (SELECT COUNT(*) FROM concept_relationships cr
                    WHERE cr.source_concept_id = c.concept_id
                       OR cr.target_concept_id = c.concept_id) as connection_count
            FROM concepts c
            ORDER BY connection_count DESC
            LIMIT %s
            """,
            (limit,),
            fetch='all'
        )

        nodes = []
        for r in results or []:
            node = ConceptNode(
                concept_id=r['concept_id'],
                name=r['name'],
                category=r['category'],
                aliases=r['aliases'] or []
            )
            nodes.append((node, r['connection_count']))

        return nodes


# =============================================================================
# RELATIONSHIP MANAGEMENT
# =============================================================================

def add_relationship(
    source_concept: str | int,
    target_concept: str | int,
    relationship_type: str,
    weight: float = 1.0,
    confidence: float = 1.0,
    document_id: str = None,
    chunk_id: str = None,
    bidirectional: bool = None
) -> bool:
    """
    Add a relationship between two concepts.

    Args:
        source_concept: Source concept ID or name
        target_concept: Target concept ID or name
        relationship_type: Type of relationship
        weight: Relationship weight (higher = stronger)
        confidence: Confidence score
        document_id: Source document (if extracted from text)
        chunk_id: Source chunk
        bidirectional: Override default bidirectionality

    Returns:
        True if successful
    """
    graph = SemanticGraph()

    source_node = graph.get_concept(source_concept)
    target_node = graph.get_concept(target_concept)

    if not source_node or not target_node:
        logger.error("Source or target concept not found")
        return False

    # Determine bidirectionality
    if bidirectional is None:
        rel_info = RELATIONSHIP_TYPES.get(relationship_type, {})
        bidirectional = rel_info.get('bidirectional', False)

    # Insert relationship
    execute_query(
        """
        INSERT INTO concept_relationships
            (source_concept_id, target_concept_id, relationship_type,
             weight, confidence, document_id, chunk_id, bidirectional)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s)
        ON CONFLICT (source_concept_id, target_concept_id, relationship_type)
        DO UPDATE SET
            weight = GREATEST(concept_relationships.weight, EXCLUDED.weight),
            confidence = GREATEST(concept_relationships.confidence, EXCLUDED.confidence)
        """,
        (source_node.concept_id, target_node.concept_id, relationship_type,
         weight, confidence, document_id, chunk_id, bidirectional)
    )

    logger.info(f"Added relationship: {source_node.name} -[{relationship_type}]-> {target_node.name}")
    return True


def verify_relationship(
    source_concept: str | int,
    target_concept: str | int,
    relationship_type: str
) -> bool:
    """Mark a relationship as human-verified."""
    graph = SemanticGraph()

    source_node = graph.get_concept(source_concept)
    target_node = graph.get_concept(target_concept)

    if not source_node or not target_node:
        return False

    execute_query(
        """
        UPDATE concept_relationships
        SET verified = TRUE
        WHERE source_concept_id = %s
          AND target_concept_id = %s
          AND relationship_type = %s
        """,
        (source_node.concept_id, target_node.concept_id, relationship_type)
    )

    return True


# =============================================================================
# CLI
# =============================================================================

def main():
    import argparse

    parser = argparse.ArgumentParser(
        description='Semantic Graph Traversal',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Find concepts related to "Anthroposophy"
  %(prog)s --concept "Anthroposophy" --related

  # Find what supports a concept
  %(prog)s --concept "Eurythmy" --supporters

  # Find what a concept influenced
  %(prog)s --concept "Theosophy" --influences

  # Trace origins of a concept
  %(prog)s --concept "Anthroposophy" --trace-origins --depth 3

  # Find paths between two concepts
  %(prog)s --from "Theosophy" --to "Waldorf Education" --max-depth 4

  # Show graph statistics
  %(prog)s --stats

  # Show most connected concepts
  %(prog)s --most-connected 20
        """
    )

    # Concept queries
    parser.add_argument('--concept', '-c', help='Concept to query')
    parser.add_argument('--related', action='store_true', help='Show related concepts')
    parser.add_argument('--supporters', action='store_true', help='Show supporting concepts')
    parser.add_argument('--influences', action='store_true', help='Show influenced concepts')
    parser.add_argument('--origins', action='store_true', help='Show origin concepts')
    parser.add_argument('--contradicts', action='store_true', help='Show contradicting concepts')

    # Path finding
    parser.add_argument('--from', dest='from_concept', help='Source concept for path finding')
    parser.add_argument('--to', dest='to_concept', help='Target concept for path finding')
    parser.add_argument('--trace-origins', action='store_true', help='Trace origin chain')

    # Options
    parser.add_argument('--relationship', '-r', help='Filter by relationship type')
    parser.add_argument('--direction', choices=['forward', 'backward', 'both'], default='both')
    parser.add_argument('--depth', '--max-depth', type=int, default=4, help='Max traversal depth')
    parser.add_argument('--limit', type=int, default=20, help='Max results')

    # Statistics
    parser.add_argument('--stats', action='store_true', help='Show graph statistics')
    parser.add_argument('--most-connected', type=int, metavar='N', help='Show N most connected concepts')

    # Search
    parser.add_argument('--search', '-s', help='Search for concepts by name')

    args = parser.parse_args()

    graph = SemanticGraph()

    # Handle commands
    if args.stats:
        stats = graph.get_statistics()
        print("\nSemantic Graph Statistics")
        print("=" * 50)
        print(f"Total concepts: {stats['total_concepts']}")
        print(f"Total relationships: {stats['total_relationships']}")
        print(f"Avg connections per concept: {stats['avg_connections_per_concept']}")

        if stats['relationships_by_type']:
            print("\nRelationships by type:")
            for rel_type, count in stats['relationships_by_type'].items():
                print(f"  {rel_type}: {count}")

        if stats['concepts_by_category']:
            print("\nConcepts by category:")
            for category, count in stats['concepts_by_category'].items():
                print(f"  {category}: {count}")

    elif args.most_connected:
        results = graph.get_most_connected(args.most_connected)
        print(f"\nTop {args.most_connected} Most Connected Concepts")
        print("=" * 50)
        for node, count in results:
            print(f"  [{count:3d}] {node.name}")
            if node.category:
                print(f"        Category: {node.category}")

    elif args.search:
        results = graph.search_concepts(args.search, limit=args.limit)
        print(f"\nSearch results for '{args.search}':")
        print("=" * 50)
        for node in results:
            print(f"  {node.name} (ID: {node.concept_id})")
            if node.category:
                print(f"    Category: {node.category}")
            if node.aliases:
                print(f"    Aliases: {', '.join(node.aliases)}")

    elif args.from_concept and args.to_concept:
        paths = graph.find_paths(
            args.from_concept,
            args.to_concept,
            max_depth=args.depth,
            relationship_filter=args.relationship,
            limit=args.limit
        )
        print(f"\nPaths from '{args.from_concept}' to '{args.to_concept}':")
        print("=" * 60)

        if not paths:
            print("  No paths found")
        else:
            for i, path in enumerate(paths, 1):
                print(f"\n  Path {i} (length: {path.length}, weight: {path.total_weight:.2f}):")
                print(f"    {path.to_string()}")

    elif args.concept:
        node = graph.get_concept(args.concept)
        if not node:
            print(f"Concept not found: {args.concept}")
            return

        print(f"\nConcept: {node.name}")
        print("=" * 50)
        if node.category:
            print(f"Category: {node.category}")
        if node.aliases:
            print(f"Aliases: {', '.join(node.aliases)}")
        print(f"Document count: {node.document_count}")

        if args.trace_origins:
            result = graph.trace_path(
                node.concept_id,
                direction='forward',
                relationship='derived_from',
                max_depth=args.depth
            )
            print(f"\nOrigin trace ({len(result.related_concepts)} concepts found):")
            for path in result.paths[:args.limit]:
                print(f"  {path.to_string()}")

        elif args.supporters:
            supporters = graph.get_supporters(node.concept_id, limit=args.limit)
            print(f"\nSupporting concepts ({len(supporters)}):")
            for s in supporters:
                print(f"  - {s.name}")

        elif args.influences:
            influenced = graph.get_influences(node.concept_id, limit=args.limit)
            print(f"\nInfluenced concepts ({len(influenced)}):")
            for i in influenced:
                print(f"  - {i.name}")

        elif args.origins:
            origins = graph.get_origins(node.concept_id, limit=args.limit)
            print(f"\nOrigin concepts ({len(origins)}):")
            for o in origins:
                print(f"  - {o.name}")

        elif args.contradicts:
            contradictions = graph.get_contradictions(node.concept_id, limit=args.limit)
            print(f"\nContradicting concepts ({len(contradictions)}):")
            for c in contradictions:
                print(f"  - {c.name}")

        elif args.related:
            related = graph.find_related(
                node.concept_id,
                relationship=args.relationship,
                direction=args.direction,
                limit=args.limit
            )
            print(f"\nRelated concepts ({len(related)}):")
            for rel_node, edge in related:
                direction = "<-->" if edge.bidirectional else "-->"
                print(f"  {direction} [{edge.relationship_type}] {rel_node.name}")
                if edge.confidence < 1.0:
                    print(f"      (confidence: {edge.confidence:.2f})")

        else:
            # Default: show summary of all relationships
            related = graph.find_related(node.concept_id, direction='both', limit=10)
            print(f"\nRelationships ({len(related)} shown):")
            for rel_node, edge in related:
                direction = "<-->" if edge.bidirectional else "-->"
                print(f"  {direction} [{edge.relationship_type}] {rel_node.name}")

    else:
        parser.print_help()


if __name__ == '__main__':
    main()
