#!/usr/bin/env python3
"""
Review Queue System for Human-in-the-Loop Operations

Provides structured queues for human review of:
- Research gaps (gap approval queue)
- Concept merges / alias learning (merge review queue)
- Auto-fix metadata changes (metadata review queue)

Each queue supports:
- Priority ordering
- Confidence thresholds
- Batch approval/rejection
- Audit trail
"""

import json
import sqlite3
from dataclasses import dataclass, field, asdict
from datetime import datetime
from enum import Enum
from pathlib import Path
from typing import List, Dict, Any, Optional, Tuple
import hashlib


class ReviewStatus(Enum):
    """Status of a review item."""
    PENDING = "pending"
    APPROVED = "approved"
    REJECTED = "rejected"
    DEFERRED = "deferred"
    AUTO_APPROVED = "auto_approved"  # Met confidence threshold


class ReviewPriority(Enum):
    """Priority levels for review items."""
    CRITICAL = 1
    HIGH = 2
    MEDIUM = 3
    LOW = 4


class QueueType(Enum):
    """Types of review queues."""
    GAP = "gap"
    CONCEPT_MERGE = "concept_merge"
    ALIAS = "alias"
    METADATA_FIX = "metadata_fix"
    OCR_QUALITY = "ocr_quality"


class RejectionReason(Enum):
    """Standardized rejection reason codes."""
    INSUFFICIENT_EVIDENCE = "insufficient_evidence"  # Not enough support for this query
    WRONG_SCOPE = "wrong_scope"                      # Outside chapter/project scope
    DUPLICATE = "duplicate"                          # Already covered elsewhere
    NEEDS_WEB = "needs_web"                          # Library insufficient, use web search
    NEEDS_LIBRARY = "needs_library"                  # Web not appropriate, use library
    NEEDS_CONSTRAINTS = "needs_constraints"          # Query too broad, add filters
    DEFER = "defer"                                  # Not now, re-review later
    QUALITY_CONCERN = "quality_concern"              # Source reliability issue
    OUT_OF_SCOPE = "out_of_scope"                    # Not relevant to research
    MERGED = "merged"                                # Combined with another item
    OTHER = "other"                                  # See notes for details


@dataclass
class FeedbackConstraints:
    """Structured constraints for rejected items that should be retried."""
    strategy_override: Optional[str] = None          # "web", "library", "hybrid"
    source_type: Optional[str] = None                # "academic", "primary", "any"
    date_range_start: Optional[int] = None           # Year
    date_range_end: Optional[int] = None             # Year
    domain_whitelist: List[str] = field(default_factory=list)  # Allowed domains
    domain_blacklist: List[str] = field(default_factory=list)  # Blocked domains
    exclude_terms: List[str] = field(default_factory=list)     # Terms to exclude
    include_terms: List[str] = field(default_factory=list)     # Required terms
    modified_query: Optional[str] = None             # Alternative query text

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary, excluding None values."""
        return {k: v for k, v in asdict(self).items() if v is not None and v != []}

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'FeedbackConstraints':
        """Create from dictionary."""
        return cls(**{k: v for k, v in data.items() if k in cls.__dataclass_fields__})


@dataclass
class ReviewItem:
    """Base class for all review queue items."""
    item_id: str
    queue_type: QueueType
    status: ReviewStatus = ReviewStatus.PENDING
    priority: ReviewPriority = ReviewPriority.MEDIUM
    confidence: float = 0.0
    created_at: str = field(default_factory=lambda: datetime.utcnow().isoformat())
    reviewed_at: Optional[str] = None
    reviewed_by: Optional[str] = None
    notes: str = ""
    context: Dict[str, Any] = field(default_factory=dict)
    # New feedback fields for enhanced rejection handling
    rejection_reason: Optional[RejectionReason] = None
    feedback: Optional[str] = None  # Free-text feedback
    feedback_constraints: Optional[FeedbackConstraints] = None  # Structured constraints
    requeued_from: Optional[str] = None  # ID of original item if this was requeued
    requeued_to: Optional[str] = None    # ID of new item if this was requeued

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        d = asdict(self)
        d['queue_type'] = self.queue_type.value
        d['status'] = self.status.value
        d['priority'] = self.priority.value
        if self.rejection_reason:
            d['rejection_reason'] = self.rejection_reason.value
        if self.feedback_constraints:
            d['feedback_constraints'] = self.feedback_constraints.to_dict()
        return d

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'ReviewItem':
        """Create from dictionary."""
        data['queue_type'] = QueueType(data['queue_type'])
        data['status'] = ReviewStatus(data['status'])
        data['priority'] = ReviewPriority(data['priority'])
        if data.get('rejection_reason'):
            data['rejection_reason'] = RejectionReason(data['rejection_reason'])
        if data.get('feedback_constraints'):
            data['feedback_constraints'] = FeedbackConstraints.from_dict(data['feedback_constraints'])
        return cls(**data)


@dataclass
class GapReviewItem(ReviewItem):
    """Review item for research gaps."""
    gap_query: str = ""
    chapter_id: Optional[str] = None
    subject: str = ""
    suggested_search: str = ""
    occurrence_count: int = 1

    def __post_init__(self):
        self.queue_type = QueueType.GAP
        if not self.item_id:
            self.item_id = f"gap_{hashlib.md5(self.gap_query.encode()).hexdigest()[:12]}"


@dataclass
class ConceptMergeReviewItem(ReviewItem):
    """Review item for concept merge suggestions."""
    source_concept: str = ""
    target_concept: str = ""
    source_id: Optional[int] = None
    target_id: Optional[int] = None
    similarity_score: float = 0.0
    merge_reason: str = ""  # fuzzy_match, abbreviation, alias, duplicate
    affected_documents: int = 0

    def __post_init__(self):
        self.queue_type = QueueType.CONCEPT_MERGE
        if not self.item_id:
            self.item_id = f"merge_{self.source_id}_{self.target_id}"


@dataclass
class AliasReviewItem(ReviewItem):
    """Review item for alias learning."""
    alias: str = ""
    canonical_name: str = ""
    concept_id: Optional[int] = None
    detection_method: str = ""  # abbreviation, last_name, fuzzy, learned
    example_contexts: List[str] = field(default_factory=list)

    def __post_init__(self):
        self.queue_type = QueueType.ALIAS
        if not self.item_id:
            self.item_id = f"alias_{hashlib.md5(f'{self.alias}_{self.canonical_name}'.encode()).hexdigest()[:12]}"


@dataclass
class MetadataFixReviewItem(ReviewItem):
    """Review item for auto-fix metadata changes."""
    doc_id: str = ""
    field_name: str = ""  # title, author, year, etc.
    original_value: str = ""
    suggested_value: str = ""
    fix_source: str = ""  # filename_parse, llm_extraction, crossref, openlibrary

    def __post_init__(self):
        self.queue_type = QueueType.METADATA_FIX
        if not self.item_id:
            self.item_id = f"meta_{self.doc_id}_{self.field_name}"


@dataclass
class ReviewQueueConfig:
    """Configuration for review queue behavior."""
    # Auto-approve thresholds (items above this confidence are auto-approved)
    gap_auto_approve_threshold: float = 0.0  # Never auto-approve gaps
    merge_auto_approve_threshold: float = 0.95
    alias_auto_approve_threshold: float = 0.90
    metadata_auto_approve_threshold: float = 0.85

    # Review required thresholds (items below this require human review)
    merge_review_required_threshold: float = 0.70
    alias_review_required_threshold: float = 0.60
    metadata_review_required_threshold: float = 0.50

    # Queue limits
    max_pending_per_queue: int = 1000
    batch_size: int = 20

    # Persistence
    queue_file: str = "review_queue.json"
    audit_log_file: str = "review_audit.jsonl"


class ReviewQueue:
    """
    Manages review queues for human-in-the-loop operations.

    Supports multiple queue types with configurable auto-approve thresholds
    and priority ordering.
    """

    def __init__(self, base_dir: Path, config: Optional[ReviewQueueConfig] = None):
        self.base_dir = Path(base_dir)
        self.config = config or ReviewQueueConfig()
        self.queue_file = self.base_dir / self.config.queue_file
        self.audit_file = self.base_dir / self.config.audit_log_file

        # In-memory queues
        self._queues: Dict[QueueType, List[ReviewItem]] = {
            qt: [] for qt in QueueType
        }

        # Load existing queue
        self._load_queue()

    def _load_queue(self):
        """Load queue from disk."""
        if self.queue_file.exists():
            try:
                with open(self.queue_file, 'r') as f:
                    data = json.load(f)

                for qt in QueueType:
                    queue_data = data.get(qt.value, [])
                    self._queues[qt] = []
                    for item_data in queue_data:
                        item_class = self._get_item_class(qt)
                        try:
                            item = item_class.from_dict(item_data)
                            self._queues[qt].append(item)
                        except Exception:
                            # Skip malformed items
                            pass
            except Exception:
                pass

    def _save_queue(self):
        """Save queue to disk."""
        self.base_dir.mkdir(parents=True, exist_ok=True)
        data = {}
        for qt, items in self._queues.items():
            data[qt.value] = [item.to_dict() for item in items]

        with open(self.queue_file, 'w') as f:
            json.dump(data, f, indent=2)

    def _get_item_class(self, queue_type: QueueType):
        """Get the appropriate item class for a queue type."""
        mapping = {
            QueueType.GAP: GapReviewItem,
            QueueType.CONCEPT_MERGE: ConceptMergeReviewItem,
            QueueType.ALIAS: AliasReviewItem,
            QueueType.METADATA_FIX: MetadataFixReviewItem,
            QueueType.OCR_QUALITY: ReviewItem,
        }
        return mapping.get(queue_type, ReviewItem)

    def _get_auto_approve_threshold(self, queue_type: QueueType) -> float:
        """Get auto-approve threshold for a queue type."""
        thresholds = {
            QueueType.GAP: self.config.gap_auto_approve_threshold,
            QueueType.CONCEPT_MERGE: self.config.merge_auto_approve_threshold,
            QueueType.ALIAS: self.config.alias_auto_approve_threshold,
            QueueType.METADATA_FIX: self.config.metadata_auto_approve_threshold,
        }
        return thresholds.get(queue_type, 0.0)

    def _log_audit(self, action: str, item: ReviewItem, details: Dict[str, Any] = None):
        """Log an audit entry."""
        self.base_dir.mkdir(parents=True, exist_ok=True)
        entry = {
            "timestamp": datetime.utcnow().isoformat(),
            "action": action,
            "item_id": item.item_id,
            "queue_type": item.queue_type.value,
            "status": item.status.value,
            "details": details or {}
        }
        with open(self.audit_file, 'a') as f:
            f.write(json.dumps(entry) + "\n")

    def add_item(self, item: ReviewItem) -> Tuple[str, bool]:
        """
        Add an item to the review queue.

        Returns:
            Tuple of (item_id, auto_approved)
        """
        queue = self._queues[item.queue_type]

        # Check for duplicates
        existing = next((i for i in queue if i.item_id == item.item_id), None)
        if existing:
            # Update occurrence count for gaps
            if isinstance(item, GapReviewItem) and isinstance(existing, GapReviewItem):
                existing.occurrence_count += 1
                existing.priority = ReviewPriority.HIGH if existing.occurrence_count >= 3 else existing.priority
            self._save_queue()
            return item.item_id, False

        # Check auto-approve threshold
        threshold = self._get_auto_approve_threshold(item.queue_type)
        if item.confidence >= threshold and threshold > 0:
            item.status = ReviewStatus.AUTO_APPROVED
            item.reviewed_at = datetime.utcnow().isoformat()
            item.reviewed_by = "system"
            self._log_audit("auto_approved", item, {"confidence": item.confidence, "threshold": threshold})
            queue.append(item)
            self._save_queue()
            return item.item_id, True

        # Add to queue
        queue.append(item)
        self._log_audit("added", item)
        self._save_queue()
        return item.item_id, False

    def get_pending(self, queue_type: QueueType, limit: Optional[int] = None) -> List[ReviewItem]:
        """Get pending items from a queue, sorted by priority."""
        queue = self._queues[queue_type]
        pending = [i for i in queue if i.status == ReviewStatus.PENDING]

        # Sort by priority (lower value = higher priority), then by created_at
        pending.sort(key=lambda x: (x.priority.value, x.created_at))

        if limit:
            pending = pending[:limit]

        return pending

    def get_batch(self, queue_type: QueueType) -> List[ReviewItem]:
        """Get a batch of items for review."""
        return self.get_pending(queue_type, self.config.batch_size)

    def approve(self, item_id: str, reviewed_by: str = "user", notes: str = "") -> bool:
        """Approve an item."""
        return self._update_status(item_id, ReviewStatus.APPROVED, reviewed_by, notes)

    def reject(
        self,
        item_id: str,
        reviewed_by: str = "user",
        notes: str = "",
        reason: Optional[RejectionReason] = None,
        feedback: Optional[str] = None,
        constraints: Optional[FeedbackConstraints] = None
    ) -> bool:
        """
        Reject an item with optional structured feedback.

        Args:
            item_id: ID of the item to reject
            reviewed_by: Who is rejecting (user/agent/system)
            notes: Free-text notes about the rejection
            reason: Standardized rejection reason code
            feedback: Free-text feedback for improvement
            constraints: Structured constraints for re-queuing

        Returns:
            True if item was found and rejected
        """
        for queue in self._queues.values():
            for item in queue:
                if item.item_id == item_id:
                    item.status = ReviewStatus.REJECTED
                    item.reviewed_at = datetime.utcnow().isoformat()
                    item.reviewed_by = reviewed_by
                    item.notes = notes
                    item.rejection_reason = reason
                    item.feedback = feedback
                    item.feedback_constraints = constraints

                    audit_details = {
                        "reviewed_by": reviewed_by,
                        "notes": notes
                    }
                    if reason:
                        audit_details["reason"] = reason.value
                    if feedback:
                        audit_details["feedback"] = feedback
                    if constraints:
                        audit_details["constraints"] = constraints.to_dict()

                    self._log_audit("rejected", item, audit_details)
                    self._save_queue()
                    return True
        return False

    def reject_with_requeue(
        self,
        item_id: str,
        reviewed_by: str = "user",
        notes: str = "",
        reason: Optional[RejectionReason] = None,
        feedback: Optional[str] = None,
        constraints: Optional[FeedbackConstraints] = None
    ) -> Optional[str]:
        """
        Reject an item and create a new modified item in the queue.

        This is useful when a gap should be retried with different parameters
        (e.g., different search strategy or constraints).

        Args:
            item_id: ID of the item to reject
            reviewed_by: Who is rejecting
            notes: Free-text notes
            reason: Rejection reason code
            feedback: Free-text feedback
            constraints: Structured constraints for the new item

        Returns:
            New item ID if requeued, None if original not found
        """
        # Find the original item
        original = self.get_item(item_id)
        if not original:
            return None

        # Reject the original
        self.reject(item_id, reviewed_by, notes, reason, feedback, constraints)

        # Create a new item based on the original
        if isinstance(original, GapReviewItem):
            # Apply constraints to create modified gap
            modified_query = original.gap_query
            if constraints and constraints.modified_query:
                modified_query = constraints.modified_query

            new_item = GapReviewItem(
                item_id="",  # Will be generated
                queue_type=QueueType.GAP,
                gap_query=modified_query,
                chapter_id=original.chapter_id,
                subject=original.subject,
                suggested_search=modified_query,
                priority=original.priority,
                context={
                    **original.context,
                    "constraints": constraints.to_dict() if constraints else {},
                    "retry_count": original.context.get("retry_count", 0) + 1
                }
            )
            new_item.requeued_from = item_id
            new_id, _ = self.add_item(new_item)

            # Update original with requeue reference
            original.requeued_to = new_id
            self._save_queue()

            self._log_audit("requeued", original, {
                "new_item_id": new_id,
                "constraints": constraints.to_dict() if constraints else {}
            })

            return new_id

        # For other item types, just reject without requeue
        return None

    def defer(self, item_id: str, reviewed_by: str = "user", notes: str = "") -> bool:
        """Defer an item for later review."""
        return self._update_status(item_id, ReviewStatus.DEFERRED, reviewed_by, notes)

    def _update_status(self, item_id: str, status: ReviewStatus, reviewed_by: str, notes: str) -> bool:
        """Update the status of an item."""
        for queue in self._queues.values():
            for item in queue:
                if item.item_id == item_id:
                    item.status = status
                    item.reviewed_at = datetime.utcnow().isoformat()
                    item.reviewed_by = reviewed_by
                    item.notes = notes
                    self._log_audit(status.value, item, {"reviewed_by": reviewed_by, "notes": notes})
                    self._save_queue()
                    return True
        return False

    def batch_approve(self, item_ids: List[str], reviewed_by: str = "user") -> int:
        """Approve multiple items at once."""
        count = 0
        for item_id in item_ids:
            if self.approve(item_id, reviewed_by):
                count += 1
        return count

    def batch_reject(self, item_ids: List[str], reviewed_by: str = "user") -> int:
        """Reject multiple items at once."""
        count = 0
        for item_id in item_ids:
            if self.reject(item_id, reviewed_by):
                count += 1
        return count

    def get_stats(self) -> Dict[str, Any]:
        """Get queue statistics."""
        stats = {
            "total_items": 0,
            "by_queue": {},
            "by_status": {s.value: 0 for s in ReviewStatus}
        }

        for qt, queue in self._queues.items():
            queue_stats = {
                "total": len(queue),
                "pending": sum(1 for i in queue if i.status == ReviewStatus.PENDING),
                "approved": sum(1 for i in queue if i.status == ReviewStatus.APPROVED),
                "rejected": sum(1 for i in queue if i.status == ReviewStatus.REJECTED),
                "auto_approved": sum(1 for i in queue if i.status == ReviewStatus.AUTO_APPROVED),
            }
            stats["by_queue"][qt.value] = queue_stats
            stats["total_items"] += queue_stats["total"]

            for item in queue:
                stats["by_status"][item.status.value] += 1

        return stats

    def get_item(self, item_id: str) -> Optional[ReviewItem]:
        """Get a specific item by ID."""
        for queue in self._queues.values():
            for item in queue:
                if item.item_id == item_id:
                    return item
        return None

    def clear_completed(self, queue_type: Optional[QueueType] = None, older_than_days: int = 30):
        """Remove completed (approved/rejected) items older than specified days."""
        from datetime import timedelta
        cutoff = datetime.utcnow() - timedelta(days=older_than_days)
        cutoff_str = cutoff.isoformat()

        queues_to_clean = [queue_type] if queue_type else list(QueueType)

        for qt in queues_to_clean:
            self._queues[qt] = [
                item for item in self._queues[qt]
                if item.status == ReviewStatus.PENDING or
                   item.status == ReviewStatus.DEFERRED or
                   (item.reviewed_at and item.reviewed_at > cutoff_str)
            ]

        self._save_queue()

    def export_pending(self, queue_type: QueueType, output_file: Path, format: str = "json") -> int:
        """Export pending items to a file for external review."""
        pending = self.get_pending(queue_type)

        if format == "json":
            with open(output_file, 'w') as f:
                json.dump([item.to_dict() for item in pending], f, indent=2)
        elif format == "markdown":
            with open(output_file, 'w') as f:
                f.write(self._render_markdown(queue_type, pending))

        return len(pending)

    def _render_markdown(self, queue_type: QueueType, items: List[ReviewItem]) -> str:
        """Render items as markdown for human review."""
        lines = [
            f"# Review Queue: {queue_type.value.replace('_', ' ').title()}",
            "",
            f"**Generated:** {datetime.utcnow().isoformat()}",
            f"**Pending Items:** {len(items)}",
            "",
            "## Instructions",
            "1. Mark items to approve: `[ ]` → `[x]`",
            "2. Mark items to reject: `[ ]` → `[-]`",
            "3. Leave unchanged to defer",
            "",
            "---",
            ""
        ]

        for i, item in enumerate(items, 1):
            lines.append(f"### Item {i}: {item.item_id}")
            lines.append(f"- [ ] **Approve this item**")
            lines.append(f"- Priority: {item.priority.name}")
            lines.append(f"- Confidence: {item.confidence:.2%}")

            if isinstance(item, GapReviewItem):
                lines.append(f"- Gap Query: {item.gap_query}")
                lines.append(f"- Subject: {item.subject}")
                lines.append(f"- Suggested Search: `{item.suggested_search}`")
                lines.append(f"- Occurrences: {item.occurrence_count}")
            elif isinstance(item, ConceptMergeReviewItem):
                lines.append(f"- Merge: \"{item.source_concept}\" → \"{item.target_concept}\"")
                lines.append(f"- Similarity: {item.similarity_score:.2%}")
                lines.append(f"- Reason: {item.merge_reason}")
                lines.append(f"- Affected Documents: {item.affected_documents}")
            elif isinstance(item, AliasReviewItem):
                lines.append(f"- Alias: \"{item.alias}\" → \"{item.canonical_name}\"")
                lines.append(f"- Detection: {item.detection_method}")
                if item.example_contexts:
                    lines.append(f"- Examples:")
                    for ctx in item.example_contexts[:3]:
                        lines.append(f"  - {ctx[:100]}...")
            elif isinstance(item, MetadataFixReviewItem):
                lines.append(f"- Document: {item.doc_id}")
                lines.append(f"- Field: {item.field_name}")
                lines.append(f"- Original: \"{item.original_value}\"")
                lines.append(f"- Suggested: \"{item.suggested_value}\"")
                lines.append(f"- Source: {item.fix_source}")

            lines.append("")

        return "\n".join(lines)

    def import_decisions(self, input_file: Path) -> Tuple[int, int]:
        """
        Import decisions from a reviewed markdown file.

        Returns:
            Tuple of (approved_count, rejected_count)
        """
        approved = 0
        rejected = 0

        with open(input_file, 'r') as f:
            content = f.read()

        import re
        # Find all items and their checkbox states
        item_pattern = r'### Item \d+: (\S+)\n- \[([ x\-])\] \*\*Approve'
        matches = re.findall(item_pattern, content)

        for item_id, checkbox in matches:
            if checkbox == 'x':
                if self.approve(item_id, reviewed_by="markdown_import"):
                    approved += 1
            elif checkbox == '-':
                if self.reject(item_id, reviewed_by="markdown_import"):
                    rejected += 1

        return approved, rejected


class GapReviewQueue(ReviewQueue):
    """Specialized queue for research gaps with gap-specific helpers."""

    def add_gap(self, gap_query: str, chapter_id: str = None, subject: str = "",
                suggested_search: str = "", priority: ReviewPriority = ReviewPriority.MEDIUM) -> str:
        """Add a research gap for review."""
        item = GapReviewItem(
            item_id="",  # Will be generated
            queue_type=QueueType.GAP,
            gap_query=gap_query,
            chapter_id=chapter_id,
            subject=subject,
            suggested_search=suggested_search or gap_query,
            priority=priority
        )
        item_id, _ = self.add_item(item)
        return item_id

    def get_approved_gaps(self) -> List[GapReviewItem]:
        """Get all approved gaps ready for research."""
        return [
            i for i in self._queues[QueueType.GAP]
            if i.status == ReviewStatus.APPROVED and isinstance(i, GapReviewItem)
        ]


class ConceptMergeQueue(ReviewQueue):
    """Specialized queue for concept merge suggestions."""

    def suggest_merge(self, source_concept: str, target_concept: str,
                      source_id: int, target_id: int, similarity: float,
                      reason: str, affected_docs: int = 0) -> Tuple[str, bool]:
        """Suggest a concept merge for review."""
        item = ConceptMergeReviewItem(
            item_id="",
            queue_type=QueueType.CONCEPT_MERGE,
            source_concept=source_concept,
            target_concept=target_concept,
            source_id=source_id,
            target_id=target_id,
            similarity_score=similarity,
            merge_reason=reason,
            affected_documents=affected_docs,
            confidence=similarity,
            priority=ReviewPriority.HIGH if affected_docs > 5 else ReviewPriority.MEDIUM
        )
        return self.add_item(item)

    def get_approved_merges(self) -> List[ConceptMergeReviewItem]:
        """Get approved merges ready to execute."""
        return [
            i for i in self._queues[QueueType.CONCEPT_MERGE]
            if i.status in (ReviewStatus.APPROVED, ReviewStatus.AUTO_APPROVED)
            and isinstance(i, ConceptMergeReviewItem)
        ]


class MetadataFixQueue(ReviewQueue):
    """Specialized queue for metadata auto-fix suggestions."""

    def suggest_fix(self, doc_id: str, field_name: str, original: str,
                    suggested: str, source: str, confidence: float) -> Tuple[str, bool]:
        """Suggest a metadata fix for review."""
        item = MetadataFixReviewItem(
            item_id="",
            queue_type=QueueType.METADATA_FIX,
            doc_id=doc_id,
            field_name=field_name,
            original_value=original,
            suggested_value=suggested,
            fix_source=source,
            confidence=confidence,
            priority=ReviewPriority.LOW if confidence > 0.8 else ReviewPriority.MEDIUM
        )
        return self.add_item(item)

    def get_approved_fixes(self) -> List[MetadataFixReviewItem]:
        """Get approved fixes ready to apply."""
        return [
            i for i in self._queues[QueueType.METADATA_FIX]
            if i.status in (ReviewStatus.APPROVED, ReviewStatus.AUTO_APPROVED)
            and isinstance(i, MetadataFixReviewItem)
        ]


# CLI Interface
def main():
    """Command-line interface for review queue management."""
    import argparse

    parser = argparse.ArgumentParser(description="Review Queue Manager")
    subparsers = parser.add_subparsers(dest="command", help="Commands")

    # Stats command
    stats_parser = subparsers.add_parser("stats", help="Show queue statistics")
    stats_parser.add_argument("--format", choices=["text", "json"], default="text")

    # List command
    list_parser = subparsers.add_parser("list", help="List pending items")
    list_parser.add_argument("queue", choices=["gap", "merge", "alias", "metadata"])
    list_parser.add_argument("--limit", type=int, default=20)
    list_parser.add_argument("--format", choices=["text", "json", "markdown"], default="text")

    # Export command
    export_parser = subparsers.add_parser("export", help="Export pending items for review")
    export_parser.add_argument("queue", choices=["gap", "merge", "alias", "metadata"])
    export_parser.add_argument("-o", "--output", required=True, help="Output file")
    export_parser.add_argument("--format", choices=["json", "markdown"], default="markdown")

    # Import command
    import_parser = subparsers.add_parser("import", help="Import decisions from reviewed file")
    import_parser.add_argument("file", help="Reviewed markdown file")

    # Approve/Reject commands
    approve_parser = subparsers.add_parser("approve", help="Approve an item")
    approve_parser.add_argument("item_id", help="Item ID to approve")
    approve_parser.add_argument("--notes", default="", help="Review notes")

    reject_parser = subparsers.add_parser("reject", help="Reject an item with optional feedback")
    reject_parser.add_argument("item_id", help="Item ID to reject")
    reject_parser.add_argument("--notes", default="", help="Review notes")
    reject_parser.add_argument(
        "--reason", "-r",
        choices=[r.value for r in RejectionReason],
        help="Rejection reason code (use 'reasons' command to see all codes)"
    )
    reject_parser.add_argument(
        "--feedback", "-f",
        help="Free-text feedback for improvement"
    )
    reject_parser.add_argument(
        "--requeue",
        action="store_true",
        help="Re-queue with modified constraints (gaps only)"
    )
    reject_parser.add_argument(
        "--strategy",
        choices=["web", "library", "hybrid"],
        help="Override search strategy for requeue"
    )
    reject_parser.add_argument(
        "--source-type",
        choices=["academic", "primary", "any"],
        help="Constrain source type for requeue"
    )
    reject_parser.add_argument(
        "--date-range",
        help="Date range constraint (e.g., '1100-1300')"
    )
    reject_parser.add_argument(
        "--modified-query",
        help="Alternative query text for requeue"
    )
    reject_parser.add_argument("--format", choices=["text", "json"], default="text")

    # Reasons command - list all rejection reason codes
    reasons_parser = subparsers.add_parser("reasons", help="List all rejection reason codes")
    reasons_parser.add_argument("--format", choices=["text", "json"], default="text")

    # Clear command
    clear_parser = subparsers.add_parser("clear", help="Clear old completed items")
    clear_parser.add_argument("--older-than", type=int, default=30, help="Days old")

    args = parser.parse_args()

    # Get base directory
    base_dir = Path(__file__).parent.parent / "review_queues"
    queue = ReviewQueue(base_dir)

    queue_type_map = {
        "gap": QueueType.GAP,
        "merge": QueueType.CONCEPT_MERGE,
        "alias": QueueType.ALIAS,
        "metadata": QueueType.METADATA_FIX,
    }

    if args.command == "stats":
        stats = queue.get_stats()
        if args.format == "json":
            print(json.dumps(stats, indent=2))
        else:
            print("Review Queue Statistics")
            print("=" * 40)
            print(f"Total items: {stats['total_items']}")
            print()
            for qt, qs in stats["by_queue"].items():
                print(f"{qt}:")
                print(f"  Pending: {qs['pending']}")
                print(f"  Approved: {qs['approved']}")
                print(f"  Auto-approved: {qs['auto_approved']}")
                print(f"  Rejected: {qs['rejected']}")

    elif args.command == "list":
        qt = queue_type_map[args.queue]
        pending = queue.get_pending(qt, args.limit)

        if args.format == "json":
            print(json.dumps([i.to_dict() for i in pending], indent=2))
        elif args.format == "markdown":
            print(queue._render_markdown(qt, pending))
        else:
            print(f"Pending {args.queue} items ({len(pending)}):")
            for item in pending:
                print(f"  [{item.item_id}] {item.priority.name} - confidence: {item.confidence:.2%}")

    elif args.command == "export":
        qt = queue_type_map[args.queue]
        output = Path(args.output)
        count = queue.export_pending(qt, output, args.format)
        print(f"Exported {count} pending items to {output}")

    elif args.command == "import":
        approved, rejected = queue.import_decisions(Path(args.file))
        print(f"Imported decisions: {approved} approved, {rejected} rejected")

    elif args.command == "approve":
        if queue.approve(args.item_id, notes=args.notes):
            print(f"Approved: {args.item_id}")
        else:
            print(f"Item not found: {args.item_id}")

    elif args.command == "reject":
        # Parse rejection reason
        reason = None
        if args.reason:
            reason = RejectionReason(args.reason)

        # Build constraints if any constraint flags provided
        constraints = None
        if args.requeue or args.strategy or args.source_type or args.date_range or args.modified_query:
            date_start = None
            date_end = None
            if args.date_range:
                try:
                    parts = args.date_range.split("-")
                    date_start = int(parts[0])
                    date_end = int(parts[1]) if len(parts) > 1 else None
                except (ValueError, IndexError):
                    print(f"Invalid date range format: {args.date_range}. Use 'YYYY-YYYY' format.")
                    return

            constraints = FeedbackConstraints(
                strategy_override=args.strategy,
                source_type=args.source_type,
                date_range_start=date_start,
                date_range_end=date_end,
                modified_query=args.modified_query
            )

        # Perform rejection
        if args.requeue:
            new_id = queue.reject_with_requeue(
                args.item_id,
                notes=args.notes,
                reason=reason,
                feedback=args.feedback,
                constraints=constraints
            )
            if new_id:
                result = {
                    "status": "success",
                    "action": "rejected_and_requeued",
                    "original_id": args.item_id,
                    "new_id": new_id,
                    "constraints": constraints.to_dict() if constraints else {}
                }
                if args.format == "json":
                    print(json.dumps(result, indent=2))
                else:
                    print(f"Rejected: {args.item_id}")
                    print(f"Re-queued as: {new_id}")
                    if constraints:
                        print(f"Constraints: {constraints.to_dict()}")
            else:
                if args.format == "json":
                    print(json.dumps({"status": "error", "message": f"Item not found: {args.item_id}"}, indent=2))
                else:
                    print(f"Item not found or not requeuable: {args.item_id}")
        else:
            if queue.reject(args.item_id, notes=args.notes, reason=reason, feedback=args.feedback, constraints=constraints):
                result = {
                    "status": "success",
                    "action": "rejected",
                    "item_id": args.item_id,
                    "reason": reason.value if reason else None,
                    "feedback": args.feedback
                }
                if args.format == "json":
                    print(json.dumps(result, indent=2))
                else:
                    print(f"Rejected: {args.item_id}")
                    if reason:
                        print(f"Reason: {reason.value}")
                    if args.feedback:
                        print(f"Feedback: {args.feedback}")
            else:
                if args.format == "json":
                    print(json.dumps({"status": "error", "message": f"Item not found: {args.item_id}"}, indent=2))
                else:
                    print(f"Item not found: {args.item_id}")

    elif args.command == "reasons":
        # Display all rejection reason codes
        reasons_info = {
            RejectionReason.INSUFFICIENT_EVIDENCE: "Not enough support for this query",
            RejectionReason.WRONG_SCOPE: "Outside chapter/project scope",
            RejectionReason.DUPLICATE: "Already covered elsewhere",
            RejectionReason.NEEDS_WEB: "Library insufficient, use web search",
            RejectionReason.NEEDS_LIBRARY: "Web not appropriate, use library",
            RejectionReason.NEEDS_CONSTRAINTS: "Query too broad, add filters",
            RejectionReason.DEFER: "Not now, re-review later",
            RejectionReason.QUALITY_CONCERN: "Source reliability issue",
            RejectionReason.OUT_OF_SCOPE: "Not relevant to research",
            RejectionReason.MERGED: "Combined with another item",
            RejectionReason.OTHER: "See notes for details",
        }

        if args.format == "json":
            print(json.dumps({
                "reasons": {r.value: desc for r, desc in reasons_info.items()}
            }, indent=2))
        else:
            print("Rejection Reason Codes")
            print("=" * 50)
            for reason, desc in reasons_info.items():
                print(f"  {reason.value:24} {desc}")

    elif args.command == "clear":
        queue.clear_completed(older_than_days=args.older_than)
        print(f"Cleared completed items older than {args.older_than} days")

    else:
        parser.print_help()


if __name__ == "__main__":
    main()
