#!/bin/bash
# =============================================================================
# Research Session Wrapper
# =============================================================================
# Unifies search, output, and project tracking into a single command.
#
# Usage:
#   ./research_session.sh PROJECT_ID "Search Query" [options]
#
# Options:
#   --semantic    Use semantic (embedding) search instead of keyword
#   --category    Filter by document category
#   --year-from   Filter by minimum year
#   --year-to     Filter by maximum year
#   --limit       Max results (default: 20)
#
# Examples:
#   ./research_session.sh PROJ_123 "etheric body memory"
#   ./research_session.sh PROJ_123 "spiritual development" --semantic
#   ./research_session.sh PROJ_123 "alchemy" --category "Philosophy" --year-from 1800
#
# =============================================================================

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(dirname "$SCRIPT_DIR")"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Help
show_help() {
    cat << 'EOF'
Research Session Wrapper - Unified search, output, and project tracking

USAGE:
    ./research_session.sh PROJECT_ID "Search Query" [OPTIONS]

ARGUMENTS:
    PROJECT_ID      The project ID (e.g., PROJ_20251211123456_abc123)
    "Search Query"  The search query in quotes

OPTIONS:
    --semantic      Use semantic (embedding) search instead of keyword search
    --category CAT  Filter results by document category
    --year-from Y   Filter results from this year onwards
    --year-to Y     Filter results up to this year
    --limit N       Maximum number of results (default: 20)
    --batch FILE    Process multiple queries from a file (one per line)
    --help          Show this help message

EXAMPLES:
    # Basic keyword search
    ./research_session.sh PROJ_123 "mystery schools initiation"

    # Semantic search for conceptual matches
    ./research_session.sh PROJ_123 "stages of inner development" --semantic

    # Filtered historical search
    ./research_session.sh PROJ_123 "alchemy" --category "Philosophy" --year-from 1850 --year-to 1920

    # Batch processing from a questions file
    ./research_session.sh PROJ_123 --batch chapter3_questions.txt

OUTPUT:
    Results are saved to: projects/PROJECT_ID/TIMESTAMP_query.md
    The search is automatically logged to the project tracker.

EOF
    exit 0
}

# Parse arguments
if [[ $# -lt 1 ]] || [[ "$1" == "--help" ]]; then
    show_help
fi

PROJ_ID="$1"
shift

# Check if project exists
PROJ_FILE="$PROJECT_ROOT/research_projects/${PROJ_ID}.json"
if [[ ! -f "$PROJ_FILE" ]]; then
    echo -e "${RED}Error: Project not found: $PROJ_ID${NC}"
    echo "Available projects:"
    python3 "$SCRIPT_DIR/research_project.py" list 2>/dev/null | grep -E "^\s+PROJ_" || echo "  (none)"
    echo ""
    echo "Create a new project with:"
    echo "  python3 pipeline/research_project.py create \"Project Name\""
    exit 1
fi

# Initialize variables
QUERY=""
SEMANTIC=""
CATEGORY=""
YEAR_FROM=""
YEAR_TO=""
LIMIT="20"
BATCH_FILE=""

# Parse remaining arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --semantic)
            SEMANTIC="--semantic"
            shift
            ;;
        --category)
            CATEGORY="--category $2"
            shift 2
            ;;
        --year-from)
            YEAR_FROM="--year-from $2"
            shift 2
            ;;
        --year-to)
            YEAR_TO="--year-to $2"
            shift 2
            ;;
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --batch)
            BATCH_FILE="$2"
            shift 2
            ;;
        --help)
            show_help
            ;;
        *)
            # Assume it's the query if not a flag
            if [[ -z "$QUERY" ]]; then
                QUERY="$1"
            else
                QUERY="$QUERY $1"
            fi
            shift
            ;;
    esac
done

# Create output directory
OUTPUT_DIR="$PROJECT_ROOT/projects/$PROJ_ID"
mkdir -p "$OUTPUT_DIR"

# Function to run a single search
run_search() {
    local query="$1"
    local timestamp=$(date +%Y%m%d_%H%M%S)

    # Create safe filename from query
    local safe_query=$(echo "$query" | tr ' ' '_' | tr -cd '[:alnum:]_' | cut -c1-50)
    local output_file="$OUTPUT_DIR/${timestamp}_${safe_query}.md"

    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${YELLOW}Query:${NC} $query"
    echo -e "${YELLOW}Project:${NC} $PROJ_ID"
    echo -e "${YELLOW}Output:${NC} $output_file"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

    # Build search command
    local search_cmd="python3 $SCRIPT_DIR/search_export.py \"$query\" --full-text --format markdown --limit $LIMIT --output \"$output_file\""

    # Add optional filters
    [[ -n "$SEMANTIC" ]] && search_cmd="$search_cmd $SEMANTIC"
    [[ -n "$CATEGORY" ]] && search_cmd="$search_cmd $CATEGORY"
    [[ -n "$YEAR_FROM" ]] && search_cmd="$search_cmd $YEAR_FROM"
    [[ -n "$YEAR_TO" ]] && search_cmd="$search_cmd $YEAR_TO"

    # Run search
    echo -e "${YELLOW}Running search...${NC}"
    eval $search_cmd

    if [[ -f "$output_file" ]]; then
        # Count results (rough estimate from file)
        local result_count=$(grep -c "^## " "$output_file" 2>/dev/null || echo "0")

        # Log to project tracker
        echo -e "${YELLOW}Logging to project tracker...${NC}"
        python3 "$SCRIPT_DIR/research_project.py" add-query "$PROJ_ID" "$query" \
            --notes "Results: ${result_count} chunks. Saved to: $(basename "$output_file")" 2>/dev/null

        echo -e "${GREEN}✓ Search complete!${NC}"
        echo -e "  Results saved to: ${BLUE}$output_file${NC}"
        echo -e "  Logged to project: ${BLUE}$PROJ_ID${NC}"
        echo ""
    else
        echo -e "${RED}✗ Search produced no results${NC}"
        python3 "$SCRIPT_DIR/research_project.py" add-query "$PROJ_ID" "$query" \
            --notes "No results found" 2>/dev/null
    fi
}

# Main execution
cd "$PROJECT_ROOT"

if [[ -n "$BATCH_FILE" ]]; then
    # Batch mode: process queries from file
    if [[ ! -f "$BATCH_FILE" ]]; then
        echo -e "${RED}Error: Batch file not found: $BATCH_FILE${NC}"
        exit 1
    fi

    echo -e "${GREEN}Batch processing queries from: $BATCH_FILE${NC}"
    echo ""

    while IFS= read -r line || [[ -n "$line" ]]; do
        # Skip empty lines and comments
        [[ -z "$line" ]] && continue
        [[ "$line" =~ ^# ]] && continue

        run_search "$line"
    done < "$BATCH_FILE"

    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}Batch processing complete!${NC}"
    echo -e "All results saved to: ${BLUE}$OUTPUT_DIR/${NC}"

else
    # Single query mode
    if [[ -z "$QUERY" ]]; then
        echo -e "${RED}Error: No query provided${NC}"
        echo "Usage: ./research_session.sh PROJECT_ID \"Search Query\" [options]"
        exit 1
    fi

    run_search "$QUERY"
fi

echo ""
echo -e "${YELLOW}View project status:${NC}"
echo "  python3 pipeline/research_project.py show $PROJ_ID"
echo ""
echo -e "${YELLOW}Export project report:${NC}"
echo "  python3 pipeline/research_project.py export $PROJ_ID --output projects/$PROJ_ID/report.md"
