#!/usr/bin/env python3
"""
Concept Merge Utility - Clean up duplicate and similar concepts.

This tool helps maintain data hygiene by identifying and merging similar concepts
that may have been created with slight variations (e.g., "Rudolf Steiner" vs
"Steiner, Rudolf" vs "Steiner").

Features:
- Fuzzy matching to find similar concept names
- Interactive merge mode with confirmation
- Batch merge from a mapping file
- Dry-run mode to preview changes
- Audit trail of all merges

Usage:
    # Find similar concepts (preview only)
    python merge_concepts.py --find-similar --threshold 80

    # Interactive merge mode
    python merge_concepts.py --interactive

    # Merge specific concepts into a primary
    python merge_concepts.py --merge "Steiner, Rudolf" "R. Steiner" --into "Rudolf Steiner"

    # Batch merge from mapping file
    python merge_concepts.py --from-file merge_mappings.json

    # List all concepts with document/chunk counts
    python merge_concepts.py --list

    # Show concept details
    python merge_concepts.py --show "Rudolf Steiner"
"""

import os
import sys
import argparse
import json
from datetime import datetime
from pathlib import Path
from typing import List, Dict, Tuple, Optional
from collections import defaultdict

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from db_utils import get_db_connection

# Try to import fuzzy matching library
try:
    from rapidfuzz import fuzz, process
    FUZZY_AVAILABLE = True
except ImportError:
    try:
        from fuzzywuzzy import fuzz, process
        FUZZY_AVAILABLE = True
    except ImportError:
        FUZZY_AVAILABLE = False


# =============================================================================
# CONCEPT QUERIES
# =============================================================================

def get_all_concepts() -> List[Dict]:
    """Get all concepts with their document and chunk counts."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            cur.execute("""
                SELECT
                    c.concept_id,
                    c.name,
                    c.category,
                    c.description,
                    c.aliases,
                    COALESCE(dc.doc_count, 0) as document_count,
                    COALESCE(dc.total_doc_mentions, 0) as total_doc_mentions,
                    COALESCE(cc.chunk_count, 0) as chunk_count,
                    COALESCE(cc.total_chunk_mentions, 0) as total_chunk_mentions,
                    c.created_at
                FROM concepts c
                LEFT JOIN (
                    SELECT concept_id,
                           COUNT(DISTINCT document_id) as doc_count,
                           SUM(mention_count) as total_doc_mentions
                    FROM document_concepts
                    GROUP BY concept_id
                ) dc ON c.concept_id = dc.concept_id
                LEFT JOIN (
                    SELECT concept_id,
                           COUNT(DISTINCT chunk_id) as chunk_count,
                           SUM(mention_count) as total_chunk_mentions
                    FROM chunk_concepts
                    GROUP BY concept_id
                ) cc ON c.concept_id = cc.concept_id
                ORDER BY c.name
            """)

            columns = ['concept_id', 'name', 'category', 'description', 'aliases',
                       'document_count', 'total_doc_mentions', 'chunk_count',
                       'total_chunk_mentions', 'created_at']
            return [dict(zip(columns, row)) for row in cur.fetchall()]


def get_concept_by_name(name: str) -> Optional[Dict]:
    """Get a concept by exact name match."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            cur.execute("""
                SELECT concept_id, name, category, description, aliases, created_at
                FROM concepts WHERE name = %s
            """, (name,))
            row = cur.fetchone()
            if row:
                return dict(zip(['concept_id', 'name', 'category', 'description',
                                'aliases', 'created_at'], row))
            return None


def get_concept_by_id(concept_id: int) -> Optional[Dict]:
    """Get a concept by ID."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            cur.execute("""
                SELECT concept_id, name, category, description, aliases, created_at
                FROM concepts WHERE concept_id = %s
            """, (concept_id,))
            row = cur.fetchone()
            if row:
                return dict(zip(['concept_id', 'name', 'category', 'description',
                                'aliases', 'created_at'], row))
            return None


def get_concept_documents(concept_id: int) -> List[Dict]:
    """Get all documents linked to a concept."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            cur.execute("""
                SELECT d.document_id, d.title, dc.mention_count
                FROM document_concepts dc
                JOIN documents d ON dc.document_id = d.document_id
                WHERE dc.concept_id = %s
                ORDER BY dc.mention_count DESC
            """, (concept_id,))
            return [{'document_id': r[0], 'title': r[1], 'mention_count': r[2]}
                    for r in cur.fetchall()]


# =============================================================================
# FUZZY MATCHING
# =============================================================================

def find_similar_concepts(threshold: int = 80) -> List[Tuple[Dict, Dict, int]]:
    """Find pairs of similar concepts using fuzzy matching.

    Args:
        threshold: Minimum similarity score (0-100)

    Returns:
        List of (concept1, concept2, similarity_score) tuples
    """
    if not FUZZY_AVAILABLE:
        print("ERROR: Fuzzy matching requires 'rapidfuzz' or 'fuzzywuzzy' package.")
        print("Install with: pip install rapidfuzz")
        sys.exit(1)

    concepts = get_all_concepts()
    if len(concepts) < 2:
        return []

    similar_pairs = []
    seen_pairs = set()

    names = [c['name'] for c in concepts]
    name_to_concept = {c['name']: c for c in concepts}

    for concept in concepts:
        # Find similar names using fuzzy matching
        matches = process.extract(concept['name'], names, scorer=fuzz.ratio, limit=10)

        for match_name, score, _ in matches:
            if match_name == concept['name']:
                continue
            if score < threshold:
                continue

            # Create sorted pair key to avoid duplicates
            pair_key = tuple(sorted([concept['name'], match_name]))
            if pair_key in seen_pairs:
                continue
            seen_pairs.add(pair_key)

            match_concept = name_to_concept[match_name]
            similar_pairs.append((concept, match_concept, score))

    # Sort by similarity score (highest first)
    similar_pairs.sort(key=lambda x: x[2], reverse=True)
    return similar_pairs


def suggest_primary_concept(concepts: List[Dict]) -> Dict:
    """Suggest which concept should be the primary (merge target).

    Criteria:
    1. Most documents/chunks linked
    2. Has description
    3. Has aliases
    4. Older (created first)
    """
    def score_concept(c):
        score = 0
        score += c.get('document_count', 0) * 10
        score += c.get('chunk_count', 0) * 5
        if c.get('description'):
            score += 20
        if c.get('aliases'):
            score += 10
        return score

    return max(concepts, key=score_concept)


# =============================================================================
# MERGE OPERATIONS
# =============================================================================

def merge_concepts(primary_id: int, secondary_ids: List[int],
                   dry_run: bool = False, create_rules: bool = True) -> Dict:
    """Merge secondary concepts into the primary concept.

    This operation:
    1. Re-links all document_concepts from secondary to primary
    2. Re-links all chunk_concepts from secondary to primary
    3. Adds secondary names to primary's aliases
    4. Deletes the secondary concepts
    5. Logs the merge operation
    6. Creates knowledge rules for future automatic resolution (if enabled)

    Args:
        primary_id: The concept ID to merge INTO
        secondary_ids: List of concept IDs to merge FROM (will be deleted)
        dry_run: If True, only preview changes without executing
        create_rules: If True, create alias rules from merges

    Returns:
        Dict with merge statistics
    """
    primary = get_concept_by_id(primary_id)
    if not primary:
        return {'status': 'error', 'error': f'Primary concept {primary_id} not found'}

    secondaries = []
    for sid in secondary_ids:
        s = get_concept_by_id(sid)
        if not s:
            return {'status': 'error', 'error': f'Secondary concept {sid} not found'}
        if sid == primary_id:
            return {'status': 'error', 'error': 'Cannot merge concept into itself'}
        secondaries.append(s)

    result = {
        'status': 'success',
        'primary': primary['name'],
        'merged': [s['name'] for s in secondaries],
        'documents_relinked': 0,
        'chunks_relinked': 0,
        'aliases_added': [],
        'rules_created': [],
        'dry_run': dry_run
    }

    if dry_run:
        print(f"\n[DRY RUN] Would merge into: {primary['name']} (ID: {primary_id})")
        for s in secondaries:
            docs = get_concept_documents(s['concept_id'])
            print(f"  - {s['name']} (ID: {s['concept_id']}): {len(docs)} documents")
            result['documents_relinked'] += len(docs)
        return result

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            for secondary in secondaries:
                sid = secondary['concept_id']

                # Re-link document_concepts
                # Handle conflicts by summing mention counts
                cur.execute("""
                    INSERT INTO document_concepts (document_id, concept_id, mention_count)
                    SELECT document_id, %s, mention_count
                    FROM document_concepts
                    WHERE concept_id = %s
                    ON CONFLICT (document_id, concept_id)
                    DO UPDATE SET mention_count = document_concepts.mention_count + EXCLUDED.mention_count
                """, (primary_id, sid))
                docs_moved = cur.rowcount
                result['documents_relinked'] += docs_moved

                # Delete old document_concepts links
                cur.execute("DELETE FROM document_concepts WHERE concept_id = %s", (sid,))

                # Re-link chunk_concepts
                cur.execute("""
                    INSERT INTO chunk_concepts (chunk_id, concept_id, mention_count)
                    SELECT chunk_id, %s, mention_count
                    FROM chunk_concepts
                    WHERE concept_id = %s
                    ON CONFLICT (chunk_id, concept_id)
                    DO UPDATE SET chunk_concepts.mention_count = chunk_concepts.mention_count + EXCLUDED.mention_count
                """, (primary_id, sid))
                chunks_moved = cur.rowcount
                result['chunks_relinked'] += chunks_moved

                # Delete old chunk_concepts links
                cur.execute("DELETE FROM chunk_concepts WHERE concept_id = %s", (sid,))

                # Add secondary name to primary's aliases
                cur.execute("""
                    UPDATE concepts
                    SET aliases = array_append(COALESCE(aliases, ARRAY[]::text[]), %s)
                    WHERE concept_id = %s
                      AND NOT (%s = ANY(COALESCE(aliases, ARRAY[]::text[])))
                """, (secondary['name'], primary_id, secondary['name']))
                if cur.rowcount > 0:
                    result['aliases_added'].append(secondary['name'])

                # Also add any aliases from secondary to primary
                if secondary.get('aliases'):
                    for alias in secondary['aliases']:
                        cur.execute("""
                            UPDATE concepts
                            SET aliases = array_append(aliases, %s)
                            WHERE concept_id = %s
                              AND NOT (%s = ANY(COALESCE(aliases, ARRAY[]::text[])))
                        """, (alias, primary_id, alias))
                        if cur.rowcount > 0:
                            result['aliases_added'].append(alias)

                # Delete the secondary concept
                cur.execute("DELETE FROM concepts WHERE concept_id = %s", (sid,))

                # Log the merge
                cur.execute("""
                    INSERT INTO change_log (document_id, action, old_value, new_value, changed_by, changed_at)
                    VALUES (NULL, 'concept_merge', %s, %s, 'merge_concepts.py', NOW())
                """, (
                    json.dumps({'merged_concept': secondary['name'], 'concept_id': sid}),
                    json.dumps({'into_concept': primary['name'], 'concept_id': primary_id})
                ))

            conn.commit()

    # Create knowledge rules for future automatic resolution
    if create_rules and not dry_run:
        try:
            from knowledge_rules import create_alias_rule
            for secondary in secondaries:
                rule_id = create_alias_rule(
                    alias=secondary['name'],
                    canonical=primary['name'],
                    created_by='merge_concepts',
                    description=f"Auto-created from merge operation"
                )
                if rule_id:
                    result['rules_created'].append({
                        'rule_id': rule_id,
                        'alias': secondary['name'],
                        'canonical': primary['name']
                    })

                # Also create rules for any aliases the secondary had
                if secondary.get('aliases'):
                    for alias in secondary['aliases']:
                        rule_id = create_alias_rule(
                            alias=alias,
                            canonical=primary['name'],
                            created_by='merge_concepts',
                            description=f"Auto-created from merge (via {secondary['name']})"
                        )
                        if rule_id:
                            result['rules_created'].append({
                                'rule_id': rule_id,
                                'alias': alias,
                                'canonical': primary['name']
                            })
        except ImportError:
            pass  # knowledge_rules module not available
        except Exception as e:
            # Log but don't fail the merge
            print(f"Warning: Could not create knowledge rules: {e}")

    return result


# =============================================================================
# CLI COMMANDS
# =============================================================================

def cmd_list(args):
    """List all concepts with counts."""
    concepts = get_all_concepts()

    if not concepts:
        print("No concepts found.")
        return

    print(f"\n{'='*80}")
    print(f"Concepts ({len(concepts)} total)")
    print(f"{'='*80}\n")

    # Sort options
    if args.sort == 'name':
        concepts.sort(key=lambda c: c['name'].lower())
    elif args.sort == 'docs':
        concepts.sort(key=lambda c: c['document_count'], reverse=True)
    elif args.sort == 'mentions':
        concepts.sort(key=lambda c: c['total_doc_mentions'] + c['total_chunk_mentions'], reverse=True)

    # Filter by category
    if args.category:
        concepts = [c for c in concepts if c.get('category') == args.category]

    # Print table
    print(f"{'ID':>5} {'Name':<40} {'Cat':<12} {'Docs':>5} {'Chunks':>6} {'Mentions':>8}")
    print("-" * 80)

    for c in concepts:
        total_mentions = (c['total_doc_mentions'] or 0) + (c['total_chunk_mentions'] or 0)
        category = (c['category'] or '-')[:12]
        name = c['name'][:40]
        print(f"{c['concept_id']:>5} {name:<40} {category:<12} {c['document_count']:>5} {c['chunk_count']:>6} {total_mentions:>8}")

    print(f"\nTotal: {len(concepts)} concepts")


def cmd_show(args):
    """Show details for a specific concept."""
    concept = get_concept_by_name(args.name)
    if not concept:
        # Try by ID
        try:
            concept = get_concept_by_id(int(args.name))
        except ValueError:
            pass

    if not concept:
        print(f"Concept not found: {args.name}")
        sys.exit(1)

    print(f"\n{'='*60}")
    print(f"Concept: {concept['name']}")
    print(f"{'='*60}")
    print(f"ID:          {concept['concept_id']}")
    print(f"Category:    {concept.get('category') or '-'}")
    print(f"Description: {concept.get('description') or '-'}")
    print(f"Aliases:     {', '.join(concept.get('aliases') or []) or '-'}")
    print(f"Created:     {concept['created_at']}")

    # Show linked documents
    docs = get_concept_documents(concept['concept_id'])
    print(f"\nLinked Documents ({len(docs)}):")
    for d in docs[:20]:
        print(f"  - [{d['mention_count']} mentions] {d['title'][:60]}")
    if len(docs) > 20:
        print(f"  ... and {len(docs) - 20} more")


def cmd_find_similar(args):
    """Find similar concepts using fuzzy matching."""
    print(f"\nFinding similar concepts (threshold: {args.threshold}%)...")

    similar = find_similar_concepts(threshold=args.threshold)

    if not similar:
        print("No similar concepts found above threshold.")
        return

    print(f"\nFound {len(similar)} similar pairs:\n")
    print(f"{'Score':>5} {'Concept 1':<35} {'Concept 2':<35}")
    print("-" * 80)

    for c1, c2, score in similar:
        # Mark which has more usage
        c1_usage = c1['document_count'] + c1['chunk_count']
        c2_usage = c2['document_count'] + c2['chunk_count']

        c1_name = c1['name'][:33]
        c2_name = c2['name'][:33]

        if c1_usage > c2_usage:
            c1_name = f"*{c1_name}"
        elif c2_usage > c1_usage:
            c2_name = f"*{c2_name}"

        print(f"{score:>5}% {c1_name:<35} {c2_name:<35}")

    print("\n* = suggested primary (more document/chunk links)")
    print(f"\nUse --interactive to merge these, or --merge to merge specific concepts.")


def cmd_interactive(args):
    """Interactive mode to review and merge similar concepts."""
    print(f"\nFinding similar concepts (threshold: {args.threshold}%)...")

    similar = find_similar_concepts(threshold=args.threshold)

    if not similar:
        print("No similar concepts found above threshold.")
        return

    print(f"\nFound {len(similar)} similar pairs to review.\n")

    merged_count = 0
    skipped_count = 0

    for i, (c1, c2, score) in enumerate(similar):
        print(f"\n{'='*60}")
        print(f"Pair {i+1}/{len(similar)} (Similarity: {score}%)")
        print(f"{'='*60}")

        # Determine suggested primary
        suggested = suggest_primary_concept([c1, c2])
        other = c2 if suggested == c1 else c1

        print(f"\n1. {c1['name']}")
        print(f"   Docs: {c1['document_count']}, Chunks: {c1['chunk_count']}, Category: {c1.get('category') or '-'}")

        print(f"\n2. {c2['name']}")
        print(f"   Docs: {c2['document_count']}, Chunks: {c2['chunk_count']}, Category: {c2.get('category') or '-'}")

        print(f"\nSuggested: Keep '{suggested['name']}' as primary")

        while True:
            choice = input("\n[m]erge (use suggested), [1] keep first, [2] keep second, [s]kip, [q]uit: ").lower().strip()

            if choice == 'q':
                print(f"\nExiting. Merged: {merged_count}, Skipped: {skipped_count}")
                return
            elif choice == 's':
                skipped_count += 1
                break
            elif choice == 'm':
                result = merge_concepts(suggested['concept_id'], [other['concept_id']])
                if result['status'] == 'success':
                    print(f"  Merged '{other['name']}' into '{suggested['name']}'")
                    merged_count += 1
                else:
                    print(f"  Error: {result.get('error')}")
                break
            elif choice == '1':
                result = merge_concepts(c1['concept_id'], [c2['concept_id']])
                if result['status'] == 'success':
                    print(f"  Merged '{c2['name']}' into '{c1['name']}'")
                    merged_count += 1
                else:
                    print(f"  Error: {result.get('error')}")
                break
            elif choice == '2':
                result = merge_concepts(c2['concept_id'], [c1['concept_id']])
                if result['status'] == 'success':
                    print(f"  Merged '{c1['name']}' into '{c2['name']}'")
                    merged_count += 1
                else:
                    print(f"  Error: {result.get('error')}")
                break
            else:
                print("Invalid choice. Try again.")

    print(f"\n{'='*60}")
    print(f"Complete! Merged: {merged_count}, Skipped: {skipped_count}")
    print(f"{'='*60}")


def cmd_merge(args):
    """Merge specific concepts by name."""
    # Get primary concept
    primary = get_concept_by_name(args.into)
    if not primary:
        print(f"Primary concept not found: {args.into}")
        sys.exit(1)

    # Get secondary concepts
    secondary_ids = []
    for name in args.merge:
        concept = get_concept_by_name(name)
        if not concept:
            print(f"Concept not found: {name}")
            sys.exit(1)
        secondary_ids.append(concept['concept_id'])

    print(f"\nMerge Plan:")
    print(f"  Primary (keep): {primary['name']} (ID: {primary['concept_id']})")
    print(f"  Secondary (delete): {', '.join(args.merge)}")

    if args.dry_run:
        result = merge_concepts(primary['concept_id'], secondary_ids, dry_run=True)
        print(f"\n[DRY RUN] Would relink {result['documents_relinked']} documents")
        return

    if not args.yes:
        confirm = input("\nProceed with merge? [y/N]: ").lower().strip()
        if confirm != 'y':
            print("Aborted.")
            return

    result = merge_concepts(primary['concept_id'], secondary_ids,
                            create_rules=not getattr(args, 'no_rules', False))

    if result['status'] == 'success':
        print(f"\nMerge complete!")
        print(f"  Documents relinked: {result['documents_relinked']}")
        print(f"  Chunks relinked: {result['chunks_relinked']}")
        print(f"  Aliases added: {', '.join(result['aliases_added']) or 'none'}")
        if result.get('rules_created'):
            print(f"  Knowledge rules created: {len(result['rules_created'])}")
            for rule in result['rules_created']:
                print(f"    - Rule #{rule['rule_id']}: {rule['alias']} → {rule['canonical']}")
    else:
        print(f"\nError: {result.get('error')}")
        sys.exit(1)


def cmd_from_file(args):
    """Batch merge from a mapping file."""
    if not args.file.exists():
        print(f"File not found: {args.file}")
        sys.exit(1)

    with open(args.file) as f:
        mappings = json.load(f)

    """
    Expected format:
    {
        "merges": [
            {"primary": "Rudolf Steiner", "merge": ["Steiner, Rudolf", "R. Steiner"]},
            {"primary": "Consciousness", "merge": ["consciousness", "Consciouness"]}
        ]
    }
    """

    if 'merges' not in mappings:
        print("Invalid file format. Expected {'merges': [...]}")
        sys.exit(1)

    total_merged = 0
    errors = []

    for mapping in mappings['merges']:
        primary_name = mapping['primary']
        secondary_names = mapping['merge']

        primary = get_concept_by_name(primary_name)
        if not primary:
            errors.append(f"Primary not found: {primary_name}")
            continue

        secondary_ids = []
        for name in secondary_names:
            concept = get_concept_by_name(name)
            if concept:
                secondary_ids.append(concept['concept_id'])
            else:
                errors.append(f"Secondary not found: {name}")

        if not secondary_ids:
            continue

        if args.dry_run:
            print(f"[DRY RUN] Would merge {secondary_names} into {primary_name}")
            continue

        result = merge_concepts(primary['concept_id'], secondary_ids)
        if result['status'] == 'success':
            print(f"Merged {result['merged']} into {result['primary']}")
            total_merged += len(result['merged'])
        else:
            errors.append(f"Merge failed: {result.get('error')}")

    print(f"\nComplete. Merged: {total_merged} concepts")
    if errors:
        print(f"\nErrors ({len(errors)}):")
        for e in errors:
            print(f"  - {e}")


# =============================================================================
# MAIN
# =============================================================================

def main():
    parser = argparse.ArgumentParser(
        description='Concept merge utility for data hygiene',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s --list                              # List all concepts
  %(prog)s --list --sort docs                  # Sort by document count
  %(prog)s --show "Rudolf Steiner"             # Show concept details
  %(prog)s --find-similar                      # Find similar concept pairs
  %(prog)s --find-similar --threshold 90       # Higher similarity threshold
  %(prog)s --interactive                       # Interactive merge mode
  %(prog)s --merge "Steiner, R" --into "Rudolf Steiner"
  %(prog)s --from-file mappings.json           # Batch merge from file

Merge File Format (JSON):
  {
    "merges": [
      {"primary": "Rudolf Steiner", "merge": ["Steiner, Rudolf", "R. Steiner"]},
      {"primary": "Consciousness", "merge": ["consciousness"]}
    ]
  }
        """
    )

    # Commands
    parser.add_argument('--list', '-l', action='store_true',
                        help='List all concepts with document/chunk counts')
    parser.add_argument('--show', '-s', metavar='NAME',
                        help='Show details for a specific concept')
    parser.add_argument('--find-similar', '-f', action='store_true',
                        help='Find similar concepts using fuzzy matching')
    parser.add_argument('--interactive', '-i', action='store_true',
                        help='Interactive mode to review and merge similar concepts')
    parser.add_argument('--merge', '-m', nargs='+', metavar='NAME',
                        help='Concept name(s) to merge (will be deleted)')
    parser.add_argument('--into', metavar='NAME',
                        help='Primary concept name to merge into (required with --merge)')
    parser.add_argument('--from-file', type=Path, metavar='FILE',
                        help='Batch merge from JSON mapping file')

    # Options
    parser.add_argument('--threshold', '-t', type=int, default=80,
                        help='Similarity threshold for fuzzy matching (default: 80)')
    parser.add_argument('--sort', choices=['name', 'docs', 'mentions'], default='name',
                        help='Sort order for --list (default: name)')
    parser.add_argument('--category', '-c',
                        help='Filter by category (for --list)')
    parser.add_argument('--dry-run', '-n', action='store_true',
                        help='Preview changes without executing')
    parser.add_argument('--yes', '-y', action='store_true',
                        help='Skip confirmation prompts')
    parser.add_argument('--no-rules', action='store_true',
                        help='Do not create knowledge rules from merges')

    args = parser.parse_args()

    # Dispatch to command
    if args.list:
        cmd_list(args)
    elif args.show:
        cmd_show(args)
    elif args.find_similar:
        cmd_find_similar(args)
    elif args.interactive:
        cmd_interactive(args)
    elif args.merge:
        if not args.into:
            print("ERROR: --into is required with --merge")
            sys.exit(1)
        cmd_merge(args)
    elif args.from_file:
        cmd_from_file(args)
    else:
        parser.print_help()
        print("\nError: No command specified. Use --list, --find-similar, --interactive, --merge, or --from-file")
        sys.exit(1)


if __name__ == '__main__':
    main()
