#!/usr/bin/env python3
"""
Knowledge Rules Engine

Implements "Living Glossaries" - the system learns from user corrections
and applies rules automatically to future ingestions.

Features:
- Concept alias rules (user merges become permanent)
- Auto-tagging rules (pattern-based suggestions)
- Domain extraction hints (improve concept detection)
- Rule import/export for sharing domain knowledge

Usage:
    # List active rules
    python knowledge_rules.py --list

    # Create alias rule manually
    python knowledge_rules.py --create-alias "R. Steiner" "Rudolf Steiner"

    # Import rules from file
    python knowledge_rules.py --import rules.json

    # Export rules for sharing
    python knowledge_rules.py --export my_rules.json

    # Apply rules to existing library (re-normalize)
    python knowledge_rules.py --apply-to-library --dry-run

    # Show statistics
    python knowledge_rules.py --stats
"""

import argparse
import json
import logging
import sys
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any, Tuple
from dataclasses import dataclass, field

# Setup path
sys.path.insert(0, str(Path(__file__).parent))

from db_utils import get_db_connection

# Try to import LOG_LEVEL from config, fallback to INFO
try:
    from config import LOG_LEVEL
except ImportError:
    LOG_LEVEL = 'INFO'

# Setup logging
logging.basicConfig(
    level=getattr(logging, LOG_LEVEL, logging.INFO),
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


# =============================================================================
# DATA CLASSES
# =============================================================================

@dataclass
class KnowledgeRule:
    """Represents a knowledge rule."""
    rule_id: Optional[int] = None
    rule_type: str = ''
    rule_definition: Dict[str, Any] = field(default_factory=dict)
    rule_name: Optional[str] = None
    description: Optional[str] = None
    source: str = 'user'
    confidence: float = 1.0
    applied_count: int = 0
    is_active: bool = True
    priority: int = 100
    created_at: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        return {
            'rule_id': self.rule_id,
            'rule_type': self.rule_type,
            'rule_definition': self.rule_definition,
            'rule_name': self.rule_name,
            'description': self.description,
            'source': self.source,
            'confidence': self.confidence,
            'applied_count': self.applied_count,
            'is_active': self.is_active,
            'priority': self.priority,
        }

    @classmethod
    def from_row(cls, row: tuple, columns: List[str]) -> 'KnowledgeRule':
        """Create from database row."""
        data = dict(zip(columns, row))
        return cls(
            rule_id=data.get('rule_id'),
            rule_type=data.get('rule_type', ''),
            rule_definition=data.get('rule_definition', {}),
            rule_name=data.get('rule_name'),
            description=data.get('description'),
            source=data.get('source', 'user'),
            confidence=float(data.get('confidence', 1.0)),
            applied_count=data.get('applied_count', 0),
            is_active=data.get('is_active', True),
            priority=data.get('priority', 100),
            created_at=str(data.get('created_at')) if data.get('created_at') else None,
        )


# =============================================================================
# KNOWLEDGE RULE ENGINE
# =============================================================================

class KnowledgeRuleEngine:
    """
    Manages knowledge rules and applies them during processing.

    The engine caches rules in memory and refreshes periodically or on demand.
    """

    def __init__(self, auto_load: bool = True):
        """
        Initialize the rule engine.

        Args:
            auto_load: Whether to load rules from database on init
        """
        self.rules: Dict[str, List[KnowledgeRule]] = {}
        self._last_refresh = None

        if auto_load:
            self.refresh_rules()

    def refresh_rules(self) -> None:
        """Load all active rules from database."""
        try:
            conn = get_db_connection()
            cur = conn.cursor()

            cur.execute("""
                SELECT rule_id, rule_type, rule_definition, rule_name,
                       description, source, confidence, applied_count,
                       is_active, priority, created_at
                FROM knowledge_rules
                WHERE is_active = TRUE
                ORDER BY priority DESC, confidence DESC
            """)

            columns = [desc[0] for desc in cur.description]
            rows = cur.fetchall()
            conn.close()

            # Group by type
            self.rules = {}
            for row in rows:
                rule = KnowledgeRule.from_row(row, columns)
                if rule.rule_type not in self.rules:
                    self.rules[rule.rule_type] = []
                self.rules[rule.rule_type].append(rule)

            self._last_refresh = datetime.now()
            logger.debug(f"Loaded {sum(len(r) for r in self.rules.values())} rules")

        except Exception as e:
            logger.warning(f"Could not load rules from database: {e}")
            self.rules = {}

    def get_canonical_name(self, name: str) -> Tuple[str, Optional[int]]:
        """
        Apply alias rules to get canonical concept name.

        Args:
            name: Original concept name

        Returns:
            Tuple of (canonical_name, rule_id_used or None)
        """
        alias_rules = self.rules.get('concept_alias', [])

        for rule in alias_rules:
            alias = rule.rule_definition.get('alias', '')
            if alias.lower() == name.lower():
                canonical = rule.rule_definition.get('canonical', name)
                return canonical, rule.rule_id

        return name, None

    def get_domain_hints(self, text: str = None, domain: str = None) -> List[str]:
        """
        Get domain-specific terms to look for during extraction.

        Args:
            text: Text to analyze (optional, for domain detection)
            domain: Specific domain to get hints for

        Returns:
            List of terms to look for
        """
        hints = []
        hint_rules = self.rules.get('extraction_hint', [])

        for rule in hint_rules:
            rule_domain = rule.rule_definition.get('domain')
            terms = rule.rule_definition.get('terms', [])

            # If domain specified, filter to that domain
            if domain and rule_domain != domain:
                continue

            # If text provided, check if domain terms appear
            if text:
                if any(term.lower() in text.lower() for term in terms[:3]):
                    hints.extend(terms)
            else:
                hints.extend(terms)

        return list(set(hints))

    def get_auto_tags(self, text: str) -> List[Tuple[str, float]]:
        """
        Get tag suggestions based on text patterns.

        Args:
            text: Text to analyze

        Returns:
            List of (tag, confidence) tuples
        """
        import fnmatch

        suggestions = []
        tag_rules = self.rules.get('auto_tag', [])

        for rule in tag_rules:
            pattern = rule.rule_definition.get('pattern', '')
            tags = rule.rule_definition.get('tags', [])

            # Convert glob pattern to search
            search_pattern = pattern.replace('*', '')
            if search_pattern.lower() in text.lower():
                for tag in tags:
                    suggestions.append((tag, rule.confidence))

        # Deduplicate, keeping highest confidence
        tag_map = {}
        for tag, conf in suggestions:
            if tag not in tag_map or conf > tag_map[tag]:
                tag_map[tag] = conf

        return [(t, c) for t, c in tag_map.items()]

    def should_ignore(self, term: str) -> bool:
        """
        Check if a term should be ignored during extraction.

        Args:
            term: Term to check

        Returns:
            True if term should be ignored
        """
        ignore_rules = self.rules.get('ignore_term', [])

        for rule in ignore_rules:
            ignore_term = rule.rule_definition.get('term', '')
            if ignore_term.lower() == term.lower():
                return True

        return False

    def record_application(self, rule_id: int, input_val: str, output_val: str,
                          document_id: str = None, chunk_id: str = None,
                          concept_id: int = None) -> None:
        """
        Record that a rule was applied (for tracking effectiveness).

        Args:
            rule_id: ID of the applied rule
            input_val: Original value
            output_val: Value after rule application
            document_id: Document context (optional)
            chunk_id: Chunk context (optional)
            concept_id: Concept context (optional)
        """
        try:
            conn = get_db_connection()
            cur = conn.cursor()

            cur.execute("""
                INSERT INTO rule_applications
                (rule_id, input_value, output_value, document_id, chunk_id, concept_id)
                VALUES (%s, %s, %s, %s, %s, %s)
            """, (rule_id, input_val[:500], output_val[:500], document_id, chunk_id, concept_id))

            # Update rule stats
            cur.execute("""
                UPDATE knowledge_rules
                SET applied_count = applied_count + 1,
                    last_applied_at = CURRENT_TIMESTAMP
                WHERE rule_id = %s
            """, (rule_id,))

            conn.commit()
            conn.close()

        except Exception as e:
            logger.warning(f"Could not record rule application: {e}")


# =============================================================================
# RULE MANAGEMENT FUNCTIONS
# =============================================================================

def create_alias_rule(alias: str, canonical: str, created_by: str = 'user',
                     description: str = None) -> Optional[int]:
    """
    Create a concept alias rule.

    Args:
        alias: The variant name
        canonical: The canonical name to map to
        created_by: Who created this rule
        description: Optional description

    Returns:
        Rule ID if created, None otherwise
    """
    conn = get_db_connection()
    cur = conn.cursor()

    rule_name = f"{alias} → {canonical}"
    desc = description or f"Maps '{alias}' to canonical name '{canonical}'"

    try:
        cur.execute("""
            INSERT INTO knowledge_rules
            (rule_type, rule_definition, rule_name, description, source, created_by)
            VALUES ('concept_alias', %s, %s, %s, 'user', %s)
            ON CONFLICT DO NOTHING
            RETURNING rule_id
        """, (
            json.dumps({'alias': alias, 'canonical': canonical}),
            rule_name,
            desc,
            created_by
        ))

        result = cur.fetchone()
        conn.commit()
        conn.close()

        if result:
            logger.info(f"Created alias rule: {rule_name}")
            return result[0]
        else:
            logger.info(f"Alias rule already exists: {rule_name}")
            return None

    except Exception as e:
        logger.error(f"Failed to create alias rule: {e}")
        conn.rollback()
        conn.close()
        return None


def create_auto_tag_rule(pattern: str, tags: List[str], description: str = None) -> Optional[int]:
    """
    Create an auto-tagging rule.

    Args:
        pattern: Pattern to match (e.g., "*anthroposoph*")
        tags: Tags to suggest when pattern matches

    Returns:
        Rule ID if created
    """
    conn = get_db_connection()
    cur = conn.cursor()

    rule_name = f"Auto-tag: {pattern} → {', '.join(tags)}"

    try:
        cur.execute("""
            INSERT INTO knowledge_rules
            (rule_type, rule_definition, rule_name, description, source)
            VALUES ('auto_tag', %s, %s, %s, 'user')
            RETURNING rule_id
        """, (
            json.dumps({'pattern': pattern, 'tags': tags}),
            rule_name,
            description
        ))

        result = cur.fetchone()
        conn.commit()
        conn.close()

        if result:
            logger.info(f"Created auto-tag rule: {rule_name}")
            return result[0]
        return None

    except Exception as e:
        logger.error(f"Failed to create auto-tag rule: {e}")
        conn.rollback()
        conn.close()
        return None


def create_extraction_hint_rule(domain: str, terms: List[str],
                                description: str = None) -> Optional[int]:
    """
    Create a domain extraction hint rule.

    Args:
        domain: Domain name (e.g., "esoteric", "philosophy")
        terms: Domain-specific terms to look for

    Returns:
        Rule ID if created
    """
    conn = get_db_connection()
    cur = conn.cursor()

    rule_name = f"Domain hints: {domain}"

    try:
        cur.execute("""
            INSERT INTO knowledge_rules
            (rule_type, rule_definition, rule_name, description, source)
            VALUES ('extraction_hint', %s, %s, %s, 'user')
            RETURNING rule_id
        """, (
            json.dumps({'domain': domain, 'terms': terms}),
            rule_name,
            description or f"Extraction hints for {domain} domain"
        ))

        result = cur.fetchone()
        conn.commit()
        conn.close()

        if result:
            logger.info(f"Created extraction hint rule for domain: {domain}")
            return result[0]
        return None

    except Exception as e:
        logger.error(f"Failed to create extraction hint rule: {e}")
        conn.rollback()
        conn.close()
        return None


def disable_rule(rule_id: int) -> bool:
    """Disable a rule (soft delete)."""
    conn = get_db_connection()
    cur = conn.cursor()

    try:
        cur.execute("""
            UPDATE knowledge_rules
            SET is_active = FALSE, updated_at = CURRENT_TIMESTAMP
            WHERE rule_id = %s
        """, (rule_id,))

        conn.commit()
        affected = cur.rowcount
        conn.close()

        if affected:
            logger.info(f"Disabled rule {rule_id}")
            return True
        return False

    except Exception as e:
        logger.error(f"Failed to disable rule: {e}")
        conn.rollback()
        conn.close()
        return False


def enable_rule(rule_id: int) -> bool:
    """Re-enable a disabled rule."""
    conn = get_db_connection()
    cur = conn.cursor()

    try:
        cur.execute("""
            UPDATE knowledge_rules
            SET is_active = TRUE, updated_at = CURRENT_TIMESTAMP
            WHERE rule_id = %s
        """, (rule_id,))

        conn.commit()
        affected = cur.rowcount
        conn.close()

        if affected:
            logger.info(f"Enabled rule {rule_id}")
            return True
        return False

    except Exception as e:
        logger.error(f"Failed to enable rule: {e}")
        conn.rollback()
        conn.close()
        return False


# =============================================================================
# IMPORT/EXPORT
# =============================================================================

def export_rules(output_path: Path, rule_types: List[str] = None) -> int:
    """
    Export rules to JSON file.

    Args:
        output_path: Path to output file
        rule_types: Types to export (None = all)

    Returns:
        Number of rules exported
    """
    conn = get_db_connection()
    cur = conn.cursor()

    query = """
        SELECT rule_type, rule_definition, rule_name, description,
               source, confidence, priority
        FROM knowledge_rules
        WHERE is_active = TRUE
    """
    params = []

    if rule_types:
        query += " AND rule_type = ANY(%s)"
        params.append(rule_types)

    cur.execute(query, params)

    rules = []
    for row in cur.fetchall():
        rules.append({
            'rule_type': row[0],
            'rule_definition': row[1],
            'rule_name': row[2],
            'description': row[3],
            'source': row[4],
            'confidence': float(row[5]) if row[5] else 1.0,
            'priority': row[6] or 100,
        })

    conn.close()

    export_data = {
        'version': '1.0',
        'exported_at': datetime.now().isoformat(),
        'rule_count': len(rules),
        'rules': rules,
    }

    with open(output_path, 'w', encoding='utf-8') as f:
        json.dump(export_data, f, indent=2, ensure_ascii=False)

    logger.info(f"Exported {len(rules)} rules to {output_path}")
    return len(rules)


def import_rules(input_path: Path, overwrite: bool = False) -> Tuple[int, int]:
    """
    Import rules from JSON file.

    Args:
        input_path: Path to input file
        overwrite: If True, update existing rules; if False, skip

    Returns:
        Tuple of (imported_count, skipped_count)
    """
    with open(input_path, 'r', encoding='utf-8') as f:
        data = json.load(f)

    rules = data.get('rules', [])
    imported = 0
    skipped = 0

    conn = get_db_connection()
    cur = conn.cursor()

    for rule in rules:
        try:
            if overwrite:
                cur.execute("""
                    INSERT INTO knowledge_rules
                    (rule_type, rule_definition, rule_name, description, source, confidence, priority)
                    VALUES (%s, %s, %s, %s, 'imported', %s, %s)
                    ON CONFLICT DO UPDATE SET
                        rule_definition = EXCLUDED.rule_definition,
                        description = EXCLUDED.description,
                        updated_at = CURRENT_TIMESTAMP
                """, (
                    rule['rule_type'],
                    json.dumps(rule['rule_definition']),
                    rule.get('rule_name'),
                    rule.get('description'),
                    rule.get('confidence', 1.0),
                    rule.get('priority', 100),
                ))
            else:
                cur.execute("""
                    INSERT INTO knowledge_rules
                    (rule_type, rule_definition, rule_name, description, source, confidence, priority)
                    VALUES (%s, %s, %s, %s, 'imported', %s, %s)
                    ON CONFLICT DO NOTHING
                """, (
                    rule['rule_type'],
                    json.dumps(rule['rule_definition']),
                    rule.get('rule_name'),
                    rule.get('description'),
                    rule.get('confidence', 1.0),
                    rule.get('priority', 100),
                ))

            if cur.rowcount > 0:
                imported += 1
            else:
                skipped += 1

        except Exception as e:
            logger.warning(f"Failed to import rule: {e}")
            skipped += 1

    conn.commit()
    conn.close()

    logger.info(f"Imported {imported} rules, skipped {skipped}")
    return imported, skipped


# =============================================================================
# STATISTICS AND LISTING
# =============================================================================

def get_rules_summary() -> Dict[str, Any]:
    """Get summary statistics of knowledge rules."""
    conn = get_db_connection()
    cur = conn.cursor()

    # Get counts by type
    cur.execute("""
        SELECT rule_type, COUNT(*), SUM(applied_count)
        FROM knowledge_rules
        WHERE is_active = TRUE
        GROUP BY rule_type
    """)

    by_type = {}
    total_rules = 0
    total_applications = 0

    for row in cur.fetchall():
        by_type[row[0]] = {
            'count': row[1],
            'applications': row[2] or 0
        }
        total_rules += row[1]
        total_applications += row[2] or 0

    # Get top rules
    cur.execute("""
        SELECT rule_id, rule_type, rule_name, applied_count
        FROM knowledge_rules
        WHERE is_active = TRUE
        ORDER BY applied_count DESC
        LIMIT 10
    """)

    top_rules = [
        {'rule_id': r[0], 'type': r[1], 'name': r[2], 'applications': r[3]}
        for r in cur.fetchall()
    ]

    conn.close()

    return {
        'total_rules': total_rules,
        'total_applications': total_applications,
        'by_type': by_type,
        'top_rules': top_rules,
    }


def list_rules(rule_type: str = None, limit: int = 50) -> List[KnowledgeRule]:
    """
    List knowledge rules.

    Args:
        rule_type: Filter to specific type
        limit: Maximum rules to return
    """
    conn = get_db_connection()
    cur = conn.cursor()

    query = """
        SELECT rule_id, rule_type, rule_definition, rule_name,
               description, source, confidence, applied_count,
               is_active, priority, created_at
        FROM knowledge_rules
        WHERE is_active = TRUE
    """
    params = []

    if rule_type:
        query += " AND rule_type = %s"
        params.append(rule_type)

    query += " ORDER BY applied_count DESC, priority DESC LIMIT %s"
    params.append(limit)

    cur.execute(query, params)

    columns = [desc[0] for desc in cur.description]
    rules = [KnowledgeRule.from_row(row, columns) for row in cur.fetchall()]

    conn.close()
    return rules


# =============================================================================
# CLI INTERFACE
# =============================================================================

def create_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        description='Manage knowledge rules for the research framework'
    )

    # Actions
    parser.add_argument('--list', action='store_true',
                       help='List active rules')
    parser.add_argument('--stats', action='store_true',
                       help='Show rule statistics')
    parser.add_argument('--type', type=str,
                       help='Filter to rule type (concept_alias, auto_tag, extraction_hint)')

    # Create rules
    parser.add_argument('--create-alias', nargs=2, metavar=('ALIAS', 'CANONICAL'),
                       help='Create alias rule: --create-alias "R. Steiner" "Rudolf Steiner"')
    parser.add_argument('--create-tag', nargs='+', metavar='ARG',
                       help='Create auto-tag: --create-tag "pattern" tag1 tag2')
    parser.add_argument('--create-hint', nargs='+', metavar='ARG',
                       help='Create domain hint: --create-hint domain term1 term2')

    # Manage rules
    parser.add_argument('--disable-rule', type=int, metavar='RULE_ID',
                       help='Disable a rule')
    parser.add_argument('--enable-rule', type=int, metavar='RULE_ID',
                       help='Enable a disabled rule')

    # Import/Export
    parser.add_argument('--import', dest='import_file', type=str, metavar='FILE',
                       help='Import rules from JSON file')
    parser.add_argument('--export', type=str, metavar='FILE',
                       help='Export rules to JSON file')
    parser.add_argument('--overwrite', action='store_true',
                       help='Overwrite existing rules during import')

    # Apply to library
    parser.add_argument('--apply-to-library', action='store_true',
                       help='Apply alias rules to existing concepts')
    parser.add_argument('--dry-run', action='store_true',
                       help='Show what would be changed without making changes')

    # Output format
    parser.add_argument('--format', choices=['text', 'json'], default='text',
                       help='Output format')

    return parser


def print_rules_list(rules: List[KnowledgeRule], format: str = 'text'):
    """Print rules in requested format."""
    if format == 'json':
        print(json.dumps([r.to_dict() for r in rules], indent=2))
        return

    print(f"\n{'='*70}")
    print(f"KNOWLEDGE RULES ({len(rules)})")
    print(f"{'='*70}\n")

    # Group by type
    by_type: Dict[str, List[KnowledgeRule]] = {}
    for rule in rules:
        if rule.rule_type not in by_type:
            by_type[rule.rule_type] = []
        by_type[rule.rule_type].append(rule)

    for rule_type, type_rules in by_type.items():
        print(f"\n{rule_type.upper().replace('_', ' ')} ({len(type_rules)}):")
        print("-" * 60)

        for rule in type_rules[:20]:  # Limit display
            apps = f"[{rule.applied_count} uses]" if rule.applied_count else ""
            print(f"  [{rule.rule_id}] {rule.rule_name or 'Unnamed'} {apps}")

            if rule.rule_type == 'concept_alias':
                alias = rule.rule_definition.get('alias', '')
                canonical = rule.rule_definition.get('canonical', '')
                print(f"      {alias} → {canonical}")

            elif rule.rule_type == 'auto_tag':
                pattern = rule.rule_definition.get('pattern', '')
                tags = rule.rule_definition.get('tags', [])
                print(f"      Pattern: {pattern}")
                print(f"      Tags: {', '.join(tags)}")

            elif rule.rule_type == 'extraction_hint':
                domain = rule.rule_definition.get('domain', '')
                terms = rule.rule_definition.get('terms', [])[:5]
                print(f"      Domain: {domain}")
                print(f"      Terms: {', '.join(terms)}...")

        if len(type_rules) > 20:
            print(f"  ... and {len(type_rules) - 20} more")


def print_stats(stats: Dict[str, Any], format: str = 'text'):
    """Print statistics."""
    if format == 'json':
        print(json.dumps(stats, indent=2))
        return

    print(f"\n{'='*70}")
    print("KNOWLEDGE RULES STATISTICS")
    print(f"{'='*70}\n")

    print(f"Total Rules: {stats['total_rules']}")
    print(f"Total Applications: {stats['total_applications']}")
    print()

    print("BY TYPE:")
    for rule_type, data in stats['by_type'].items():
        print(f"  {rule_type}: {data['count']} rules, {data['applications']} applications")

    print()
    print("TOP RULES BY USAGE:")
    for i, rule in enumerate(stats['top_rules'][:10], 1):
        print(f"  {i}. [{rule['type']}] {rule['name']} ({rule['applications']} uses)")


def apply_rules_to_library(dry_run: bool = True) -> Dict[str, int]:
    """
    Apply alias rules to existing concepts in the library.

    Args:
        dry_run: If True, only show what would change

    Returns:
        Dict with counts of changes
    """
    engine = KnowledgeRuleEngine()
    conn = get_db_connection()
    cur = conn.cursor()

    # Get all concepts
    cur.execute("SELECT concept_id, name FROM concepts")
    concepts = cur.fetchall()

    changes = []
    for concept_id, name in concepts:
        canonical, rule_id = engine.get_canonical_name(name)
        if canonical != name:
            changes.append({
                'concept_id': concept_id,
                'old_name': name,
                'new_name': canonical,
                'rule_id': rule_id,
            })

    if dry_run:
        print(f"\n[DRY RUN] Would rename {len(changes)} concepts:")
        for c in changes[:20]:
            print(f"  {c['old_name']} → {c['new_name']}")
        if len(changes) > 20:
            print(f"  ... and {len(changes) - 20} more")
    else:
        for change in changes:
            cur.execute("""
                UPDATE concepts SET name = %s WHERE concept_id = %s
            """, (change['new_name'], change['concept_id']))

            if change['rule_id']:
                engine.record_application(
                    change['rule_id'],
                    change['old_name'],
                    change['new_name'],
                    concept_id=change['concept_id']
                )

        conn.commit()
        print(f"Renamed {len(changes)} concepts")

    conn.close()
    return {'renamed': len(changes)}


def main():
    parser = create_parser()
    args = parser.parse_args()

    # List rules
    if args.list:
        rules = list_rules(rule_type=args.type)
        print_rules_list(rules, args.format)
        return 0

    # Show stats
    if args.stats:
        stats = get_rules_summary()
        print_stats(stats, args.format)
        return 0

    # Create alias
    if args.create_alias:
        alias, canonical = args.create_alias
        rule_id = create_alias_rule(alias, canonical)
        if rule_id:
            print(f"Created alias rule {rule_id}: {alias} → {canonical}")
        return 0

    # Create auto-tag
    if args.create_tag:
        pattern = args.create_tag[0]
        tags = args.create_tag[1:]
        if not tags:
            print("Error: Need at least one tag")
            return 1
        rule_id = create_auto_tag_rule(pattern, tags)
        if rule_id:
            print(f"Created auto-tag rule {rule_id}")
        return 0

    # Create extraction hint
    if args.create_hint:
        domain = args.create_hint[0]
        terms = args.create_hint[1:]
        if not terms:
            print("Error: Need at least one term")
            return 1
        rule_id = create_extraction_hint_rule(domain, terms)
        if rule_id:
            print(f"Created extraction hint rule {rule_id}")
        return 0

    # Disable/Enable rules
    if args.disable_rule:
        if disable_rule(args.disable_rule):
            print(f"Disabled rule {args.disable_rule}")
        return 0

    if args.enable_rule:
        if enable_rule(args.enable_rule):
            print(f"Enabled rule {args.enable_rule}")
        return 0

    # Import
    if args.import_file:
        imported, skipped = import_rules(Path(args.import_file), args.overwrite)
        print(f"Imported {imported} rules, skipped {skipped}")
        return 0

    # Export
    if args.export:
        count = export_rules(Path(args.export), [args.type] if args.type else None)
        print(f"Exported {count} rules to {args.export}")
        return 0

    # Apply to library
    if args.apply_to_library:
        apply_rules_to_library(args.dry_run)
        return 0

    # No action specified
    parser.print_help()
    return 0


if __name__ == '__main__':
    sys.exit(main())
