#!/usr/bin/env python3
"""
Gap Pinning System - Mark research gaps for prioritization.

Allows users to:
- Pin important gaps for targeted research
- Ignore irrelevant gaps
- Track gap status (pending, pinned, ignored, filled)
- Prioritize gaps by importance

Usage:
    python gap_pinning.py --list
    python gap_pinning.py --pin GAP_ID
    python gap_pinning.py --ignore GAP_ID
    python gap_pinning.py --fill GAP_ID
    python gap_pinning.py --add "Description" --query "suggested query"
"""

import argparse
import sys
import json
import logging
from pathlib import Path
from datetime import datetime
from typing import List, Dict, Any, Optional

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent))

from db_utils import execute_query

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


# =============================================================================
# CONFIGURATION
# =============================================================================

GAP_STATUS_OPTIONS = ['pending', 'pinned', 'ignored', 'filled']
MAX_PRIORITY = 10


# =============================================================================
# GAP MANAGEMENT FUNCTIONS
# =============================================================================

def add_gap(
    description: str,
    suggested_query: str = None,
    source_session_id: str = None,
    source_project_id: str = None,
    source_subject: str = None,
    priority: int = 0
) -> int:
    """
    Add a new research gap.

    Args:
        description: Description of the gap
        suggested_query: Suggested search query to fill the gap
        source_session_id: Research session that identified this gap
        source_project_id: Associated project ID
        source_subject: Subject area being researched
        priority: Initial priority (0-10)

    Returns:
        The new gap_id
    """
    result = execute_query(
        """
        INSERT INTO research_gaps (
            description, suggested_query, source_session_id,
            source_project_id, source_subject, priority, status
        )
        VALUES (%s, %s, %s, %s, %s, %s, 'pending')
        RETURNING gap_id
        """,
        (description, suggested_query, source_session_id,
         source_project_id, source_subject, min(priority, MAX_PRIORITY)),
        fetch='one'
    )

    gap_id = result['gap_id'] if result else None
    logger.info(f"Added gap {gap_id}: {description[:50]}...")

    return gap_id


def pin_gap(gap_id: int, priority: int = 5) -> bool:
    """
    Pin a gap for prioritization.

    Args:
        gap_id: Gap to pin
        priority: Priority level (1-10, higher = more important)

    Returns:
        True if successful
    """
    result = execute_query(
        """
        UPDATE research_gaps
        SET status = 'pinned',
            priority = %s,
            pinned_at = CURRENT_TIMESTAMP
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (min(priority, MAX_PRIORITY), gap_id),
        fetch='one'
    )

    if result:
        logger.info(f"Pinned gap {gap_id} with priority {priority}")
        return True

    logger.warning(f"Gap not found: {gap_id}")
    return False


def ignore_gap(gap_id: int) -> bool:
    """
    Mark a gap as ignored (not relevant).

    Args:
        gap_id: Gap to ignore

    Returns:
        True if successful
    """
    result = execute_query(
        """
        UPDATE research_gaps
        SET status = 'ignored',
            priority = 0
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (gap_id,),
        fetch='one'
    )

    if result:
        logger.info(f"Ignored gap {gap_id}")
        return True

    logger.warning(f"Gap not found: {gap_id}")
    return False


def fill_gap(gap_id: int, search_results_count: int = 0) -> bool:
    """
    Mark a gap as filled (research complete).

    Args:
        gap_id: Gap to mark as filled
        search_results_count: Number of results found

    Returns:
        True if successful
    """
    result = execute_query(
        """
        UPDATE research_gaps
        SET status = 'filled',
            filled_at = CURRENT_TIMESTAMP,
            search_results_count = %s
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (search_results_count, gap_id),
        fetch='one'
    )

    if result:
        logger.info(f"Filled gap {gap_id} with {search_results_count} results")
        return True

    logger.warning(f"Gap not found: {gap_id}")
    return False


def set_gap_priority(gap_id: int, priority: int) -> bool:
    """
    Set the priority level for a gap.

    Args:
        gap_id: Gap ID
        priority: New priority (0-10)

    Returns:
        True if successful
    """
    result = execute_query(
        """
        UPDATE research_gaps
        SET priority = %s
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (min(max(priority, 0), MAX_PRIORITY), gap_id),
        fetch='one'
    )

    if result:
        logger.info(f"Set priority {priority} for gap {gap_id}")
        return True

    return False


def reset_gap(gap_id: int) -> bool:
    """
    Reset a gap to pending status.

    Args:
        gap_id: Gap to reset

    Returns:
        True if successful
    """
    result = execute_query(
        """
        UPDATE research_gaps
        SET status = 'pending',
            priority = 0,
            pinned_at = NULL,
            filled_at = NULL
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (gap_id,),
        fetch='one'
    )

    return bool(result)


def delete_gap(gap_id: int) -> bool:
    """
    Delete a gap.

    Args:
        gap_id: Gap to delete

    Returns:
        True if successful
    """
    result = execute_query(
        """
        DELETE FROM research_gaps
        WHERE gap_id = %s
        RETURNING gap_id
        """,
        (gap_id,),
        fetch='one'
    )

    if result:
        logger.info(f"Deleted gap {gap_id}")
        return True

    return False


# =============================================================================
# QUERY FUNCTIONS
# =============================================================================

def get_gap(gap_id: int) -> Optional[Dict[str, Any]]:
    """Get a single gap by ID."""
    result = execute_query(
        "SELECT * FROM research_gaps WHERE gap_id = %s",
        (gap_id,),
        fetch='one'
    )

    return dict(result) if result else None


def list_gaps(
    status: str = None,
    project_id: str = None,
    subject: str = None,
    limit: int = 100
) -> List[Dict[str, Any]]:
    """
    List research gaps with optional filters.

    Args:
        status: Filter by status (pending, pinned, ignored, filled)
        project_id: Filter by project
        subject: Filter by subject
        limit: Maximum results

    Returns:
        List of gap dictionaries
    """
    conditions = []
    params = []

    if status:
        conditions.append("status = %s")
        params.append(status)

    if project_id:
        conditions.append("source_project_id = %s")
        params.append(project_id)

    if subject:
        conditions.append("source_subject ILIKE %s")
        params.append(f"%{subject}%")

    where_clause = f"WHERE {' AND '.join(conditions)}" if conditions else ""

    params.append(limit)

    results = execute_query(
        f"""
        SELECT * FROM research_gaps
        {where_clause}
        ORDER BY
            CASE status
                WHEN 'pinned' THEN 1
                WHEN 'pending' THEN 2
                WHEN 'filled' THEN 3
                WHEN 'ignored' THEN 4
            END,
            priority DESC,
            created_at DESC
        LIMIT %s
        """,
        tuple(params),
        fetch='all'
    )

    return [dict(r) for r in results] if results else []


def get_pinned_gaps(limit: int = 50) -> List[Dict[str, Any]]:
    """Get all pinned gaps ordered by priority."""
    return list_gaps(status='pinned', limit=limit)


def get_pending_gaps(limit: int = 100) -> List[Dict[str, Any]]:
    """Get all pending gaps."""
    return list_gaps(status='pending', limit=limit)


def get_gaps_for_project(project_id: str) -> List[Dict[str, Any]]:
    """Get all gaps for a specific project."""
    return list_gaps(project_id=project_id)


def get_gap_stats() -> Dict[str, Any]:
    """
    Get gap statistics.

    Returns:
        Dict with status counts and priority distribution
    """
    status_counts = execute_query(
        """
        SELECT status, COUNT(*) as count
        FROM research_gaps
        GROUP BY status
        """,
        fetch='all'
    )

    priority_dist = execute_query(
        """
        SELECT priority, COUNT(*) as count
        FROM research_gaps
        WHERE status = 'pinned'
        GROUP BY priority
        ORDER BY priority DESC
        """,
        fetch='all'
    )

    total = execute_query(
        "SELECT COUNT(*) as count FROM research_gaps",
        fetch='one'
    )

    recent = execute_query(
        """
        SELECT COUNT(*) as count
        FROM research_gaps
        WHERE created_at > CURRENT_TIMESTAMP - INTERVAL '7 days'
        """,
        fetch='one'
    )

    return {
        'total': total['count'] if total else 0,
        'recent_7_days': recent['count'] if recent else 0,
        'by_status': {r['status']: r['count'] for r in (status_counts or [])},
        'pinned_by_priority': {r['priority']: r['count'] for r in (priority_dist or [])}
    }


# =============================================================================
# BULK OPERATIONS
# =============================================================================

def add_gaps_batch(gaps: List[Dict[str, Any]], source_session_id: str = None) -> List[int]:
    """
    Add multiple gaps at once.

    Args:
        gaps: List of dicts with 'description' and optionally 'suggested_query', 'priority'
        source_session_id: Session that identified these gaps

    Returns:
        List of new gap_ids
    """
    gap_ids = []

    for gap in gaps:
        gap_id = add_gap(
            description=gap.get('description', ''),
            suggested_query=gap.get('suggested_query'),
            source_session_id=source_session_id,
            source_project_id=gap.get('source_project_id'),
            source_subject=gap.get('source_subject'),
            priority=gap.get('priority', 0)
        )
        if gap_id:
            gap_ids.append(gap_id)

    return gap_ids


def pin_gaps_batch(gap_ids: List[int], priority: int = 5) -> Dict[int, bool]:
    """
    Pin multiple gaps at once.

    Args:
        gap_ids: List of gap IDs to pin
        priority: Priority level for all

    Returns:
        Dict mapping gap_id to success status
    """
    results = {}
    for gap_id in gap_ids:
        results[gap_id] = pin_gap(gap_id, priority)
    return results


def ignore_gaps_batch(gap_ids: List[int]) -> Dict[int, bool]:
    """Ignore multiple gaps at once."""
    results = {}
    for gap_id in gap_ids:
        results[gap_id] = ignore_gap(gap_id)
    return results


def clear_filled_gaps(older_than_days: int = 30) -> int:
    """
    Delete filled gaps older than specified days.

    Args:
        older_than_days: Delete gaps filled more than this many days ago

    Returns:
        Number of gaps deleted
    """
    result = execute_query(
        """
        DELETE FROM research_gaps
        WHERE status = 'filled'
          AND filled_at < CURRENT_TIMESTAMP - INTERVAL '%s days'
        """,
        (older_than_days,),
        fetch='none'
    )

    # Can't easily get affected count, return 0
    logger.info(f"Cleared filled gaps older than {older_than_days} days")
    return 0


# =============================================================================
# INTEGRATION HELPERS
# =============================================================================

def get_gaps_for_research(
    project_id: str = None,
    include_pinned: bool = True,
    include_pending: bool = True,
    limit: int = 20
) -> List[Dict[str, Any]]:
    """
    Get gaps suitable for research, ordered by priority.

    Args:
        project_id: Filter by project
        include_pinned: Include pinned gaps
        include_pending: Include pending gaps
        limit: Maximum results

    Returns:
        List of gap dicts with suggested queries
    """
    statuses = []
    if include_pinned:
        statuses.append("'pinned'")
    if include_pending:
        statuses.append("'pending'")

    if not statuses:
        return []

    conditions = [f"status IN ({', '.join(statuses)})"]
    params = []

    if project_id:
        conditions.append("source_project_id = %s")
        params.append(project_id)

    where_clause = f"WHERE {' AND '.join(conditions)}"
    params.append(limit)

    results = execute_query(
        f"""
        SELECT gap_id, description, suggested_query, priority, status, source_subject
        FROM research_gaps
        {where_clause}
        ORDER BY
            CASE status WHEN 'pinned' THEN 0 ELSE 1 END,
            priority DESC,
            created_at ASC
        LIMIT %s
        """,
        tuple(params),
        fetch='all'
    )

    return [dict(r) for r in results] if results else []


def format_gaps_for_display(gaps: List[Dict[str, Any]]) -> str:
    """
    Format gaps for text display.

    Args:
        gaps: List of gap dicts

    Returns:
        Formatted string for display
    """
    if not gaps:
        return "No gaps found."

    lines = []

    for gap in gaps:
        status_icon = {
            'pinned': '📌',
            'pending': '⏳',
            'ignored': '🚫',
            'filled': '✅'
        }.get(gap.get('status'), '?')

        priority = gap.get('priority', 0)
        priority_bar = '█' * priority + '░' * (MAX_PRIORITY - priority)

        lines.append(f"\n{status_icon} Gap #{gap['gap_id']} [{priority_bar}]")
        lines.append(f"   {gap.get('description', 'No description')[:80]}")

        if gap.get('suggested_query'):
            lines.append(f"   Query: {gap['suggested_query']}")

        if gap.get('source_subject'):
            lines.append(f"   Subject: {gap['source_subject']}")

    return '\n'.join(lines)


# =============================================================================
# CLI
# =============================================================================

def main():
    parser = argparse.ArgumentParser(
        description='Gap Pinning System',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # List all gaps
  %(prog)s --list

  # List only pinned gaps
  %(prog)s --list --status pinned

  # Add a new gap
  %(prog)s --add "Missing information about X" --query "X historical context"

  # Pin a gap with priority
  %(prog)s --pin 123 --priority 8

  # Ignore a gap
  %(prog)s --ignore 123

  # Mark gap as filled
  %(prog)s --fill 123 --results 5

  # Show statistics
  %(prog)s --stats

  # Delete a gap
  %(prog)s --delete 123
        """
    )

    # Actions
    parser.add_argument('--list', action='store_true', help='List gaps')
    parser.add_argument('--add', metavar='DESC', help='Add a new gap with description')
    parser.add_argument('--pin', type=int, metavar='GAP_ID', help='Pin a gap')
    parser.add_argument('--ignore', type=int, metavar='GAP_ID', help='Ignore a gap')
    parser.add_argument('--fill', type=int, metavar='GAP_ID', help='Mark gap as filled')
    parser.add_argument('--reset', type=int, metavar='GAP_ID', help='Reset gap to pending')
    parser.add_argument('--delete', type=int, metavar='GAP_ID', help='Delete a gap')
    parser.add_argument('--stats', action='store_true', help='Show statistics')

    # Options
    parser.add_argument('--query', '-q', help='Suggested query (use with --add)')
    parser.add_argument('--priority', '-p', type=int, default=5,
                        help='Priority level 0-10 (default: 5)')
    parser.add_argument('--status', '-s', choices=GAP_STATUS_OPTIONS,
                        help='Filter by status (use with --list)')
    parser.add_argument('--project', help='Filter by project ID')
    parser.add_argument('--subject', help='Filter by subject')
    parser.add_argument('--results', type=int, default=0,
                        help='Results count (use with --fill)')
    parser.add_argument('--limit', type=int, default=50, help='Max results to display')
    parser.add_argument('--format', choices=['text', 'json'], default='text',
                        help='Output format')

    args = parser.parse_args()

    # Handle actions
    if args.stats:
        stats = get_gap_stats()
        if args.format == 'json':
            print(json.dumps(stats, indent=2))
        else:
            print("\nGap Statistics")
            print("=" * 40)
            print(f"Total gaps: {stats['total']}")
            print(f"Added in last 7 days: {stats['recent_7_days']}")
            print("\nBy Status:")
            for status, count in stats.get('by_status', {}).items():
                print(f"  {status}: {count}")
            if stats.get('pinned_by_priority'):
                print("\nPinned by Priority:")
                for priority, count in sorted(stats['pinned_by_priority'].items(), reverse=True):
                    print(f"  Priority {priority}: {count}")

    elif args.add:
        gap_id = add_gap(
            description=args.add,
            suggested_query=args.query,
            source_project_id=args.project,
            priority=args.priority if args.status == 'pinned' else 0
        )
        print(f"Added gap #{gap_id}")

    elif args.pin is not None:
        success = pin_gap(args.pin, args.priority)
        if success:
            print(f"Pinned gap #{args.pin} with priority {args.priority}")
        else:
            print(f"Failed to pin gap #{args.pin}")

    elif args.ignore is not None:
        success = ignore_gap(args.ignore)
        if success:
            print(f"Ignored gap #{args.ignore}")
        else:
            print(f"Failed to ignore gap #{args.ignore}")

    elif args.fill is not None:
        success = fill_gap(args.fill, args.results)
        if success:
            print(f"Marked gap #{args.fill} as filled")
        else:
            print(f"Failed to fill gap #{args.fill}")

    elif args.reset is not None:
        success = reset_gap(args.reset)
        if success:
            print(f"Reset gap #{args.reset} to pending")
        else:
            print(f"Failed to reset gap #{args.reset}")

    elif args.delete is not None:
        success = delete_gap(args.delete)
        if success:
            print(f"Deleted gap #{args.delete}")
        else:
            print(f"Failed to delete gap #{args.delete}")

    elif args.list:
        gaps = list_gaps(
            status=args.status,
            project_id=args.project,
            subject=args.subject,
            limit=args.limit
        )

        if args.format == 'json':
            # Convert datetime objects to strings for JSON
            for gap in gaps:
                for key in ['created_at', 'updated_at', 'pinned_at', 'filled_at']:
                    if gap.get(key):
                        gap[key] = gap[key].isoformat()
            print(json.dumps(gaps, indent=2))
        else:
            print(f"\nResearch Gaps ({len(gaps)} found)")
            print("=" * 60)
            print(format_gaps_for_display(gaps))

    else:
        parser.print_help()


if __name__ == '__main__':
    main()
